package com.elitesland.fin.application.service.workflow.adjust;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.log.model.bo.OperationLogDTO;
import com.elitescloud.boot.log.service.OperationLogMqMessageService;
import com.elitesland.fin.common.FinBusinessObjectEnum;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.entity.adjustorder.AdjustOrderDO;
import com.elitesland.fin.entity.adjustorder.QAdjustOrderDO;
import com.elitesland.fin.param.flow.AccountFlowRpcParam;
import com.elitesland.fin.repo.adjustorder.AdjustOrderRepo;
import com.elitesland.fin.service.flow.AccountFlowRpcService;
import com.elitesland.workflow.CommentInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;

/**
 * @author Jason.zhao
 * @date 2022/5/7 10:23
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class FinAdjustOrderProcessServiceImpl implements FinAdjustOrderProcessService {

    private final AdjustOrderRepo adjustOrderRepo;

    private final JPAQueryFactory jpaQueryFactory;

    private final AccountFlowRpcService accountFlowRpcService;
    private final OperationLogMqMessageService operationLogMqMessageService;
    /**
     * 第一个用户任务节点的KEY
     */
    public static String FIRST_TASK_DEF_KEY = "createrSubmit";

    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void processStatusChange(String businessKey, ProcInstStatus procInstStatus, CommentInfo commentInfo) {
        AdjustOrderDO adjustOrderDO = getAdjustOrderDO(businessKey);
        log.info("开始财务调整单工作流回调，流程状态为:" + procInstStatus.getDesc());
        QAdjustOrderDO qAdjustOrderDo = QAdjustOrderDO.adjustOrderDO;
        // 设置工作流状态
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qAdjustOrderDo)
                .set(qAdjustOrderDo.workflowProcInstStatus, procInstStatus)
                .set(qAdjustOrderDo.auditUser, commentInfo.getUserName())
                .set(qAdjustOrderDo.auditTime, commentInfo.getTime())
                .where(qAdjustOrderDo.id.eq(adjustOrderDO.getId()));
        /***人工记录业务操作日志****/
        OperationLogDTO dto ;
        // 单据状态
        switch (procInstStatus) {
            case NOTSUBMIT:
                // 未提交 将单据状态变成"草稿"
                jpaUpdateClause.set(qAdjustOrderDo.state, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                dto = operationLogMqMessageService.quickNewOperationLogDTO(
                    FinBusinessObjectEnum.FIN_ADJUST,
                    adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_WITHDRAWAL,
                        OperationTypeEnum.APPROVE_WITHDRAWAL.getDescription());
                break;
            case INTERRUPT:
                // 中断执行 将单据状态变成"草稿",并且将单据上的"流程实例状态"，"流程实例ID"清成null(不是空字符串)
                jpaUpdateClause.set(qAdjustOrderDo.state, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                jpaUpdateClause.setNull(qAdjustOrderDo.workflowProcInstId);
                // 未提交 将单据状态变成"草稿"
                jpaUpdateClause.set(qAdjustOrderDo.state, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                dto = operationLogMqMessageService.quickNewOperationLogDTO(
                        FinBusinessObjectEnum.FIN_ADJUST,
                        adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_WITHDRAWAL,
                        OperationTypeEnum.APPROVE_WITHDRAWAL.getDescription());
                break;
            case REJECTED:
                // 审批拒绝 一般情况将单据状态变成草稿
                // 工作流驳回到创建人节点/工作流审核拒绝 将单据状态设置为审批拒绝
                jpaUpdateClause.set(qAdjustOrderDo.state, UdcEnum.APPLY_STATUS_DRAFT.getValueCode());
                jpaUpdateClause.set(qAdjustOrderDo.auditRejectReason, commentInfo.getComment());
                dto = operationLogMqMessageService.quickNewOperationLogDTO(
                        FinBusinessObjectEnum.FIN_ADJUST,
                        adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_REJECT,
                        OperationTypeEnum.APPROVE_REJECT.getDescription());
                break;
            case INVALID:
                // 作废 一般情况将单据状态变成"作废" ，直接删除单据
                jpaUpdateClause.set(qAdjustOrderDo.state, UdcEnum.APPLY_STATUS_VOID.getValueCode())
                        ;
                // 审批拒绝 一般情况将单据状态变成草稿
                jpaUpdateClause.set(qAdjustOrderDo.auditRejectReason, commentInfo.getComment());
                dto = operationLogMqMessageService.quickNewOperationLogDTO(
                        FinBusinessObjectEnum.FIN_ADJUST,
                        adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_CANCELLATION,
                        OperationTypeEnum.APPROVE_CANCELLATION.getDescription());
                break;
            case APPROVING:
                // 审批中 一般情况将单据状态变成"审批 中
                jpaUpdateClause.set(qAdjustOrderDo.state, UdcEnum.APPLY_STATUS_DOING.getValueCode())
                        .set(qAdjustOrderDo.workflowSubmitTime, LocalDateTime.now());
                dto = operationLogMqMessageService.quickNewOperationLogDTO(
                        FinBusinessObjectEnum.FIN_ADJUST,
                        adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_SUBMIT,
                        OperationTypeEnum.APPROVE_SUBMIT.getDescription());
                break;
            case APPROVED:
                // 最终节点审批通过 一般情况将单据状态变成"审批通过"
                jpaUpdateClause.set(qAdjustOrderDo.state, UdcEnum.APPLY_STATUS_COMPLETE.getValueCode())
                        .set(qAdjustOrderDo.workflowEndTime, commentInfo.getTime());
                dto = operationLogMqMessageService.quickNewOperationLogDTO(
                        FinBusinessObjectEnum.FIN_ADJUST,
                        adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_OK,
                        OperationTypeEnum.APPROVE_OK.getDescription());

                break;
            default:
                dto = operationLogMqMessageService.quickNewOperationLogDTO(
                        FinBusinessObjectEnum.FIN_ADJUST,
                        adjustOrderDO.getDocNo(), OperationTypeEnum.APPROVE_OK,
                        "流程状态:"+procInstStatus.getDesc());
                break;
        }

        log.info("开始处理工作流回调业务--------");
        log.info(" 流程实例ID:{} , \n审批状态:{}, \n审批信息 {}", adjustOrderDO.getWorkflowProcInstId(), procInstStatus, commentInfo);

        AccountFlowRpcParam flowRpcParam = new AccountFlowRpcParam();
        // 调入单
        if (StrUtil.equals(adjustOrderDO.getAdjType(), UdcEnum.ADJUST_TYPE_1.getValueCode())) {
            if (procInstStatus.equals(procInstStatus.APPROVED)) {
                // 1.创建增加的流水
                flowRpcParam = adjustDoToFlowParam(adjustOrderDO,
                        UdcEnum.ADJUST_TYPE_1.getValueCode(),
                        UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
            }
        }
        // 调出单
        if (StrUtil.equals(adjustOrderDO.getAdjType(), UdcEnum.ADJUST_TYPE_2.getValueCode())) {
            if (procInstStatus.equals(ProcInstStatus.APPROVED)) {
                // 1.释放占用的流水
                // 2.创建扣除的流水 账户规则配置状态为已完成就能够创建两条
                flowRpcParam = adjustDoToFlowParam(adjustOrderDO,
                        UdcEnum.ADJUST_TYPE_2.getValueCode(),
                        UdcEnum.APPLY_STATUS_COMPLETE.getValueCode());
            }
            if (procInstStatus.equals(ProcInstStatus.REJECTED)) {
                // 拒绝不是第一个节点
                // 释放占用的流水
                flowRpcParam = adjustDoToFlowParam(adjustOrderDO,
                        UdcEnum.ADJUST_TYPE_2.getValueCode(),
                        UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
            }
			// 撤回到提交人
			if (procInstStatus.equals(ProcInstStatus.NOTSUBMIT) || procInstStatus.equals(ProcInstStatus.INTERRUPT)) {
				// 释放占用的流水
				flowRpcParam = adjustDoToFlowParam(adjustOrderDO,
						UdcEnum.ADJUST_TYPE_2.getValueCode(),
						UdcEnum.APPLY_STATUS_REJECTED.getValueCode());
			}
        }
        // 执行
        jpaUpdateClause.execute();
        if (ObjectUtil.isNotNull(flowRpcParam.getSourceId())) {
             accountFlowRpcService.generateAccountFlow(flowRpcParam);
        }

        //发送埋点日志
        operationLogMqMessageService.sendAsyncOperationLogMqMessage(dto);
    }

    @Override
    public ArrayList<String> taskAssignee(String businessKey, String customParams) {
        return null;
    }

    private AccountFlowRpcParam adjustDoToFlowParam(AdjustOrderDO adjustOrderDO, String sourceDcoType, String sourceDocStatus) {
        AccountFlowRpcParam flowRpcParam = new AccountFlowRpcParam();
        flowRpcParam.setSourceNo(adjustOrderDO.getDocNo());
        flowRpcParam.setSourceId(adjustOrderDO.getId());
        flowRpcParam.setSourceDoc(UdcEnum.DOC_CLS_AO.getValueCode());
        flowRpcParam.setSourceDocType(sourceDcoType);
        flowRpcParam.setSourceDocStatus(sourceDocStatus);
        flowRpcParam.setSourceDocAmount(adjustOrderDO.getTotalAmt());
        flowRpcParam.setAccountCode(adjustOrderDO.getAccCode());
        // 用于判断是否启用工作流
        flowRpcParam.setWorkflowProcInstId(adjustOrderDO.getWorkflowProcInstId());
        return flowRpcParam;
    }

    /**
     * 查询单据
     */
    private AdjustOrderDO getAdjustOrderDO(String businessKey) {
        String docNo = businessKey.split("#")[0].trim();
        AdjustOrderDO adjustOrderDO = jpaQueryFactory.selectFrom(QAdjustOrderDO.adjustOrderDO)
                .where(QAdjustOrderDO.adjustOrderDO.docNo.eq(docNo))
                .fetchOne();
        if (adjustOrderDO == null) {
            throw new RuntimeException("调整单(编码:" + docNo + ")不存在，无法审批");
        }

        return adjustOrderDO;
    }
}
