package com.elitesland.fin.infr.repo.payorder;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.core.support.customfield.service.impl.CustomFieldJpaServiceUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.fin.application.service.excel.entity.PayOrderExportEntity;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.payorder.PayOrderDO;
import com.elitesland.fin.domain.entity.payorder.QPayOrderDO;
import com.elitesland.fin.domain.entity.payorder.QPayOrderDtlDO;
import com.elitesland.fin.domain.param.payorder.PayOrderPageParam;
import com.elitesland.fin.domain.param.payorder.PayOrderParam;
import com.elitesland.fin.infr.dto.common.ApVerDTO;
import com.elitesland.fin.infr.dto.payorder.PayOrderDTO;
import com.elitesland.fin.param.payorder.OrderAmtRpcParam;
import com.elitesland.fin.utils.BusinessSecurityUtil;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * @author zhiyu.he
 * @date 2022/3/16 9:40
 */
@Component
@RequiredArgsConstructor
public class PayOrderRepoProc {
    private final JPAQueryFactory jpaQueryFactory;
    private final QPayOrderDO qPayOrderDO = QPayOrderDO.payOrderDO;
    private final QPayOrderDtlDO qPayOrderDtlDO = QPayOrderDtlDO.payOrderDtlDO;

    public List<PayOrderDO> queryBySourceNo(String sourceNo) {
        return jpaQueryFactory.selectFrom(qPayOrderDO).where(qPayOrderDO.sourceNo.eq(sourceNo)).fetch();
    }

    public List<PayOrderDO> queryBySourceNoList(List<String> sourceNoList) {
        return jpaQueryFactory.selectFrom(qPayOrderDO).where(qPayOrderDO.sourceNo.in(sourceNoList)).fetch();
    }

    public PayOrderDTO queryById(Long id) {
        return select(PayOrderDTO.class)
                .where(qPayOrderDO.id.eq(id))
                .where(qPayOrderDO.deleteFlag.eq(0))
                .fetchOne();
    }

    public void deleteByIds(List<Long> ids) {
        jpaQueryFactory.delete(qPayOrderDO)
                .where(qPayOrderDO.id.in(ids))
                .execute();
    }

    public List<PayOrderDTO> queryByPayTypeId(List<Long> ids) {
        return select(PayOrderDTO.class)
                .where(qPayOrderDO.payTypeId.in(ids))
                .fetch();
    }


    public Long audit(List<Long> ids, String auditRejection, SysUserDTO user) {
        return jpaQueryFactory.update(qPayOrderDO)
                .set(qPayOrderDO.auditDate, LocalDateTime.now())
                .set(qPayOrderDO.auditRejection, auditRejection)
                .set(qPayOrderDO.auditUser, user.getUsername())
                .set(qPayOrderDO.auditUserId, user.getId())
                .where(qPayOrderDO.id.in(ids))
                .execute();
    }

    public List<PayOrderDTO> queryByParam(PayOrderParam payOrderParam) {
        // 条件拼接
        List<Predicate> predicates = new ArrayList<>();
        if (!StringUtils.isEmpty(payOrderParam.getCurrCode())) {
            predicates.add(qPayOrderDO.currCode.eq(payOrderParam.getCurrCode()));
        }
        if (payOrderParam.getSuppId() != null) {
            predicates.add(qPayOrderDO.suppId.eq(payOrderParam.getSuppId()));
        }
        if (payOrderParam.getOuId() != null) {
            predicates.add(qPayOrderDO.ouId.eq(payOrderParam.getOuId()));
        }
        if (payOrderParam.getBuDateStart() != null && payOrderParam.getBuDateEnd() != null) {
            predicates.add(qPayOrderDO.buDate.between(payOrderParam.getBuDateStart(), payOrderParam.getBuDateEnd()));
        }
        predicates.add(qPayOrderDO.deleteFlag.eq(0));
        predicates.add(qPayOrderDO.orderState.eq(UdcEnum.APPLY_STATUS_COMPLETE.getValueCode()));


        return select(PayOrderDTO.class).
                where(ExpressionUtils.allOf(predicates)).fetch();
    }

    public List<PayOrderDTO> selectPayOrderByParam(PayOrderParam payOrderParam) {

        List<Predicate> predicates = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(payOrderParam.getPayOrderNoList())) {
            predicates.add(qPayOrderDO.payOrderNo.in(payOrderParam.getPayOrderNoList()));
        }

        return select(PayOrderDTO.class).where(ExpressionUtils.allOf(predicates)).fetch();
    }

    public List<PayOrderDTO> queryByIds(List<Long> ids) {
        return select(PayOrderDTO.class)
                .where(qPayOrderDO.id.in(ids))
                .where(qPayOrderDO.deleteFlag.eq(0))
                .fetch();
    }

    public PagingVO<PayOrderDTO> page(PayOrderPageParam param, List<Long> ids) {
        var predicates = pageWhere(param, ids);
        JPAQuery<PayOrderDTO> query = select(PayOrderDTO.class).where(ExpressionUtils.allOf(predicates));
        //数据权限
        BusinessSecurityUtil.where(query, PayOrderDO.class);
        param.setPaging(query);
        param.fillOrders(query, qPayOrderDO);
        return PagingVO.<PayOrderDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    private List<Predicate> pageWhere(PayOrderPageParam param, List<Long> ids) {
        // 条件拼接
        List<Predicate> predicates = new ArrayList<>();
        if (CollUtil.isNotEmpty(param.getInOutCustList())) {
            predicates.add(qPayOrderDO.inOutCust.in(param.getInOutCustList()));
        }
        if (CollUtil.isNotEmpty(param.getRelevanceOuCodeList())) {
            predicates.add(qPayOrderDO.relevanceOuCode.in(param.getRelevanceOuCodeList()));
        }
        if (param.getApFlag() != null) {
            predicates.add(qPayOrderDO.apFlag.eq(param.getApFlag()));
        }
        if (!StringUtils.isEmpty(param.getCreateMode())) {
            predicates.add(qPayOrderDO.createMode.eq(param.getCreateMode()));
        }
        if (param.getInitFlag() != null) {
            predicates.add(qPayOrderDO.initFlag.eq(param.getInitFlag()));
        }
        if (param.getPayTypeId() != null) {
            predicates.add(qPayOrderDO.payTypeId.eq(param.getPayTypeId()));
        }
        if (!StringUtils.isEmpty(param.getVerState())) {
            predicates.add(qPayOrderDO.verState.eq(param.getVerState()));
        }
        // 核销状态
        if (CollectionUtil.isNotEmpty(param.getVerStateList())) {
            predicates.add(qPayOrderDO.verState.in(param.getVerStateList()));
        }
        if (Objects.nonNull(param.getRedState())) {
            predicates.add(qPayOrderDO.redState.eq(param.getRedState()));
        }
        if (Objects.nonNull(param.getProposedStatus())) {
            predicates.add(qPayOrderDO.proposedStatus.eq(param.getProposedStatus()));
        }

        if (!StringUtils.isEmpty(param.getRedSourceNo())) {
            predicates.add(qPayOrderDO.redSourceNo.like("%" + param.getRedSourceNo() + "%"));
        }
        if (!StringUtils.isEmpty(param.getOrderState())) {
            predicates.add(qPayOrderDO.orderState.eq(param.getOrderState()));
        }
        if (!StringUtils.isEmpty(param.getCurrCode())) {
            predicates.add(qPayOrderDO.currCode.eq(param.getCurrCode()));
        }
        if (!StringUtils.isEmpty(param.getSourceNo())) {
            predicates.add(qPayOrderDO.sourceNo.like("%" + param.getSourceNo() + "%"));
        }
        if (!StringUtils.isEmpty(param.getPayOrderNo())) {
            predicates.add(qPayOrderDO.payOrderNo.like("%" + param.getPayOrderNo() + "%"));
        }
        if (param.getSuppId() != null) {
            predicates.add(qPayOrderDO.suppId.eq(param.getSuppId()));
        }
        if (!StringUtils.isEmpty(param.getOperator())) {
            predicates.add(qPayOrderDO.operator.like("%" + param.getOperator() + "%"));
        }
        if (param.getOuId() != null) {
            predicates.add(qPayOrderDO.ouId.eq(param.getOuId()));
        }
        if (StringUtils.isNotBlank(param.getOuCode())) {
            predicates.add(qPayOrderDO.ouCode.eq(param.getOuCode()));
        }
        if (null != param.getBuDateStart() && null != param.getBuDateEnd()) {
            predicates.add(qPayOrderDO.buDate.between(param.getBuDateStart(), param.getBuDateEnd()));
        }
        if (!CollUtil.isEmpty(ids)) {
            predicates.add(qPayOrderDO.id.in(ids));
        }
        if (!CollUtil.isEmpty(param.getIdList())) {
            predicates.add(qPayOrderDO.id.in(param.getIdList()));
        }
        if (param.getTotalAmtFrom() != null) {
            predicates.add(qPayOrderDO.totalAmt.goe(param.getTotalAmtFrom()));
        }
        if (param.getTotalAmtTo() != null) {
            predicates.add(qPayOrderDO.totalAmt.loe(param.getTotalAmtTo()));
        }
        if (param.getOperUserId() != null) {
            predicates.add(qPayOrderDO.operUserId.eq(param.getOperUserId()));
        }
        if (null != param.getApDateStart() && null != param.getApDateEnd()) {
            predicates.add(qPayOrderDO.apDate.between(param.getApDateStart(), param.getApDateEnd()));
        }
        if (StringUtils.isNotBlank(param.getTransferStatus())) {
            predicates.add(qPayOrderDO.transferStatus.eq(param.getTransferStatus()));
        }
        if (param.getPayOuId() != null) {
            predicates.add(qPayOrderDO.payOuId.eq(param.getPayOuId()));
        }
        if (StringUtils.isNotBlank(param.getPayOuCode())) {
            predicates.add(qPayOrderDO.payOuCode.eq(param.getPayOuCode()));
        }
        if (StringUtils.isNotBlank(param.getSyncStatus())) {
            predicates.add(qPayOrderDO.syncStatus.eq(param.getSyncStatus()));
        }
        if (CollectionUtils.isNotEmpty(param.getSyncStatusList())) {
            predicates.add(qPayOrderDO.syncStatus.in(param.getSyncStatusList()));
        }
        if (StringUtils.isNotBlank(param.getPayStatus())) {
            predicates.add(qPayOrderDO.payStatus.eq(param.getPayStatus()));
        }
        if (CollectionUtils.isNotEmpty(param.getPayStatusList())) {
            predicates.add(qPayOrderDO.payStatus.in(param.getPayStatusList()));
        }
        if (StringUtils.isNotBlank(param.getOutSysKey())) {
            predicates.add(qPayOrderDO.outSysKey.eq(param.getOutSysKey()));
        }
        if (StringUtils.isNotBlank(param.getTransNo())) {
            predicates.add(qPayOrderDO.transNo.eq(param.getTransNo()));
        }
        if (StringUtils.isNotBlank(param.getRcvSeqNo())) {
            predicates.add(qPayOrderDO.rcvSeqNo.eq(param.getRcvSeqNo()));
        }
        predicates.add(qPayOrderDO.deleteFlag.eq(0));
        //增加扩展字段查询条件begin
        Predicate customFieldPredicate = CustomFieldJpaServiceUtil.getPredicate(param.getConditions(), PayOrderDO.class);
        if (customFieldPredicate != null) {
            predicates.add(customFieldPredicate);
            //predicate= ExpressionUtils.and(predicate,customFieldPredicate);
        }
        //增加扩展字段查询条件end
        return predicates;
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                        qPayOrderDO.id,
                        qPayOrderDO.sourceNo,
                        qPayOrderDO.suppId,
                        qPayOrderDO.suppCode,
                        qPayOrderDO.suppName,
                        qPayOrderDO.ouCode,
                        qPayOrderDO.ouId,
                        qPayOrderDO.ouName,
                        qPayOrderDO.apTypeId,
                        qPayOrderDO.apTypeCode,
                        qPayOrderDO.apTypeName,
                        qPayOrderDO.payTypeId,
                        qPayOrderDO.payTypeCode,
                        qPayOrderDO.payTypeName,
                        qPayOrderDO.payOrderNo,
                        qPayOrderDO.currCode,
                        qPayOrderDO.currName,
                        qPayOrderDO.operator,
                        qPayOrderDO.operUserId,
                        qPayOrderDO.buType,
                        qPayOrderDO.buDate,
                        qPayOrderDO.totalAmt,
                        qPayOrderDO.totalCurAmt,
                        qPayOrderDO.auditUserId,
                        qPayOrderDO.auditDate,
                        qPayOrderDO.auditUser,
                        qPayOrderDO.orderState,
                        qPayOrderDO.exchangeRate,
                        qPayOrderDO.initFlag,
                        qPayOrderDO.realPayAmt,
                        qPayOrderDO.realPayCurAmt,
                        qPayOrderDO.apDate,
                        qPayOrderDO.apFlag,
                        qPayOrderDO.createTime,
                        qPayOrderDO.creator,
                        qPayOrderDO.modifyTime,
                        qPayOrderDO.updater,
                        qPayOrderDO.remark,
                        qPayOrderDO.auditRejection,
                        qPayOrderDO.createMode,
                        qPayOrderDO.localCurrCode,
                        qPayOrderDO.localCurrName,
                        qPayOrderDO.verState,
                        qPayOrderDO.verAmt,
                        qPayOrderDO.auditDataVersion,
                        qPayOrderDO.procInstId,
                        qPayOrderDO.procInstStatus,
                        qPayOrderDO.approvedTime,
                        qPayOrderDO.submitTime,
                        qPayOrderDO.addrNo,
                        qPayOrderDO.suppAddrNo,
                        qPayOrderDO.redState,
                        qPayOrderDO.redSourceId,
                        qPayOrderDO.redSourceNo,
                        qPayOrderDO.proposedStatus,
                        qPayOrderDO.inOutCust,
                        qPayOrderDO.relevanceOuCode,
                        qPayOrderDO.extensionInfo,
                        qPayOrderDO.transferStatus,
                        qPayOrderDO.payOuId,
                        qPayOrderDO.payOuCode,
                        qPayOrderDO.payOuName,
                        qPayOrderDO.payOuSameFlag,
                        qPayOrderDO.syncStatus,
                        qPayOrderDO.payStatus,
                        qPayOrderDO.syncError,
                        qPayOrderDO.outSysKey,
                        qPayOrderDO.transNo,
                        qPayOrderDO.rcvSeqNo,
                        qPayOrderDO.sysAcntgSeqNo
                )
        ).from(qPayOrderDO);
    }

    public ApVerDTO queryVerAmtById(Long id) {
        return jpaQueryFactory.select(Projections.bean(ApVerDTO.class,
                qPayOrderDO.id,
                qPayOrderDO.totalAmt,
                qPayOrderDO.verAmt,
                qPayOrderDO.auditDataVersion)).from(qPayOrderDO).where(qPayOrderDO.id.eq(id)).fetchOne();
    }

    public void updateVerApplyFlag(Long id, String val) {
        jpaQueryFactory.update(qPayOrderDO).set(qPayOrderDO.verApplyFlag, val).where(qPayOrderDO.id.eq(id)).execute();
    }

    public long updateVerAmt(ApVerDTO apVerDTO) {
        return jpaQueryFactory.update(qPayOrderDO).set(qPayOrderDO.verState, apVerDTO.getVerState())
                .set(qPayOrderDO.verAmt, apVerDTO.getVerAmt())
                .set(qPayOrderDO.auditDataVersion, apVerDTO.getAuditDataVersion() + 1)
                .where(qPayOrderDO.id.eq(apVerDTO.getId())
                        .and(qPayOrderDO.totalAmt.goe(apVerDTO.getVerAmt()))
                        .and(qPayOrderDO.auditDataVersion.eq(apVerDTO.getAuditDataVersion())))
                .execute();
    }

    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qPayOrderDO)
                .set(qPayOrderDO.procInstStatus, ProcInstStatus.APPROVING)
                .set(qPayOrderDO.procInstId, processInfo.getProcInstId())
                .set(qPayOrderDO.submitTime, LocalDateTime.now())
                .where(qPayOrderDO.id.eq(id));
        if (!Objects.equals(processInfo.getProcInstStatus(), ProcInstStatus.APPROVED)) {
            jpaUpdateClause.set(qPayOrderDO.procInstStatus, ProcInstStatus.APPROVING);
        }
        jpaUpdateClause.execute();
    }

    public void setNullRedFlag(Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qPayOrderDO).setNull(qPayOrderDO.redState)
                .where(qPayOrderDO.id.eq(id));
        jpaUpdateClause.execute();
    }

    public void setRedFlag(Long id, Boolean flag) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qPayOrderDO).set(qPayOrderDO.redState, flag)
                .where(qPayOrderDO.id.eq(id));
        jpaUpdateClause.execute();
    }

    public PagingVO<PayOrderExportEntity> defaultExport(PayOrderPageParam param) {
        List<Predicate> predicates = pageWhere(param, null);
        Predicate predicate = ExpressionUtils.allOf(predicates);
        var query = jpaQueryFactory.select(Projections.bean(
                        PayOrderExportEntity.class,
                        qPayOrderDO.sourceNo,
                        qPayOrderDO.suppCode,
                        qPayOrderDO.suppName,
                        qPayOrderDO.ouCode,
                        qPayOrderDO.ouName,
                        qPayOrderDO.apTypeCode,
                        qPayOrderDO.apTypeName,
                        qPayOrderDO.payTypeCode,
                        qPayOrderDO.payTypeName,
                        qPayOrderDO.payOrderNo,
                        qPayOrderDO.currCode,
                        qPayOrderDO.currName,
                        qPayOrderDO.operator,
                        qPayOrderDO.buType,
                        qPayOrderDO.buDate,
                        qPayOrderDO.totalAmt,
                        qPayOrderDO.totalCurAmt,
                        qPayOrderDO.auditUserId,
                        qPayOrderDO.auditDate,
                        qPayOrderDO.auditUser,
                        qPayOrderDO.orderState,
                        qPayOrderDO.exchangeRate,
                        qPayOrderDO.initFlag,
                        qPayOrderDO.realPayAmt,
                        qPayOrderDO.realPayCurAmt,
                        qPayOrderDO.apDate,
                        qPayOrderDO.apFlag,
                        qPayOrderDO.createTime,
                        qPayOrderDO.creator,
                        qPayOrderDO.modifyTime,
                        qPayOrderDO.updater,
                        qPayOrderDO.remark,
                        qPayOrderDO.auditRejection,
                        qPayOrderDO.createMode,
                        qPayOrderDO.localCurrCode,
                        qPayOrderDO.localCurrName,
                        qPayOrderDO.verState,
                        qPayOrderDO.verAmt,
                        qPayOrderDO.totalAmt.subtract(qPayOrderDO.verAmt).as("unVerAmt"),
                        qPayOrderDO.approvedTime,
                        qPayOrderDO.submitTime,
                        qPayOrderDO.addrNo,
                        qPayOrderDO.suppAddrNo,
                        qPayOrderDO.redState,
                        qPayOrderDO.redSourceNo,
                        qPayOrderDO.proposedStatus
                ))
                .from(qPayOrderDO)
                .where(predicate);
        param.setPaging(query);
        param.fillOrders(query, qPayOrderDO);
        return new PagingVO<>(query.fetchCount(), query.fetch());
    }


    public List<PayOrderDTO> selectInOutCustByPayOrderNo(List<String> payOrderNoList) {
        if (CollectionUtil.isEmpty(payOrderNoList)) {
            return Collections.EMPTY_LIST;
        }
        List<PayOrderDTO> payOrderDTOList = jpaQueryFactory.select(
                        Projections.bean(PayOrderDTO.class,
                                qPayOrderDO.payOrderNo,
                                qPayOrderDO.inOutCust,
                                qPayOrderDO.relevanceOuCode))
                .from(qPayOrderDO)
                .where(qPayOrderDO.payOrderNo.in(payOrderNoList))
                .fetch();

        return payOrderDTOList;
    }

    public void updateVerState(String verState, Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qPayOrderDO)
                .set(qPayOrderDO.verState, verState)
                .where(qPayOrderDO.id.eq(id));
        jpaUpdateClause.execute();
    }

    public Long countExportPayOrder(PayOrderPageParam param) {
        var jpaQuery = jpaQueryFactory.select(qPayOrderDtlDO.count()).from(qPayOrderDtlDO)
                .leftJoin(qPayOrderDO).on(qPayOrderDtlDO.masId.eq(qPayOrderDO.id));
        var predicates = pageWhere(param, null);
        if (StrUtil.isNotBlank(param.getSourceNoDtl())) {
            predicates.add(qPayOrderDtlDO.sourceNo.eq(param.getSourceNoDtl()));
        }
        if (CollUtil.isNotEmpty(param.getDtlSourceNoList())) {
            predicates.add(qPayOrderDtlDO.sourceNo.in(param.getDtlSourceNoList()));
        }
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return jpaQuery.fetchCount();
    }

    public List<PayOrderExportEntity> exportPayOrder(PayOrderPageParam param) {
        var jpaQuery = jpaQueryFactory.select(Projections.bean(
                        PayOrderExportEntity.class,
                        qPayOrderDO.id,
                        qPayOrderDO.sourceNo,
                        qPayOrderDO.suppId,
                        qPayOrderDO.suppCode,
                        qPayOrderDO.suppName,
                        qPayOrderDO.ouCode,
                        qPayOrderDO.ouId,
                        qPayOrderDO.ouName,
                        qPayOrderDO.apTypeId,
                        qPayOrderDO.apTypeCode,
                        qPayOrderDO.apTypeName,
                        qPayOrderDO.payTypeId,
                        qPayOrderDO.payTypeCode,
                        qPayOrderDO.payTypeName.as("payTypeCodeName"),
                        qPayOrderDO.payOrderNo,
                        qPayOrderDO.currCode,
                        qPayOrderDO.currName,
                        qPayOrderDO.operator,
                        qPayOrderDO.operUserId,
                        qPayOrderDO.buType,
                        qPayOrderDO.buDate,
                        qPayOrderDO.totalAmt,
                        qPayOrderDO.auditUserId,
                        qPayOrderDO.auditDate,
                        qPayOrderDO.auditUser,
                        qPayOrderDO.orderState,
                        qPayOrderDO.exchangeRate,
                        qPayOrderDO.initFlag,
                        qPayOrderDO.realPayAmt,
                        qPayOrderDO.apDate,
                        qPayOrderDO.apFlag,
                        qPayOrderDO.createTime,
                        qPayOrderDO.creator,
                        qPayOrderDO.modifyTime,
                        qPayOrderDO.updater,
                        qPayOrderDO.remark,
                        qPayOrderDO.auditRejection,
                        qPayOrderDO.createMode,
                        qPayOrderDO.auditDataVersion,
                        qPayOrderDO.procInstId,
                        qPayOrderDO.procInstStatus,
                        qPayOrderDO.approvedTime,
                        qPayOrderDO.submitTime,
                        qPayOrderDO.addrNo,
                        qPayOrderDO.suppAddrNo,
                        qPayOrderDO.redState,
                        qPayOrderDO.redSourceId,
                        qPayOrderDO.redSourceNo,
                        qPayOrderDO.proposedStatus,
                        qPayOrderDO.inOutCust,
                        qPayOrderDO.relevanceOuCode,
                        qPayOrderDtlDO.sourceNo.as("detailSourceNo"),
                        qPayOrderDtlDO.sourceLine,
                        qPayOrderDtlDO.payType,
                        qPayOrderDtlDO.payBank,
                        qPayOrderDtlDO.payAccount,
                        qPayOrderDtlDO.recBank,
                        qPayOrderDtlDO.recAccount,
                        qPayOrderDtlDO.totalAmt.as("detailTotalAmt"),
                        qPayOrderDtlDO.realPayAmt.as("detailRealPayAmt"),
                        qPayOrderDtlDO.verAmt,
                        qPayOrderDtlDO.unVerAmt,
                        qPayOrderDtlDO.applyVerAmTing
                )).from(qPayOrderDtlDO)
                .leftJoin(qPayOrderDO).on(qPayOrderDtlDO.masId.eq(qPayOrderDO.id));
        var predicates = pageWhere(param, null);
        if (StrUtil.isNotBlank(param.getSourceNoDtl())) {
            predicates.add(qPayOrderDtlDO.sourceNo.eq(param.getSourceNoDtl()));
        }
        if (CollUtil.isNotEmpty(param.getDtlSourceNoList())) {
            predicates.add(qPayOrderDtlDO.sourceNo.in(param.getDtlSourceNoList()));
        }
        //数据权限
        BusinessSecurityUtil.where(jpaQuery, PayOrderDO.class);
        param.setPaging(jpaQuery);
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        jpaQuery.orderBy(qPayOrderDO.id.desc(), qPayOrderDtlDO.id.desc());
        return jpaQuery.fetch();
    }

    public void updateSyncStatus(PayOrderDTO payOrderDTO) {
        JPAUpdateClause update = jpaQueryFactory.update(qPayOrderDO);
        update.set(qPayOrderDO.syncError, payOrderDTO.getSyncError());
        update.set(qPayOrderDO.modifyTime, LocalDateTime.now());
        if(StringUtils.isNotEmpty(payOrderDTO.getSyncStatus())){
            update.set(qPayOrderDO.syncStatus, payOrderDTO.getSyncStatus());
        }
        if(StringUtils.isNotEmpty(payOrderDTO.getPayStatus())){
            update.set(qPayOrderDO.payStatus, payOrderDTO.getPayStatus());
        }
        if(StringUtils.isNotEmpty(payOrderDTO.getOutSysKey())){
            update.set(qPayOrderDO.outSysKey, payOrderDTO.getOutSysKey());
        }
        if(StringUtils.isNotEmpty(payOrderDTO.getTransNo())){
            update.set(qPayOrderDO.transNo, payOrderDTO.getTransNo());
        }
        if(StringUtils.isNotEmpty(payOrderDTO.getRcvSeqNo())){
            update.set(qPayOrderDO.rcvSeqNo, payOrderDTO.getRcvSeqNo());
        }
        if(StringUtils.isNotEmpty(payOrderDTO.getSysAcntgSeqNo())){
            update.set(qPayOrderDO.sysAcntgSeqNo, payOrderDTO.getSysAcntgSeqNo());
        }
        update.where(qPayOrderDO.id.eq(payOrderDTO.getId())).execute();
    }

    public List<PayOrderDTO> queryOrderAmt(OrderAmtRpcParam param) {
        var jpaQuery = jpaQueryFactory.select(Projections.bean(
                        PayOrderDTO.class,
                        qPayOrderDO.id,
                        qPayOrderDO.sourceNo,
                        qPayOrderDO.suppId,
                        qPayOrderDO.suppCode,
                        qPayOrderDO.suppName,
                        qPayOrderDO.ouCode,
                        qPayOrderDO.ouId,
                        qPayOrderDO.ouName,
                        qPayOrderDO.payOrderNo,
                        qPayOrderDO.totalAmt,
                        qPayOrderDO.totalCurAmt,
                        qPayOrderDO.realPayAmt,
                        qPayOrderDO.realPayCurAmt
                )).from(qPayOrderDO);
        List<Predicate> predicates = new ArrayList<>();
        if (param.getOuId() != null) {
            predicates.add(qPayOrderDO.ouId.eq(param.getOuId()));
        }
        if (param.getSuppId() != null) {
            predicates.add(qPayOrderDO.suppId.eq(param.getSuppId()));
        }
        if (CollectionUtils.isNotEmpty(param.getFilterDocStatusList())) {
            predicates.add(qPayOrderDO.orderState.notIn(param.getFilterDocStatusList()));
        }
        jpaQuery.where(ExpressionUtils.allOf(predicates));
        return jpaQuery.fetch();
    }
}
