package com.elitesland.fin.infinity.http;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.HttpUtil;
import com.elitesland.fin.infinity.http.param.HttpParam;
import com.elitesland.fin.infinity.vo.ResponseVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * @author eric.hao
 * @since 2023/04/19
 */
@Slf4j
@Component
public class RestClient {

    @Resource
    private RestTemplate restTemplate;

    /**
     * 统一接口调用
     *
     * @param param 接口请求参数
     * @param <T>   返回数据结果的实体类型
     * @param <K>   响应值的实体类型
     * @return 请求结果
     */
    public <T, K extends ResponseVO> String exchange(HttpParam<T, K> param) {
        String responseStr;
        // GET
        if (param.getRequestMethod().equals(RequestMethod.GET)) {
            return get(param.getUrl(), param.getBody(), param.getHeaders());
        }
        // POST
        else if (param.getRequestMethod().equals(RequestMethod.POST)) {
            //  form-data
            if (MediaType.MULTIPART_FORM_DATA.equals(param.getMediaType())) {
                responseStr = postFormData(param.getUrl(), param.getBody(), param.getHeaders());
            }
            // x-www-form-urlencoded
            else if (MediaType.APPLICATION_FORM_URLENCODED.equals(param.getMediaType())) {
                responseStr = postFormUrlencoded(param.getUrl(), param.getBody(), param.getHeaders());
            }
            // other: application/json
            else {
                responseStr = postJsonData(param.getUrl(), param.getBody(), param.getHeaders());
            }
            return responseStr;
        }
        throw new RuntimeException("[INFINITY] only support: GET and POST.");
    }

    /**
     * 统一接口调用：做了结果集的对象简单封装
     *
     * @param param 接口请求参数
     * @param <T>   返回数据结果的实体类型
     * @param <K>   响应值的实体类型
     * @return 请求结果
     */
    public <T, K extends ResponseVO> K exchangeToBean(HttpParam<T, K> param) {
        if (ObjectUtil.isNull(param.getResponseType())) {
            throw new RuntimeException("[INFINITY] NULL OF RESPONSE_TYPE.");
        }
        return ResponseVO.result(exchange(param), param.getResponseType());
    }

    /**
     * get请求
     *
     * @param url    请求url
     * @param reqObj 请求request实体
     * @return 请求结果
     */
    public String get(String url, Map<String, Object> reqObj) {
        return get(url, reqObj, null);
    }

    /**
     * get请求
     *
     * @param url        请求url
     * @param reqObj     请求request实体
     * @param addHeaders 添加的头部 如需要使用jwt，key名称需设为"Bearer-Auth"
     * @return 请求结果
     */
    public String get(String url, Map<String, Object> reqObj, Map<String, String> addHeaders) {
        String queryParams = HttpUtil.toParams(reqObj, StandardCharsets.UTF_8);
        String requestUrl = url + "?" + queryParams;
        HttpEntity<String> entity = getMethodHttpEntity(addHeaders);
        return restTemplate.exchange(URI.create(requestUrl), HttpMethod.GET, entity, String.class).getBody();
    }

    /**
     * form-data的post请求
     *
     * @param requestUrl 请求url
     * @param reqObj     请求request实体
     * @return 请求结果
     */
    public String postFormData(String requestUrl, Map<String, Object> reqObj) {
        return postFormData(requestUrl, reqObj, null);
    }

    /**
     * form-data的post请求
     *
     * @param requestUrl 请求url
     * @param reqObj     请求request实体
     * @param addHeaders 添加的头部 如需要使用jwt,key名称需设为"Bearer-Auth"
     * @return 请求结果
     */
    public String postFormData(String requestUrl, Map<String, Object> reqObj, Map<String, String> addHeaders) {
        HttpEntity<MultiValueMap<String, Object>> entity = getFormRequestHttpEntity(reqObj, MediaType.MULTIPART_FORM_DATA_VALUE, addHeaders);
        return restTemplate.postForObject(requestUrl, entity, String.class);
    }

    /**
     * x-www-form-urlencoded的post请求
     *
     * @param requestUrl 请求url
     * @param reqObj     请求request实体
     * @return 请求结果
     */
    public String postFormUrlencoded(String requestUrl, Map<String, Object> reqObj) {
        return postFormUrlencoded(requestUrl, reqObj, null);
    }

    /**
     * x-www-form-urlencoded的post请求
     *
     * @param requestUrl 请求url
     * @param reqObj     请求request实体
     * @param addHeaders 添加的头部 如需要使用jwt,key名称需设为"Bearer-Auth"
     * @return 请求结果
     */
    public String postFormUrlencoded(String requestUrl, Map<String, Object> reqObj, Map<String, String> addHeaders) {
        HttpEntity<MultiValueMap<String, Object>> entity = getFormRequestHttpEntity(reqObj, MediaType.APPLICATION_FORM_URLENCODED_VALUE, addHeaders);
        return restTemplate.postForObject(requestUrl, entity, String.class);
    }

    /**
     * json的post请求
     *
     * @param requestUrl 请求url
     * @param reqObj     请求request实体
     * @return 请求结果
     */
    public String postJsonData(String requestUrl, Map<String, Object> reqObj) {
        return postJsonData(requestUrl, reqObj, null);
    }

    /**
     * json的post请求
     *
     * @param requestUrl 请求url
     * @param reqObj     请求request实体
     * @param addHeaders 添加的头部 如需要使用jwt,key名称需设为"Bearer-Auth"
     * @return 请求结果
     */
    public String postJsonData(String requestUrl, Map<String, Object> reqObj, Map<String, String> addHeaders) {
        HttpEntity<?> entity = getJsonRequestHttpEntity(reqObj, addHeaders);
        return restTemplate.exchange(requestUrl, HttpMethod.POST, entity, String.class).getBody();
    }

    /**
     * 请求request实体转form-data的map
     *
     * @param reqObj 请求request实体
     * @return 请求参数
     */
    public MultiValueMap<String, Object> requestBeanToFormParams(Map<String, Object> reqObj) {
        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        for (String key : reqObj.keySet()) {
            multiValueMap.add(key, reqObj.get(key));
        }
        return multiValueMap;
    }

    /**
     * 获取get方法的HttpEntity
     *
     * @param addHeaders 添加的头部
     * @return 请求参数
     */
    public HttpEntity<String> getMethodHttpEntity(Map<String, String> addHeaders) {
        HttpHeaders headers = getHttpHeaders(MediaType.APPLICATION_JSON);
        addHeaders(headers, addHeaders);
        return new HttpEntity<>(null, headers);
    }

    /**
     * 获取form请求的HttpEntity
     *
     * @param reqObj      请求request实体
     * @param contentType form请求类型
     * @param addHeaders  添加的头部
     * @return 构建参数体
     */
    public HttpEntity<MultiValueMap<String, Object>> getFormRequestHttpEntity(Map<String, Object> reqObj, String contentType, Map<String, String> addHeaders) {
        HttpHeaders headers = null;
        if (MediaType.APPLICATION_FORM_URLENCODED_VALUE.equals(contentType)) {
            headers = getHttpHeaders(MediaType.APPLICATION_FORM_URLENCODED);
        } else if (MediaType.MULTIPART_FORM_DATA_VALUE.equals(contentType)) {
            headers = getHttpHeaders(MediaType.MULTIPART_FORM_DATA);
        }
        addHeaders(headers, addHeaders);
        MultiValueMap<String, Object> multiValueMap = requestBeanToFormParams(reqObj);
        return new HttpEntity<>(multiValueMap, headers);
    }

    /**
     * 获取json请求的HttpEntity
     *
     * @param obj        请求request实体
     * @param addHeaders 添加的头部
     * @return 请求参数
     */
    public HttpEntity<?> getJsonRequestHttpEntity(Map<String, Object> obj, Map<String, String> addHeaders) {
        HttpHeaders headers = getHttpHeaders(MediaType.APPLICATION_JSON);
        addHeaders(headers, addHeaders);
        return new HttpEntity<>(obj, headers);
    }

    /**
     * 获取指定mediaType请求头
     */
    public HttpHeaders getHttpHeaders(MediaType mediaType) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(mediaType);
        return headers;
    }

    /**
     * 添加自定义的头部
     *
     * @param headers    头部参数
     * @param addHeaders 自定义参数
     */
    public void addHeaders(HttpHeaders headers, Map<String, String> addHeaders) {
        if (ObjectUtil.isEmpty(addHeaders)) {
            return;
        }
        for (String key : addHeaders.keySet()) {
            // jwt授权
            if ("Bearer-Auth".equals(key)) {
                headers.setBearerAuth(addHeaders.get(key));
                continue;
            }
            headers.set(key, addHeaders.get(key));
        }
    }
}

