package com.elitesland.fin.infinity.utils;

import com.elitescloud.boot.exception.BusinessException;
import com.opencsv.CSVReader;
import com.opencsv.exceptions.CsvException;
import io.swagger.annotations.ApiModelProperty;
import lombok.extern.slf4j.Slf4j;

import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
public class CsvReaderUtil {

    private static final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private static final DateTimeFormatter formatter2 = DateTimeFormatter.ofPattern("yyyyMMdd");

    private static final DateTimeFormatter formatter3 = DateTimeFormatter.ofPattern("yyyy-MM-d HH:mm");


    /**
     * 读取CSV文件并通过监听器处理
     *
     * @param filePath CSV文件路径
     * @param clazz    数据模型类
     * @param listener 处理监听器
     */
    public static <T> void readCsv(String filePath, Class<T> clazz, CsvReadListener<T> listener) {
        log.info("开始读取CSV文件：{}", filePath);
        try (InputStreamReader isr = new InputStreamReader(
                new FileInputStream(filePath),
                Charset.forName("GBK")); CSVReader reader = new CSVReader(isr)) {
            List<String[]> rows = reader.readAll();
            if (rows.isEmpty()) {
                return;
            }

            // 解析表头
            String[] headers = rows.get(0);
            // 创建表头到索引的映射
            Map<String, Integer> headerIndexMap = new HashMap<>();
            for (int i = 0; i < headers.length; i++) {
                headerIndexMap.put(headers[i].trim(), i);
            }

            // 获取类字段与CSV列名的映射关系
            Map<String, Field> fieldMapping = getFieldMapping(clazz);

            // 逐行处理数据
            for (int i = 1; i < rows.size(); i++) {
                String[] data = rows.get(i);
                T obj = mapToObject(data, headerIndexMap, fieldMapping, clazz);
                listener.invoke(obj);
            }

            listener.doAfterAllAnalysed();

        } catch (IOException | CsvException e) {
            log.error("CSV读取失败", e);
            throw new BusinessException("CSV读取失败", e);
        }
    }

    /**
     * 获取类字段与CSV列名的映射关系（基于@ApiModelProperty的name属性）
     */
    private static <T> Map<String, Field> getFieldMapping(Class<T> clazz) {
        Map<String, Field> fieldMapping = new HashMap<>();
        Field[] fields = clazz.getDeclaredFields();

        for (Field field : fields) {
            // 获取ApiModelProperty注解
            ApiModelProperty annotation = field.getAnnotation(ApiModelProperty.class);
            if (annotation != null && !annotation.value().isEmpty()) {
                // 使用注解中指定的name作为CSV列名
                fieldMapping.put(annotation.value().trim(), field);
                field.setAccessible(true); // 设置可访问私有字段
            }
        }

        return fieldMapping;
    }

    /**
     * 将CSV行数据映射到Java对象
     */
    private static <T> T mapToObject(String[] data, Map<String, Integer> headerIndexMap,
                                     Map<String, Field> fieldMapping, Class<T> clazz) {
        try {
            T obj = clazz.getDeclaredConstructor().newInstance();

            // 遍历所有映射字段
            for (Map.Entry<String, Field> entry : fieldMapping.entrySet()) {
                String csvColumnName = entry.getKey();
                Field field = entry.getValue();

                // 检查CSV中是否存在该列
                if (headerIndexMap.containsKey(csvColumnName)) {
                    int columnIndex = headerIndexMap.get(csvColumnName);
                    String value = columnIndex < data.length ? data[columnIndex].trim() : null;

                    // 根据字段类型设置值
                    setFieldValue(obj, field, value);
                }
            }

            return obj;
        } catch (Exception e) {
            log.error("对象映射失败", e);
            throw new RuntimeException("对象映射失败", e);
        }
    }

    /**
     * 根据字段类型设置值
     */
    private static void setFieldValue(Object obj, Field field, String value) throws IllegalAccessException {
        if (value == null || value.isEmpty()) {
            return;
        }

        Class<?> fieldType = field.getType();

        // 处理常见类型
        if (fieldType == String.class) {
            field.set(obj, value);
        } else if (fieldType == Integer.class || fieldType == int.class) {
            field.set(obj, Integer.parseInt(value));
        } else if (fieldType == Long.class || fieldType == long.class) {
            field.set(obj, Long.parseLong(value));
        } else if (fieldType == Double.class || fieldType == double.class) {
            field.set(obj, Double.parseDouble(value));
        } else if (fieldType == BigDecimal.class || fieldType == float.class) {
            field.set(obj, new BigDecimal(value));
        } else if (fieldType == Boolean.class || fieldType == boolean.class) {
            field.set(obj, Boolean.parseBoolean(value));
        } else if (fieldType == LocalDateTime.class) {
            // 这里简化处理，实际可根据需要添加日期格式化逻辑
            field.set(obj, LocalDateTime.parse(value.replace("/", "-"), formatter));
        } else if (fieldType == LocalDate.class) {
            field.set(obj, LocalDate.parse(value.replace("/", "-"), formatter2));

        }

    }
}

