package com.elitesland.fin.utils.tcsl.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.nacos.shaded.com.google.gson.Gson;
import com.cloudt.apm.common.exception.BusinessException;
import com.elitesland.fin.application.convert.financial.ManagementFeeBillConvert;
import com.elitesland.fin.application.facade.dto.financial.ManagementFeeBillDTO;
import com.elitesland.fin.utils.tcsl.config.LakalaProperties;
import com.elitesland.fin.utils.tcsl.vo.BillQueryParamVO;
import com.elitesland.fin.utils.tcsl.vo.SyncResult;
import com.elitesland.fin.utils.tcsl.vo.TcslDataRecordResponse;
import com.elitesland.fin.utils.tcsl.vo.TcslResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Component;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

/**
 * 用来请求商龙开放平台接口的deme，前提引入httpClient依赖包
 * <dependency>
 * <groupId>org.apache.httpcomponents</groupId>
 * <artifactId>httpclient</artifactId>
 * <version>4.5.13</version>
 * </dependency>
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class TcslApiClient {
    // 商龙开放平台接口访问地址,例如 测试环境：http://open-test.tcsl.com.cn/api/，
    //                          线上环境：https://openapi.tcsl.com.cn/
//    String host = "http://open-test.tcsl.com.cn/api/";
//
//    // 接口名，
//    String pay = "yzj/wuuxiangcloudfunds/third/external/payment/checkoutPay";//替换成开放平台接口文档里对应的接口名称
//    String queryAccountDtl = "yzj/wuuxiangcloudfunds/third/external/account/accountDetails";//替换成开放平台接口文档里对应的接口名称
//    String queryBalance = "yzj/wuuxiangcloudfunds/third/external/account/balance/query";//替换成开放平台接口文档里对应的接口名称
//    String queryResult = "yzj/wuuxiangcloudfunds/third/external/payment/checkoutPay/query";//替换成开放平台接口文档里对应的接口名称
//
//
//    //开放平台业务接口，目前只支持HTTP POST一种请求方式
//    String appSecret = "****";//你的appSecret，注意替换对应环境的appsecret
//    String appKey = "*****";

    private final String TCSL_THIRD_APP = "TSCL";

    private final String TCSL_BUSINESS_CODE = "TCSL_ACCOUNT_BILL";

    private final LakalaProperties lakalaProperties;

    // 在方法上添加重试注解
//    @Retryable(value = {Exception.class}, maxAttempts = 3, backoff = @Backoff(delay = 1000))
    public List<ManagementFeeBillDTO> queryBill(BillQueryParamVO paramVO) {
        //        测试环境：10
//        生产环境：1306373713372925967
        paramVO.setBrandId(lakalaProperties.getBrandId());
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        // 如果没有开始时间，默认给前一天时间的0点
        if(StringUtils.isEmpty(paramVO.getStartTime())){
            LocalDateTime localDateTime = LocalDateTime.now().minusDays(1).withHour(0).withMinute(0).withSecond(0);
            paramVO.setStartTime(localDateTime.format(dateTimeFormatter));
        }
        // 如果没有结束时间，默认给今天时间的0点
        if(StringUtils.isEmpty(paramVO.getEndTime())){
            LocalDateTime localDateTime = LocalDateTime.now().withHour(0).withMinute(0).withSecond(0);
            paramVO.setEndTime(localDateTime.format(dateTimeFormatter));
        }
        SyncResult send = send(JSONObject.toJSONString(paramVO), lakalaProperties.getBillQuery());
        if(send.getSuccess() != null && !send.getSuccess()){
            throw new BusinessException("账单查询异常：" + send.getReqFailMsg());
        }
        TcslResponse response = new Gson().fromJson(send.getData(), TcslResponse.class);
        TcslDataRecordResponse data = response.getData().getData();
        return ManagementFeeBillConvert.INSTANCE.dataToDTO(data.getRecords());
    }

    public SyncResult send(String params,String methodName) {

        SyncResult syncResult = null;

        //参数1：头参数
        Map<String, String> headerMap = new HashMap<>();
        headerMap.put("groupID", lakalaProperties.getGroupID()); //集团id，不可为空，在开放平台进行开通时候填写的那个G号
        headerMap.put("traceID", UUID.randomUUID().toString());//业务方的traceID，问题排查时，提供该traceID,注意保证唯一性,自己去生成就行
        headerMap.put("Proxy-Client-IP", "192.168.28.30");//请传入真实IP，用于校验ip白名单

        //参数2：系统级必传参数，每个接口必传这几个值，签名算法中会用到，不能缺少
        Map<String, String> systemParamsMap = new HashMap<>();
        systemParamsMap.put("timestamp", String.valueOf(System.currentTimeMillis()));
        systemParamsMap.put("appKey", lakalaProperties.getAppKey());//注意替换对应环境的appkey

        systemParamsMap.put("version", "1");//固定值
        systemParamsMap.put("charset", "UTF-8");//固定值

        //参数类型3：业务入参数
        Map<String, String> bizParamsMap = new HashMap<>();
        bizParamsMap.put("biz", params);
//        bizParamsMap.put("biz", "{\"outTradeNo\":\"00441207916460685358035645863974\",\"channelCode\":\"LklSmb\",\"outTradeTime\":\"2023-07-17 14:44:12\",\"payerAccountNo\":\"123payer\",\"payeeList\":[{\"payeeAccountNo\":\"222123\",\"subOutTradeNo\":\"122222222222222222222\",\"subTradeAmount\":1.01}]}");//把请求报文进行json压缩后放入这里就行，我这里目前传了一个空对象
        try {
            //生成用来签名的完整url路径
            String urlForGenSig = genUrlForGenSig(lakalaProperties.getHost(), methodName, systemParamsMap, bizParamsMap);
            log.info("完整url路径: " + urlForGenSig);
            //生成签名
            String sig = getSig(lakalaProperties.getAppSecret(), urlForGenSig);
            //发送请求
            log.info("Request Body : " + JSONObject.toJSONString(params));
            String result = sendRequest(urlForGenSig, sig, headerMap, bizParamsMap);
            // 输出响应响应体
            log.info("Response Body : " + result);
//            TcslResponse response = new Gson().fromJson(result, TcslResponse.class);
            TcslResponse response = JSON.parseObject(result, TcslResponse.class);
            if (response.isSuccess() && response.getData() != null && response.getData().isSuccess()) {
                syncResult = SyncResult.success(params, result, HttpMethod.POST, lakalaProperties.getHost(), methodName, true,
                        TCSL_THIRD_APP, null);
            } else {
                syncResult = SyncResult.fail(params, result, false, response.getMsg(), HttpMethod.POST, lakalaProperties.getHost(),
                        methodName, true, TCSL_THIRD_APP, null);
            }
        } catch (Exception e) {
            e.printStackTrace();
            syncResult = SyncResult.fail(params, null, false, e.getMessage(), HttpMethod.POST,
                    lakalaProperties.getHost(), methodName, true, TCSL_THIRD_APP, null);
        } finally {
            ThirdSysLogUtil.logUpload(TCSL_BUSINESS_CODE, "管理费账单查询", syncResult, false);
        }
        return syncResult;
    }

    private String sendRequest(String url, String sig, Map<String, String> headersMap, Map<String, String> paramsMap)
            throws RuntimeException {
        String requestUrl = url + "&sig=" + sig;
        HttpPost httpPost = new HttpPost(requestUrl);
        CloseableHttpResponse response = null;
        try {
            //设置参数
//            List<BasicNameValuePair> nameValuePairList = convertToEntity(paramsMap);
//            UrlEncodedFormEntity uefEntity = new UrlEncodedFormEntity(nameValuePairList, "UTF-8");
//            httpPost.setEntity(uefEntity);
            headersMap.forEach((k, v) -> httpPost.addHeader(k, v));
            CloseableHttpClient httpClient = HttpClients.createDefault();
            // 设置请求头
            httpPost.setHeader("Content-Type", "application/json");
            httpPost.setHeader("Accept", "application/json");

            // 设置请求体
            String requestBody = paramsMap.get("biz");
            httpPost.setEntity(new StringEntity(requestBody, StandardCharsets.UTF_8));
            response = httpClient.execute(httpPost);
            HttpEntity entity = response.getEntity();

            return EntityUtils.toString(entity, "UTF-8");

        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        } finally {
            httpPost.releaseConnection();
            try {
                if (response != null) {
                    response.close();
                }
            } catch (IOException e) {
                throw new RuntimeException(e.getMessage());
            }
        }

    }

    public String getSig(String appSecret, String baseUrl) {
        try {
            Mac hs256 = Mac.getInstance("HmacSHA256");
            SecretKeySpec secretKey = new SecretKeySpec(appSecret.getBytes("UTF-8"), "HmacSHA256");
            hs256.init(secretKey);
            byte[] bytes = hs256.doFinal(baseUrl.getBytes("UTF-8"));

            return toHex(bytes);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("不支持HmacSHA256算法", e);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("不支持utf8编码", e);
        } catch (Exception e) {
            throw new RuntimeException("hs256签名失败", e);
        }

    }

    public String toHex(byte[] bytes) {
        StringBuffer hexstr = new StringBuffer();
        for (int i = 0; i < bytes.length; i++) {
            String b = Integer.toHexString(bytes[i] & 0xFF);
            if (b.length() < 2) {
                hexstr.append(0);
            }
            hexstr.append(b);
        }

        return hexstr.toString();
    }

    public String genUrlForGenSig(String host, String methodName, Map<String, String> systemParamsMap,
                                         Map<String, String> applicationParamsMap) throws UnsupportedEncodingException {
        Map<String, String> paramMap = new HashMap<>();
        paramMap.putAll(systemParamsMap);
        //如果应用级参数不为空，则组合应用级参数
        if (applicationParamsMap != null) {
            paramMap.putAll(applicationParamsMap);
        }
        return host + methodName + "?" + getSortedParamStr(paramMap);
    }

    /**
     * 构造自然排序请求参数
     *
     * @param params 请求
     * @return 字符串
     */
    public String getSortedParamStr(Map<String, String> params) throws UnsupportedEncodingException {
        Set<String> sortedParams = new TreeSet<>(params.keySet());
        StringBuilder sortedParamStr = new StringBuilder();
        for (String key : sortedParams) {
            String value = params.get(key);
            // 排除sign
            if (null != value && !"".equals(value) && !"sig".equalsIgnoreCase(key) && !"biz".equalsIgnoreCase(key)) {
                sortedParamStr.append(key + "=" + URLEncoder.encode(value, "UTF-8") + "&");
            }
        }
        return sortedParamStr.substring(0, sortedParamStr.length() - 1);
    }

    /**
     * 对象转化为参数列表
     *
     * @param applicationParamsMap
     * @return
     */
    public List<BasicNameValuePair> convertToEntity(Map<String, String> applicationParamsMap)
            throws RuntimeException {
        List<BasicNameValuePair> formParam = new ArrayList<>();
        try {
            if (applicationParamsMap != null) {
                for (Map.Entry<String, String> entry : applicationParamsMap.entrySet()) {
                    BasicNameValuePair nameValuePair = new BasicNameValuePair(entry.getKey(), entry.getValue());
                    formParam.add(nameValuePair);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException("1");
        }
        return formParam;
    }
}
