package com.elitesland.scp.application.service.calendar;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetComputeParamVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetLineParamVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetPageParamVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetParamVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetLineRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetRespVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetApproveVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetImportDTO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetSaveVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetSubmitVO;
import com.elitesland.scp.domain.convert.calendar.ScpStoreCalendarSetConvert;
import com.elitesland.scp.domain.entity.calendar.ScpStoreDemandCalendarDO;
import com.elitesland.scp.domain.service.calendar.ScpStoreCalendarSetDomainService;
import com.elitesland.scp.domain.service.calendar.ScpStoreCalendarSetLineDomainService;
import com.elitesland.scp.domain.service.calendar.ScpStoreDemandCalendarDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.calendar.ScpStoreCalendarSetDTO;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.utils.CalendarBuilder;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.workflow.enums.ProcInstStatus;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.TemporalAdjusters;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Service
@Slf4j
public class ScpStoreCalendarSetServiceImpl implements ScpStoreCalendarSetService {

    private final ScpStoreCalendarSetDomainService scpStoreCalendarSetDomainService;
    private final ScpStoreCalendarSetLineDomainService scpStoreCalendarSetLineDomainService;
    private final TransactionTemplate transactionTemplate;
    private final ScpStoreDemandCalendarDomainService scpStoreDemandCalendarDomainService;
    private final TaskExecutor taskExecutor;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;

    @Override
    @SysCodeProc
    public PagingVO<ScpStoreCalendarSetPageRespVO> page(ScpStoreCalendarSetPageParamVO paramVO) {
        return scpStoreCalendarSetDomainService.queryCalendarSetList(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveCalendarSet(ScpStoreCalendarSetSaveVO saveVO) {
        // 校验单据类型配置
        checkSetting(saveVO);
        return scpStoreCalendarSetDomainService.saveCalendarSet(saveVO);
    }

    @Override
    @SysCodeProc
    public Optional<ScpStoreCalendarSetRespVO> findCalendarSetById(Long id) {
        return scpStoreCalendarSetDomainService.findCalendarSetById(id).map(ScpStoreCalendarSetConvert.INSTANCE::dtoToRespVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpStoreCalendarSetDomainService.deleteByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submit(ScpStoreCalendarSetSubmitVO submitVO) {
        ScpStoreCalendarSetLineParamVO scpStoreCalendarSetLineParamVO = new ScpStoreCalendarSetLineParamVO();
        scpStoreCalendarSetLineParamVO.setMasId(submitVO.getId());
        Long count = scpStoreCalendarSetLineDomainService.countCalendarSetLine(scpStoreCalendarSetLineParamVO);
        if (count == 0) {
            throw new BusinessException("门店订货日历配置明细不能为空");
        }
        scpStoreCalendarSetDomainService.applyChangeUpdate(submitVO.getId(), ScpUdcEnum.CALENDAR_SET_STATUS_APPRING.getValueCode(), ProcInstStatus.APPROVING);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchApprove(ScpStoreCalendarSetApproveVO approveSaveParam) {
        //调用scpStoreCalendarSetDomainService的approveUpdateApplyAlter方法，传入approveSaveParam的id，CALENDAR_SET_STATUS_COMPLETE的valueCode，ProcInstStatus.APPROVED，approveSaveParam的approveComment，LocalDateTime.now()作为参数
        scpStoreCalendarSetDomainService.approveUpdateApplyAlter(approveSaveParam.getIds(), ScpUdcEnum.CALENDAR_SET_STATUS_COMPLETE.getValueCode(), ProcInstStatus.APPROVED,
                approveSaveParam.getApproveComment(), LocalDateTime.now());
        //异步执行写入门店订货日历表
        for (Long id : approveSaveParam.getIds()) {
            //调用batchSaveCalendar方法，传入approveSaveParam的id作为参数
            batchSaveCalendar(id);
        }
    }

    private void batchSaveCalendar(Long id) {
        // 根据id查询门店订货日历配置信息
        Optional<ScpStoreCalendarSetDTO> calendarSetDTO = scpStoreCalendarSetDomainService.findCalendarSetById(id);
        // 如果查询不到门店订货日历配置信息，抛出异常
        if (calendarSetDTO.isEmpty()) {
            throw new BusinessException("查询不到门店订货日历配置信息");
        }
        // 记录日志，门店订货日历保存开始
        log.info("门店订货日历保存开始：{}", JSONObject.toJSONString(calendarSetDTO));
        // 获取门店订货日历配置信息
        ScpStoreCalendarSetDTO scpStoreCalendarSetDTO = calendarSetDTO.get();
        // 获取开始年份
        Integer startYear = scpStoreCalendarSetDTO.getStartYear();
        // 获取开始日期
        LocalDate startOfYear = LocalDate.of(startYear, 1, 1);
        // 根据masId和开始年份查询门店区域列表
        List<ScpStoreCalendarSetLineRespVO> storeRegionList = scpStoreCalendarSetLineDomainService.findStoreCodeByMasId(id, startYear);
        // 获取门店区域编码列表
        List<String> storeRegionCodes = storeRegionList.stream().map(ScpStoreCalendarSetLineRespVO::getStoreCode).collect(Collectors.toList());

        //删除区域或门店订货日历历史记录
        scpStoreDemandCalendarDomainService.deleteByStoreAndYearAndDeliveryType(scpStoreCalendarSetDTO.getType(), storeRegionCodes, startYear.toString(), scpStoreCalendarSetDTO.getDeliveryType());
        // 如果日历类型是门店类型，调用buildAndSaveCalendar方法保存门店订货日历
        if (ScpUdcEnum.CALENDAR_TYPE_STORE.getValueCode().equals(scpStoreCalendarSetDTO.getType())) {
            buildAndSaveCalendar(id, storeRegionList, startOfYear, scpStoreCalendarSetDTO.getDeliveryType(), scpStoreCalendarSetDTO.getType());
        } else {
//            // 创建门店区域列表
//            List<ScpStoreCalendarSetLineRespVO> storeRegionList_ = new ArrayList<>();
            // 删除门店订货日历历史记录
            List<OrgStoreBaseRpcDTO> storeList = rmiOrgStoreRpcService.findSimpleByRegion(storeRegionCodes);
            if (CollectionUtil.isNotEmpty(storeList)) {
                // 获取门店编码列表
                List<String> storeCodes = storeList.stream().map(OrgStoreBaseRpcDTO::getStoreCode).distinct().collect(Collectors.toList());
                // 这个部分意义极其不明，并不需要覆盖，直接删除再插入即可
//                // 获取门店编码和区域映射关系
//                Map<String, List<OrgStoreBaseRpcDTO>> storeMap = storeList.stream().collect(Collectors.groupingBy(OrgStoreBaseRpcDTO::getRegion));
//                // 遍历门店区域列表
//                storeRegionList.forEach(region -> {
//                    // 如果门店编码和门店信息映射关系包含该门店区域编码
//                    if (storeMap.containsKey(region.getStoreCode())) {
//                        // 获取该门店区域编码对应的门店信息列表
//                        List<OrgStoreBaseRpcDTO> stores = storeMap.get(region.getStoreCode());
//                        // 遍历门店信息列表
//                        stores.forEach(store -> {
//                            // 创建门店订货日历配置信息对象
//                            ScpStoreCalendarSetLineRespVO setLineRespVO = new ScpStoreCalendarSetLineRespVO();
//                            // 设置门店编码
//                            setLineRespVO.setStoreCode(store.getStoreCode());
//                            // 设置门店id
//                            setLineRespVO.setStoreId(store.getId());
//                            // 设置门店名称
//                            setLineRespVO.setStoreName(store.getStoreName());
//                            // 设置masId
//                            setLineRespVO.setMasId(region.getMasId());
//                            // 设置日历类型
//                            setLineRespVO.setType(ScpUdcEnum.CALENDAR_TYPE_STORE.getValueCode());
//                            // 设置今日周
//                            setLineRespVO.setTodayWeek(region.getTodayWeek());
//                            // 设置交货类型
//                            setLineRespVO.setDeliveryType(region.getDeliveryType());
//                            // 将门店订货日历配置信息对象添加到门店区域列表
//                            storeRegionList_.add(setLineRespVO);
//                        });
//                    }
//                });
                // 删除门店订货日历历史记录
                scpStoreDemandCalendarDomainService.deleteByStoreAndYearAndDeliveryType(ScpUdcEnum.CALENDAR_TYPE_STORE.getValueCode(), storeCodes, startYear.toString(), scpStoreCalendarSetDTO.getDeliveryType());
            }
            // 记录日志，批量保存门店订货日历开始
            log.info("批量保存门店订货日历开始：{}", storeRegionList);
            // 调用buildAndSaveCalendar方法保存门店订货日历
            buildAndSaveCalendar(id, storeRegionList, startOfYear, scpStoreCalendarSetDTO.getDeliveryType(), scpStoreCalendarSetDTO.getType());
        }
    }

// 根据传入的id、storeRegionList和startOfYear构建并保存日历，deliveryType为配送类型，type为区域/门店标识，在ScpStoreCalendarSetDTO中被传递
    private void buildAndSaveCalendar(Long id, List<ScpStoreCalendarSetLineRespVO> storeRegionList, LocalDate startOfYear, String deliveryType, String type) {
        // 获取计算参数
        var computeParamVO = getScpStoreCalendarSetComputeParamVO(id, storeRegionList);
        // 循环12次，分别构建1-12月的日历
        for (int i = 0; i < 12; i++) {
            log.info("开始写入:{}月份区域/门店订货日历，computeParamVO为：{}", i + 1, JSONObject.toJSONString(computeParamVO));
            // 设置事务传播行为为新事务
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            int finalI = i;
            // 执行事务
            transactionTemplate.execute(trans -> {
                try {
                    // 计算当前月份的开始日期和结束日期
                    LocalDate startDate = startOfYear.plusMonths(finalI);
                    LocalDate endDate = startDate.with(TemporalAdjusters.lastDayOfMonth());
                    // 构建当前月份的日历
                    List<ScpStoreDemandCalendarDO> scpStoreDemandCalendarDOS = CalendarBuilder.buildCalendarSetList(startDate, endDate, computeParamVO, deliveryType, type);
                    // 批量插入日历
                    scpStoreDemandCalendarDomainService.batchInsert(scpStoreDemandCalendarDOS, 10000);
                    return "OK";
                } catch (Exception e) {
                    // 如果发生异常，回滚事务并抛出业务异常
                    trans.setRollbackOnly();
                    throw new BusinessException(ApiCode.FAIL, e.getMessage());
                }
            });
            log.info("结束写入:{}月份区域/门店订货日历", i + 1);
        }
    }

    private ScpStoreCalendarSetComputeParamVO getScpStoreCalendarSetComputeParamVO(Long masId, List<ScpStoreCalendarSetLineRespVO> storeInfos) {
        List<ScpStoreCalendarSetLineRespVO> monList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 1);
        List<ScpStoreCalendarSetLineRespVO> tuesList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 2);
        List<ScpStoreCalendarSetLineRespVO> wendsList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 3);
        List<ScpStoreCalendarSetLineRespVO> thursList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 4);
        List<ScpStoreCalendarSetLineRespVO> friList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 5);
        List<ScpStoreCalendarSetLineRespVO> saturList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 6);
        List<ScpStoreCalendarSetLineRespVO> sunList = scpStoreCalendarSetLineDomainService.findCalendarSetByParam(masId, 7);
        ScpStoreCalendarSetComputeParamVO computeParamVO = new ScpStoreCalendarSetComputeParamVO();
        computeParamVO.setMonList(monList);
        computeParamVO.setTuesList(tuesList);
        computeParamVO.setWendsList(wendsList);
        computeParamVO.setThursList(thursList);
        computeParamVO.setFriList(friList);
        computeParamVO.setSaturList(saturList);
        computeParamVO.setSunList(sunList);
        computeParamVO.setStoreInfos(storeInfos);
        computeParamVO.setNonMonList(findMissingList(monList, storeInfos));
        computeParamVO.setNonTuesList(findMissingList(tuesList, storeInfos));
        computeParamVO.setNonWendsList(findMissingList(wendsList, storeInfos));
        computeParamVO.setNonThursList(findMissingList(thursList, storeInfos));
        computeParamVO.setNonFriList(findMissingList(friList, storeInfos));
        computeParamVO.setNonSaturList(findMissingList(saturList, storeInfos));
        computeParamVO.setNonSunList(findMissingList(sunList, storeInfos));
        return computeParamVO;
    }

    public List<ScpStoreCalendarSetLineRespVO> findMissingList(List<ScpStoreCalendarSetLineRespVO> filterList, List<ScpStoreCalendarSetLineRespVO> storeInfos) {
        Set<String> storeCodesInList = filterList.stream()
                .map(ScpStoreCalendarSetLineRespVO::getStoreCode)
                .collect(Collectors.toCollection(HashSet::new));

        return storeInfos.stream()
                .filter(storeInfo -> !storeCodesInList.contains(storeInfo.getStoreCode()))
                .collect(Collectors.toList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchReject(ScpStoreCalendarSetApproveVO approveSaveParam) {
        scpStoreCalendarSetDomainService.approveUpdateApplyAlter(approveSaveParam.getIds(), ScpUdcEnum.CALENDAR_SET_STATUS_REJECTED.getValueCode(), ProcInstStatus.REJECTED,
                approveSaveParam.getApproveComment(), LocalDateTime.now());
    }

    private void checkSetting(ScpStoreCalendarSetSaveVO saveVO) {
        if (saveVO.getEndYear() == null) {
            saveVO.setEndYear(saveVO.getStartYear());
        }
        ScpStoreCalendarSetParamVO scpStoreCalendarSetParamVO = new ScpStoreCalendarSetParamVO();
        scpStoreCalendarSetParamVO.setSetCode(saveVO.getSetCode());
        scpStoreCalendarSetParamVO.setEndYear(saveVO.getEndYear());
        scpStoreCalendarSetParamVO.setStartYear(saveVO.getStartYear());
        List<ScpStoreCalendarSetRespVO> calendarSets = scpStoreCalendarSetDomainService.findCalendarSet(scpStoreCalendarSetParamVO);
        if (CollUtil.isNotEmpty(calendarSets)) {
            var scpStoreCalendarSetRespVO = calendarSets.get(0);
            if (saveVO.getId() == null || !saveVO.getId().equals(scpStoreCalendarSetRespVO.getId())) {
                if (saveVO.getSetCode() != null && !saveVO.getSetCode().equals(scpStoreCalendarSetRespVO.getSetCode())) {
                    throw new BusinessException("门店订货日历配置编码已存在，请检查");
                }
            }
        }
        if (saveVO.getId() == null) {
            saveVO.setDocStatus(ScpUdcEnum.CALENDAR_SET_STATUS_DR.getValueCode());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void executeImport(List<ScpStoreCalendarSetImportDTO> scpStoreCalendarSetImportDTO) {
        // 获取门店区域编码列表
        for (ScpStoreCalendarSetImportDTO scpStoreCalendarSetDTO : scpStoreCalendarSetImportDTO) {
            // 获取开始年份
            Integer startYear = scpStoreCalendarSetDTO.getStartYear();
            // 获取开始日期
            LocalDate startOfYear = LocalDate.of(startYear, 1, 1);

            List<String> storeRegionCodes = scpStoreCalendarSetDTO.getStoreRegionCodeList();
            //删除区域或门店订货日历历史记录
            scpStoreDemandCalendarDomainService.deleteByStoreAndYearAndDeliveryType(scpStoreCalendarSetDTO.getType(), storeRegionCodes, startYear.toString(), scpStoreCalendarSetDTO.getDeliveryType());
            // 如果日历类型是门店类型，调用buildAndSaveCalendar方法保存门店订货日历
            if (ScpUdcEnum.CALENDAR_TYPE_STORE.getValueCode().equals(scpStoreCalendarSetDTO.getType())) {
                buildAndSaveCalendar2(scpStoreCalendarSetDTO, startOfYear, scpStoreCalendarSetDTO.getDeliveryType(), scpStoreCalendarSetDTO.getType());
            } else {
                // 删除门店订货日历历史记录
                List<OrgStoreBaseRpcDTO> storeList = rmiOrgStoreRpcService.findSimpleByRegion(storeRegionCodes);
                if (CollectionUtil.isNotEmpty(storeList)) {
                    // 获取门店编码列表
                    List<String> storeCodes = storeList.stream().map(OrgStoreBaseRpcDTO::getStoreCode).distinct().collect(Collectors.toList());
                    // 删除门店订货日历历史记录
                    scpStoreDemandCalendarDomainService.deleteByStoreAndYearAndDeliveryType(ScpUdcEnum.CALENDAR_TYPE_STORE.getValueCode(), storeCodes, startYear.toString(), scpStoreCalendarSetDTO.getDeliveryType());
                }
                // 记录日志，批量保存门店订货日历开始
                log.info("批量保存门店订货日历开始：{}", JSONObject.toJSONString(scpStoreCalendarSetDTO));
                // 调用buildAndSaveCalendar方法保存门店订货日历
                buildAndSaveCalendar2(scpStoreCalendarSetDTO, startOfYear, scpStoreCalendarSetDTO.getDeliveryType(), scpStoreCalendarSetDTO.getType());
            }
        }
    }

    private void buildAndSaveCalendar2(ScpStoreCalendarSetImportDTO importDTO, LocalDate startOfYear, String deliveryType, String type) {
        ScpStoreCalendarSetComputeParamVO computeParamVO = new ScpStoreCalendarSetComputeParamVO();
        computeParamVO.setMonList(importDTO.getMonday());
        computeParamVO.setTuesList(importDTO.getTuesday());
        computeParamVO.setWendsList(importDTO.getWednesday());
        computeParamVO.setThursList(importDTO.getThursday());
        computeParamVO.setFriList(importDTO.getFriday());
        computeParamVO.setSaturList(importDTO.getSaturday());
        computeParamVO.setSunList(importDTO.getSunday());

        computeParamVO.setNonMonList(importDTO.getMonday2());
        computeParamVO.setNonTuesList(importDTO.getTuesday2());
        computeParamVO.setNonWendsList(importDTO.getWednesday2());
        computeParamVO.setNonThursList(importDTO.getThursday2());
        computeParamVO.setNonFriList(importDTO.getFriday2());
        computeParamVO.setNonSaturList(importDTO.getSaturday2());
        computeParamVO.setNonSunList(importDTO.getSunday2());

        // 循环12次，分别构建1-12月的日历
        for (int i = 0; i < 12; i++) {
            log.info("开始写入:{}月份区域/门店订货日历，computeParamVO为：{}", i + 1, JSONObject.toJSONString(computeParamVO));
            // 设置事务传播行为为新事务
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            int finalI = i;
            // 执行事务
            transactionTemplate.execute(trans -> {
                try {
                    // 计算当前月份的开始日期和结束日期
                    LocalDate startDate = startOfYear.plusMonths(finalI);
                    LocalDate endDate = startDate.with(TemporalAdjusters.lastDayOfMonth());
                    // 构建当前月份的日历
                    List<ScpStoreDemandCalendarDO> scpStoreDemandCalendarDOS = CalendarBuilder.buildCalendarSetList(startDate, endDate, computeParamVO, deliveryType, type);
                    // 批量插入日历
                    scpStoreDemandCalendarDomainService.batchInsert(scpStoreDemandCalendarDOS, 10000);
                    return "OK";
                } catch (Exception e) {
                    // 如果发生异常，回滚事务并抛出业务异常
                    trans.setRollbackOnly();
                    throw new BusinessException(ApiCode.FAIL, e.getMessage());
                }
            });
            log.info("结束写入:{}月份区域/门店订货日历", i + 1);
        }
    }

}
