package com.elitesland.scp.application.service.mrp;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitesland.scp.application.facade.vo.param.item.ScpMrpDImportSaveVO;
import com.elitesland.scp.application.facade.vo.param.mrp.ScpMrpDImportEntity;
import com.elitesland.scp.application.facade.vo.param.mrp.ScpThousandUseParam;
import com.elitesland.scp.application.facade.vo.param.mrp.ScpThousandUseRefParam;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpThousandUseRefExportVO;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpThousandUseRefVO;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpThousandUseVO;
import com.elitesland.scp.enums.MrpStoreTypeEnum;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemRpcDtoParam;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Component
@Slf4j
@RequiredArgsConstructor
public class ScpThousandUseImportServiceImpl implements DataImport<ScpThousandUseRefExportVO> {
    private static final String LINE = "\n";
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 校验异常: {2}; ";

    private final TransactionTemplate transactionTemplate;
    private final ScpThousandUseService scpThousandUseService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RmiItemService rmiItemService;

    @Override
    public Set<Integer> sheetNoList() {
        return Collections.singleton(1);
    }

    @Override
    public Integer stepSize() {
        return 1000000;
    }

    @Override
    public String getTmplCode() {
        return "yst_scp_mrp_thousand_use";
    }

    @Override
    public List<String> executeImport(List<ScpThousandUseRefExportVO> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errors = new ArrayList<>();

        try {
            List<String> errorMsg = checkAndSaveData(dataList, startRowIndex);
            if (CollectionUtil.isNotEmpty(errorMsg)) {
                log.info("千元用量导入完成,错误信息如下:{}", JSONUtil.toJsonStr(errorMsg));
            } else {
                log.info("千元用量导入完成,未发生错误");
            }
            return errorMsg;
        } catch (Exception e) {
            for (ScpThousandUseRefExportVO entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
    }

    private List<String> checkAndSaveData(List<ScpThousandUseRefExportVO> dataList, int startRowIndex) {

        Map<Integer, String> errMsgMap = new LinkedHashMap<>();
        Integer index = 0;
        //初始化行号，和错误信息容器
        for (ScpThousandUseRefExportVO entity : dataList) {
            entity.setRowNo(index);
            errMsgMap.put(index, null);
            index++;
        }

        log.info("startRowIndex的值:{}", startRowIndex);
        // 记录明细有错误的明细id
        Set<String> errorIdSet = new HashSet<>();
        List<ScpThousandUseParam> dataToSave = prepareData(dataList, errMsgMap, errorIdSet);
        log.info("千元用量导入，准备好的数据:{},错误信息:{}", JSONUtil.toJsonStr(dataToSave), JSONUtil.toJsonStr(errMsgMap));
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        //一个一个保存
        for (ScpThousandUseParam vo : dataToSave) {
            transactionTemplate.execute(
                    transactionStatus -> {
                        try {
                           // 保存数据
                            scpThousandUseService.save(vo);
                        } catch (Exception e) {
                            log.error("MRP计划导入报错：" + e);
                            // 回滚数据
                            transactionStatus.setRollbackOnly();
                        }
                        return null;
                    }
            );
        }
   //     long docNumber = dataList.stream().map(ScpMrpDImportEntity::getId).distinct().count();
//        String orderMsg = String.format("文件共包含%d条数据，其中导入成功%d单，失败%d单", dataList.size(), docNumber,
//                docNumber - errorIdSet.size(),
//                errorIdSet.size());

        log.info("写入自定义提示语:{}", "导入成功");
        HttpServletUtil.currentRequest().setAttribute("orderMsg", "导入成功");
        return new ArrayList<>(errMsgMap.values());
    }

    private List<ScpThousandUseParam> prepareData(List<ScpThousandUseRefExportVO> dataList,Map<Integer, String> errMsgMap,Set<String> errorIdSet){
        Map<String, List<ScpThousandUseRefExportVO>> exportMap = dataList.stream().collect(Collectors.groupingBy(s->s.getStWhCode()+","+s.getReferenceDataStart()+","+s.getReferenceDataEnd()+","+s.getValidDataStart()+","+s.getValidDataEnd()));

        Set<String> storeCodes = new HashSet<>();
        Set<String> itemCodeList = new HashSet<>();
        dataList.stream().forEach(p->{
            if(StringUtils.hasText(p.getStWhCode()) && StringUtils.hasText(p.getItemCode()) ){
                storeCodes.add(p.getStWhCode());
                itemCodeList.add(p.getItemCode());
            }
        });
        // 查询门店信息
        log.info("门店编码集合:{}", JSONUtil.toJsonStr(storeCodes));
        List<OrgStoreDetailRpcDTO> queryByStoreCodes = rmiOrgStoreRpcService.queryByStoreCodes(new ArrayList<>(storeCodes));
        Map<String, OrgStoreDetailRpcDTO> storeMap =
                queryByStoreCodes.stream().collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode, Function.identity()));
        // 查询商品信息
        ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
        itmItemRpcDtoParam.setItemCodes(new ArrayList<>(itemCodeList));
        List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
        Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));
        List<ScpThousandUseParam> paramList = new ArrayList<>();
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        exportMap.forEach((k,v)->{
            ScpThousandUseParam param = new ScpThousandUseParam();
            String[] split = k.split(",");
            if(StringUtils.hasText(split[0]) && storeMap.containsKey(split[0])){
                OrgStoreDetailRpcDTO orgStoreDetailRpcDTO = storeMap.get(split[0]);
                param.setStWhId(orgStoreDetailRpcDTO.getId());
                param.setStWhCode(split[0]);
                param.setStWhName(orgStoreDetailRpcDTO.getStoreName());
                String storeType1 = v.get(0).getStWhType();
                if(StringUtils.hasText(storeType1) && StringUtils.hasText(storeType1.trim())){
                    String storeType = "";
                    for(MrpStoreTypeEnum mrpStoreTypeEnum:MrpStoreTypeEnum.values()){
                        if(mrpStoreTypeEnum.getDesc().equals(storeType1.trim())){
                            storeType = mrpStoreTypeEnum.getCode();
                        }
                    }
                    param.setStWhType(storeType);
                }
                try {
                    param.setReferenceDataStart(simpleDateFormat.parse(split[1]));
                    param.setReferenceDataEnd(simpleDateFormat.parse(split[2]));
                    param.setValidDataStart(simpleDateFormat.parse(split[3]));
                    param.setValidDataEnd(simpleDateFormat.parse(split[4]));
                } catch (ParseException e) {
                  return;
                }
                List<ScpThousandUseRefParam> refParams = new ArrayList<>();
               v.forEach(p->{
                   if(itemRpcDTOMap.containsKey(p.getItemCode())){
                       ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(p.getItemCode());
                       ScpThousandUseRefParam refParam = new ScpThousandUseRefParam();
                       refParam.setItemId(itmItemRpcDTO.getId());
                       refParam.setUom2(itmItemRpcDTO.getUom2Name());
                       refParam.setItemName(itmItemRpcDTO.getItemName());
                       refParam.setItemAttr(itmItemRpcDTO.getSpec());
                       refParam.setItemType(itmItemRpcDTO.getItemCateFullName());
                       refParam.setItemCode(p.getItemCode());
                       refParam.setUom(itmItemRpcDTO.getUomName());
                       refParam.setAdjustThousNum(p.getAdjustThousNum());
                       refParam.setEstimateThousNum(p.getEstimateThousNum());
                       refParam.setExt1(itmItemRpcDTO.getBrandName());
                       refParams.add(refParam);
                   }
               });
               if(!CollectionUtils.isEmpty(refParams)){
                   param.setScpThousandUseRefParamList(refParams);
               }
            }
            paramList.add(param);
        });
        return paramList;
    }



    public static void addErrMsg(Map<Integer, String> errMsgMap, Integer rowNo, String msg) {
        addErrMsg(errMsgMap, Collections.singletonList(rowNo), msg);
    }

    public static void addErrMsg(Map<Integer, String> errMsgMap, List<Integer> rowNos, String msg) {
        for (Integer rowNo : rowNos) {
            if (errMsgMap.containsKey(rowNo) && errMsgMap.get(rowNo) != null) {
                String newMsg = errMsgMap.get(rowNo) + LINE + msg;
                errMsgMap.put(rowNo, newMsg);
            } else {
                errMsgMap.put(rowNo, msg);
            }
        }
    }
}
