package com.elitesland.scp.application.service.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderImportEntity;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderImportSaveVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderItemBatchParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderItemParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderDRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetSelectRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDMgmtSaveVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderSaveVO;
import com.elitesland.scp.application.facade.vo.serviceconfig.ScpServiceConfigRespVO;
import com.elitesland.scp.application.service.UserService;
import com.elitesland.scp.application.service.serviceconfig.ScpServiceConfigService;
import com.elitesland.scp.common.CurrentUserDTO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDO;
import com.elitesland.scp.domain.entity.scpsman.ScpsmanInfoDO;
import com.elitesland.scp.domain.service.order.ScpDemandSetDomainService;
import com.elitesland.scp.domain.service.setting.ScpOrderSettingDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepo;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.scp.utils.ValidatedList;
import com.elitesland.support.provider.item.dto.ItmItemBusinessRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.MessageFormat;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Auther: ryan.xu
 * @Date: 2024/9/10 17:01
 * @Description:
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class ScpDemandOrderImportService implements DataImport<ScpDemandOrderImportEntity> {


    private static final String LINE = "\n";
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 校验异常: {2}; ";

    private final TransactionTemplate transactionTemplate;
    private final ScpDemandOrderService scpDemandOrderService;
    private final ScpDemandOrderDService scpDemandOrderDService;
    private final ScpOrderSettingDomainService scpOrderSettingDomainService;
    private final ScpDemandSetDomainService scpDemandSetDomainService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final RmiItemService rmiItemService;
    private final UdcProvider sysUdcService;
    private final ScpDemandOrderSetService scpDemandOrderSetService;
    private final ScpServiceConfigService scpServiceConfigService;
    private final ScpsmanInfoRepo scpsmanInfoRepo;

    @Override
    public Set<Integer> sheetNoList() {
        return Collections.singleton(1);
    }

    @Override
    public Integer stepSize() {
        return 1000000;
    }

    @Override
    public String getTmplCode() {
        return "yst_scp_demand_order_import";
    }

    @Override
    public List<String> executeImport(List<ScpDemandOrderImportEntity> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errors = new ArrayList<>();

        try {
            List<String> errorMsg = checkAndSaveData(dataList, startRowIndex);
            if (errorMsg != null) {
                log.info("订货单导入完成,错误信息如下:{}", JSONUtil.toJsonStr(errorMsg));
            } else {
                log.info("订货单导入完成,未发生错误");
            }
            return errorMsg;
        } catch (Exception e) {
            for (ScpDemandOrderImportEntity entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
    }

    private List<String> checkAndSaveData(List<ScpDemandOrderImportEntity> dataList, int startRowIndex) {
        Map<Integer, String> errMsgMap = new LinkedHashMap<>();
        Integer index = 0;
        //初始化行号，和错误信息容器
        for (ScpDemandOrderImportEntity entity : dataList) {
            entity.setRowNo(index);
            errMsgMap.put(index, null);
            index++;
        }

        log.info("startRowIndex的值:{}", startRowIndex);
        // 记录明细有错误的订单序号
        Set<String> errorDocNumSet = new HashSet<>();
        List<ScpDemandOrderImportSaveVO> dataToSave = prepareData(dataList, errMsgMap, errorDocNumSet);
        log.info("订货单导入，准备好的数据:{},错误信息:{}", JSONUtil.toJsonStr(dataToSave), JSONUtil.toJsonStr(errMsgMap));
        if (CollUtil.isNotEmpty(dataToSave)) {
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            for (ScpDemandOrderImportSaveVO vo : dataToSave) {
                transactionTemplate.execute(
                        transactionStatus -> {
                            try {
                                ScpDemandOrderDO scpDemandOrderDO = scpDemandOrderService.saveDemandOrder(vo.getHeaderSaveVO());
                                scpDemandOrderDService.batchSaveDemandOrderDMgmt(scpDemandOrderDO.getId(), scpDemandOrderDO.getDemandWhStCode(), vo.getDetailSaveVOs(), null);
                            } catch (Exception e) {
                                log.error("订货单导入报错：{}", e.getMessage(), e);
                                // 回滚数据
                                transactionStatus.setRollbackOnly();
                            }
                            return null;
                        }
                );
            }
            long docNumber = dataList.stream().map(ScpDemandOrderImportEntity::getSeqNo).distinct().count();
            String orderMsg = String.format("文件共包含%d条数据，识别出订货单%d单，其中导入成功%d单，失败%d单", dataList.size(), docNumber,
                    docNumber - errorDocNumSet.size(),
                    errorDocNumSet.size());
            log.info("写入自定义提示语:{}", orderMsg);
            HttpServletUtil.currentRequest().setAttribute("orderMsg", orderMsg);
        }
        return new ArrayList<>(errMsgMap.values());
    }

    private List<ScpDemandOrderImportSaveVO> prepareData(List<ScpDemandOrderImportEntity> dataList,
                                                         Map<Integer, String> errMsgMap,
                                                         Set<String> errorDocNumSet) {
        List<ScpDemandOrderImportSaveVO> resultList = new ArrayList<>();

        for (ScpDemandOrderImportEntity data : dataList) {
            if (Strings.isBlank(data.getSeqNo())) {
                addErrMsg(errMsgMap, data.getRowNo(), "序号不能为空");
                return null;
            }
        }

        // 获取当前用户类型
        //查询当前登录用户
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (currentUserDTO == null) {
            throw new BusinessException("当前登录人查询异常");
        }
        // 查询订货账号类型
        ScpsmanInfoDO byLoginAccount = scpsmanInfoRepo.findByLoginAccount(currentUserDTO.getDetail().getUsername());
        log.info("当前登录人订货账号:{}", JSONUtil.toJsonStr(byLoginAccount));
        if (byLoginAccount == null) {
            throw new BusinessException("当前登录人订货账号查询异常");
        }
        String scpManType = byLoginAccount.getScpsmanType();
        Map<String, List<ScpDemandOrderImportEntity>> seqNoMap = dataList.stream().collect(Collectors.groupingBy(ScpDemandOrderImportEntity::getSeqNo));

        Map<String, String> uomMap = sysUdcService.getValueMapByUdcCode("yst-supp", "UOM");
        Set<String> docTypeCodeList = new HashSet<>();
        dataList.forEach(entity -> {
            Optional.ofNullable(entity.getDocTypeCode()).ifPresent(docTypeCodeList::add);
        });

        List<ScpOrderSettingRespVO> enabledSetting = scpOrderSettingDomainService.findCacheByBusinessType(new ArrayList<>(docTypeCodeList), ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode());
        Map<String, ScpOrderSettingRespVO> settingMap = enabledSetting.stream().collect(Collectors.toMap(ScpOrderSettingRespVO::getBusinessType, Function.identity()));


        for (Map.Entry<String, List<ScpDemandOrderImportEntity>> dataEntry : seqNoMap.entrySet()) {
            ScpDemandOrderImportSaveVO scpDemandOrderImportSaveVO = new ScpDemandOrderImportSaveVO();
            ScpDemandOrderSaveVO result = new ScpDemandOrderSaveVO();
            List<ScpDemandOrderImportEntity> importEntityList = dataEntry.getValue();
            ScpDemandOrderImportEntity header = importEntityList.get(0);
            List<Integer> rowNoList = importEntityList.stream().map(ScpDemandOrderImportEntity::getRowNo).collect(Collectors.toList());

            // 设置默认订货集
            ScpDemandSetSelectRespVO scpDemandSetDTO = scpDemandOrderService.getScpDemandSetSelectRespVO(ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode(), header.getDemandWhStCode());
            result.setDemandId(scpDemandSetDTO.getId());
            result.setDemandCode(scpDemandSetDTO.getDemandCode());
            result.setDemandName(scpDemandSetDTO.getDemandName());
            result.setDemandDate(scpDemandSetDTO.getDemandDate());
            result.setType(scpDemandSetDTO.getType());
            result.setDemandWhStCode(header.getDemandWhStCode());
            result.setDemandWhStName(header.getDemandWhStName());
            String docCls = ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode().equals(scpDemandSetDTO.getType()) ? ScpUdcEnum.OB_DOC_CLS_ST.getValueCode() : ScpUdcEnum.OB_DOC_CLS_WH.getValueCode();
            result.setDocCls(docCls);
            //单据类型配置
            ScpOrderSettingRespVO scpOrderSettingRespVO = settingMap.get(header.getDocTypeCode());
            if (scpOrderSettingRespVO == null) {
                addErrMsg(errMsgMap, rowNoList, "单据类型:" + header.getDocTypeCode() + "不存在");
                errorDocNumSet.add(dataEntry.getKey());
                continue;
            }

            String businessType = scpOrderSettingRespVO.getBusinessType();
            if (Objects.equals(UdcEnum.SALE_SCPSMAN_TYPE_PREPARATION.getValueCode(), scpManType)) {
                //        1、筹建只能下【新店首批设备】类型的代客下单
                if (!Objects.equals(businessType, ScpUdcEnum.BUSINESS_TYPE_20.getValueCode())) {
                    addErrMsg(errMsgMap, rowNoList, "筹建只能下【首批设备】的单据类型");
                    errorDocNumSet.add(dataEntry.getKey());
                    continue;
                }
            } else if (Objects.equals(UdcEnum.SALE_SCPSMAN_TYPE_STOREGROUP.getValueCode(), scpManType)) {
                //        2、门店组只能下【新店首批物料】类型的代客下单
                if (!Objects.equals(businessType, ScpUdcEnum.BUSINESS_TYPE_10.getValueCode())) {
                    addErrMsg(errMsgMap, rowNoList, "门店组只能下【新店首批物料】的单据类型");
                    errorDocNumSet.add(dataEntry.getKey());
                    continue;
                }
            } else if (Objects.equals(UdcEnum.SALE_SCPSMAN_TYPE_SCPSMAN.getValueCode(), scpManType)) {
                //        3、督导只能下【老店增补】类型的代客下单
                if (!Objects.equals(businessType, ScpUdcEnum.BUSINESS_TYPE_30.getValueCode())) {
                    addErrMsg(errMsgMap, rowNoList, "督导只能下【老店增补】的单据类型");
                    errorDocNumSet.add(dataEntry.getKey());
                    continue;
                }
            }
            result.setDocType(businessType);
            result.setBusinessType(businessType);

            List<OrgStoreBaseRpcDTO> orgStores = rmiOrgStoreRpcService.findOrgStoreBaseByParam(List.of(result.getDemandWhStCode()));
            if (CollUtil.isEmpty(orgStores)) {
                addErrMsg(errMsgMap, rowNoList, "门店编码:" + result.getDemandWhStCode() + "不存在");
                errorDocNumSet.add(dataEntry.getKey());
                continue;
            }
            OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = orgStores.get(0);

            Map<String, ScpServiceConfigRespVO> configRespVOMap = scpServiceConfigService.findServiceConfigGroupByStore(orgStoreBaseRpcDTO.getStoreCode(),
                    orgStoreBaseRpcDTO.getRegion(), orgStoreBaseRpcDTO.getStoreLevel());
            log.info("订货单导入的configRespVOMap:{}", JSONUtil.toJsonStr(configRespVOMap));
            if (configRespVOMap == null) {
                addErrMsg(errMsgMap, rowNoList, "门店:" + result.getDemandWhStCode() + "信息未完善");
                errorDocNumSet.add(dataEntry.getKey());
                continue;
            }

            result.setDemandWhStId(orgStoreBaseRpcDTO.getId());
            result.setDemandWhStName(orgStoreBaseRpcDTO.getStoreName());
            result.setStoreLevel(orgStoreBaseRpcDTO.getStoreLevel());
            result.setStoreType2(orgStoreBaseRpcDTO.getStoreType2());
            result.setOuCode(orgStoreBaseRpcDTO.getOuCode());
            result.setBrandCode(orgStoreBaseRpcDTO.getBrandCode());
            scpDemandOrderImportSaveVO.setHeaderSaveVO(result);

            List<String> itemCodes = importEntityList.stream().map(ScpDemandOrderImportEntity::getItemCode).distinct().collect(Collectors.toList());
            List<ItmItemBusinessRpcDTO> itmItemBusinessList =
                    rmiItemService.findItmItemBusinessByBUCodeAndBrandAndItemList(result.getOuCode(), result.getBrandCode(), itemCodes);
            Map<String, List<ItmItemBusinessRpcDTO>> itemMap = itmItemBusinessList.stream().collect(Collectors.groupingBy(ItmItemBusinessRpcDTO::getItemCode));

            // 原子integer
            AtomicBoolean isValid = new AtomicBoolean(true);
            List<ScpDemandOrderItemParamVO> paramVOList = new ArrayList<>();
            for (ScpDemandOrderImportEntity scpDemandOrderImportEntity : importEntityList) {
                // 校验需求数量
                if (scpDemandOrderImportEntity.getDemandQuantity() == null) {
                    addErrMsg(errMsgMap, scpDemandOrderImportEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "需求数量", "需求数量不能为空"));
                    isValid.getAndSet(false);
                    continue;
                } else if (scpDemandOrderImportEntity.getDemandQuantity().compareTo(BigDecimal.ZERO) <= 0) {
                    addErrMsg(errMsgMap, scpDemandOrderImportEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "需求数量", "需求数量不能小于0"));
                    isValid.getAndSet(false);
                    continue;
                }

                List<ItmItemBusinessRpcDTO> businessItems = itemMap.get(scpDemandOrderImportEntity.getItemCode());
                if (CollUtil.isEmpty(businessItems)) {
                    addErrMsg(errMsgMap, scpDemandOrderImportEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "商品", "商品编码:" + scpDemandOrderImportEntity.getItemCode() + "商品信息不存在"));
                    isValid.getAndSet(false);
                    continue;
                }
                ItmItemBusinessRpcDTO businessItem = businessItems.get(0);
                ScpDemandOrderItemBatchParamVO.Item item = new ScpDemandOrderItemBatchParamVO.Item();
                item.setItemId(businessItem.getId());
                item.setItemCode(businessItem.getItemCode());
                item.setItemName(businessItem.getItemName());
                item.setItemCateCode(businessItem.getItemCateCode());
                item.setSpuId(businessItem.getSpuId());
                item.setSpuItemCode(businessItem.getSpuCode());
                item.setSpuItemName(businessItem.getSpuName());
                item.setUom(businessItem.getUom2());
                item.setUomName(uomMap.get(businessItem.getUom2()));
                item.setDemandQuantity(scpDemandOrderImportEntity.getDemandQuantity());
                item.setQty2(SysUtils.processQtyScale(scpDemandOrderImportEntity.getDemandQuantity().multiply(businessItem.getUomRatio2())));
                item.setUom2(businessItem.getUom());
                item.setUom2Name(uomMap.get(businessItem.getUom()));
                item.setDecimalPlaces(businessItem.getDecimalPlaces());
                item.setUomRatio(businessItem.getUomRatio2());
                item.setItemType(businessItem.getItemType2());

                ScpDemandOrderItemParamVO paramVO = new ScpDemandOrderItemParamVO();
                paramVO.setItemCode(item.getItemCode());
                paramVO.setItemName(item.getItemName());
                paramVO.setItemId(item.getItemId());
                paramVO.setSpuId(item.getSpuId());
                paramVO.setSpuItemCode(item.getSpuItemCode());
                paramVO.setSpuItemName(item.getSpuItemName());
                paramVO.setUnit(item.getUom2());
                paramVO.setUnitName(item.getUom2Name());
                paramVO.setDemandQuantity(item.getDemandQuantity());
                paramVO.setUom2(item.getUom());
                paramVO.setUom2Name(item.getUomName());
                paramVO.setUomRatio(item.getUomRatio());
                paramVO.setDecimalPlaces(item.getDecimalPlaces());
                paramVO.setItemType(item.getItemType());
                paramVO.setDeliveryType(item.getItemType());
                paramVO.setDemandWhStCode(result.getDemandWhStCode());
                paramVO.setDemandDate(result.getDemandDate());
                paramVO.setType(result.getType());
                paramVO.setBusinessType(result.getBusinessType());
                paramVOList.add(paramVO);
            }
            if (!isValid.get()) {
                errorDocNumSet.add(dataEntry.getKey());
                continue;
            }

            List<ScpDemandOrderDRespVO> demandOrderDRespVOS = new ArrayList<>(scpDemandOrderService.whnetStrategy(paramVOList, orgStoreBaseRpcDTO, scpOrderSettingRespVO));
            if (demandOrderDRespVOS.isEmpty()) {
                addErrMsg(errMsgMap, rowNoList, "商品仓网供应关系或者供应商份额获取失败");
                errorDocNumSet.add(dataEntry.getKey());
                continue;
            }
            log.info("获取商品仓网关系成功:{}", JSONUtil.toJsonStr(demandOrderDRespVOS));
            // 获取仓网等商品详细信息
            Map<String, List<ScpDemandOrderDRespVO>> itemListBatchMap = demandOrderDRespVOS.stream().collect(Collectors.groupingBy(ScpDemandOrderDRespVO::getItemCode));

            Integer pricePlace = SysUtils.getAppPricePlace();

            for (ScpDemandOrderImportEntity importEntity : importEntityList) {
                if (Strings.isBlank(importEntity.getDiscountType())) {
                    addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "折扣方式", "折扣方式不能为空"));
                    isValid.getAndSet(false);
                    continue;
                }

                if ("1".equals(importEntity.getDiscountType())) {
                    if (importEntity.getDiscount() == null) {
                        addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "折扣比例", "商品编码【" + importEntity.getItemCode() + "】,折扣比例不能为空"));
                        isValid.getAndSet(false);
                        continue;
                    }
                    if (importEntity.getDiscount().compareTo(new BigDecimal("100")) > 0) {
                        addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "折扣比例", "商品编码【" + importEntity.getItemCode() + "】,折扣比例不能大于1"));
                        isValid.getAndSet(false);
                        continue;
                    }
                    if (importEntity.getDiscount().compareTo(BigDecimal.ZERO) <= 0) {
                        addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "折扣比例", "商品编码【" + importEntity.getItemCode() + "】,折扣比例不能小于等于0"));
                        isValid.getAndSet(false);
                        continue;
                    }
                } else if ("2".equals(importEntity.getDiscountType())) {
                    if (importEntity.getSettlementPrice() == null) {
                        addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "折后货款总单价", "商品编码【" + importEntity.getItemCode() + "】,折后货款总单价不能为空"));
                        isValid.getAndSet(false);
                        continue;
                    }
                }

                List<ScpDemandOrderDRespVO> orderDRespVOS = itemListBatchMap.get(importEntity.getItemCode());
                if (CollUtil.isEmpty(orderDRespVOS)) {
                    addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "商品供货关系", "商品供货关系不存在,请联系管理员维护"));
                    isValid.getAndSet(false);
                    continue;
                }

                for (ScpDemandOrderDRespVO row : orderDRespVOS) {
                    if (row.getPrice() == null) {
                        addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "价格", "商品编码【" + row.getItemCode() + "】,未配置结算价格信息"));
                        isValid.getAndSet(false);
                        continue;
                    }
                    BigDecimal price = row.getPrice();
                    if (importEntity.getDiscountType() != null) {
                        if ("1".equals(importEntity.getDiscountType())) {
                            if (importEntity.getDiscount() != null) {
                                BigDecimal newSettlePrice = price
                                        .multiply(importEntity.getDiscount().divide(new BigDecimal("100"), pricePlace, RoundingMode.HALF_UP))
                                        .setScale(2, RoundingMode.HALF_UP);
                                row.setSettlementPrice(newSettlePrice);
                                row.setSettlementSalePrice(null);
                                row.setDiscount(importEntity.getDiscount());
                                log.info("商品编码【{}】,折扣前价格【{}】,折扣方式【{}】,折扣比例【{}】,折扣后的结算价【{}】",
                                        row.getItemCode(),
                                        price,
                                        importEntity.getDiscountType(),
                                        importEntity.getDiscount(),
                                        newSettlePrice);
                            }
                        } else if ("2".equals(importEntity.getDiscountType())) {
                            log.info("打折方式：{},结算单价{}", importEntity.getDiscountType(), importEntity.getSettlementPrice());
                            if (importEntity.getSettlementPrice() != null) {
                                BigDecimal settlePrice = importEntity.getSettlementPrice();
                                // 改价，修改map里的结算价，为后续计算价格做准备
                                log.info("商品编码【{}】,已配置结算价格信息，已修改为：{}", row.getItemCode(), settlePrice);
                                row.setSettlementPrice(settlePrice);
                                row.setSettlementSalePrice(null);
                            }
                        }

                        if (("1".equals(importEntity.getDiscountType()) || "2".equals(importEntity.getDiscountType())) && price.subtract(row.getSettlementPrice()).compareTo(BigDecimal.ONE) < 0) {
                            addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "价格", "商品编码【" + row.getItemCode() + "】,折后单价应低于原单价，且折扣差额需大于1元"));
                            isValid.getAndSet(false);
                            continue;
                        }
                    }

                    if (ScpConstant.PUR_DELIVERY_TYPES.contains(row.getDeliveryType())) {
                        if (row.getSuppId() == null || row.getSuppCode() == null) {
                            addErrMsg(errMsgMap, importEntity.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, header.getRowNo(), "供应商", "直送商品编码【" + row.getItemCode() + "】未获取到供应商"));
                            isValid.getAndSet(false);
                        }
                    }
                }
            }
            if (!isValid.get()) {
                errorDocNumSet.add(dataEntry.getKey());
                continue;
            }

            List<ScpDemandOrderDMgmtSaveVO> mgmtSaveVOS = new ArrayList<>();
            for (Map.Entry<String, List<ScpDemandOrderDRespVO>> entry : itemListBatchMap.entrySet()) {
                List<ScpDemandOrderDRespVO> value = entry.getValue();
                ScpDemandOrderDRespVO scpDemandOrderDRespVO = value.get(0);
                ScpDemandOrderDMgmtSaveVO saveVO = new ScpDemandOrderDMgmtSaveVO();
                saveVO.setSpuItemName(scpDemandOrderDRespVO.getSpuItemName());
                saveVO.setSpuItemCode(scpDemandOrderDRespVO.getSpuItemCode());
                saveVO.setItemId(scpDemandOrderDRespVO.getItemId());
                saveVO.setItemCode(scpDemandOrderDRespVO.getItemCode());
                saveVO.setItemName(scpDemandOrderDRespVO.getItemName());
                saveVO.setItemType(scpDemandOrderDRespVO.getItemType());
                saveVO.setDeliveryType(scpDemandOrderDRespVO.getDeliveryType());
                saveVO.setDemandQuantity(scpDemandOrderDRespVO.getDemandQuantity());
                saveVO.setCurrency("CNY");

                ValidatedList<ScpDemandOrderDMgmtSaveVO.SupplyItem> supplyItemList = value.stream().map(row -> {
                    ScpDemandOrderDMgmtSaveVO.SupplyItem supplyItem = new ScpDemandOrderDMgmtSaveVO.SupplyItem();
                    supplyItem.setRatio(row.getRatio());
                    supplyItem.setAllocationDeQuantity(row.getAllocationDeQuantity());
                    supplyItem.setAllocationQuantity(row.getAllocationQuantity());
                    supplyItem.setSupplyType(row.getSupplyType());
                    supplyItem.setSuppWhId(row.getSuppWhId());
                    supplyItem.setSuppWhCode(row.getSuppWhCode());
                    supplyItem.setSuppWhName(row.getSuppWhName());
                    supplyItem.setSuppId(row.getSuppId());
                    supplyItem.setSuppCode(row.getSuppCode());
                    supplyItem.setSuppName(row.getSuppName());
                    supplyItem.setOuId(row.getOuId());
                    supplyItem.setOuCode(row.getOuCode());
                    supplyItem.setOuName(row.getOuName());
                    supplyItem.setIsCalculated(Boolean.FALSE);
                    supplyItem.setIsPushed(Boolean.FALSE);
                    supplyItem.setUnit(row.getUnit());
                    supplyItem.setUnitName(row.getUnitName());
                    supplyItem.setUom2(row.getUom2());
                    supplyItem.setUom2Name(row.getUom2Name());
                    supplyItem.setUomRatio(row.getUomRatio());
                    supplyItem.setDecimalPlaces(row.getDecimalPlaces());
                    supplyItem.setQty2(row.getQty2());
                    supplyItem.setIsProjFeeCharged(row.getIsProjFeeCharged());

                    // 图片
                    supplyItem.setImgUrl(row.getImgUrl());

                    // 价格
                    supplyItem.setPrice(row.getPrice());
                    supplyItem.setSettlementPrice(row.getSettlementPrice());
                    supplyItem.setSalePrice(row.getSalePrice());
                    supplyItem.setOefPrice(row.getOefPrice());
                    supplyItem.setMefPrice(row.getMefPrice());
                    supplyItem.setTefPrice(row.getTefPrice());

                    supplyItem.setTefFeeOuCode(row.getTefFeeOuCode());
                    supplyItem.setTefFeeOuName(row.getTefFeeOuName());
                    supplyItem.setOefFeeOuCode(row.getOefFeeOuCode());
                    supplyItem.setOefFeeOuName(row.getOefFeeOuName());
                    supplyItem.setMefFeeOuCode(row.getMefFeeOuCode());
                    supplyItem.setMefFeeOuName(row.getMefFeeOuName());

                    // 折扣比例
                    supplyItem.setDiscount(row.getDiscount());

                    supplyItem.setFreightFlag(row.getFreightFlag());
                    supplyItem.setFreightRatio(row.getFreightRatio());
                    supplyItem.setSaleCustCode(row.getSaleCustCode());
                    supplyItem.setSaleOuName(row.getSaleOuName());
                    supplyItem.setSaleOuCode(row.getSaleOuCode());
                    supplyItem.setAllocationQuantity(row.getAllocationQuantity());
                    return supplyItem;
                }).collect(Collectors.toCollection(ValidatedList::new));
                saveVO.setItemList(supplyItemList);
                mgmtSaveVOS.add(saveVO);
            }
            scpDemandOrderImportSaveVO.setDetailSaveVOs(mgmtSaveVOS);
            resultList.add(scpDemandOrderImportSaveVO);
        }
        return resultList;
    }


    public static void addErrMsg(Map<Integer, String> errMsgMap, Integer rowNo, String msg) {
        addErrMsg(errMsgMap, Collections.singletonList(rowNo), msg);
    }

    public static void addErrMsg(Map<Integer, String> errMsgMap, List<Integer> rowNos, String msg) {
        for (Integer rowNo : rowNos) {
            if (errMsgMap.containsKey(rowNo) && errMsgMap.get(rowNo) != null) {
                String newMsg = errMsgMap.get(rowNo) + LINE + msg;
                errMsgMap.put(rowNo, newMsg);
            } else {
                errMsgMap.put(rowNo, msg);
            }
        }
    }
}
