package com.elitesland.scp.application.service.app;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysEmployeeDetailDTO;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeePageQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.provider.org.EmployeeRpcService;
import com.elitesland.scp.application.facade.vo.param.app.AppLoginAccountStoreProtocolRespVO;
import com.elitesland.scp.application.facade.vo.param.app.AppStoreProtocolAgreeParamVO;
import com.elitesland.scp.application.facade.vo.param.authority.ScpManAuthorityParam;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpManAuthorityPageRespVO;
import com.elitesland.scp.application.facade.vo.scpsman.SalesmanInfoRespVO;
import com.elitesland.scp.application.service.authority.ScpsmanAuthorityManager;
import com.elitesland.scp.application.service.scpsman.ScpsmanInfoService;
import com.elitesland.scp.application.web.app.enums.LoginAccountType;
import com.elitesland.scp.application.web.app.util.AccountIdentifierUtil;
import com.elitesland.scp.domain.convert.scpsman.SalesmanInfoConvert;
import com.elitesland.scp.domain.entity.scpsman.ScpsmanInfoDO;
import com.elitesland.scp.domain.service.article.ArticleManageService;
import com.elitesland.scp.domain.service.authority.ScpDemandAuthorityService;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepo;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreProtocolSaveRpcParam;
import com.elitesland.support.provider.org.param.OrgStoreRpcParam;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import feign.RequestTemplate;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static com.elitesland.scp.common.ScpConstant.CART_PREFIX;


/**
 * @author jeesie
 * @description:
 * @datetime 2025年 09月 16日 17:24
 * @version: 1.0
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class AppStoreProtocolServiceImpl implements AppStoreProtocolService {

    private final ScpsmanInfoService scpsmanInfoService;

    private final ScpDemandAuthorityService scpDemandAuthorityService;

    private final OrgStoreRpcService rmiOrgStoreRpcService;

    private final EmployeeRpcService employeeRpcService;

    private final RedisTemplate redisClient;

    private final ArticleManageService articleManageService;

    private final TenantDataIsolateProvider tenantDataIsolateProvider;
    private final TenantClientProvider tenantClientProvider;

    private final ScpsmanAuthorityManager scpsmanAuthorityManager;

    private final ScpsmanInfoRepo scpsmanInfoRepo;


    @Value("${app.protocol.tenantCode:}")
    private String tenantCode;

    private static final List<String> SCPSMAN_TYPE_CHECK_LIST = List.of(
            UdcEnum.SALE_SCPSMAN_TYPE_SHOPOWNER.getValueCode(),
            UdcEnum.SALE_SCPSMAN_TYPE_SHOPASSISTANT.getValueCode(),
            UdcEnum.SALE_SCPSMAN_TYPE_CONTRACTOR.getValueCode());

    private static final String ARTICLE_CODE_APP_STORE_PROTOCOL = "小程序登录页协议";
    @Override
    public AppLoginAccountStoreProtocolRespVO queryStoreProtocolByAccount(String account) {
        log.info("开始根据账号查询门店协议，账号: {}", account);
        // 1. 尝试从缓存获取
//        AppLoginAccountStoreProtocolRespVO cachedResult = getFromCache(account);
//        if (cachedResult != null) {
//            return cachedResult;
//        }
        AppLoginAccountStoreProtocolRespVO protocolRespVO = getProtocolRespVO(account);
        final SysTenantDTO sysTenantDTO = TenantClient.getTenant(tenantCode);
        // 验证用户身份后设置租户上下文
        SysTenantDTO sessionTenant = tenantClientProvider.getSessionTenant();
        log.info("当前用户租户信息：{},{}", sessionTenant,sysTenantDTO);
        TenantContextHolder.setCurrentTenant(sysTenantDTO);
        try {
            // 3. 验证账号
            String cleanAccount = account.trim();
            if (StringUtils.isEmpty(cleanAccount)) {
                log.warn("账号为空或无效: {}", account);
                protocolRespVO.setMsg("账号为空或无效");
                return protocolRespVO;
            }
            List<SalesmanInfoRespVO> salesmanInfoRespVOS = getSalesmanInfoRespVOS(cleanAccount, protocolRespVO);
            if (CollectionUtils.isEmpty(salesmanInfoRespVOS)) {
                log.warn("未查询到对应的门店协议信息，账号: {}", cleanAccount);
                protocolRespVO.setMsg("未查询到账号信息");
                return protocolRespVO;
            }
            // 将查询结果设置到返回对象中
            log.info("查询到门店协议订货员信息：{}", salesmanInfoRespVOS);
            processSalesmanInfo(salesmanInfoRespVOS, protocolRespVO);
            //查询订货员工下的门店
            List<OrgStoreBaseRpcDTO> storeBaseRpcDTOS = rmiOrgStoreRpcService.findSimpleByCodes(List.of(account)).computeData();
            if(CollectionUtils.isNotEmpty(storeBaseRpcDTOS)){
                OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = storeBaseRpcDTOS.get(0);
                if(orgStoreBaseRpcDTO != null){
                    protocolRespVO.setStoreCode(orgStoreBaseRpcDTO.getStoreCode());
                    protocolRespVO.setStoreId(orgStoreBaseRpcDTO.getId());
                    protocolRespVO.setProtocolAgreeStatus(orgStoreBaseRpcDTO.getProtocolAgreeStatus());
                    protocolRespVO.setProtocolAgreeUserName(orgStoreBaseRpcDTO.getLegalName());
                    protocolRespVO.setProtocolAgreeUserPhone(orgStoreBaseRpcDTO.getLegalPhone());
                    protocolRespVO.setSuccess(true);
                }else{
                    processStoreInfo(salesmanInfoRespVOS, protocolRespVO);
                }
            }else{
                processStoreInfo(salesmanInfoRespVOS, protocolRespVO);
            }
//            cacheResult(account, protocolRespVO);
            log.info("成功查询到{}条门店协议信息", salesmanInfoRespVOS.size());
        } catch (Exception e) {
            log.error("查询门店协议信息异常，账号: {}", account, e);
            // 可以根据需要设置错误信息
            throw new BusinessException(ApiCode.FAIL, "查询门店协议信息异常");
        }finally {
            TenantContextHolder.clearCurrentTenant();
        }
        return protocolRespVO;
    }

    @Override
    @Transactional
    public String setProtocolAgreeStatus(AppStoreProtocolAgreeParamVO param) {
        log.info("开始设置门店协议同意状态，参数: {}", param);
        try{
            final SysTenantDTO sysTenantDTO = TenantClient.getTenant(tenantCode);
            log.info("当前用户租户信息：{}", sysTenantDTO);
            TenantContextHolder.setCurrentTenant(sysTenantDTO);
            List<OrgStoreBaseRpcDTO> storeBaseRpcDTOS = rmiOrgStoreRpcService.findSimpleByCodes(List.of(param.getStoreCode())).computeData();
            if (CollectionUtils.isEmpty(storeBaseRpcDTOS)) {
                log.warn("门店不存在，门店编码: {}", param.getStoreCode());
                throw new BusinessException("门店不存在");
            }
            Boolean protocolAgreeStatus = storeBaseRpcDTOS.get(0).getProtocolAgreeStatus();
            if(Boolean.TRUE.equals(protocolAgreeStatus)){
                return param.getStoreCode();
            }
            OrgStoreProtocolSaveRpcParam saveRpcParam = new OrgStoreProtocolSaveRpcParam();
            saveRpcParam.setStoreCode(param.getStoreCode());
            saveRpcParam.setProtocolAgreeStatus(Boolean.TRUE);
            saveRpcParam.setProtocolAgreeTime(LocalDateTime.now());
            saveRpcParam.setProtocolAgreeUserName(param.getProtocolAgreeUserName());
            saveRpcParam.setProtocolAgreeUserPhone(param.getProtocolAgreeUserPhone());
            saveProtocol(saveRpcParam);
        }catch (Exception e){
            throw new BusinessException(ApiCode.FAIL, e.getMessage());
        }finally {
            TenantContextHolder.clearCurrentTenant();
        }
        return param.getStoreCode();

    }

    private void saveProtocol(OrgStoreProtocolSaveRpcParam saveRpcParam) {
        log.info("开始设置门店协议同意状态：{}",JSON.toJSONString(saveRpcParam));
        rmiOrgStoreRpcService.saveStoreProtocolStatus(saveRpcParam);
    }

    @Override
    public String serchStoreProtocol() {
        try{
            final SysTenantDTO sysTenantDTO = TenantClient.getTenant(tenantCode);
//            SysTenantDTO sessionTenant = tenantClientProvider.getSessionTenant();
            log.info("当前用户租户信息：{}", sysTenantDTO);
            TenantContextHolder.setCurrentTenant(sysTenantDTO);
            return articleManageService.queryAppStoreProtocolArticle(ARTICLE_CODE_APP_STORE_PROTOCOL);
        }catch (Exception e){
            log.error(e.getMessage());
            throw new BusinessException(ApiCode.FAIL, "查询门店协议信息异常");
        }finally {
            TenantContextHolder.clearCurrentTenant();
        }
//        return tenantDataIsolateProvider.byTenantDirectly(() -> {
//
//
//        },tenantCode);
    }

    private void processStoreInfo(List<SalesmanInfoRespVO> salesmanInfoRespVOS, AppLoginAccountStoreProtocolRespVO protocolRespVO) {
        // 查询权限门店信息
        SalesmanInfoRespVO salesmanInfoRespVO = salesmanInfoRespVOS.get(0);
        Optional<ScpsmanInfoDO> infoDO = scpsmanInfoRepo.findById(salesmanInfoRespVO.getId());
        if(infoDO.isEmpty()){
            protocolRespVO.setSuccess(false);
            protocolRespVO.setMsg("未查询到账号");
            return ;
        }

        Map<String, OrgStoreBaseRpcDTO> storeMap = scpsmanAuthorityManager.queryStoreList(null, infoDO.get(),1);
        log.info("协议门店信息：{}", JSON.toJSONString(storeMap));
        if (storeMap.isEmpty()) {
            protocolRespVO.setSuccess(false);
            protocolRespVO.setMsg("未查询到门店");
            return ;
        }
        Optional<Map.Entry<String, OrgStoreBaseRpcDTO>> firstEntry = storeMap.entrySet().stream()
                .findFirst();
        OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = firstEntry.get().getValue();
        if(orgStoreBaseRpcDTO == null){
            protocolRespVO.setSuccess(false);
            protocolRespVO.setMsg("未查询到门店");
        }
        protocolRespVO.setStoreCode(orgStoreBaseRpcDTO.getStoreCode());
        protocolRespVO.setStoreId(orgStoreBaseRpcDTO.getId());
        protocolRespVO.setProtocolAgreeStatus(orgStoreBaseRpcDTO.getProtocolAgreeStatus());
        protocolRespVO.setProtocolAgreeUserName(orgStoreBaseRpcDTO.getLegalName());
        protocolRespVO.setProtocolAgreeUserPhone(orgStoreBaseRpcDTO.getLegalPhone());
        protocolRespVO.setSuccess(true);
    }

    private void processSalesmanInfo(List<SalesmanInfoRespVO> salesmanInfoRespVOS, AppLoginAccountStoreProtocolRespVO protocolRespVO) {
        if(SCPSMAN_TYPE_CHECK_LIST.contains(salesmanInfoRespVOS.get(0).getScpsmanType())){
            protocolRespVO.setForceCheck(Boolean.TRUE);
        }else{
            protocolRespVO.setForceCheck(Boolean.FALSE);
        }
        SysEmployeeDetailDTO employeeDetailDTO = employeeRpcService.getDetailByCode(salesmanInfoRespVOS.get(0).getScpsmanNo()).computeData();
        protocolRespVO.setLoginName(employeeDetailDTO.getFullName());
        protocolRespVO.setMobile(employeeDetailDTO.getPhone());
        protocolRespVO.setScpmanNo(salesmanInfoRespVOS.get(0).getScpsmanNo());
    }


    private void cacheResult(String account, AppLoginAccountStoreProtocolRespVO result) {
        if (result.getSuccess()) {
            String key = getKey(account);
            try {
                redisClient.opsForValue().set(key, JSON.toJSONString(result));
                log.debug("门店协议信息缓存成功，账号: {}", account);
            } catch (Exception e) {
                log.warn("缓存门店协议信息异常，账号: {}，错误: {}", account, e.getMessage());
                // 缓存异常不应影响主流程
            }
        }
    }

    @NotNull
    private String getKey(String account) {
        String key = "APP_LOGIN_PROTOCOL:" + account;
        return key;
    }

    private List<SalesmanInfoRespVO> getSalesmanInfoRespVOS( String cleanAccount, AppLoginAccountStoreProtocolRespVO protocolRespVO) {
        LoginAccountType loginAccountType = AccountIdentifierUtil.identifyAccountType(cleanAccount);
        List<SalesmanInfoRespVO> salesmanInfoRespVOS = new ArrayList<>();
        switch (loginAccountType) {
            case PHONE_NUMBER:
                log.debug("识别为手机号登录，手机号: {}", cleanAccount);
                salesmanInfoRespVOS = scpsmanInfoService.queryByMobile(cleanAccount);
                protocolRespVO.setAccountType(LoginAccountType.PHONE_NUMBER.name());
                break;
            case EMPLOYEE_ID:
                log.debug("识别为员工账号登录，员工号: {}", cleanAccount);
                salesmanInfoRespVOS = scpsmanInfoService.queryByLoginAccount(cleanAccount);
                protocolRespVO.setAccountType(LoginAccountType.EMPLOYEE_ID.name());
                break;
            case INVALID:
            default:
                log.warn("无法识别的账号格式: {}", cleanAccount);
                // 可以尝试两种方式都查询（根据业务需求）
                salesmanInfoRespVOS = tryBothQueryMethods(cleanAccount, protocolRespVO);
                break;
        }
        return salesmanInfoRespVOS;
    }

    @NotNull
    private AppLoginAccountStoreProtocolRespVO getProtocolRespVO(String account) {
        AppLoginAccountStoreProtocolRespVO protocolRespVO = new AppLoginAccountStoreProtocolRespVO();
        protocolRespVO.setLoginAccount(account);
        protocolRespVO.setProtocolAgreeStatus( false);
        protocolRespVO.setSuccess(false);
        return protocolRespVO;
    }


    private AppLoginAccountStoreProtocolRespVO getFromCache(String account) {
        String key = getKey(account);
        try {
            log.debug("尝试从Redis中获取门店协议信息key，账号: {}", key);
            String cachedJson = (String) redisClient.opsForValue().get(key);
            if (cachedJson != null) {
                log.info("从Redis中获取门店协议信息成功，账号: {}", account);
                return JSON.parseObject(cachedJson, AppLoginAccountStoreProtocolRespVO.class);
            }
        } catch (Exception e) {
            log.warn("从Redis获取数据异常，账号: {}，错误: {}", account, e.getMessage());
            // 缓存异常不应影响主流程
        }
        return null;
    }
    

    private List<SalesmanInfoRespVO> tryBothQueryMethods(String account, AppLoginAccountStoreProtocolRespVO protocolRespVO) {
        List<SalesmanInfoRespVO> result = new ArrayList<>();
        // 先尝试按员工号查询
        result = scpsmanInfoService.queryByLoginAccount(account);
        if (CollectionUtils.isNotEmpty(result)) {
            log.debug("通过员工号查询成功: {}", account);
            protocolRespVO.setLoginAccount(LoginAccountType.EMPLOYEE_ID.name());
            return result;
        }
        // 再尝试按手机号查询
        result = scpsmanInfoService.queryByMobile(account);
        if (CollectionUtils.isNotEmpty(result)) {
            log.debug("通过手机号查询成功: {}", account);
            protocolRespVO.setLoginAccount(LoginAccountType.PHONE_NUMBER.name());
            return result;
        }
        return result;
    }



    public void apply(RequestTemplate template) {
        if (!this.tenantClientProvider.enabledTenant()) {
            template.removeHeader("X-Tenant-Id");
        } else {
            SysTenantDTO tenant = this.tenantClientProvider.getSessionTenant();
            if (tenant == null) {
                template.removeHeader("X-Tenant-Id");
            } else {
                template.header("X-Tenant-Id", new String[]{tenant.getId().toString()});
            }
        }
    }
}
