package com.elitesland.scp.application.service.boh.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.boot.provider.IdFactory;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.inv.dto.boh.StoreOrderDRpcDTO;
import com.elitesland.inv.dto.boh.StoreOrderRpcDTO;
import com.elitesland.inv.dto.boh.StoreRtnApplyResult;
import com.elitesland.inv.dto.invTrn.StoreCanRtnQtyRpcDTO;
import com.elitesland.inv.dto.invTrn.StoreReceiveRpcDTO;
import com.elitesland.inv.dto.invTrn.StoreRtnRpcDTO;
import com.elitesland.inv.provider.StoreOrderTrnProvider;
import com.elitesland.pur.provider.PurRnsProvider;
import com.elitesland.scp.application.facade.vo.boh.*;
import com.elitesland.scp.application.facade.vo.param.order.StoreReceiveSubmitMqParam;
import com.elitesland.scp.application.service.boh.StoreReceiveService;
import com.elitesland.scp.application.service.order.ScpDemandOrderDService;
import com.elitesland.scp.application.service.order.ScpDemandOrderService;
import com.elitesland.scp.domain.entity.storereceive.StoreReceiveDDO;
import com.elitesland.scp.domain.entity.storereceive.StoreReceiveDO;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDomainService;
import com.elitesland.scp.infr.dto.boh.DemandOrderDtl;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepo;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveDRepo;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveRepo;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveRepoProc;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.scp.utils.BeanUtils;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.item.dto.ItmItemLotRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemSimpleRpcDTO;
import com.elitesland.support.provider.item.dto.ItmUomRpcDTO;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgAddressRpcDTO;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.param.OrgAddressRpcDtoParam;
import com.elitesland.support.provider.org.service.OrgAddrRpcService;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.core.task.TaskExecutor;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

@Component
@Slf4j
@RequiredArgsConstructor
public class StoreReceiveServiceImpl implements StoreReceiveService {

    private final StoreReceiveRepo storeReceiveRepo;
    private final StoreReceiveDRepo storeReceiveDRepo;
    private final StoreReceiveRepoProc storeReceiveRepoProc;
    private final OrgOuRpcService orgOuRpcService;
    private final ItmItemRpcService itmItemRpcService;
    private final UdcProvider udcProvider;
    private final OrgAddrRpcService orgAddrRpcService;
    private final StoreOrderTrnProvider storeOrderTrnProvider;
    private final PurRnsProvider purRnsProvider;
    private final TaskExecutor taskExecutor;
    private final MessageQueueTemplate messageQueueTemplate;
    private final RedisTemplate<String, Object> redisTemplate;
    private final ScpDemandOrderService scpDemandOrderService;
    private final ScpDemandOrderDService scpDemandOrderDService;
    private final ScpDemandOrderDomainService scpDemandOrderDomainService;
    private final ScpDemandOrderDDomainService scpDemandOrderDDomainService;
    private final ScpDemandOrderDRepo scpDemandOrderDRepo;
    private final RmiSysUDCService rmiSysUDCService;

    private static final Map<String, String> receiveStatusMap = new HashMap<>();
    private static final Map<String, String> rtnStatusMap = new HashMap<>();

    static {
        rtnStatusMap.put("DR", "退货待审");
        rtnStatusMap.put("CE", "待退货");
        rtnStatusMap.put("CF", "已退货");
        rtnStatusMap.put("CL", "已取消");
        rtnStatusMap.put("APPROVED", "审批通过");

        receiveStatusMap.put("CE", "待收货");
        receiveStatusMap.put("CF", "已收货");
        receiveStatusMap.put("CL", "已取消");
    }

    @Override
    public PagingVO<StoreReceivePageVO> page(StoreReceivePageParam param) {
        if (StrUtil.isNotBlank(param.getItemName()) || StrUtil.isNotBlank(param.getItemCode()) || StrUtil.isNotBlank(param.getItemKeyword())) {
            List<Long> itemIdByCodeName = itmItemRpcService.findItemIdByCodeName(param.getItemCode(), param.getItemName(), param.getItemKeyword());
            if (CollectionUtil.isEmpty(itemIdByCodeName)) {
                return new PagingVO<>();
            }

            if (CollectionUtil.isEmpty(param.getItemIds())) {
                param.setItemIds(itemIdByCodeName);
            } else {
                param.getItemIds().retainAll(itemIdByCodeName);
            }
            if (CollectionUtil.isEmpty(param.getItemIds())) {
                return new PagingVO<>();
            }
        }

        if (CollectionUtil.isNotEmpty(param.getCateIds())) {
            List<Long> allLeafCateId = itmItemRpcService.findAllLeafCateId(param.getCateIds());
            param.setCateIds(allLeafCateId);

            List<Long> itemIdsByCate = storeReceiveRepoProc.findItemIdsByCate(allLeafCateId);
            if (CollectionUtil.isEmpty(itemIdsByCate)) {
                return new PagingVO<>();
            }
            if (CollectionUtil.isEmpty(param.getItemIds())) {
                param.setItemIds(itemIdsByCate);
            } else {
                param.getItemIds().retainAll(itemIdsByCate);
            }
            if (CollectionUtil.isEmpty(param.getItemIds())) {
                return new PagingVO<>();
            }
        }

        if (CollectionUtil.isNotEmpty(param.getItemIds())) {
            List<Long> docId = storeReceiveRepoProc.findDocId(param.getItemIds());
            if (CollectionUtil.isEmpty(docId)) {
                return new PagingVO<>();
            }
            if (CollectionUtil.isEmpty(param.getIds())) {
                param.setIds(docId);
            } else {
                param.getIds().retainAll(docId);
            }
            if (CollectionUtil.isEmpty(param.getIds())) {
                return new PagingVO<>();
            }
        }

        PagingVO<StoreReceivePageVO> page = storeReceiveRepoProc.page(param);
        if (CollectionUtil.isNotEmpty(page.getRecords())) {
            Map<String, String> refundType = rmiSysUDCService.getCodeMap("yst-suplan", "REFUND_TYPE");
            for (StoreReceivePageVO record : page.getRecords()) {
                if (record.getRtType().equals("R")) {
                    record.setStatusName(receiveStatusMap.get(record.getStatus()));
                } else {
                    record.setStatusName(rtnStatusMap.get(record.getStatus()));
                }
                if (record.getDocType().equals("TRN")) {
                    record.setDocTypeName("仓配");
                } else {
                    record.setDocTypeName("供应商直发");
                }
                
                // REFUND_TYPE
                record.setRefundTypeName(refundType.get(record.getRefundType()));
            }
        }

        CompletableFuture<Void> future1 = CompletableFuture.runAsync(() -> {
            List<StoreReceivePageVO> rnsRtnList = page.getRecords().stream().filter(i -> i.getRtType().equals("T") && i.getStatus().equals("DR"))
                    .filter(i -> i.getDocType().equals("GR")).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(rnsRtnList)) {
                List<Long> approvingIds = purRnsProvider.findApproving(rnsRtnList.stream().map(StoreReceivePageVO::getDocId).collect(Collectors.toList())).computeData();
                if (CollectionUtil.isEmpty(approvingIds)) {
                    approvingIds = new ArrayList<>();
                }
                List<Long> finalApprovingIds = approvingIds;
                rnsRtnList.stream().filter(i -> !finalApprovingIds.contains(i.getDocId())).forEach(i -> i.setCanAbort(true));
            }
        }, taskExecutor);

        CompletableFuture<Void> future2 = CompletableFuture.runAsync(() -> {
            List<StoreReceivePageVO> trnRtnList = page.getRecords().stream().filter(i -> i.getRtType().equals("T") && i.getStatus().equals("DR"))
                    .filter(i -> i.getDocType().equals("TRN")).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(trnRtnList)) {
                List<Long> approvingIds = storeOrderTrnProvider.findApproving(trnRtnList.stream().map(StoreReceivePageVO::getDocId).collect(Collectors.toList())).computeData();
                if (CollectionUtil.isEmpty(approvingIds)) {
                    approvingIds = new ArrayList<>();
                }
                List<Long> finalApprovingIds = approvingIds;
                trnRtnList.stream().filter(i -> !finalApprovingIds.contains(i.getDocId())).forEach(i -> i.setCanAbort(true));
            }
        }, taskExecutor);

        CompletableFuture.allOf(future1, future2).join();

        return page;
    }

    @Override
    public PagingVO<StoreReceivePageVO> pageAndDetail(StoreReceivePageParam param) {
        PagingVO<StoreReceivePageVO> page = storeReceiveRepoProc.page(param);
        List<StoreReceivePageVO> list = page.getRecords();
        List<StoreReceivePageVO> result = getStoreReceivePageVOS(list);
        if (!result.isEmpty()) {
            result.forEach(i -> {
                String recvKey = "RECEIVE" + i.getDocId().toString();
                // 检查是否可以确认收货
                i.setCanConfirm(!redisTemplate.hasKey(recvKey) && Objects.equals(i.getStatus(), "CE"));
            });
            page.setRecords(result);
        }
        return page;
    }

    private Map<Long, StoreCanRtnQtyRpcDTO> getCanRtnQtyRpcDTOMap(List<StoreCanRtnQtyRpcDTO> rpcDTOS, String docType) {
        Map<Long, StoreCanRtnQtyRpcDTO> canRtnQtyRpcDTOHashMap;
        if (CollectionUtil.isNotEmpty(rpcDTOS)) {
            canRtnQtyRpcDTOHashMap = rpcDTOS.stream().peek(row -> {
                if ("GR".equals(docType)) {
                    row.setCanRtnQty(row.getCanRtnQty());
                } else {
                    if (row.getCanRtnQty() != null && row.getUomRatio() != null && row.getUomRatio().compareTo(BigDecimal.ZERO) > 0) {
                        row.setCanRtnQty(row.getCanRtnQty().divide(row.getUomRatio(), row.getPlaces(), RoundingMode.HALF_UP));
                    }
                }
                if (row.getStkQty() != null && row.getUomRatio() != null && row.getUomRatio().compareTo(BigDecimal.ZERO) > 0) {
                    row.setStkQty(row.getStkQty().divide(row.getUomRatio(), row.getPlaces(), RoundingMode.HALF_UP));
                }
            }).collect(Collectors.toMap(StoreCanRtnQtyRpcDTO::getId, i -> i));
        } else {
            canRtnQtyRpcDTOHashMap = new HashMap<>();
        }
        return canRtnQtyRpcDTOHashMap;
    }

    @Nullable
    private StoreReceiveDDetailVO getStoreReceiveDDetailVO(StoreReceivePageVO receivePageVO,
                                                           StoreOrderDRpcDTO invTrnDRpcDTO,
                                                           ItmItemLotRpcDTO itmItemSimpleRpcDTO,
                                                           String orderItemSetting,
                                                           Map<Long, ScpDemandOrderDDTO> demandOrderDDTOMap,
                                                           Map<String, Map<String, String>> suppUdcMap,
                                                           Map<Long, List<DemandOrderDtl>> itemImgUrlMap,
                                                           Map<String, Map<String, String>> invUdcMap) {
        // 退货单不需要显示运费行
        if (receivePageVO.getRtType().equals("T") && itmItemSimpleRpcDTO.getItemCode().equals(orderItemSetting)) {
            return null;
        }
        StoreReceiveDDetailVO storeReceiveDDetailVO = new StoreReceiveDDetailVO();
        ScpDemandOrderDDTO demandOrderDDTO = demandOrderDDTOMap.get(invTrnDRpcDTO.getRelateDocDid());
        if (demandOrderDDTO != null) {
            storeReceiveDDetailVO.setCombineItemCode(demandOrderDDTO.getCombineItemCode());
            storeReceiveDDetailVO.setCombineItemName(demandOrderDDTO.getCombineItemName());

            storeReceiveDDetailVO.setReturnQty(demandOrderDDTO.getReturnQty());
            storeReceiveDDetailVO.setCompensateQty(demandOrderDDTO.getCompensateQty());
        }

        storeReceiveDDetailVO.setGiftFlag(invTrnDRpcDTO.getGiftLineFlag());
        storeReceiveDDetailVO.setId(invTrnDRpcDTO.getId());
        storeReceiveDDetailVO.setManuDate(invTrnDRpcDTO.getManuDate());
        storeReceiveDDetailVO.setManuFactory(invTrnDRpcDTO.getManuFactory());
        storeReceiveDDetailVO.setManuFactoryName(invTrnDRpcDTO.getManuFactoryName());
        storeReceiveDDetailVO.setOrderDid(invTrnDRpcDTO.getRelateDocDid());
        storeReceiveDDetailVO.setItemId(invTrnDRpcDTO.getItemId());
        storeReceiveDDetailVO.setPrice(invTrnDRpcDTO.getTotalPrice() == null ? invTrnDRpcDTO.getPrice() : invTrnDRpcDTO.getTotalPrice());
        if (receivePageVO.getRtType().equals("R")) {
            storeReceiveDDetailVO.setLotNo(invTrnDRpcDTO.getInLotNo());
            storeReceiveDDetailVO.setLimit1(invTrnDRpcDTO.getInLimit1());
            if (invTrnDRpcDTO.getOutQty() == null || BigDecimal.ZERO.compareTo(invTrnDRpcDTO.getOutQty()) == 0) {
                return null;
            }
        } else {
            if (receivePageVO.getStatus().equals("CF")) {
                storeReceiveDDetailVO.setLotNo(invTrnDRpcDTO.getOutLotNo());
                storeReceiveDDetailVO.setLimit1(invTrnDRpcDTO.getOutLimit1());
            } else {
                storeReceiveDDetailVO.setLotNo(invTrnDRpcDTO.getLotNo());
                storeReceiveDDetailVO.setLimit1(invTrnDRpcDTO.getOutLimit1());
            }
        }
        if (receivePageVO.getDocType().equals("GR")) {
            storeReceiveDDetailVO.setDeliveryQty(invTrnDRpcDTO.getOutQty());
            storeReceiveDDetailVO.setOrderQty(invTrnDRpcDTO.getQty());
            storeReceiveDDetailVO.setReceiveQty(invTrnDRpcDTO.getInQty());
            storeReceiveDDetailVO.setUom(invTrnDRpcDTO.getUom());
            storeReceiveDDetailVO.setUomName(suppUdcMap.get("UOM").get(invTrnDRpcDTO.getUom()));
        } else {
            if (invTrnDRpcDTO.getQty2() == null) {
                return null;
            }
            storeReceiveDDetailVO.setDeliveryQty(invTrnDRpcDTO.getOutQty2());
            storeReceiveDDetailVO.setOrderQty(invTrnDRpcDTO.getQty2());
            if (receivePageVO.getRtType().equals("R")) {
                storeReceiveDDetailVO.setReceiveQty(invTrnDRpcDTO.getOutQty2());
            } else {
                storeReceiveDDetailVO.setReceiveQty(invTrnDRpcDTO.getInQty2());
            }
            storeReceiveDDetailVO.setUom(invTrnDRpcDTO.getUom2());
            storeReceiveDDetailVO.setUomName(suppUdcMap.get("UOM").get(invTrnDRpcDTO.getUom2()));
        }

        if (itmItemSimpleRpcDTO != null) {
            storeReceiveDDetailVO.setItemCode(itmItemSimpleRpcDTO.getItemCode());
            storeReceiveDDetailVO.setItemName(itmItemSimpleRpcDTO.getItemName());
            storeReceiveDDetailVO.setSpec(itmItemSimpleRpcDTO.getSpec());
            storeReceiveDDetailVO.setLotFlag(itmItemSimpleRpcDTO.getLotFlag());
            List<DemandOrderDtl> demandOrderDtls = itemImgUrlMap.get(itmItemSimpleRpcDTO.getId());
            if (CollectionUtil.isNotEmpty(demandOrderDtls)) {
                storeReceiveDDetailVO.setImageUrl(demandOrderDtls.get(0).getImgUrl());
            }
        }
        storeReceiveDDetailVO.setLimit1Name(invUdcMap.get("STK_ITM_STATUS").get(storeReceiveDDetailVO.getLimit1()));
        return storeReceiveDDetailVO;
    }

    @NotNull
    private List<StoreReceivePageVO> getStoreReceivePageVOS(List<StoreReceivePageVO> list) {
        List<StoreReceivePageVO> result = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(list)) {
            for (StoreReceivePageVO receivePageVO : list) {
                if (receivePageVO.getRtType().equals("R")) {
                    receivePageVO.setStatusName(receiveStatusMap.get(receivePageVO.getStatus()));
                } else {
                    receivePageVO.setStatusName(rtnStatusMap.get(receivePageVO.getStatus()));
                }
                if (receivePageVO.getDocType().equals("TRN")) {
                    receivePageVO.setDocTypeName("仓配");
                } else {
                    receivePageVO.setDocTypeName("供应商直发");
                }

                // 查询商品信息
                ApiResult<StoreOrderRpcDTO> detail = storeOrderTrnProvider.detail(receivePageVO.getDocId(), receivePageVO.getDocType(), receivePageVO.getRtType());
                Map<Long, ScpDemandOrderDDTO> demandOrderDDTOMap = scpDemandOrderDDomainService.findDemandOrderDByMasId(receivePageVO.getOrderId()).stream().collect(Collectors.toMap(ScpDemandOrderDDTO::getId, c -> c));
                if (detail != null && detail.isSuccess() && detail.getData() != null) {
                    // 获取可退数量和库存数量
                    List<StoreCanRtnQtyRpcDTO> rpcDTOS = storeOrderTrnProvider.canRtn(receivePageVO.getDocId(), receivePageVO.getDocType()).computeData();
                    Map<Long, StoreCanRtnQtyRpcDTO> canRtnQtyRpcDTOHashMap = getCanRtnQtyRpcDTOMap(rpcDTOS, receivePageVO.getDocType());
                    StoreOrderRpcDTO storeOrderRpcDTO = detail.getData();

                    // 获取标准行
                    List<StoreOrderDRpcDTO> standard = storeOrderRpcDTO.getInvTrnDRpcDTOS().stream().filter(i -> Objects.equals(i.getLineType(), "standard")).toList();

                    List<Long> itemIds = standard.stream().map(StoreOrderDRpcDTO::getItemId).distinct().collect(Collectors.toList());
                    Map<Long, ItmItemLotRpcDTO> itmItemMap = new HashMap<>();
                    if (CollectionUtil.isNotEmpty(itemIds)) {
                        List<ItmItemLotRpcDTO> simpleItem = itmItemRpcService.findLotRpcDTOS(itemIds);
                        itmItemMap = simpleItem.stream().collect(Collectors.toMap(ItmItemLotRpcDTO::getId, i -> i));
                    }
                    //获取商品图片信息
                    Map<Long, List<DemandOrderDtl>> itemImgUrlMap = getItemImgUrlMap(receivePageVO.getOrderId());
                    Map<String, Map<String, String>> suppUdcMap = udcProvider.getValueMapByUdcCode("yst-supp", Set.of("UOM"));
                    Map<String, Map<String, String>> invUdcMap = udcProvider.getValueMapByUdcCode("yst-inv", Set.of("STK_ITM_STATUS"));
                    //查询运费商品行
                    String orderItemSetting = SysUtils.getOrderItemSetting();

                    List<StoreReceiveDDetailVO> items = new ArrayList<>();

                    BigDecimal totalItemQty = BigDecimal.ZERO;
                    for (StoreOrderDRpcDTO invTrnDRpcDTO : standard) {
                        ItmItemLotRpcDTO itmItemSimpleRpcDTO = itmItemMap.get(invTrnDRpcDTO.getItemId());

                        // 组装商品行信息
                        StoreReceiveDDetailVO storeReceiveDDetailVO = getStoreReceiveDDetailVO(receivePageVO,
                                invTrnDRpcDTO,
                                itmItemSimpleRpcDTO,
                                orderItemSetting,
                                demandOrderDDTOMap,
                                suppUdcMap,
                                itemImgUrlMap,
                                invUdcMap);

                        if (storeReceiveDDetailVO == null) continue;
                        totalItemQty = totalItemQty.add(storeReceiveDDetailVO.getDeliveryQty());

                        // 计算可退数量和库存数量
                        if (CollectionUtil.isNotEmpty(canRtnQtyRpcDTOHashMap)) {
                            StoreCanRtnQtyRpcDTO storeCanRtnQtyRpcDTO = canRtnQtyRpcDTOHashMap.get(storeReceiveDDetailVO.getId());
                            if (storeCanRtnQtyRpcDTO != null) {
                                storeReceiveDDetailVO.setCanRtnQty(storeCanRtnQtyRpcDTO.getCanRtnQty());
                                storeReceiveDDetailVO.setStkQty(storeCanRtnQtyRpcDTO.getStkQty());
                            }
                        }

                        // 获取商品的多单位
                        try {
                            ApiResult<ItmUomRpcDTO> itmUomRpcByItemId = itmItemRpcService.findItmUomRpcByItemId(storeReceiveDDetailVO.getItemId());
                            if (itmUomRpcByItemId != null && itmUomRpcByItemId.isSuccess()) {
                                ItmUomRpcDTO data = itmUomRpcByItemId.getData();
                                Optional.ofNullable(data).ifPresent(itmUomRpcDTO -> {
                                    List<StoreReceiveItemUomDetail> itemUomList = new ArrayList<>();
                                    for (ItmUomRpcDTO.ItemUomDetail itemUomDetail : itmUomRpcDTO.getItemUomList()) {
                                        StoreReceiveItemUomDetail storeReceiveItemUomDetail = new StoreReceiveItemUomDetail();
                                        storeReceiveItemUomDetail.setUomCode(itemUomDetail.getUomCode());
                                        storeReceiveItemUomDetail.setUomName(itemUomDetail.getUomName());
                                        storeReceiveItemUomDetail.setRatio(itemUomDetail.getRatio());
                                        storeReceiveItemUomDetail.setRatioRevert(itemUomDetail.getRatioRevert());
                                        storeReceiveItemUomDetail.setDefaultFlag(itemUomDetail.getDefaultFlag());
                                        storeReceiveItemUomDetail.setDecimalPlaces(data.getDecimalPlaces());
                                        itemUomList.add(storeReceiveItemUomDetail);
                                    }
                                    storeReceiveDDetailVO.setItemUomList(itemUomList);
                                });
                            }
                        } catch (Exception e) {
                            log.error("获取商品单位数据失败", e);
                        }
                        items.add(storeReceiveDDetailVO);
                    }
                    receivePageVO.setTotalItemQty(totalItemQty);

                    // 组合商品处理
                    List<StoreReceiveDDetailVO> resultItems = items.stream().filter(d -> StringUtils.isBlank(d.getCombineItemCode())).collect(Collectors.toList());
                    Map<String, List<StoreReceiveDDetailVO>> subItemMap = items.stream().filter(d -> StringUtils.isNotBlank(d.getCombineItemCode())).collect(Collectors.groupingBy(StoreReceiveDDetailVO::getCombineItemCode));
                    if (!subItemMap.isEmpty()) {
                        subItemMap.forEach((k, v) -> {
                            StoreReceiveDDetailVO combineOrderDRespVO = new StoreReceiveDDetailVO();
                            BeanUtils.copyProperties(v.get(0), combineOrderDRespVO);
                            combineOrderDRespVO.setCombineItemCode(k);
                            combineOrderDRespVO.setCombineItemName(v.get(0).getCombineItemName());
                            combineOrderDRespVO.setSubItemList(v);
                            resultItems.add(combineOrderDRespVO);
                        });
                    }
                    receivePageVO.setItems(resultItems);
                }
                result.add(receivePageVO);
            }
        }
        return result;
    }

    private List<StoreReceivePageVO> getStoreReceivePageVOS2(List<StoreReceivePageVO> list) {
        List<StoreReceivePageVO> result = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(list)) {
            for (StoreReceivePageVO receivePageVO : list) {
                if (receivePageVO.getRtType().equals("R")) {
                    receivePageVO.setStatusName(receiveStatusMap.get(receivePageVO.getStatus()));
                } else {
                    receivePageVO.setStatusName(rtnStatusMap.get(receivePageVO.getStatus()));
                }
                if (receivePageVO.getDocType().equals("TRN")) {
                    receivePageVO.setDocTypeName("仓配");
                } else {
                    receivePageVO.setDocTypeName("供应商直发");
                }

                // 查询商品信息
                ApiResult<StoreOrderRpcDTO> detail = storeOrderTrnProvider.detail(receivePageVO.getDocId(), receivePageVO.getDocType(), receivePageVO.getRtType());
                Map<Long, ScpDemandOrderDDTO> orderDDTOMap = scpDemandOrderDDomainService.findDemandOrderDByMasId(receivePageVO.getOrderId()).stream().collect(Collectors.toMap(ScpDemandOrderDDTO::getId, c -> c));
                if (detail != null && detail.isSuccess() && detail.getData() != null) {
                    StoreOrderRpcDTO storeOrderRpcDTO = detail.getData();
                    List<StoreOrderDRpcDTO> standard = storeOrderRpcDTO.getInvTrnDRpcDTOS().stream().filter(i -> Objects.equals(i.getLineType(), "standard")).toList();
                    List<Long> itemIds = standard.stream().map(StoreOrderDRpcDTO::getItemId).distinct().collect(Collectors.toList());
                    Map<Long, ItmItemLotRpcDTO> itmItemMap = new HashMap<>();
                    if (CollectionUtil.isNotEmpty(itemIds)) {
                        List<ItmItemLotRpcDTO> simpleItem = itmItemRpcService.findLotRpcDTOS(itemIds);
                        itmItemMap = simpleItem.stream().collect(Collectors.toMap(ItmItemLotRpcDTO::getId, i -> i));
                    }
                    //获取商品图片信息
                    Map<Long, List<DemandOrderDtl>> itemImgUrlMap = getItemImgUrlMap(receivePageVO.getOrderId());
                    Map<String, Map<String, String>> suppUdcMap = udcProvider.getValueMapByUdcCode("yst-supp", Set.of("UOM"));
                    Map<String, Map<String, String>> invUdcMap = udcProvider.getValueMapByUdcCode("yst-inv", Set.of("STK_ITM_STATUS"));
                    //查询运费商品行
                    String orderItemSetting = SysUtils.getOrderItemSetting();

                    List<StoreReceiveDDetailVO> items = new ArrayList<>();
                    BigDecimal totalItemQty = BigDecimal.ZERO;
                    for (StoreOrderDRpcDTO invTrnDRpcDTO : standard) {
                        ItmItemLotRpcDTO itmItemSimpleRpcDTO = itmItemMap.get(invTrnDRpcDTO.getItemId());
                        StoreReceiveDDetailVO storeReceiveDDetailVO = getStoreReceiveDDetailVO(receivePageVO, invTrnDRpcDTO, itmItemSimpleRpcDTO, orderItemSetting, orderDDTOMap, suppUdcMap, itemImgUrlMap, invUdcMap);
                        if (storeReceiveDDetailVO == null) continue;
                        totalItemQty = totalItemQty.add(storeReceiveDDetailVO.getDeliveryQty());
                        items.add(storeReceiveDDetailVO);
                    }
                    receivePageVO.setTotalItemQty(totalItemQty);

                    // 获取预计到货时间
                    LocalDateTime etaDate = storeOrderRpcDTO.getEtaDate();
                    receivePageVO.setEtaDate(etaDate);

                    // 获取承运商
                    receivePageVO.setCarrier(storeOrderRpcDTO.getCarrier());
                    receivePageVO.setCarrierCode(storeOrderRpcDTO.getCarrierCode());
                    receivePageVO.setLogisDocNo(storeOrderRpcDTO.getLogisDocNo());

                    // 组合商品处理
                    List<StoreReceiveDDetailVO> resultItems = items.stream().filter(d -> StringUtils.isBlank(d.getCombineItemCode())).collect(Collectors.toList());
                    Map<String, List<StoreReceiveDDetailVO>> subItemMap = items.stream().filter(d -> StringUtils.isNotBlank(d.getCombineItemCode())).collect(Collectors.groupingBy(StoreReceiveDDetailVO::getCombineItemCode));
                    if (!subItemMap.isEmpty()) {
                        for (Map.Entry<String, List<StoreReceiveDDetailVO>> entry : subItemMap.entrySet()) {
                            String k = entry.getKey();
                            List<StoreReceiveDDetailVO> v = entry.getValue();
                            StoreReceiveDDetailVO combineOrderDRespVO = new StoreReceiveDDetailVO();
                            combineOrderDRespVO.setCombineItemCode(k);
                            combineOrderDRespVO.setCombineItemName(v.get(0).getCombineItemName());
                            combineOrderDRespVO.setSubItemList(v);
                            resultItems.add(combineOrderDRespVO);
                        }
                    }

                    // 合并相同商品，如果遇到相同商品，将数量累加到第一个商品上
                    Map<String, StoreReceiveDDetailVO> itemMap = new LinkedHashMap<>();
                    for (StoreReceiveDDetailVO item : resultItems) {
                        String key = item.getItemCode(); // 使用商品编码作为唯一标识
                        if (itemMap.containsKey(key)) {
                            StoreReceiveDDetailVO existingItem = itemMap.get(key);
                            // 累加数量和金额
                            existingItem.setDeliveryQty(getDecimal(existingItem.getDeliveryQty()).add(getDecimal(item.getDeliveryQty())));
                            existingItem.setOrderQty(getDecimal(existingItem.getOrderQty()).add(getDecimal(item.getOrderQty())));
                            existingItem.setReceiveQty(getDecimal(existingItem.getReceiveQty()).add(getDecimal(item.getReceiveQty())));
                            existingItem.setAmt(getDecimal(existingItem.getAmt()).add(getDecimal(item.getAmt())));
                            existingItem.setCanRtnQty(getDecimal(existingItem.getCanRtnQty()).add(getDecimal(item.getCanRtnQty())));
                            existingItem.setStkQty(getDecimal(existingItem.getStkQty()).add(getDecimal(item.getStkQty())));
                            if (item.getSubItemList() != null && !item.getSubItemList().isEmpty()) {
                                // 如果有子项，也需要合并子项
                                if (existingItem.getSubItemList() == null) {
                                    existingItem.setSubItemList(new ArrayList<>());
                                }
                                existingItem.getSubItemList().addAll(item.getSubItemList());
                            }
                            itemMap.put(key, existingItem);
                        } else {
                            itemMap.put(key, item);
                        }
                    }
                    resultItems = new ArrayList<>(itemMap.values());
                    receivePageVO.setItems(resultItems);
                }
                result.add(receivePageVO);
            }
        }
        return result;
    }

    @Override
    public StoreReceiveCategoryVO classifyByWhOrDirect(StoreReceivePageParam param) {
        StoreReceiveCategoryVO storeReceiveCategoryVO = new StoreReceiveCategoryVO();
        ScpDemandOrderDTO demandOrderDTO = scpDemandOrderDomainService.findDemandOrderById(param.getOrderId()).orElseThrow(() -> new BusinessException("订货单不存在"));
        BeanUtils.copyProperties(demandOrderDTO, storeReceiveCategoryVO);

        List<StoreReceivePageVO> list = storeReceiveRepoProc.list(param);
        List<StoreReceivePageVO> result = getStoreReceivePageVOS(list);

        List<StoreReceivePageVO> trnRtnList = new ArrayList<>();
        List<StoreReceivePageVO> rnsRtnList = new ArrayList<>();
        result.forEach(i -> {
            String recvKey = "RECEIVE" + i.getDocId().toString();
            // 检查是否可以确认收货
            if (redisTemplate.hasKey(recvKey) || !Objects.equals(i.getStatus(), "CE")) {
                i.setCanConfirm(false);
            }
            if (i.getDocType().equals("TRN")) {
                trnRtnList.add(i);
            }
            if (i.getDocType().equals("GR")) {
                rnsRtnList.add(i);
            }
        });
        storeReceiveCategoryVO.setWhDeliveryStatus(trnRtnList);
        storeReceiveCategoryVO.setDirectDeliveryStatus(rnsRtnList);
        return storeReceiveCategoryVO;
    }

    @Override
    public StoreReceiveCategoryInfoVO classifyItemByWhOrDirect(StoreReceivePageParam param) {
        StoreReceiveCategoryInfoVO storeReceiveCategoryVO = new StoreReceiveCategoryInfoVO();
        ScpDemandOrderDTO demandOrderDTO = scpDemandOrderDomainService.findDemandOrderById(param.getOrderId()).orElseThrow(() -> new BusinessException("订货单不存在"));
        BeanUtils.copyProperties(demandOrderDTO, storeReceiveCategoryVO);
        List<StoreReceivePageVO> list = storeReceiveRepoProc.list(param);
        List<StoreReceivePageVO> result = getStoreReceivePageVOS2(list);
        boolean canConfirm = false;
        List<StoreReceivePageVO> trnRtnList = new ArrayList<>();
        List<StoreReceivePageVO> rnsRtnList = new ArrayList<>();
        for (StoreReceivePageVO i : result) {
            if (i.getDocType().equals("TRN")) {
                trnRtnList.add(i);
                if (Objects.equals(i.getStatus(), "CE")) {
                    canConfirm = true;
                }
            }
            if (i.getDocType().equals("GR")) {
                rnsRtnList.add(i);
                if (Objects.equals(i.getStatus(), "CE")) {
                    canConfirm = true;
                }
            }
        }
        storeReceiveCategoryVO.setWhDeliveryStatus(trnRtnList);
        storeReceiveCategoryVO.setDirectDeliveryStatus(rnsRtnList);
        storeReceiveCategoryVO.setCanConfirm(canConfirm);
        return storeReceiveCategoryVO;
    }

    private BigDecimal getDecimal(BigDecimal existingItem) {
        return existingItem == null ? BigDecimal.ZERO : existingItem;
    }

    @Override
    public StoreReceiveDetailVO find(Long docId) {
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(docId);
        Assert.notEmpty(receiveDOS, "单据不存在");
        StoreReceiveDO thisDO = receiveDOS.get(0);

        StoreOrderRpcDTO storeOrderRpcDTO = storeOrderTrnProvider.detail(docId, thisDO.getDocType(), thisDO.getRtType()).computeData();
        StoreReceiveDetailVO storeReceiveDetailVO = new StoreReceiveDetailVO();

//        ScpDemandOrderDTO demandOrderDTO = scpDemandOrderDomainService.findDemandOrderById(thisDO.getOrderId()).orElseThrow(() -> new BusinessException("订货单不存在"));
        Map<Long, ScpDemandOrderDDTO> orderDDTOMap = scpDemandOrderDDomainService.findDemandOrderDByMasId(thisDO.getOrderId()).stream().collect(Collectors.toMap(ScpDemandOrderDDTO::getId, c -> c));
        storeReceiveDetailVO.setCreditAccAmt(storeOrderRpcDTO.getCreditAccAmt());
        storeReceiveDetailVO.setFlzhAccAmt(storeOrderRpcDTO.getFlzhAccAmt());
        storeReceiveDetailVO.setStoreAccAmt(storeOrderRpcDTO.getStoreAccAmt());

        // 总退款付款金额
        BigDecimal paymentAmt = getDecimal(storeOrderRpcDTO.getCreditAccAmt()).add(getDecimal(storeOrderRpcDTO.getFlzhAccAmt())).add(getDecimal(storeOrderRpcDTO.getStoreAccAmt()));

        storeReceiveDetailVO.setStoreName(storeReceiveRepoProc.findStoreName(thisDO.getOrderId()));
        assignAddress(storeReceiveDetailVO, storeOrderRpcDTO.getAddrNo()); // 填充地址信息
        storeReceiveDetailVO.setDocId(docId);
        storeReceiveDetailVO.setDocNo(storeOrderRpcDTO.getDocNo());
        storeReceiveDetailVO.setRelatedDocId(thisDO.getRelatedDocId());
        storeReceiveDetailVO.setRelatedDocNo(thisDO.getRelatedDocNo());
        storeReceiveDetailVO.setOrderId(thisDO.getOrderId());
        storeReceiveDetailVO.setOrderNo(thisDO.getOrderNo());

        storeReceiveDetailVO.setRefundType(thisDO.getRefundType());
        storeReceiveDetailVO.setRefundTypeName(thisDO.getRefundTypeName());

        storeReceiveDetailVO.setRealAmt(thisDO.getRealAmt());
        if (thisDO.getDocType().equals("TRN") && thisDO.getRtType().equals("R")) {
            List<OrgOuRpcSimpleDTO> simpleOuDto = orgOuRpcService.findSimpleOuDto(List.of(storeOrderRpcDTO.getFromOuId()));
            if (CollectionUtil.isNotEmpty(simpleOuDto)) {
                storeReceiveDetailVO.setDeliveyOuName(simpleOuDto.get(0).getOuName());
            }
            storeReceiveDetailVO.setDeliveryWhId(storeOrderRpcDTO.getFromWhId());
            storeReceiveDetailVO.setDeliveryWhName(storeOrderRpcDTO.getFromWhName());
            storeReceiveDetailVO.setReceWhName(storeOrderRpcDTO.getToWhName());
            storeReceiveDetailVO.setRecvWhId(storeOrderRpcDTO.getToWhId());
        } else if (thisDO.getDocType().equals("GR") && thisDO.getRtType().equals("R")) {
            storeReceiveDetailVO.setSuppName(storeOrderRpcDTO.getSuppName());
            storeReceiveDetailVO.setReceWhName(storeOrderRpcDTO.getToWhName());
        } else if (thisDO.getDocType().equals("GR") && thisDO.getRtType().equals("T")) {
            storeReceiveDetailVO.setSuppName(storeOrderRpcDTO.getSuppName());
            storeReceiveDetailVO.setDeliveryWhId(storeOrderRpcDTO.getFromWhId());
            storeReceiveDetailVO.setDeliveryWhName(storeOrderRpcDTO.getFromWhName());
        } else {
            List<OrgOuRpcSimpleDTO> simpleOuDto = orgOuRpcService.findSimpleOuDto(List.of(storeOrderRpcDTO.getToOuId()));
            if (CollectionUtil.isNotEmpty(simpleOuDto)) {
                storeReceiveDetailVO.setReceiveOuName(simpleOuDto.get(0).getOuName());
            }
            storeReceiveDetailVO.setDeliveryWhName(storeOrderRpcDTO.getFromWhName());
            storeReceiveDetailVO.setReceWhName(storeOrderRpcDTO.getToWhName());
        }

        storeReceiveDetailVO.setCreator(thisDO.getCreator());
        storeReceiveDetailVO.setCreateDate(storeOrderRpcDTO.getCreateTime());
        storeReceiveDetailVO.setReceiveDate(thisDO.getReceiveDate());
        storeReceiveDetailVO.setDocStatus(storeOrderRpcDTO.getDocStatus());
        storeReceiveDetailVO.setNote(storeOrderRpcDTO.getRemark());

        // 图片
        storeReceiveDetailVO.setPics(storeOrderRpcDTO.getPics());
        if (StringUtils.isNotEmpty(storeOrderRpcDTO.getPics())) {
            storeReceiveDetailVO.setPicList(Arrays.asList(storeOrderRpcDTO.getPics().split(";")));
        } else {
            storeReceiveDetailVO.setPicList(Collections.emptyList());
        }

        // 关联单据
        if (thisDO.getRtType().equals("R")) {
            storeReceiveDetailVO.setRelateDocClsName("门店订货单");
            storeReceiveDetailVO.setRelateDocNo(thisDO.getOrderNo());
        } else {
            storeReceiveDetailVO.setRelateDocClsName("门店收货单");
            storeReceiveDetailVO.setRelateDocNo(thisDO.getReceiveDocNo());
        }

        // 单据类型
        storeReceiveDetailVO.setDocType(thisDO.getDocType());
        if (thisDO.getDocType().equals("TRN")) {
            storeReceiveDetailVO.setDocTypeName("仓配");
        } else {
            storeReceiveDetailVO.setDocTypeName("供应商直发");
        }

        // 查询商品信息
        List<StoreOrderDRpcDTO> storeOrderDRpcDTOS = new ArrayList<>();
        List<Long> itemIds = new ArrayList<>();
        for (StoreOrderDRpcDTO invTrnDRpcDTO : storeOrderRpcDTO.getInvTrnDRpcDTOS()) {
            // 排除项目费用
            if (!"SF".equals(invTrnDRpcDTO.getLineType())){
                storeOrderDRpcDTOS.add(invTrnDRpcDTO);
                itemIds.add(invTrnDRpcDTO.getItemId());
            }
        }

        // 商品信息
        Map<Long, ItmItemLotRpcDTO> itmItemLotMap = new HashMap<>();
        Map<Long, ItmItemSimpleRpcDTO> itmItemSimplemap = new HashMap<>();
        if (CollectionUtil.isNotEmpty(itemIds)) {
            List<ItmItemLotRpcDTO> lotRpcDTOS = itmItemRpcService.findLotRpcDTOS(itemIds);
            itmItemLotMap = lotRpcDTOS.stream().collect(Collectors.toMap(ItmItemLotRpcDTO::getId, Function.identity(), (v1, v2) -> v1));
            List<ItmItemSimpleRpcDTO> simpleItem = itmItemRpcService.findSimpleItem(itemIds);
            itmItemSimplemap = simpleItem.stream().collect(Collectors.toMap(ItmItemSimpleRpcDTO::getId, Function.identity(), (v1, v2) -> v1));
        }

        //获取商品图片信息
        Map<Long, List<DemandOrderDtl>> itemImgUrlMap = getItemImgUrlMap(thisDO.getOrderId());
        // 获取商品Udc信息
        Map<String, Map<String, String>> suppUdcMap = udcProvider.getValueMapByUdcCode("yst-supp", Set.of("UOM"));
        // 查询库存商品行
        Map<String, Map<String, String>> invUdcMap = udcProvider.getValueMapByUdcCode("yst-inv", Set.of("STK_ITM_STATUS"));
        //查询运费商品行
        String orderItemSetting = SysUtils.getOrderItemSetting();

        List<StoreReceiveDDetailVO> items = new ArrayList<>();

        List<StoreCanRtnQtyRpcDTO> rpcDTOS = new ArrayList<>();
        if (thisDO.getRtType().equals("R")) {
                rpcDTOS = storeOrderTrnProvider.canRtn(docId, thisDO.getDocType()).computeData();
        } else if (thisDO.getRtType().equals("T")) {
            if (thisDO.getDocType().equals("TRO")) {
                rpcDTOS = storeOrderTrnProvider.canRtn(thisDO.getReceiveDocId(), "TRN").computeData();
            } else {
                rpcDTOS = storeOrderTrnProvider.canRtn(thisDO.getReceiveDocId(), thisDO.getDocType()).computeData();
            }
        }

        // 获取可退货数量
        Map<Long, StoreCanRtnQtyRpcDTO> canRtnQtyRpcDTOHashMap = getCanRtnQtyRpcDTOMap(rpcDTOS, thisDO.getDocType());

        for (StoreOrderDRpcDTO storeOrderDRpcDTO : storeOrderDRpcDTOS) {
            ItmItemLotRpcDTO itmItemLotRpcDTO = itmItemLotMap.get(storeOrderDRpcDTO.getItemId());
            // 退货单不需要显示运费行
            if (thisDO.getRtType().equals("T") && itmItemLotRpcDTO.getItemCode().equals(orderItemSetting)) {
                continue;
            }
            ItmItemSimpleRpcDTO itmItemSimpleRpcDTO = itmItemSimplemap.get(storeOrderDRpcDTO.getItemId());
            // 商品处理
            StoreReceiveDDetailVO storeReceiveDDetailVO = getStoreReceiveDDetailVO(storeOrderDRpcDTO,
                    thisDO,
                    suppUdcMap,
                    itmItemLotRpcDTO,
                    itemImgUrlMap,
                    invUdcMap,
                    canRtnQtyRpcDTOHashMap,
                    orderDDTOMap,
                    itmItemSimpleRpcDTO);
            if (storeReceiveDDetailVO == null) continue;
            items.add(storeReceiveDDetailVO);
        }

        // 组合商品处理
        List<StoreReceiveDDetailVO> result = items.stream().filter(d -> StringUtils.isBlank(d.getCombineItemCode())).collect(Collectors.toList());
        Map<String, List<StoreReceiveDDetailVO>> subItemMap = items.stream().filter(d -> StringUtils.isNotBlank(d.getCombineItemCode())).collect(Collectors.groupingBy(StoreReceiveDDetailVO::getCombineItemCode));
        if (!subItemMap.isEmpty()) {
            subItemMap.forEach((k, v) -> {
                StoreReceiveDDetailVO combineOrderDRespVO = new StoreReceiveDDetailVO();
                combineOrderDRespVO.setCombineItemCode(k);
                combineOrderDRespVO.setCombineItemName(v.get(0).getCombineItemName());
                combineOrderDRespVO.setSubItemList(v);
                result.add(combineOrderDRespVO);
            });
        }
        storeReceiveDetailVO.setItems(result);

        if (thisDO.getRtType().equals("R")) {
            storeReceiveDetailVO.setDocStatusName(receiveStatusMap.get(thisDO.getStatus()));
        } else {
            storeReceiveDetailVO.setDocStatusName(rtnStatusMap.get(thisDO.getStatus()));
        }
        //获取系统设置
        String invOidoc = SysUtils.getInvOidoc();
        storeReceiveDetailVO.setIoDO("1".equals(invOidoc));

        storeReceiveDetailVO.setCarrier(storeOrderRpcDTO.getCarrier());
        storeReceiveDetailVO.setLogisCarrierId(storeOrderRpcDTO.getLogisCarrierId());
        storeReceiveDetailVO.setLogisCarrierCode(storeOrderRpcDTO.getLogisCarrierCode());
        storeReceiveDetailVO.setLogisCarrierName(storeReceiveDetailVO.getLogisCarrierName());
        storeReceiveDetailVO.setLogisContactName(storeReceiveDetailVO.getLogisContactName());
        storeReceiveDetailVO.setLogisContactTel(storeReceiveDetailVO.getLogisContactTel());
        if (StrUtil.isNotBlank(storeOrderRpcDTO.getDeliverMethod())) {
            storeReceiveDetailVO.setDeliverMethod(storeOrderRpcDTO.getDeliverMethod());
            Map<String, Map<String, String>> orderUdcMap = udcProvider.getValueMapByUdcCode("yst-order", Set.of("SO_DELIVER_METHOD"));
            storeReceiveDetailVO.setDeliverMethodName(orderUdcMap.get("SO_DELIVER_METHOD").get(storeReceiveDetailVO.getDeliverMethod()));
        }
        storeReceiveDetailVO.setLogisDocNo(storeOrderRpcDTO.getLogisDocNo());
        List<StoreReceiveDDO> receiveDDOS = storeReceiveDRepo.findAllByMasId(thisDO.getId());
        Map<Long, StoreReceiveDDO> ddoMap = receiveDDOS.stream().collect(Collectors.toMap(StoreReceiveDDO::getDid, i -> i));
        BigDecimal totalAmt = BigDecimal.ZERO;
        BigDecimal totalQty = BigDecimal.ZERO;

        Integer amtPlace = SysUtils.getAmtPlace();

        for (StoreReceiveDDetailVO item : storeReceiveDetailVO.getItems()) {
            StoreReceiveDDO storeReceiveDDO = ddoMap.get(item.getId());
            if (storeReceiveDDO != null) {
                BigDecimal price = getDecimal(storeReceiveDDO.getPrice());
                item.setPrice(price);
                item.setOrderQty(storeReceiveDDO.getQty());
                item.setAmt(item.getOrderQty().multiply(item.getPrice()).setScale(amtPlace, RoundingMode.HALF_UP));
            }

            BigDecimal price = getDecimal(item.getPrice());
            if (thisDO.getRtType().equals("T") && Objects.equals(storeReceiveDetailVO.getDocStatus(), "CF")) {
                totalAmt = totalAmt.add(getDecimal(item.getDeliveryQty()).multiply(price).setScale(amtPlace, RoundingMode.HALF_UP));
                totalQty = totalQty.add(getDecimal(item.getDeliveryQty()));
            } else if (thisDO.getRtType().equals("R") && Objects.equals(storeReceiveDetailVO.getDocStatus(), "CF")) {
                totalAmt = totalAmt.add(getDecimal(item.getReceiveQty()).multiply(price).setScale(amtPlace, RoundingMode.HALF_UP));
                totalQty = totalQty.add(getDecimal(item.getReceiveQty()));
            } else {
                totalAmt = totalAmt.add(getDecimal(item.getOrderQty()).multiply(price).setScale(amtPlace, RoundingMode.HALF_UP));
                totalQty = totalQty.add(getDecimal(item.getOrderQty()));
            }
        }

        if (paymentAmt.compareTo(BigDecimal.ZERO) > 0) {
            BigDecimal diff = totalAmt.subtract(paymentAmt);
            // 如果amt2比saleAmt2多，则取value里settlementAmt最多的一行去做减去差值,然后再
            if (diff.compareTo(BigDecimal.ZERO) != 0) {
                // 直接查找金额最大的项目并调整其金额，避免排序和复制对象
                StoreReceiveDDetailVO maxItem = storeReceiveDetailVO.getItems().stream()
                        .max(Comparator.comparing(StoreReceiveDDetailVO::getAmt))
                        .orElse(null);
                if (maxItem != null) {
                    // 直接修改最大金额项目的金额，避免创建新对象
                    BigDecimal adjustedAmt = maxItem.getAmt().subtract(diff);
                    maxItem.setAmt(adjustedAmt);
                }
            }
            storeReceiveDetailVO.setTotalAmt(paymentAmt);
        } else {
            storeReceiveDetailVO.setTotalAmt(totalAmt.setScale(amtPlace, RoundingMode.HALF_UP));
        }
        storeReceiveDetailVO.setTotalQty(totalQty);
        storeReceiveDetailVO.setDocStatus(thisDO.getStatus());
        if (thisDO.getRtType().equals("T")) {
            storeReceiveDetailVO.setDocStatusName(rtnStatusMap.get(storeReceiveDetailVO.getDocStatus()));
        } else {
            storeReceiveDetailVO.setDocStatusName(receiveStatusMap.get(storeReceiveDetailVO.getDocStatus()));
        }

        return storeReceiveDetailVO;
    }

    private StoreReceiveDDetailVO getStoreReceiveDDetailVO(StoreOrderDRpcDTO invTrnDRpcDTO,
                                                           StoreReceiveDO thisDO,
                                                           Map<String, Map<String, String>> suppUdcMap,
                                                           ItmItemLotRpcDTO itmItemLotRpcDTO,
                                                           Map<Long, List<DemandOrderDtl>> itemImgUrlMap,
                                                           Map<String, Map<String, String>> invUdcMap,
                                                           Map<Long, StoreCanRtnQtyRpcDTO> canRtnQtyRpcDTOHashMap,
                                                           Map<Long, ScpDemandOrderDDTO> orderDDTOMap,
                                                           ItmItemSimpleRpcDTO itmItemSimpleRpcDTO) {
        StoreReceiveDDetailVO storeReceiveDDetailVO = new StoreReceiveDDetailVO();
        ScpDemandOrderDDTO demandOrderDDTO = orderDDTOMap.get(invTrnDRpcDTO.getRelateDocDid());
        if (demandOrderDDTO != null) {
            storeReceiveDDetailVO.setCombineItemCode(demandOrderDDTO.getCombineItemCode());
            storeReceiveDDetailVO.setCombineItemName(demandOrderDDTO.getCombineItemName());
        }
        storeReceiveDDetailVO.setGiftFlag(invTrnDRpcDTO.getGiftLineFlag());
        storeReceiveDDetailVO.setId(invTrnDRpcDTO.getId());
        storeReceiveDDetailVO.setManuDate(invTrnDRpcDTO.getManuDate());
        storeReceiveDDetailVO.setManuFactory(invTrnDRpcDTO.getManuFactory());
        storeReceiveDDetailVO.setManuFactoryName(invTrnDRpcDTO.getManuFactoryName());
        storeReceiveDDetailVO.setOrderDid(invTrnDRpcDTO.getRelateDocDid());
        storeReceiveDDetailVO.setItemId(invTrnDRpcDTO.getItemId());
        storeReceiveDDetailVO.setPrice(invTrnDRpcDTO.getTotalPrice() == null ? invTrnDRpcDTO.getPrice() : invTrnDRpcDTO.getTotalPrice());
        if (thisDO.getRtType().equals("R")) {
            storeReceiveDDetailVO.setLotNo(invTrnDRpcDTO.getInLotNo());
            storeReceiveDDetailVO.setLimit1(invTrnDRpcDTO.getInLimit1());
            if (invTrnDRpcDTO.getOutQty() == null || BigDecimal.ZERO.compareTo(invTrnDRpcDTO.getOutQty()) == 0) {
                return null;
            }
        } else {
            if (thisDO.getStatus().equals("CF")) {
                storeReceiveDDetailVO.setLotNo(invTrnDRpcDTO.getOutLotNo());
                storeReceiveDDetailVO.setLimit1(invTrnDRpcDTO.getOutLimit1());
            } else {
                storeReceiveDDetailVO.setLotNo(invTrnDRpcDTO.getLotNo());
                storeReceiveDDetailVO.setLimit1(invTrnDRpcDTO.getOutLimit1());
            }
        }
        if (thisDO.getDocType().equals("GR")) {
            storeReceiveDDetailVO.setDeliveryQty(invTrnDRpcDTO.getOutQty());
            storeReceiveDDetailVO.setOrderQty(invTrnDRpcDTO.getQty());
            storeReceiveDDetailVO.setReceiveQty(invTrnDRpcDTO.getInQty());
            storeReceiveDDetailVO.setUom(invTrnDRpcDTO.getUom());
            storeReceiveDDetailVO.setUomName(suppUdcMap.get("UOM").get(invTrnDRpcDTO.getUom()));
        } else {
            if (invTrnDRpcDTO.getQty2() == null) {
                return null;
            }
            storeReceiveDDetailVO.setDeliveryQty(invTrnDRpcDTO.getOutQty2());
            storeReceiveDDetailVO.setOrderQty(invTrnDRpcDTO.getQty2());
            storeReceiveDDetailVO.setReceiveQty(invTrnDRpcDTO.getInQty2());
            storeReceiveDDetailVO.setUom(invTrnDRpcDTO.getUom2());
            storeReceiveDDetailVO.setUomName(suppUdcMap.get("UOM").get(invTrnDRpcDTO.getUom2()));
        }

        if (itmItemLotRpcDTO != null) {
            storeReceiveDDetailVO.setItemCode(itmItemLotRpcDTO.getItemCode());
            storeReceiveDDetailVO.setItemName(itmItemLotRpcDTO.getItemName());
            storeReceiveDDetailVO.setSpec(itmItemLotRpcDTO.getSpec());
            storeReceiveDDetailVO.setLotFlag(itmItemLotRpcDTO.getLotFlag());
            List<DemandOrderDtl> demandOrderDtls = itemImgUrlMap.get(itmItemLotRpcDTO.getId());
            if (CollectionUtil.isNotEmpty(demandOrderDtls)) {
                storeReceiveDDetailVO.setImageUrl(demandOrderDtls.get(0).getImgUrl());
            }
        }

        // 设置保留小数位
        if (itmItemSimpleRpcDTO != null) {
            storeReceiveDDetailVO.setDecimalPlaces(itmItemSimpleRpcDTO.getDecimalPlaces());
        }

        storeReceiveDDetailVO.setLimit1Name(invUdcMap.get("STK_ITM_STATUS").get(storeReceiveDDetailVO.getLimit1()));

        // 计算可退数量和库存数量
        if (CollectionUtil.isNotEmpty(canRtnQtyRpcDTOHashMap)) {
            StoreCanRtnQtyRpcDTO storeCanRtnQtyRpcDTO;
            if (thisDO.getRtType().equals("T")) {
                if (thisDO.getDocType().equals("TRO")){
                    storeCanRtnQtyRpcDTO = canRtnQtyRpcDTOHashMap.get(invTrnDRpcDTO.getRelateDocDid());
                } else {
                    storeCanRtnQtyRpcDTO = canRtnQtyRpcDTOHashMap.get(invTrnDRpcDTO.getRelate2Did());
                }
            } else {
                storeCanRtnQtyRpcDTO = canRtnQtyRpcDTOHashMap.get(storeReceiveDDetailVO.getId());
            }
            if (storeCanRtnQtyRpcDTO != null) {
                storeReceiveDDetailVO.setCanRtnQty(storeCanRtnQtyRpcDTO.getCanRtnQty());
                storeReceiveDDetailVO.setStkQty(storeCanRtnQtyRpcDTO.getStkQty());
            }
        }
        return storeReceiveDDetailVO;
    }

    private void assignAddress(StoreReceiveDetailVO storeReceiveDetailVO, Long addrNo) {
        if (addrNo == null || addrNo <= 0) {
            return;
        }

        OrgAddressRpcDtoParam orgAddressRpcDtoParam = new OrgAddressRpcDtoParam();
        orgAddressRpcDtoParam.setAddrNos(List.of(addrNo));
        List<OrgAddressRpcDTO> addrAddressRpcDtoByParam = orgAddrRpcService.findAddrAddressRpcDtoByParam(orgAddressRpcDtoParam);
        if (CollectionUtil.isEmpty(addrAddressRpcDtoByParam)) {
            return;
        }
        OrgAddressRpcDTO orgAddressRpcDTO = addrAddressRpcDtoByParam.stream().filter(i -> i.getDefaultFlag() != null && i.getDefaultFlag()).findFirst().orElseGet(() -> null);
        if (orgAddressRpcDTO == null) {
            return;
        }
        storeReceiveDetailVO.setCountry(orgAddressRpcDTO.getCountry());
        storeReceiveDetailVO.setCountryName(orgAddressRpcDTO.getCountryName());
        storeReceiveDetailVO.setProvince(orgAddressRpcDTO.getProvince());
        storeReceiveDetailVO.setProvinceName(orgAddressRpcDTO.getProvinceName());
        storeReceiveDetailVO.setCity(orgAddressRpcDTO.getCity());
        storeReceiveDetailVO.setCityName(orgAddressRpcDTO.getCityName());
        storeReceiveDetailVO.setCounty(orgAddressRpcDTO.getCounty());
        storeReceiveDetailVO.setCountyName(orgAddressRpcDTO.getCountyName());
        storeReceiveDetailVO.setDetailaddr(orgAddressRpcDTO.getDetailAddr());
        storeReceiveDetailVO.setContPerson(orgAddressRpcDTO.getContPerson());
        storeReceiveDetailVO.setTel(orgAddressRpcDTO.getTel());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void receiveConfirm(StoreReceiveConfirmParam confirmParam) {
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(confirmParam.getDocId());
        Assert.notEmpty(receiveDOS, "单据不存在");
        StoreReceiveDO thisDO = receiveDOS.get(0);

        String recvKey = "RECEIVE" + confirmParam.getDocId().toString();
        // 检查键是否存在
        if (redisTemplate.hasKey(recvKey)) {
            throw new BusinessException("正在收货中,请稍后重试");
        }
        // 使用SETNX设置键，确保原子性
        Boolean isPoSet = redisTemplate.opsForValue().setIfAbsent(recvKey, "1", 60, TimeUnit.SECONDS);
        // 检查是否成功设置键
        if (Boolean.FALSE.equals(isPoSet)) {
            throw new BusinessException("正在收货中,请稍后重试");
        }
        Assert.isTrue(Objects.equals(thisDO.getStatus(), "CE") && Objects.equals(thisDO.getRtType(), "R"), "单据不是待收货状态");
        StoreReceiveRpcDTO storeReceiveRpcDTO = new StoreReceiveRpcDTO();
        storeReceiveRpcDTO.setDocId(thisDO.getDocId());
        storeReceiveRpcDTO.setDocNo(thisDO.getDocNo());
        storeReceiveRpcDTO.setDocType(thisDO.getDocType());

        List<StoreReceiveRpcDTO.StoreReceiveRpcDTODetail> details = new ArrayList<>();
        for (StoreReceiveConfirmParam.StoreReceiveConfirmDetail detail : confirmParam.getDetails()) {
            StoreReceiveRpcDTO.StoreReceiveRpcDTODetail rpcDetail = new StoreReceiveRpcDTO.StoreReceiveRpcDTODetail();
            rpcDetail.setQty(detail.getQty());
            rpcDetail.setId(detail.getId());
            rpcDetail.setItemCode(detail.getItemCode());
            rpcDetail.setItemId(detail.getItemId());
            rpcDetail.setManuDate(detail.getManuDate());
            rpcDetail.setLimit1(detail.getLimit1());
            rpcDetail.setLotNo(detail.getLotNo());
            details.add(rpcDetail);
        }
        storeReceiveRpcDTO.setDetails(details);

        try {
            //写入mq队列,进行收货确认
            StoreReceiveSubmitMqParam orderSubmitMqParam = new StoreReceiveSubmitMqParam();
            orderSubmitMqParam.setReceiveRpcDTO(storeReceiveRpcDTO);
            orderSubmitMqParam.setDocId(confirmParam.getDocId());
            orderSubmitMqParam.setBusinessKey(StoreReceiveSubmitMqParam.RECEIVE_SUBMIT_CHANNEL);
            messageQueueTemplate.publishMessageSync("yst-suplan", StoreReceiveSubmitMqParam.RECEIVE_SUBMIT_CHANNEL, orderSubmitMqParam);
        } catch (Exception e) {
            log.info("门店收货提交：{},发送mq消息失败：{}", confirmParam.getDocId(), e.toString());
            redisTemplate.delete(recvKey);
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void rtnConfirm(StoreRtnConfirmParam confirmParam) {
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(confirmParam.getDocId());
        Assert.notEmpty(receiveDOS, "单据不存在");
        StoreReceiveDO thisDO = receiveDOS.get(0);

        Assert.isTrue(Objects.equals(thisDO.getStatus(), "CE") && Objects.equals(thisDO.getRtType(), "T"), "单据不是待退货状态");
        StoreRtnRpcDTO storeRtnRpcDTO = new StoreRtnRpcDTO();
        storeRtnRpcDTO.setDocId(thisDO.getDocId());
        storeRtnRpcDTO.setDeliveryMethod(confirmParam.getDeliveryMethod());
        storeRtnRpcDTO.setDocType(thisDO.getDocType());
        storeRtnRpcDTO.setLogisDocNo(confirmParam.getLogisDocNo());
        storeRtnRpcDTO.setLogisCarrierId(confirmParam.getLogisCarrierId());
        storeRtnRpcDTO.setLogisCarrierCode(confirmParam.getLogisCarrierCode());
        storeRtnRpcDTO.setLogisCarrierName(confirmParam.getLogisCarrierName());
        storeRtnRpcDTO.setNote(confirmParam.getNote());
        storeRtnRpcDTO.setLogisContactTel(confirmParam.getLogisContactTel());
        storeRtnRpcDTO.setLogisContactName(confirmParam.getLogisContactName());

        List<StoreReceiveDDO> receiveDDOS = storeReceiveDRepo.findAllByMasId(thisDO.getId());
        Map<Long, StoreReceiveDDO> ddoMap = receiveDDOS.stream().collect(Collectors.toMap(StoreReceiveDDO::getDid, i -> i));

        List<StoreRtnRpcDTO.StoreRtnRpcDTODetail> details = new ArrayList<>();
        for (StoreRtnConfirmParam.StoreRtnConfirmDetail detail : confirmParam.getDetails()) {
            StoreReceiveDDO storeReceiveDDO = ddoMap.get(detail.getId());
            if (storeReceiveDDO == null) {
                throw new BusinessException("商品" + detail.getItemCode() + "不在退货申请单内");
            }
            if (storeReceiveDDO.getQty().compareTo(detail.getQty()) < 0) {
                throw new BusinessException("商品" + detail.getItemCode() + "退货数量" + detail.getQty() + "大于申请数量" + storeReceiveDDO.getQty());
            }

            StoreRtnRpcDTO.StoreRtnRpcDTODetail rpcDetail = new StoreRtnRpcDTO.StoreRtnRpcDTODetail();
            rpcDetail.setQty(detail.getQty());
            rpcDetail.setId(detail.getId());
            rpcDetail.setItemCode(detail.getItemCode());
            rpcDetail.setItemId(detail.getItemId());
            details.add(rpcDetail);
        }
        storeRtnRpcDTO.setDetails(details);
        storeOrderTrnProvider.rtn(storeRtnRpcDTO).computeData();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void rtnApply(StoreRtnApplyParam applyParam) {
        // 是否存在未完结的退单
        List<StoreReceiveDO> statusNotIn = storeReceiveRepo.findAllByOrderIdAndRtTypeAndStatusNotIn(applyParam.getOrderId(), "T", Arrays.asList("CF", "APPROVED", "CL"));
        if (CollUtil.isNotEmpty(statusNotIn)) {
            throw new BusinessException("存在未完结的退单");
        }
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(applyParam.getDocId());
        Assert.notEmpty(receiveDOS, "单据不存在");
        StoreReceiveDO thisDO = receiveDOS.get(0);

        Assert.isTrue(Objects.equals(thisDO.getStatus(), "CF") && Objects.equals(thisDO.getRtType(), "R"), "单据不是已收货状态");
        StoreRtnRpcDTO storeRtnRpcDTO = new StoreRtnRpcDTO();
        storeRtnRpcDTO.setDocId(thisDO.getDocId());
        storeRtnRpcDTO.setDocType(thisDO.getDocType());
        storeRtnRpcDTO.setRefundType("RETURN");
        storeRtnRpcDTO.setRefundTypeName("退货退款");
        storeRtnRpcDTO.setNote(applyParam.getNote());
        storeRtnRpcDTO.setPicList(applyParam.getPicList());
        storeRtnRpcDTO.setDeliveryMethod(applyParam.getDeliveryMethod());

        List<StoreRtnRpcDTO.StoreRtnRpcDTODetail> details = new ArrayList<>();
        for (StoreRtnApplyParam.StoreRtnApplyDetail detail : applyParam.getDetails()) {
            StoreRtnRpcDTO.StoreRtnRpcDTODetail rpcDetail = new StoreRtnRpcDTO.StoreRtnRpcDTODetail();
            rpcDetail.setQty(detail.getQty());
            rpcDetail.setId(detail.getId());
            rpcDetail.setItemCode(detail.getItemCode());
            rpcDetail.setItemName(detail.getItemName());
            details.add(rpcDetail);
        }
        storeRtnRpcDTO.setDetails(details);
        StoreRtnApplyResult result = storeOrderTrnProvider.rtnApply(storeRtnRpcDTO).computeData();

        StoreReceiveDO receiveDO = new StoreReceiveDO();
        receiveDO.setDocCreateDate(LocalDateTime.now());
        receiveDO.setStatus("DR"); // 退货待审
        receiveDO.setDocId(result.getDocId());
        receiveDO.setDocNo(result.getDocNo());
        receiveDO.setOrderId(thisDO.getOrderId());
        receiveDO.setOrderNo(thisDO.getOrderNo());
        receiveDO.setOrderDate(thisDO.getOrderDate());
        receiveDO.setReceiveDate(thisDO.getReceiveDate());
        receiveDO.setDeliveryDate(thisDO.getDeliveryDate());
        receiveDO.setStoreId(thisDO.getStoreId());
        receiveDO.setOrderSetId(thisDO.getOrderSetId());
        String docType = "GR".equals(thisDO.getDocType()) ? "GR" : "TRO";
        receiveDO.setDocType(docType);
        receiveDO.setReceiveDocId(thisDO.getDocId());
        receiveDO.setReceiveDocNo(thisDO.getDocNo());
        receiveDO.setTotalAmt(applyParam.getDetails().stream().filter(i -> i.getQty() != null && i.getPrice() != null)
                .map(i -> i.getQty().multiply(i.getPrice())).reduce(BigDecimal.ZERO, BigDecimal::add));
        receiveDO.setTotalQty(applyParam.getDetails().stream().map(StoreRtnApplyParam.StoreRtnApplyDetail::getQty).reduce(BigDecimal.ZERO,
                BigDecimal::add));
        receiveDO.setRtType("T");
        receiveDO.setRefundType("RETURN");
        receiveDO.setRefundTypeName("退货退款");
        StoreReceiveDO saved = storeReceiveRepo.save(receiveDO);

        Map<Long, BigDecimal> priceMap = applyParam.getDetails().stream().filter(i -> i.getPrice() != null).collect(Collectors.toMap(StoreRtnApplyParam.StoreRtnApplyDetail::getId, i -> i.getPrice()));
        List<StoreReceiveDDO> receiveDDOS = new ArrayList<>();
        for (StoreRtnApplyResult.OrderItem item : result.getItems()) {
            StoreReceiveDDO storeReceiveDDO = new StoreReceiveDDO();
            storeReceiveDDO.setDid(item.getDid());
            storeReceiveDDO.setId(IdFactory.generateLong());
            storeReceiveDDO.setItemId(item.getItemId());
            storeReceiveDDO.setMasId(saved.getId());
            storeReceiveDDO.setQty(item.getQty());
            storeReceiveDDO.setPrice(priceMap.get(item.getRelateDocDid()));
            receiveDDOS.add(storeReceiveDDO);
        }
        storeReceiveDRepo.saveAll(receiveDDOS);

//        // 更新订货单的赔付数量
//        for (StoreRtnApplyResult.OrderItem item : result.getItems()) {
//            scpDemandOrderDRepo.updateCompensateQtyByMasIdAndSrcDocIdAndItemId(item.getQty(), applyParam.getOrderId(), item.getItemId());
//        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public RtnApplyResult reFundApply(List<StoreRtnApplyParam> storeRtnApplyParamList) {
        RtnApplyResult rtnApplyResult = new RtnApplyResult();
        // 是否存在未完结的退单
        List<StoreReceiveDO> statusNotIn = storeReceiveRepo.findAllByOrderIdAndRtTypeAndStatusNotIn(storeRtnApplyParamList.get(0).getOrderId(), "T", Arrays.asList("CF", "APPROVED", "CL"));
        if (CollUtil.isNotEmpty(statusNotIn)) {
            throw new BusinessException("存在未完结的退单");
        }
        for (StoreRtnApplyParam applyParam : storeRtnApplyParamList) {
            try {
                List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(applyParam.getDocId());
                Assert.notEmpty(receiveDOS, "单据不存在");
                StoreReceiveDO storeReceiveDO = receiveDOS.get(0);

                Assert.isTrue(Objects.equals(storeReceiveDO.getStatus(), "CF") && Objects.equals(storeReceiveDO.getRtType(), "R"), "单据不是已收货状态");
                StoreRtnRpcDTO storeRtnRpcDTO = new StoreRtnRpcDTO();
                storeRtnRpcDTO.setDocId(storeReceiveDO.getDocId());
                storeRtnRpcDTO.setDocType(storeReceiveDO.getDocType());
                storeRtnRpcDTO.setRefundType("REFUND");
                storeRtnRpcDTO.setRefundTypeName("赔付");
                storeRtnRpcDTO.setNote(applyParam.getNote());
                storeRtnRpcDTO.setPicList(applyParam.getPicList());
                storeRtnRpcDTO.setDeliveryMethod(applyParam.getDeliveryMethod());

                List<StoreRtnRpcDTO.StoreRtnRpcDTODetail> details = new ArrayList<>();
                for (StoreRtnApplyParam.StoreRtnApplyDetail detail : applyParam.getDetails()) {
                    StoreRtnRpcDTO.StoreRtnRpcDTODetail rpcDetail = new StoreRtnRpcDTO.StoreRtnRpcDTODetail();
                    if (detail.getQty() == null || detail.getQty().compareTo(BigDecimal.ZERO) <= 0) {
                        throw new IllegalArgumentException("数量不能小于0");
                    }
                    rpcDetail.setQty(detail.getQty());
                    rpcDetail.setId(detail.getId());
                    rpcDetail.setUom(detail.getUom());
                    rpcDetail.setUomRatio(detail.getUomRatio());
                    rpcDetail.setPlaces(detail.getPlaces());
                    rpcDetail.setItemCode(detail.getItemCode());
                    rpcDetail.setItemName(detail.getItemName());
                    details.add(rpcDetail);
                }
                storeRtnRpcDTO.setDetails(details);
                StoreRtnApplyResult result;
                try {
                    result =storeOrderTrnProvider.rtnApply(storeRtnRpcDTO).computeData();
                } catch (Exception e) {
                    log.error("赔付失败：{}", e.getMessage(), e);
                    throw new BusinessException(e.getMessage());
                }
                StoreReceiveDO receiveDO = new StoreReceiveDO();
                receiveDO.setDocCreateDate(LocalDateTime.now());
                receiveDO.setStatus("DR"); // 退货待审
                receiveDO.setDocId(result.getDocId());
                receiveDO.setDocNo(result.getDocNo());
                receiveDO.setOrderId(storeReceiveDO.getOrderId());
                receiveDO.setOrderNo(storeReceiveDO.getOrderNo());
                receiveDO.setOrderDate(storeReceiveDO.getOrderDate());
                receiveDO.setReceiveDate(storeReceiveDO.getReceiveDate());
                receiveDO.setDeliveryDate(storeReceiveDO.getDeliveryDate());
                receiveDO.setStoreId(storeReceiveDO.getStoreId());
                receiveDO.setOrderSetId(storeReceiveDO.getOrderSetId());
                String docType = "GR".equals(storeReceiveDO.getDocType()) ? "GR" : "TRO";
                receiveDO.setDocType(docType);
                receiveDO.setReceiveDocId(storeReceiveDO.getDocId());
                receiveDO.setReceiveDocNo(storeReceiveDO.getDocNo());

                AtomicReference<BigDecimal> reduceQty = new AtomicReference<>(BigDecimal.ZERO);
                BigDecimal reduceAmt = applyParam.getDetails().stream().filter(i -> i.getQty() != null && i.getPrice() != null)
                        .map(i -> {
                            BigDecimal qty = i.getQty();
                            // 支持换单位赔付
                            if (i.getUom() != null && i.getUomRatio() != null && i.getPlaces() != null) {
                                // 现数量 = 原数量 * 转换率
                                qty = qty.multiply(i.getUomRatio()).setScale(i.getPlaces(), RoundingMode.HALF_UP);
                            }
                            reduceQty.set(reduceQty.get().add(qty));
                            return qty.multiply(i.getPrice());
                        }).reduce(BigDecimal.ZERO, BigDecimal::add);
                receiveDO.setTotalAmt(reduceAmt);
                receiveDO.setTotalQty(reduceQty.get());
                receiveDO.setRtType("T");
                receiveDO.setRefundType("REFUND");
                receiveDO.setRefundTypeName("赔付");
                StoreReceiveDO saved = storeReceiveRepo.save(receiveDO);

                Map<Long, BigDecimal> priceMap = applyParam.getDetails().stream().filter(i -> i.getPrice() != null)
                        .collect(Collectors.toMap(StoreRtnApplyParam.StoreRtnApplyDetail::getId, StoreRtnApplyParam.StoreRtnApplyDetail::getPrice));
                List<StoreReceiveDDO> receiveDDOS = new ArrayList<>();
                for (StoreRtnApplyResult.OrderItem item : result.getItems()) {
                    StoreReceiveDDO storeReceiveDDO = new StoreReceiveDDO();
                    storeReceiveDDO.setDid(item.getDid());
                    storeReceiveDDO.setId(IdFactory.generateLong());
                    storeReceiveDDO.setItemId(item.getItemId());
                    storeReceiveDDO.setMasId(saved.getId());
                    storeReceiveDDO.setQty(item.getQty());
                    storeReceiveDDO.setPrice(priceMap.get(item.getRelateDocDid()) == null ? BigDecimal.ZERO : priceMap.get(item.getRelateDocDid()).setScale(2, RoundingMode.HALF_UP));
                    receiveDDOS.add(storeReceiveDDO);
                }
                storeReceiveDRepo.saveAll(receiveDDOS);
                rtnApplyResult.getSuccessList().add(applyParam);
            } catch (IllegalArgumentException e) {
                log.error("赔付失败：{}", e.getMessage(), e);
                // 添加到失败列表
                rtnApplyResult.getFailedList().add(new RtnApplyResult.FailedRtnApply(applyParam, e.getMessage()));
            }
        }
        return rtnApplyResult;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public RtnApplyResult rtnApplyByOrder(List<StoreRtnApplyParam> applyParamList) {
        RtnApplyResult rtnApplyResult = new RtnApplyResult();

        // 是否存在未完结的退单
        List<StoreReceiveDO> statusNotIn = storeReceiveRepo.findAllByOrderIdAndRtTypeAndStatusNotIn(applyParamList.get(0).getOrderId(), "T", Arrays.asList("CF", "APPROVED", "CL"));
        if (CollUtil.isNotEmpty(statusNotIn)) {
            throw new BusinessException("存在未完结的退单");
        }

        for (StoreRtnApplyParam applyParam : applyParamList) {
            try {
                StoreReceiveDO storeReceiveDO = storeReceiveRepo.findByDocId(applyParam.getDocId());
                Assert.notNull(storeReceiveDO, "单据不存在");

                Assert.isTrue(Objects.equals(storeReceiveDO.getStatus(), "CF") && Objects.equals(storeReceiveDO.getRtType(), "R"), "单据不是已收货状态");
                StoreRtnRpcDTO storeRtnRpcDTO = new StoreRtnRpcDTO();
                storeRtnRpcDTO.setDocId(storeReceiveDO.getDocId());
                storeRtnRpcDTO.setDocType(storeReceiveDO.getDocType());
                storeRtnRpcDTO.setRefundType("RETURN");
                storeRtnRpcDTO.setRefundTypeName("退货退款");
                storeRtnRpcDTO.setNote(applyParam.getNote());
                storeRtnRpcDTO.setPicList(applyParam.getPicList());
                storeRtnRpcDTO.setDeliveryMethod(applyParam.getDeliveryMethod());
                List<StoreRtnRpcDTO.StoreRtnRpcDTODetail> details = new ArrayList<>();
                for (StoreRtnApplyParam.StoreRtnApplyDetail detail : applyParam.getDetails()) {
                    StoreRtnRpcDTO.StoreRtnRpcDTODetail rpcDetail = new StoreRtnRpcDTO.StoreRtnRpcDTODetail();
                    if (detail.getQty() == null || detail.getQty().compareTo(BigDecimal.ZERO) <= 0) {
                        throw new IllegalArgumentException("数量不能小于0");
                    }
                    rpcDetail.setQty(detail.getQty());
                    rpcDetail.setId(detail.getId());
                    rpcDetail.setItemCode(detail.getItemCode());
                    rpcDetail.setItemName(detail.getItemName());
                    details.add(rpcDetail);
                }
                storeRtnRpcDTO.setDetails(details);

                StoreRtnApplyResult result;
                try {
                    result = storeOrderTrnProvider.rtnApply(storeRtnRpcDTO).computeData();
                } catch (Exception e) {
                    log.error("退货退款失败：{}", e.getMessage(), e);
                    throw new BusinessException(e.getMessage());
                }

                StoreReceiveDO receiveDO = getStoreReceiveDO(applyParam.getDetails(), result, storeReceiveDO);
                receiveDO.setRefundType("RETURN");
                receiveDO.setRefundTypeName("退货退款");
                StoreReceiveDO saved = storeReceiveRepo.save(receiveDO);

                Map<Long, BigDecimal> priceMap = applyParam.getDetails().stream().filter(i -> i.getPrice() != null)
                        .collect(Collectors.toMap(StoreRtnApplyParam.StoreRtnApplyDetail::getId, StoreRtnApplyParam.StoreRtnApplyDetail::getPrice));
                List<StoreReceiveDDO> receiveDDOS = new ArrayList<>();
                for (StoreRtnApplyResult.OrderItem item : result.getItems()) {
                    StoreReceiveDDO storeReceiveDDO = new StoreReceiveDDO();
                    storeReceiveDDO.setDid(item.getDid());
                    storeReceiveDDO.setId(IdFactory.generateLong());
                    storeReceiveDDO.setItemId(item.getItemId());
                    storeReceiveDDO.setMasId(saved.getId());
                    storeReceiveDDO.setQty(item.getQty());
                    storeReceiveDDO.setPrice(priceMap.get(item.getRelateDocDid()));
                    receiveDDOS.add(storeReceiveDDO);
                }
                storeReceiveDRepo.saveAll(receiveDDOS);
                rtnApplyResult.getSuccessList().add(applyParam);
            } catch (Exception e) {
                log.error("退货退款失败：{}", e.getMessage(), e);
                // 添加到失败列表
                rtnApplyResult.getFailedList().add(new RtnApplyResult.FailedRtnApply(applyParam, e.getMessage()));
            }
        }
        return rtnApplyResult;
    }

    @NotNull
    private StoreReceiveDO getStoreReceiveDO(List<StoreRtnApplyParam.StoreRtnApplyDetail> details, StoreRtnApplyResult result, StoreReceiveDO thisDO) {
        Integer amtPlace = SysUtils.getAmtPlace();
        StoreReceiveDO receiveDO = new StoreReceiveDO();
        receiveDO.setDocCreateDate(LocalDateTime.now());
        receiveDO.setStatus("DR"); // 退货待审
        receiveDO.setDocId(result.getDocId());
        receiveDO.setDocNo(result.getDocNo());
        receiveDO.setOrderId(thisDO.getOrderId());
        receiveDO.setOrderNo(thisDO.getOrderNo());
        receiveDO.setOrderDate(thisDO.getOrderDate());
        receiveDO.setReceiveDate(thisDO.getReceiveDate());
        receiveDO.setDeliveryDate(thisDO.getDeliveryDate());
        receiveDO.setStoreId(thisDO.getStoreId());
        receiveDO.setOrderSetId(thisDO.getOrderSetId());
        String docType = "GR".equals(thisDO.getDocType()) ? "GR" : "TRO";
        receiveDO.setDocType(docType);
        receiveDO.setReceiveDocId(thisDO.getDocId());
        receiveDO.setReceiveDocNo(thisDO.getDocNo());
        BigDecimal totalAmt = BigDecimal.ZERO;
        BigDecimal totalQty = BigDecimal.ZERO;
        for (StoreRtnApplyParam.StoreRtnApplyDetail storeRtnApplyDetail : details) {
            if (storeRtnApplyDetail != null && storeRtnApplyDetail.getQty() != null) {
                totalQty = totalQty.add(storeRtnApplyDetail.getQty());
            }
            if (storeRtnApplyDetail != null && storeRtnApplyDetail.getQty() != null && storeRtnApplyDetail.getPrice() != null) {
                BigDecimal multiply = storeRtnApplyDetail.getQty().multiply(storeRtnApplyDetail.getPrice().setScale(amtPlace, RoundingMode.HALF_UP));
                totalAmt = totalAmt.add(multiply);
            }
        }
        receiveDO.setTotalAmt(totalAmt.setScale(amtPlace, RoundingMode.HALF_UP));
        receiveDO.setTotalQty(totalQty);
        receiveDO.setRtType("T");
        return receiveDO;
    }

    @Override
    public List<StoreCanRtnQtyRpcDTO> canRtn(Long docId) {
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(docId);
        Assert.notEmpty(receiveDOS, "单据不存在");
        StoreReceiveDO thisDO = receiveDOS.get(0);

        Assert.isTrue(Objects.equals(thisDO.getStatus(), "CF") && Objects.equals(thisDO.getRtType(), "R"), "单据不是已收货状态");

        List<StoreCanRtnQtyRpcDTO> rpcDTOS = storeOrderTrnProvider.canRtn(docId, thisDO.getDocType()).computeData();
        rpcDTOS.forEach(row -> {
            if ("GR".equals(thisDO.getDocType())) {
                row.setCanRtnQty(row.getCanRtnQty());
            } else {
                if (row.getCanRtnQty() != null && row.getUomRatio() != null && row.getUomRatio().compareTo(BigDecimal.ZERO) > 0) {
                    row.setCanRtnQty(row.getCanRtnQty().divide(row.getUomRatio(), row.getPlaces(), BigDecimal.ROUND_HALF_UP));
                }
            }
            if (row.getStkQty() != null && row.getUomRatio() != null && row.getUomRatio().compareTo(BigDecimal.ZERO) > 0) {
                row.setStkQty(row.getStkQty().divide(row.getUomRatio(), row.getPlaces(), BigDecimal.ROUND_HALF_UP));
            }
        });

        return rpcDTOS;
    }

    @Override
    public Boolean rtnPermission(Long orderId) {
        // 是否存在未完结的退单
        List<StoreReceiveDO> statusNotIn = storeReceiveRepo.findAllByOrderIdAndRtTypeAndStatusNotIn(orderId, "T", Arrays.asList("CF", "APPROVED", "CL"));
        if (CollUtil.isNotEmpty(statusNotIn)) {
            // 存在，则不能退单
            return Boolean.FALSE;
        }
        return Boolean.TRUE;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void abort(Long docId) {
        List<StoreReceiveDO> receiveDOS = storeReceiveRepo.findAllByDocId(docId);
        Assert.notEmpty(receiveDOS, "单据不存在");
        StoreReceiveDO thisDO = receiveDOS.get(0);
        Assert.isTrue(thisDO.getStatus().equals("DR"), "退货申请不可作废");

        if (thisDO.getDocType().equals("TRN")) {
            storeOrderTrnProvider.abort(docId).computeData();
        } else {
            purRnsProvider.abort(docId).computeData();
        }
        thisDO.setStatus("CL");
        storeReceiveRepo.save(thisDO);
    }

    @Override
    public Long countStoreReceive(StoreReceivePageParam param) {
        PagingVO<StoreReceivePageVO> page = storeReceiveRepoProc.page(param);
        return page.getTotal();
    }

    private Map<Long, List<DemandOrderDtl>> getItemImgUrlMap(Long orderId) {
        List<DemandOrderDtl> orderDtl = storeReceiveRepoProc.findOrderDtlWithSettlePrice(orderId);
        if (CollUtil.isNotEmpty(orderDtl)) {
            return orderDtl.stream().collect(Collectors.groupingBy(DemandOrderDtl::getItemId));
        }
        return new HashMap<>();
    }
}
