package com.elitesland.scp.domain.service.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.order.ScpDemandOrderExportRespVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpComputeDemandOrderPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpsmanAuthQueryParam;
import com.elitesland.scp.application.facade.vo.resp.app.AppPayOrderRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.*;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderSaveVO;
import com.elitesland.scp.application.service.UserService;
import com.elitesland.scp.common.CurrentUserDTO;
import com.elitesland.scp.common.OrderStatusCountdownService;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDDO;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDTO;
import com.elitesland.scp.infr.repo.order.*;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepoProc;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanRegionRepoProc;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBaseRpcParam;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpDemandOrderDomainServiceImpl implements ScpDemandOrderDomainService {
    private final ScpDemandOrderRepo scpDemandOrderRepo;
    private final ScpDemandOrderDRepo scpDemandOrderDRepo;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;
    private final ScpDemandOrderRepoProc scpDemandOrderRepoProc;
    private final ScpsmanInfoRepoProc scpsmanInfoRepoProc;
    private final ScpsmanRegionRepoProc scscpsmanRegionRepoProc;
    private final OrderStatusCountdownService orderStatusCountdownService;
    private final OrgStoreRpcService orgStoreRpcService;
    private final ScpDemandOrderExportRepoProc scpDemandOrderExportRepoProc;

    @Override
    public ScpDemandOrderDO saveDemandOrder(ScpDemandOrderSaveVO saveVO) {
        if (saveVO.getId() == null) {
            ScpDemandOrderDO intentDO = ScpDemandOrderConvert.INSTANCE.saveVoToDO(saveVO);
            return scpDemandOrderRepo.save(intentDO);
        } else {
            Optional<ScpDemandOrderDO> option = scpDemandOrderRepo.findById(saveVO.getId());
            if (option.isEmpty()) {
                throw new BusinessException("订货订单ID：" + saveVO.getId() + "不存在");
            }
            ScpDemandOrderConvert.INSTANCE.copySaveParamToDo(saveVO, option.get());
            return scpDemandOrderRepo.save(option.get());
        }
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpDemandOrderTitlePageRespVO> page(ScpDemandOrderPageParamVO paramVO) {
//        paramVO.setAuthIds(getAuthIds());
//        if (CollUtil.isEmpty(paramVO.getAuthIds())) {
//            return new PagingVO<>();
//        }

        // 查询计划员信息
        var authed = appendScpsmanAuth(paramVO);
        if (!authed) {
            return PagingVO.empty();
        }

        long count = scpDemandOrderRepoProc.countDemandOrderPage(paramVO);
        if (count > 0) {
            var demandOrder = scpDemandOrderRepoProc.queryDemandOrderPage(paramVO);
            return new PagingVO<>(count, demandOrder);
        }
        return new PagingVO<>();
    }

    private boolean appendScpsmanAuth(ScpsmanAuthQueryParam paramVO) {
        var username = SecurityContextUtil.currentUserIfUnauthorizedThrow().getUsername();
        var scpsmanInfo = scpsmanInfoRepoProc.getByScpsmanNo(username);
        if (scpsmanInfo == null) {
            // 无权限
            return false;
        }
        paramVO.setScpsmanInfoId(scpsmanInfo.getId());

        if (Boolean.TRUE.equals(scpsmanInfo.getJurisdictionAll())) {
            // 区域筛选
            if (CollectionUtil.isNotEmpty(paramVO.getRegionCodes())) {
                OrgStoreBaseRpcParam param = new OrgStoreBaseRpcParam();
                param.setRegionList(new ArrayList<>(paramVO.getRegionCodes()));
                var storeList = orgStoreRpcService.findSimpleStoreByParam(param).computeData();
                if (CollUtil.isNotEmpty(storeList)) {
                    paramVO.setRegionStoreCodes(storeList.stream().map(OrgStoreBaseRpcDTO::getStoreCode).collect(Collectors.toSet()));
                }
                paramVO.setAllRegion(false);
            } else {
                // 所有区域
                paramVO.setAllRegion(true);
            }
            return true;
        }
        paramVO.setAllRegion(false);

        // 查询区域下门店
        OrgStoreBaseRpcParam param = new OrgStoreBaseRpcParam();
        var regionCodes = scscpsmanRegionRepoProc.getRegionCodesByMasId(scpsmanInfo.getId(), com.elitesland.scp.application.enums.ScpUdcEnum.SCPSMAN_REGION_TYPE_REGION.getValueCode());
        if (!regionCodes.isEmpty()) {
            // 筛选regionCodes和regionCodes1都存在的编码
            if (CollectionUtil.isNotEmpty(paramVO.getRegionCodes())) {
                regionCodes.retainAll(paramVO.getRegionCodes());
            }
            param.setRegionList(new ArrayList<>(regionCodes));
        } else {
            if (CollectionUtil.isNotEmpty(paramVO.getRegionCodes())) {
                param.setRegionList(new ArrayList<>(paramVO.getRegionCodes()));
            }
        }

        if (CollectionUtil.isNotEmpty(param.getRegionList())) {
            var storeList = orgStoreRpcService.findSimpleStoreByParam(param).computeData();
            if (CollUtil.isNotEmpty(storeList)) {
                paramVO.setRegionStoreCodes(storeList.stream().map(OrgStoreBaseRpcDTO::getStoreCode).collect(Collectors.toSet()));
            }
        }
        return true;
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpDemandOrderPageRespVO> queryDemandOrderList(ScpDemandOrderPageParamVO paramVO) {
//        paramVO.setAuthIds(getAuthIds());
//        if (CollUtil.isEmpty(paramVO.getAuthIds())) {
//            return new PagingVO<>();
//        }

        // 查询计划员信息
        var authed = appendScpsmanAuth(paramVO);
        if (!authed) {
            return PagingVO.empty();
        }

        long count = scpDemandOrderRepoProc.countDemandOrder(paramVO);
        if (count > 0) {
            var demandOrder = scpDemandOrderRepoProc.queryDemandOrder(paramVO);
            demandOrder = demandOrder.stream()
                    .peek(demandOrderPageRespVO ->
                            demandOrderPageRespVO.setUniqueId(demandOrderPageRespVO.getDetailId() == null ?
                                    demandOrderPageRespVO.getId() :
                                    demandOrderPageRespVO.getId() + demandOrderPageRespVO.getDetailId())).toList();
            return new PagingVO<>(count, demandOrder);
        }
        return new PagingVO<>();
    }

    /**
     * 获取当前用户权限信息
     *
     * @return
     */
    private List<Long> getAuthIds() {
        //查询当前登录用户
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (ObjectUtil.isEmpty(currentUserDTO)) {
            throw new BusinessException("当前登录人查询异常");
        }
        String username = currentUserDTO.getDetail().getUsername();
        //查询权限Id信息
        return scpDemandOrderRepoProc.queryIdsByAccount(username);
    }

    @SysCodeProc
    @Override
    public PagingVO<ScpDemandOrderPageRespVO> queryDemandOrderHeader(ScpDemandOrderPageParamVO paramVO) {
        //查询当前登录用户
        Long currentUserId = UserService.currentUserId();
        paramVO.setCreateUserId(currentUserId);
        long count = scpDemandOrderRepoProc.countDemandOrderHeader(paramVO);
        if (count > 0) {
            var demandOrder = scpDemandOrderRepoProc.queryDemandOrderHeader(paramVO);
            List<Long> masId = demandOrder.stream().map(ScpDemandOrderPageRespVO::getId).collect(Collectors.toList());
            List<ScpDemandOrderPageRespVO> orderDList = scpDemandOrderRepoProc.sumDemandAmt(masId);
            if (CollUtil.isEmpty(orderDList)) {
                return new PagingVO<>(count, demandOrder);
            }

            List<ScpDemandOrderDDTO> itemList = scpDemandOrderDRepoProc.findItemListByMasId(masId);
            Map<Long, List<ScpDemandOrderDDTO>> orderMap = itemList.stream().collect(Collectors.groupingBy(ScpDemandOrderDDTO::getMasId));
            Map<Long, ScpDemandOrderPageRespVO> idMap = orderDList.stream().collect(Collectors.toMap(ScpDemandOrderPageRespVO::getId, Function.identity()));
            demandOrder.forEach(row -> {
                if (idMap.containsKey(row.getId())) {
                    ScpDemandOrderPageRespVO orderD = idMap.get(row.getId());
                    row.setDemandSumQuantity(orderD.getDemandSumQuantity());
                    row.setDemandSumAmt(orderD.getDemandSumAmt());
                }
                List<ScpDemandOrderDDTO> demandOrderDDTOS = orderMap.get(row.getId());
                if (CollUtil.isNotEmpty(demandOrderDDTOS)) {
                    List<String> imgUrls = demandOrderDDTOS.stream().map(ScpDemandOrderDDTO::getImgUrl).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                    if (CollUtil.isNotEmpty(imgUrls)) {
                        row.setItemImgUrls(imgUrls);
                    }
                }
            });
            return new PagingVO<>(count, demandOrder);
        }
        return new PagingVO<>();
    }


    @Override
    @SysCodeProc
    public List<ScpDemandOrderDTO> findDemandOrderByIds(List<Long> ids) {
        return scpDemandOrderRepo.findByIdIn(ids).stream().map(ScpDemandOrderConvert.INSTANCE::doToDto).collect(Collectors.toList());
    }

    @Override
    @SysCodeProc
    public Optional<ScpDemandOrderDTO> findDemandOrderById(Long id) {
        return scpDemandOrderRepo.findById(id).map(ScpDemandOrderConvert.INSTANCE::doToDto);
    }

    @Override
    public List<ScpDemandOrderComputeVO> getItemAndWarehouseByIds(List<Long> ids) {
        return scpDemandOrderRepoProc.getItemAndWarehouseByIds(ids);
    }

    @Override
    public List<ScpDemandOrderDTO> findDemandOrderByDemandIds(List<Long> demandIds) {
        return scpDemandOrderRepo.findByDemandIdIn(demandIds).stream().map(ScpDemandOrderConvert.INSTANCE::doToDto).collect(Collectors.toList());
    }

    @Override
    public List<ScpDemandOrderDTO> findDemandOrderByParam(ScpDemandOrderParamVO paramVO) {
        return scpDemandOrderRepoProc.findDemandOrderByParam(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateAllocStatusByDemandId(Long demandId) {
        var demandOrderComputeVOS = scpDemandOrderRepoProc.findDemandOrderInfo(demandId);
        //部分推送和全部推送无需修改分配状态
        List<String> partOrPushedList = ListUtil.toList(ScpUdcEnum.DEO_STATUS_PUSHING.getValueCode(), ScpUdcEnum.DEO_STATUS_PUSHED.getValueCode());
        Map<String, List<ScpDemandOrderComputeVO>> orderMap = demandOrderComputeVOS.stream().filter(row -> !partOrPushedList.contains(row.getDocStatus())).collect(Collectors.groupingBy(ScpDemandOrderComputeVO::getDocCode));
        for (Map.Entry<String, List<ScpDemandOrderComputeVO>> entry : orderMap.entrySet()) {
            //查询未配货的订单明细行数
            long noAllocCount = entry.getValue().stream().filter(row -> !row.getIsCalculated()).count();
            String status = (noAllocCount == 0) ? ScpUdcEnum.DEO_STATUS_DONE.getValueCode() :
                    (noAllocCount == entry.getValue().size()) ? ScpUdcEnum.DEO_STATUS_WT.getValueCode() :
                            ScpUdcEnum.DEO_STATUS_DOING.getValueCode();
            Boolean isCalculated = !ScpUdcEnum.DEO_STATUS_WT.getValueCode().equals(status) ? Boolean.TRUE : Boolean.FALSE;
            scpDemandOrderRepoProc.updateOrderStatusByCode(entry.getKey(), status, isCalculated);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePushStatusByDemandId(Long demandId) {
        var demandOrderComputeVOS = scpDemandOrderRepoProc.findDemandOrderInfo(demandId);
        Map<String, List<ScpDemandOrderComputeVO>> orderMap = demandOrderComputeVOS.stream().collect(Collectors.groupingBy(ScpDemandOrderComputeVO::getDocCode));
        for (Map.Entry<String, List<ScpDemandOrderComputeVO>> entry : orderMap.entrySet()) {
            //查询未推送的订单明细行数
            long noPushCount = entry.getValue().stream().filter(row -> !row.getIsPushed()).count();
            String status = (noPushCount == 0) ? ScpUdcEnum.DEO_STATUS_PUSHED.getValueCode() : (noPushCount == entry.getValue().size() ? ScpConstant.PUSH_DOING : ScpUdcEnum.DEO_STATUS_PUSHING.getValueCode());
            if (!ScpConstant.PUSH_DOING.equals(status)) {
                scpDemandOrderRepoProc.updateOrderStatusByCode(entry.getKey(), status, null);
            }
        }
    }

    @Override
    public List<AppDemandOrderCountRespVO> countStatus(ScpDemandOrderPageParamVO paramVO) {
        return scpDemandOrderRepoProc.countAppDemandOrder(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpDemandOrderRepo.deleteByIdIn(ids);
    }

    @Override
    public Long countUnPushedOrderByDemandId(Long demandId) {
        return scpDemandOrderRepoProc.countUnPushedItemByDemandId(demandId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateAllocStatusAndPayStatusById(Long id) {
        List<ScpDemandOrderDDO> demandOrderDDOS = scpDemandOrderDRepo.findByMasId(id);
        if (CollUtil.isEmpty(demandOrderDDOS)) {
            return;
        }

        //查询未配货的订单明细行数
        List<String> noAllocItemName = demandOrderDDOS.stream().filter(row -> !row.getIsCalculated()).map(ScpDemandOrderDDO::getItemName).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(noAllocItemName)) {
            String noAllocItemNames = String.join(",", noAllocItemName);
            throw new BusinessException("存在无法分配的订单明细行:" + noAllocItemNames);
        }

        String status = ScpUdcEnum.DEO_STATUS_DONE.getValueCode();
        //查询未支付的订单明细行数
        List<String> payStatusList = List.of(ScpUdcEnum.PAY_STATUS_PART_PAY.getValueCode(), ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode());
        long needPayCount = demandOrderDDOS.stream().filter(row -> payStatusList.contains(row.getPayStatus())).count();
        long noPayCount = demandOrderDDOS.stream().filter(row -> ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode().equals(row.getPayStatus())).count();
        String payStatus = (noPayCount == demandOrderDDOS.size()) ? ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode() :
                (needPayCount > 0) ? ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode() :
                        ScpUdcEnum.PAY_STATUS_PAYED.getValueCode();
        scpDemandOrderRepoProc.updateOrderStatusAndPayStatusById(id, status, payStatus);

    }

    @Override
    public Long countDemandOrderByDemandIds(List<Long> demandIds) {
        return scpDemandOrderRepoProc.countDemandOrderByDemandIds(demandIds);
    }

    @Override
    public List<ScpDemandOrderComputeVO> findComputeDemandOrderD(List<Long> orderIds) {
        return scpDemandOrderRepoProc.findComputeDemandOrderDByParam(orderIds);
    }

    @Override
    public List<Long> pageDemandOrderIds(ScpComputeDemandOrderPageParamVO paramVO) {
        return scpDemandOrderRepoProc.pageDemandOrderIds(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDocStatusById(Long id, String type) {
        //过滤运费商品行和项目费用明细行,还有取消的明细行
        List<ScpDemandOrderDDTO> itemList = scpDemandOrderDRepoProc.findByMasId(id).stream()
                .filter(row -> Boolean.FALSE.equals(row.getFreightLineFlag())
                        && Boolean.FALSE.equals(row.getProjectFeeFlag())
                        && !Objects.equals(row.getStatus(), ScpUdcEnum.DEO_STATUS_CLOSE.getValueCode())
                )
                .toList();
        if (ScpConstant.DELIVERY.equals(type)) {
            List<ScpDemandOrderDDTO> recvList = itemList.stream().filter(row -> row.getRecvQty() != null
                    && row.getRecvQty().compareTo(BigDecimal.ZERO) > 0).collect(Collectors.toList());
            if (CollUtil.isNotEmpty(recvList)) {
                return;
            }
            List<ScpDemandOrderDDTO> noDeliverList = itemList.stream().filter(row -> row.getQuantity() == null
                    || row.getQuantity().compareTo(BigDecimal.ZERO) == 0 || row.getQuantity().compareTo(row.getPlanQuantity()) < 0)
                    .collect(Collectors.toList());
            String status = CollUtil.isEmpty(noDeliverList) ? ScpUdcEnum.DEO_STATUS_SHIP.getValueCode() : ScpUdcEnum.DEO_STATUS_PART_SHIP.getValueCode();
            scpDemandOrderRepoProc.updateOrderStatusById(id, status, null);
        }
        if (ScpConstant.RECV.equals(type)) {
            List<ScpDemandOrderDDTO> noRecvList = itemList.stream().filter(row -> row.getRecvQty() == null || row.getRecvQty().compareTo(BigDecimal.ZERO) == 0 || row.getRecvQty().compareTo(row.getPlanQuantity()) < 0).collect(Collectors.toList());
            String status = CollUtil.isEmpty(noRecvList) ? ScpUdcEnum.DEO_STATUS_RECV.getValueCode() : ScpUdcEnum.DEO_STATUS_PART_RECV.getValueCode();
            scpDemandOrderRepoProc.updateOrderStatusById(id, status, null);
        }
        if (ScpConstant.CANCEL.equals(type)) {
            List<String> statusList = List.of(ScpUdcEnum.DEO_STATUS_PART_SHIP.getValueCode(), ScpUdcEnum.DEO_STATUS_SHIP.getValueCode(),
                    ScpUdcEnum.DEO_STATUS_RECV.getValueCode(), ScpUdcEnum.DEO_STATUS_PART_RECV.getValueCode());
            List<ScpDemandOrderComputeVO> demandOrderList = scpDemandOrderRepoProc.findDemandOrderById(id);
            if (CollUtil.isEmpty(demandOrderList)) {
                log.info("取消查询要货单数据为空:{}", id);
            }
            ScpDemandOrderComputeVO order = demandOrderList.get(0);
            if (statusList.contains(order.getDocStatus())) {
                return;
            }
            //查询未推送的订单明细行数
            long noPushCount = itemList.stream().filter(row -> !row.getIsPushed()).count();
            String status = (noPushCount == 0) ? ScpUdcEnum.DEO_STATUS_PUSHED.getValueCode() : (noPushCount == itemList.size() ? ScpUdcEnum.DEO_STATUS_DONE.getValueCode() : ScpUdcEnum.DEO_STATUS_PUSHING.getValueCode());
            Boolean isPushed = ScpUdcEnum.DEO_STATUS_DONE.getValueCode().equals(status);
            scpDemandOrderRepoProc.updateOrderStatusById(id, status, isPushed);
            this.updatePayStatusById(id);
        }
        if (ScpConstant.ZF.equals(type)) {
            List<String> statusList = List.of(ScpUdcEnum.DEO_STATUS_PART_SHIP.getValueCode(), ScpUdcEnum.DEO_STATUS_SHIP.getValueCode(),
                    ScpUdcEnum.DEO_STATUS_RECV.getValueCode(), ScpUdcEnum.DEO_STATUS_PART_RECV.getValueCode());
            List<ScpDemandOrderComputeVO> orders = scpDemandOrderRepoProc.findDemandOrderById(id);
            if (CollUtil.isEmpty(orders)) {
                log.info("没有查询到要货单据：{}", id);
                return;
            }
            ScpDemandOrderComputeVO order = orders.get(0);
            if (statusList.contains(order.getDocStatus())) {
                return;
            }
            List<ScpDemandOrderDDTO> noShipList = itemList.stream().filter(row -> row.getQuantity() == null || row.getQuantity().compareTo(BigDecimal.ZERO) == 0).collect(Collectors.toList());
            String status = (CollUtil.isNotEmpty(noShipList) && noShipList.size() != itemList.size()) ? ScpUdcEnum.DEO_STATUS_PART_SHIP.getValueCode() : ScpUdcEnum.DEO_STATUS_PUSHED.getValueCode();
            scpDemandOrderRepoProc.updateOrderStatusById(id, status, null);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePushStatusById(Long id) {
        var demandOrderComputeVOS = scpDemandOrderRepoProc.findDemandOrderById(id);
        //查询未推送的订单明细行数
        long noPushCount = demandOrderComputeVOS.stream().filter(row -> !row.getIsPushed()).count();
        String status = (noPushCount == 0) ? ScpUdcEnum.DEO_STATUS_PUSHED.getValueCode() : (noPushCount == demandOrderComputeVOS.size() ? ScpConstant.PUSH_DOING : ScpUdcEnum.DEO_STATUS_PUSHING.getValueCode());
        if (!ScpConstant.PUSH_DOING.equals(status)) {
            if (ScpUdcEnum.DEO_STATUS_PUSHED.getValueCode().equals(status)) {
                // 全部推送,需要更新全部推送时间
                LocalDateTime pushedTime = LocalDateTime.now();
                scpDemandOrderRepoProc.updateOrderStatusAndPushedTimeById(id, status, Boolean.TRUE, pushedTime);
                ScpDemandOrderComputeVO scpDemandOrderComputeVO = demandOrderComputeVOS.get(0);
                if (scpDemandOrderComputeVO != null && scpDemandOrderComputeVO.getAllowPayTimeout() != null && scpDemandOrderComputeVO.getAllowPayTimeout()) {
                    // 提交延迟队列，固定时间后未支付设置为超时状态
                    String payTimeoutSetting = SysUtils.getOrderPayTimeoutSetting();
                    long parseLong = 30;
                    try {
                        parseLong = Long.parseLong(payTimeoutSetting);
                    } catch (NumberFormatException e) {
                        log.error("支付超时设置错误，使用默认值30");
                    }
                    orderStatusCountdownService.startCountdown(id, parseLong * 60, ScpUdcEnum.DEO_STATUS_TIMEOUT.getValueCode(), pushedTime);
                }
            } else {
                scpDemandOrderRepoProc.updateOrderStatusById(id, status, Boolean.TRUE);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePayStatusById(Long id) {
        var demandOrderDDTOS = scpDemandOrderDRepoProc.findByMasId(id);
        //查询未推送的订单明细行数
        long payedCount = demandOrderDDTOS.stream().filter(row -> ScpUdcEnum.PAY_STATUS_PAYED.getValueCode().equals(row.getPayStatus())).count();
        long waitPayCount = demandOrderDDTOS.stream().filter(row -> ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode().equals(row.getPayStatus())).count();
        long noPayCount = demandOrderDDTOS.stream().filter(row -> ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode().equals(row.getPayStatus())).count();
        if (noPayCount == demandOrderDDTOS.size()) {
            scpDemandOrderRepoProc.updatePayStatusById(id, ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode());
        } else {
            String status = (waitPayCount > 0 && payedCount == 0) ? ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode() : (payedCount == (demandOrderDDTOS.size() - noPayCount) ? ScpUdcEnum.PAY_STATUS_PAYED.getValueCode() :
                    ScpUdcEnum.PAY_STATUS_PART_PAY.getValueCode());
            scpDemandOrderRepoProc.updatePayStatusById(id, status);
        }

    }

    @Override
    public PagingVO<ScpPayOrderPageRespVO> payPage(ScpDemandOrderPageParamVO paramVO) {
        long count = scpDemandOrderRepoProc.countAppPayPage(paramVO);
        if (count > 0) {
            var demandOrder = scpDemandOrderRepoProc.payPage(paramVO);
            List<Long> ids = demandOrder.stream().map(ScpPayOrderPageRespVO::getDocId).collect(Collectors.toList());
            Map<Long, ScpPayOrderPageRespVO> payOrderMap = scpDemandOrderRepoProc.findDemandOrderByIds(ids).stream().collect(Collectors.toMap(ScpPayOrderPageRespVO::getDocId, Function.identity()));
            List<ScpDemandOrderDDTO> itemList = scpDemandOrderDRepoProc.findItemListByMasId(ids);
            Map<Long, List<ScpDemandOrderDDTO>> orderMap = itemList.stream().collect(Collectors.groupingBy(ScpDemandOrderDDTO::getMasId));
            demandOrder.forEach(row -> {
                var pageRespVO = payOrderMap.get(row.getDocId());
                row.setDocCode(pageRespVO.getDocCode());
                row.setDocCls(pageRespVO.getDocCls());
                row.setDocType(pageRespVO.getDocType());
                row.setDocStatus(pageRespVO.getDocStatus());
                row.setCreateTime(pageRespVO.getCreateTime());
                List<ScpDemandOrderDDTO> demandOrderDDTOS = orderMap.get(row.getId());
                if (CollUtil.isNotEmpty(demandOrderDDTOS)) {
                    List<String> imgUrls = demandOrderDDTOS.stream().map(ScpDemandOrderDDTO::getImgUrl).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                    if (CollUtil.isNotEmpty(imgUrls)) {
                        row.setItemImgUrls(imgUrls);
                    }
                }
            });
            return new PagingVO<>(count, demandOrder);
        }
        return new PagingVO<>();
    }

    @Override
    public AppPayOrderRespVO payOrderBySrcDocId(Long srcDocId) {
        return scpDemandOrderRepoProc.findPayOrderBySrcDocId(srcDocId);
    }

    @Override
    public List<AppPayOrderRespVO> payOrderById(Long id) {
        return scpDemandOrderRepoProc.findPayOrderById(id);
    }

    @Override
    public Long countAppOrderByParam(ScpDemandOrderPageParamVO paramVO) {
        return scpDemandOrderRepoProc.countAppOrderByParam(paramVO);
    }

    @Override
    public Long countAppPayOrderByParam(ScpDemandOrderPageParamVO paramVO) {
        return scpDemandOrderRepoProc.countAppPayPage(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateReplyFlagById(Long id) {
        scpDemandOrderRepo.updateReplyById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void closeDemandOrder(Long id, String cancelOperator) {
        scpDemandOrderRepo.closeDemandOrder(id,cancelOperator);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateOrderStatusById(Long id, String docStatus) {
        scpDemandOrderRepoProc.updateOrderStatusById(id, docStatus);
    }

    @Override
    public Long findIdByCode(String docCode) {
        return scpDemandOrderRepoProc.findIdByCode(docCode);
    }

    /**
     * 订货单导出
     *
     * @param queryParamVO
     * @return
     */
    @Override
    public PagingVO<ScpDemandOrderExportRespVO> executeExport(ScpDemandOrderPageParamVO queryParamVO) {
        // 查询计划员信息
        var authed = appendScpsmanAuth(queryParamVO);
        if (!authed) {
            return PagingVO.empty();
        }
       return scpDemandOrderExportRepoProc.executeExport(queryParamVO);
    }
}
