package com.elitesland.scp.infr.repo.article;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.domain.dto.article.ArticleCategoryDTO;
import com.elitesland.scp.domain.entity.article.ArticleCategoryDO;
import com.elitesland.scp.domain.entity.article.QArticleCategoryDO;
import com.elitesland.scp.domain.vo.article.ArticleCategoryPagingParam;
import com.elitesland.scp.domain.vo.article.UpCategoryRespVO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import org.springframework.stereotype.Repository;

import java.util.List;
import java.util.Set;

@Repository
public class ArticleCategoryRepoProc extends BaseRepoProc<ArticleCategoryDO> {

    private final static QArticleCategoryDO Q_DO = QArticleCategoryDO.articleCategoryDO;

    private final static QArticleCategoryDO P_Q_DO = QArticleCategoryDO.articleCategoryDO;

    protected ArticleCategoryRepoProc() {
        super(Q_DO);
    }

    public PagingVO<ArticleCategoryDTO> findParentCategoryPagingResults(Set<Long> pidSet, ArticleCategoryPagingParam param) {
        JPAQuery<ArticleCategoryDTO> query = select(ArticleCategoryDTO.class)
                .where(whereByPid(pidSet))
                .orderBy(Q_DO.showFlag.asc())
                .orderBy(Q_DO.sortNo.desc())
                .orderBy(Q_DO.createTime.desc());
        param.setPaging(query);
        return PagingVO.<ArticleCategoryDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    private Predicate whereByPid(Set<Long> pidSet) {
        return PredicateBuilder.builder()
                .andIsNull(CollectionUtil.isEmpty(pidSet), Q_DO.pid)
                .andIn(CollectionUtil.isNotEmpty(pidSet), Q_DO.id, pidSet)
                .build();
    }

    public List<ArticleCategoryDTO> queryUpCategory() {
        return select(ArticleCategoryDTO.class)
                .where(Q_DO.pid.isNull())
                .orderBy(Q_DO.sortNo.asc())
                .orderBy(Q_DO.createTime.asc())
                .fetch();
    }

    public List<ArticleCategoryDTO> findByPidInAndShowFlag(List<Long> parentIdList, ArticleCategoryPagingParam param) {
        return select(ArticleCategoryDTO.class)
                .where(whereByPidInAndShowFlag(parentIdList, param))
                .orderBy(Q_DO.showFlag.asc())
                .orderBy(Q_DO.sortNo.desc())
                .orderBy(Q_DO.createTime.desc())
                .fetch();
    }

    private Predicate whereByPidInAndShowFlag(List<Long> parentIdList, ArticleCategoryPagingParam param) {
        return PredicateBuilder.builder()
                .andIn(CollectionUtil.isNotEmpty(parentIdList), Q_DO.pid, parentIdList)
                .andEq(StrUtil.isNotBlank(param.getShowFlag()), Q_DO.showFlag, param.getShowFlag())
                .andLike(StrUtil.isNotBlank(param.getCategoryName()), Q_DO.categoryName, param.getCategoryName())
                .build();
    }

    public List<ArticleCategoryDTO> findByCategoryNameAndShowFlag(ArticleCategoryPagingParam param) {
        return select(ArticleCategoryDTO.class)
                .where(whereByCategoryNameAndShowFlag(param))
                .orderBy(Q_DO.showFlag.asc())
                .orderBy(Q_DO.sortNo.desc())
                .orderBy(Q_DO.createTime.desc())
                .fetch();
    }

    private Predicate whereByCategoryNameAndShowFlag(ArticleCategoryPagingParam param) {
        return PredicateBuilder.builder()
                .andLike(StrUtil.isNotBlank(param.getCategoryName()), Q_DO.categoryName, param.getCategoryName())
                .andEq(StrUtil.isNotBlank(param.getShowFlag()), Q_DO.showFlag, param.getShowFlag())
                .build();
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                Q_DO.id,
                Q_DO.categoryName,
                Q_DO.showFlag,
                Q_DO.pid,
                Q_DO.sortNo,
                Q_DO.rootId
        )).from(Q_DO);
    }

    public List<ArticleCategoryDTO> findAll() {
        return select(ArticleCategoryDTO.class)
                .orderBy(Q_DO.id.asc())
                .fetch();
    }

    public List<UpCategoryRespVO> queryUpCategory(Long id) {
        return jpaQueryFactory.select(Projections.bean(UpCategoryRespVO.class,
                        P_Q_DO.id,
                        P_Q_DO.categoryName
                )).from(Q_DO).innerJoin(P_Q_DO).on(P_Q_DO.id.eq(Q_DO.pid))
                .where(Q_DO.id.eq(id))
                .fetch();
    }

    public List<ArticleCategoryDTO> findAllByShowFlag(String showFlag) {
        return select(ArticleCategoryDTO.class)
                .where(Q_DO.showFlag.eq(showFlag))
                .orderBy(Q_DO.id.asc())
                .fetch();
    }

}
