package com.elitesland.scp.application.service.authority;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.save.authority.ScpManAuthoritySaveVO;
import com.elitesland.scp.domain.convert.StoreInfoConvert;
import com.elitesland.scp.domain.convert.authority.ScpDemandAuthorityConvert;
import com.elitesland.scp.dto.authority.*;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.service.authority.StoreAuthorityRpcService;
import com.elitesland.support.provider.item.dto.ItmItemAttachmentProviderDTO;
import com.elitesland.support.provider.item.dto.ItmItemBusinessRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemBusinessRpcPagingDTO;
import com.elitesland.support.provider.item.param.ItmItemBusinessRpcPagingParam;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/3/27 10:48
 * @Description:
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class StoreAuthorityRpcServiceImpl implements StoreAuthorityRpcService {

    private final RmiOrgStoreRpcService rmiOrgStoreService;
    private final RmiItemService rmiItemService;
    private final ScpsmanAuthorityService scpsmanAuthorityService;

    @Value("${ext.server-addr}")
    private String serverAddr;


    @SysCodeProc
    @Override
    public ApiResult<PagingVO<ItemObject>> getStoreAuthorityDataPage(StoreAuthorityRpcPageDTO storeAuthorityRpcPageDTO) {
        log.info("门店要货权限数据分页获取，入参:{}", JSONUtil.toJsonStr(storeAuthorityRpcPageDTO));
        String storeCode = storeAuthorityRpcPageDTO.getStoreCode();
        if (StrUtil.isBlank(storeCode)) {
            throw new BusinessException("参数门店编码不能为空");
        }
        OrgStoreBaseRpcDTO storeDto = rmiOrgStoreService.getOrgStoreBase(storeCode);
        log.info("门店信息:{}", JSONUtil.toJsonStr(storeDto));
        if (storeDto == null) {
            return ApiResult.ok();
        }

        String ouCode = storeDto.getOuCode();
        String brandCode = storeDto.getBrandCode();

        ItmItemBusinessRpcPagingParam param = new ItmItemBusinessRpcPagingParam();
        if (StrUtil.isNotBlank(ouCode)) {
            param.setBuCodes(Collections.singletonList(ouCode));
        }
        if (StrUtil.isNotBlank(brandCode)) {
            param.setBrandList(Collections.singletonList(brandCode));
        }
        if (StrUtil.isNotBlank(storeAuthorityRpcPageDTO.getItemCodeName())) {
            param.setItemCodeName(storeAuthorityRpcPageDTO.getItemCodeName());
        }
        if (CollUtil.isNotEmpty(storeAuthorityRpcPageDTO.getItemType2s())) {
            param.setItemType2s(storeAuthorityRpcPageDTO.getItemType2s());
        }
        param.setCurrent(storeAuthorityRpcPageDTO.getCurrent() + 2);
        param.setSize(storeAuthorityRpcPageDTO.getSize());
        PagingVO<ItmItemBusinessRpcPagingDTO> itemBusinessRpcDTOS = rmiItemService.findItemBusinessRpcDtoByParam(param);

        long total = itemBusinessRpcDTOS.getTotal();
        List<ItmItemBusinessRpcPagingDTO> records = itemBusinessRpcDTOS.getRecords();

        log.info("调用基础域分页查询商品结果:{}", JSONUtil.toJsonStr(records));

        List<Long> fetchImageIds = records.stream().map(ItmItemBusinessRpcPagingDTO::getSpuId).toList();
        //获取商品图片信息
        List<ItmItemAttachmentProviderDTO> skuImgList = rmiItemService.findSkuImgByItemIds(fetchImageIds);
        Map<Long, List<ItmItemAttachmentProviderDTO>> skuImgMap = new HashMap<>();
        if (CollUtil.isNotEmpty(skuImgList)) {
            skuImgMap = skuImgList.stream().collect(Collectors.groupingBy(ItmItemAttachmentProviderDTO::getItemId));
        }

        List<ItemObject> itemInfoList = new ArrayList<>();
        for (ItmItemBusinessRpcPagingDTO dto : records) {
            ItemObject itemInfo = new ItemObject();
            itemInfo.setItemCode(dto.getItemCode());
            itemInfo.setItemId(dto.getItemId());
            itemInfo.setItemName(dto.getItemName());
            itemInfo.setUom(dto.getUom());
            itemInfo.setUomName(dto.getUomName());
            itemInfo.setTaxRate(dto.getTaxRate2());
            itemInfo.setBrand(dto.getBrand());
            itemInfo.setBrandName(dto.getBrandName());
            itemInfo.setItemCateCode(dto.getItemCateCode());
            itemInfo.setItemCateFullName(dto.getItemCateFullName());
            itemInfo.setItemCatePathName(dto.getItemCatePathName());
            itemInfo.setSpec(dto.getSpec());
            itemInfo.setSpuCode(dto.getSpuCode());
            itemInfo.setSpuName(dto.getSpuName());
            itemInfo.setPackageSpec(dto.getPackageSpec());
            itemInfo.setSuppCode(dto.getSuppCode());
            itemInfo.setSuppName(dto.getSuppName());
            itemInfo.setUom2(dto.getUom2());
            itemInfo.setUom2Name(dto.getUom2Name());
            itemInfo.setItemType(dto.getItemType2());
            itemInfo.setSpuId(dto.getSpuId());
            List<ItmItemAttachmentProviderDTO> v = skuImgMap.get(dto.getSpuId());
            if (CollUtil.isNotEmpty(v)) {
                ItmItemAttachmentProviderDTO itmItemAttachmentProviderDTO = v.stream().filter(i -> i.getMajor() != null && i.getMajor()).findFirst().orElse(null);
                if (itmItemAttachmentProviderDTO == null) {
                    itmItemAttachmentProviderDTO = v.get(0);
                }
                String fileUrl = serverAddr + "/coordinator/el-fsm-service/api/fsm/download/" + itmItemAttachmentProviderDTO.getFileCode();
                itemInfo.setImgUrl(fileUrl);
            }
            itemInfoList.add(itemInfo);
        }
        PagingVO<ItemObject> result = new PagingVO<>();
        result.setRecords(itemInfoList);
        result.setTotal(total);
        log.info("门店要货权限数据-分页:{}", JSONUtil.toJsonStr(result));
        return ApiResult.ok(result);
    }


    @SysCodeProc
    @Override
    public ApiResult<StoreAuthorityRespDTO> getStoreAuthorityData(StoreAuthorityRpcDTO storeAuthorityRpcDTO) {
        String storeCode = storeAuthorityRpcDTO.getStoreCode();
        if (StrUtil.isBlank(storeCode)) {
            throw new BusinessException("参数门店编码不能为空");
        }
        OrgStoreBaseRpcDTO storeDto = rmiOrgStoreService.getOrgStoreBase(storeCode);
        log.info("门店信息:{}", JSONUtil.toJsonStr(storeDto));
        if (storeDto == null) {
            return ApiResult.ok();
        }
        StoreAuthorityRespDTO storeAuthorityRespDTO = StoreInfoConvert.INSTANCE.toStoreAuthorityRespDTO(storeDto);
        String ouCode = storeAuthorityRespDTO.getOuCode();
        String brandCode = storeAuthorityRespDTO.getBrandCode();
        List<ItmItemBusinessRpcDTO> itmItemBusinessRpcDTOS =
                rmiItemService.findItmItemBusinessByBUCodeAndBrand(ouCode, brandCode);
        log.info("调用基础域查询商品结果:{}", JSONUtil.toJsonStr(itmItemBusinessRpcDTOS));

        List<StoreAuthorityRespDTO.ItemInfo> itemInfoList = new ArrayList<>();
        for (ItmItemBusinessRpcDTO dto : itmItemBusinessRpcDTOS) {
            StoreAuthorityRespDTO.ItemInfo itemInfo = new StoreAuthorityRespDTO.ItemInfo();
            itemInfo.setItemCode(dto.getItemCode());
            itemInfo.setItemId(dto.getId());
            itemInfo.setItemName(dto.getItemName());
            itemInfo.setUom(dto.getUom());
            itemInfo.setUomName(dto.getUomName());
            itemInfo.setTaxRate(dto.getTaxRate2());
            itemInfoList.add(itemInfo);
        }
        storeAuthorityRespDTO.setItemInfos(itemInfoList);
        log.info("门店要货权限数据:{}", JSONUtil.toJsonStr(storeAuthorityRespDTO));
        return ApiResult.ok(storeAuthorityRespDTO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> saveAuthority(ScpManAuthorityRpcSaveVO saveVO) {
        log.info("保存计划员权限RPC接口开始：{}", JSONUtil.toJsonStr(saveVO));
        ScpManAuthoritySaveVO scpManAuthoritySaveVO = ScpDemandAuthorityConvert.INSTANCE.rpc2SaveVo(saveVO);
        Long id = scpsmanAuthorityService.saveScpAuthority(scpManAuthoritySaveVO);
        log.info("保存计划员权限RPC接口结束：{}", id);
        return ApiResult.ok(id);
    }
}
