package com.elitesland.scp.rmi;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.sale.api.vo.param.crm.CustCode2BaseParam;
import com.elitesland.sale.api.vo.resp.crm.CustBaseAndBelongOuDTO;
import com.elitesland.sale.api.vo.resp.crm.CustBaseDTO;
import com.elitesland.sale.api.vo.resp.crm.CustBelongOuDTO;
import com.elitesland.sale.dto.CrmCustDTO;
import com.elitesland.sale.dto.CustBaseInfoDTO;
import com.elitesland.sale.dto.param.CustBaseRpcParam;
import com.elitesland.sale.service.CrmCustRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Component
@Slf4j
@RequiredArgsConstructor
public class RmiSalRpcService {

    private final CrmCustRpcService crmCustRpcService;

    /**
     * 根据客户编码查询客户信息
     *
     * @param custCodes
     * @return
     */
    public List<CrmCustDTO> listCustByCode(List<String> custCodes) {
        if (CollUtil.isEmpty(custCodes)) {
            return new ArrayList<>();
        }
        ApiResult<List<CrmCustDTO>> custListResult = crmCustRpcService.listCustByCode(custCodes);
        if (!custListResult.isSuccess()) {
            throw new BusinessException("查询客户数据信息接口异常");
        }
        return custListResult.getData();
    }

    public List<CustBaseInfoDTO> getCustBaseInfoByCode(List<String> custCodes) {
        if (CollectionUtil.isEmpty(custCodes)) {
            return Collections.EMPTY_LIST;
        }
        CustCode2BaseParam custCode2BaseParam = new CustCode2BaseParam();
        custCode2BaseParam.setCustCode(custCodes);
        ApiResult<List<CustBaseInfoDTO>> custListResult = crmCustRpcService.getCustBaseInfoByCode(custCode2BaseParam);
        if (!custListResult.isSuccess()) {
            throw new BusinessException("查询客户(getCustBaseInfoByCode)数据信息接口异常");
        }
        return custListResult.getData();
    }

    public List<CustBaseDTO> findBaseByParam(List<String> custCodes) {
        if (CollectionUtil.isEmpty(custCodes)) {
            return Collections.EMPTY_LIST;
        }
        CustBaseRpcParam custBaseRpcParam = new CustBaseRpcParam();
        custBaseRpcParam.setCustCodeList(custCodes);
        ApiResult<List<CustBaseDTO>> custListResult = crmCustRpcService.findBaseByParam(custBaseRpcParam);
        if (!custListResult.isSuccess()) {
            throw new BusinessException("查询客户(findBaseByParam)数据信息接口异常");
        }
        return custListResult.getData();
    }

    /**
     * 根据客户编码查询归属公司信息
     *
     * @param custCode
     * @return
     */
    public List<CustBelongOuDTO> getCustByCode(String custCode) {
        try {
            return crmCustRpcService.findCustBelongOuByCustCode(custCode);
        } catch (Exception e) {
            log.error("查询客户归属公司信息异常:{}", e);
            throw new BusinessException("查询客户归属公司信息异常");
        }
    }

    /**
     * 根据客户编码集合查询归属公司信息
     *
     * @param custCodes
     * @return
     */
    public List<CustBelongOuDTO> getCustByCodes(List<String> custCodes) {
        try {
            return crmCustRpcService.findCustBelongOuByCustCodes(custCodes);
        } catch (Exception e) {
            log.error("查询客户归属公司信息异常:{}", e);
            throw new BusinessException("查询客户归属公司信息异常");
        }
    }

    public Map<String, List<CustBelongOuDTO>> findPreCustBaseAndBelongOuMap(List<String> custCodes) {
        log.info("查询主客户归属公司信息入参:{}", JSONUtil.toJsonStr(custCodes));
        if (CollUtil.isEmpty(custCodes)) {
            return new HashMap<>();
        }
        ApiResult<Map<String, List<CustBelongOuDTO>>> custBelongOu = crmCustRpcService.findPreCustBelongOuByCustCodes(custCodes);
        if (!custBelongOu.isSuccess()) {
            log.error("查询客户数据信息接口异常:{},详情:{}", custBelongOu.getErrorMsg(), JSONUtil.toJsonStr(custBelongOu));
            throw new BusinessException("查询客户数据信息接口异常");
        }
        Map<String, List<CustBelongOuDTO>> data = custBelongOu.getData();
        if (CollUtil.isEmpty(data)) {
            return new HashMap<>();
        }
        log.info("查询主客户归属公司信息结果:{}", data);
        return data;
    }

    public List<CustBaseAndBelongOuDTO> findCustBaseAndBelongOuByParam(List<String> custCodes) {
        if (CollUtil.isEmpty(custCodes)) {
            return new ArrayList<>();
        }
        CustBaseRpcParam custBaseRpcParam = new CustBaseRpcParam();
        custBaseRpcParam.setCustCodeList(custCodes);
        ApiResult<List<CustBaseAndBelongOuDTO>> result = crmCustRpcService.findCustBaseAndBelongOuByParam(custBaseRpcParam);
        if (!result.isSuccess()) {
            throw new BusinessException("查询客户数据信息接口异常");
        }
        return result.getData();
    }

    public Map<String, CustBaseDTO> getCustBaseMapByCode(List<String> custCodes) {
        if (CollectionUtil.isEmpty(custCodes)) {
            return new HashMap<>();
        }
        CustBaseRpcParam custBaseParam = new CustBaseRpcParam();
        custBaseParam.setCustCodeList(custCodes);
        ApiResult<List<CustBaseDTO>> custListResult = crmCustRpcService.findBaseByParam(custBaseParam);
        if (!custListResult.isSuccess()) {
            throw new BusinessException("查询客户(findBaseByParam)数据信息接口异常");
        }
        return custListResult.getData().stream().collect(Collectors.toMap(CustBaseDTO::getCustCode, Function.identity()));
    }
}
