package com.elitesland.scp.application.service.app;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.scp.application.facade.vo.param.app.ScpStoreItemParamVO;
import com.elitesland.scp.domain.convert.app.ScpStoreItemConvert;
import com.elitesland.scp.domain.entity.item.ScpStoreItemDO;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.utils.ScpCacheUtils;
import com.elitesland.support.provider.item.dto.ItmItemScpBaseRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemScpBaseRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpCacheServiceImpl implements ScpCacheService {
    private final ScpCacheUtils scpCacheUtils;
    private final RedissonClient redissonClient;
    private final RmiItemService rmiItemService;

    @Override
    public List<ScpStoreItemDO> getItem(ScpStoreItemParamVO paramVO) {
        log.info("开始从缓存管理器获取商品信息，入参:{}", JSONUtil.toJsonStr(paramVO));
        List<ScpStoreItemDO> scpStoreItemDOS = scpCacheUtils.getItemBySpuId(paramVO.getSpuId());
        if (CollUtil.isNotEmpty(scpStoreItemDOS)) {
            return scpStoreItemDOS;
        }
        ItmItemScpBaseRpcParam itemScpBaseRpcParam = new ItmItemScpBaseRpcParam();
        itemScpBaseRpcParam.setBusinessBuCode(paramVO.getOuCode());
        itemScpBaseRpcParam.setSpuIds(Arrays.asList(paramVO.getSpuId()));
        List<ItmItemScpBaseRpcDTO> itemScpBaseRpcDTOS =
                rmiItemService.findItemScpBaseRpcDtoByParam(itemScpBaseRpcParam);
        if (CollectionUtils.isEmpty(itemScpBaseRpcDTOS)) {
            throw new BusinessException("商品SPU【" + paramVO.getSpuId() + "】不存在");
        }
        log.info("结束从缓存管理器获取商品信息");
        return buildItem(paramVO.getSpuId(), itemScpBaseRpcDTOS);
    }

    @Override
    public void deleteKeys(List<Long> itemIds) {
        for (Long itemId : itemIds) {
            scpCacheUtils.deleteKeys(itemId);
        }
    }

    @Override
    public void deleteAllKeys() {
        scpCacheUtils.deleteAllKeys();
    }

    @Override
    public void deleteSpecKeys(String spec) {
        scpCacheUtils.deleteSpecKeys(spec);
    }

    /**
     * SCP自定义查询商品信息
     *
     * @param spuId
     * @param itemList
     * @return
     */
    private List<ScpStoreItemDO> buildItem(Long spuId, List<ItmItemScpBaseRpcDTO> itemList) {
        String lockKey = scpCacheUtils.generateLockKey(spuId);
        RLock lock = redissonClient.getLock(lockKey);
        try {
            if (lock.tryLock(300L, 30L, TimeUnit.SECONDS)) {
                // 加锁成功，查DB前再查一次缓存
                List<ScpStoreItemDO> itemDOList = scpCacheUtils.getItemBySpuId(spuId);
                if (CollUtil.isNotEmpty(itemDOList)) {
                    return itemDOList;
                } else {
                    // 查DB构建缓存
                    List<ScpStoreItemDO> saveItems = itemList.stream().map(row -> {
                        ScpStoreItemDO itemDO = ScpStoreItemConvert.INSTANCE.dtoToDo(row);
                        itemDO.setSkuAttachmentList(row.getSkuAttchmentList());
                        itemDO.setSpuAttachmentList(row.getSpuAttchmentList());
                        itemDO.setItemId(row.getId());
                        itemDO.setUom(row.getUom2());
                        itemDO.setUomName(row.getUom2Name());
                        itemDO.setUom2(row.getUom());
                        itemDO.setUom2Name(row.getUomName());
                        itemDO.setUomRatio(row.getUomRatio2());
                        itemDO.setMoq(row.getMoq());
                        return itemDO;
                    }).collect(Collectors.toList());
                    // 更新缓存
                    scpCacheUtils.saveItem(spuId, saveItems);
                    return saveItems;
                }
            }
        } catch (BusinessException | InterruptedException e) {
            log.error("lock error:", e);
            throw new BusinessException(e.getMessage());
        } finally {
            if (lock.isLocked() && lock.isHeldByCurrentThread()) {
                lock.unlock();
                log.info("unlock success,lockKey:{}", lockKey);
            }
        }
        return new ArrayList<>();
    }
}
