package com.elitesland.scp.domain.service.survey;

import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitesland.scp.application.facade.vo.scpsman.SalesmanInfoQueryVO;
import com.elitesland.scp.application.facade.vo.scpsman.SalesmanInfoRespVO;
import com.elitesland.scp.domain.convert.survey.SurveyConvert;
import com.elitesland.scp.domain.convert.survey.SurveyOptionConvert;
import com.elitesland.scp.domain.convert.survey.SurveyPublishConvert;
import com.elitesland.scp.domain.convert.survey.SurveyQuestionConvert;
import com.elitesland.scp.domain.entity.scpsman.ScpsmanRegionDO;
import com.elitesland.scp.domain.entity.survey.SurveyDO;
import com.elitesland.scp.domain.entity.survey.SurveyOptionDO;
import com.elitesland.scp.domain.entity.survey.SurveyPublishDO;
import com.elitesland.scp.domain.entity.survey.SurveyQuestionDO;
import com.elitesland.scp.domain.vo.survey.*;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.repo.scpsman.SalesmanRegionRepo;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepoProc;
import com.elitesland.scp.infr.repo.survey.*;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@Slf4j
public class SurveyServiceImpl implements SurveyService {

    private final SurveyRepo surveyRepo;
    private final SurveyRepoProc surveyRepoProc;
    private final SurveyQuestionRepo surveyQuestionRepo;
    private final SurveyOptionRepo surveyOptionRepo;
    private final SurveyPublishRepo surveyPublishRepo;
    private final ScpsmanInfoRepoProc scpsmanInfoRepoProc;
    private final SalesmanRegionRepo salesmanRegionRepo;

    @Override
    @SysCodeProc
    public PagingVO<SurveyRespVO> search(SurveyPagingParam searchParam) {
        PagingVO<SurveyRespVO> search = surveyRepoProc.search(searchParam);
        if(search.getTotal() > 0){
            List<SurveyRespVO> records = search.getRecords();

            // 补充发布对象
            List<Long> suvIdList = records.stream().map(SurveyRespVO::getId).toList();
            Map<Long, List<SurveyPublishDO>> publishMap = surveyPublishRepo.findBySuvIdIn(suvIdList).stream().collect(Collectors.groupingBy(SurveyPublishDO::getSuvId));

            for(SurveyRespVO surveyRespVO : records){
                if(publishMap.containsKey(surveyRespVO.getId())){
                    String collect = publishMap.get(surveyRespVO.getId()).stream().map(SurveyPublishDO::getOuName).collect(Collectors.joining(","));
                    surveyRespVO.setPublishOuName(collect);
                }
            }
        }
        return search;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdate(SurveySaveParam saveVO) {
        SurveyDO surveyDO = SurveyConvert.INSTANCE.saveVOToDo(saveVO);
        surveyDO.setSurveyStatus(ScpUdcEnum.SURVEY_STATUS_10.getValueCode());
        SurveyDO save = surveyRepo.save(surveyDO);
        // 删除问题和选项
        List<SurveyQuestionDO> questionDOList = surveyQuestionRepo.findBySuvId(save.getId());
        if(CollectionUtils.isNotEmpty(questionDOList)){
            List<Long> qusIdList = questionDOList.stream().map(SurveyQuestionDO::getId).toList();
            surveyQuestionRepo.deleteByIdIn(qusIdList);
            surveyOptionRepo.deleteByQusIdIn(qusIdList);
        }
        // 保存问题和选项
        if(CollectionUtils.isNotEmpty(saveVO.getQuestionList())){
            for(SurveyQuestionSaveParam questionVO : saveVO.getQuestionList()){
                SurveyQuestionDO questionDO = SurveyQuestionConvert.INSTANCE.saveParamToDo(questionVO);
                questionDO.setSuvId(save.getId());
                SurveyQuestionDO question = surveyQuestionRepo.save(questionDO);
                if(CollectionUtils.isNotEmpty(questionVO.getOptionList())){
                    for(SurveyOptionSaveParam optionVO : questionVO.getOptionList()){
                        SurveyOptionDO optionDO = SurveyOptionConvert.INSTANCE.saveParamToDo(optionVO);
                        optionDO.setQusId(question.getId());
                        surveyOptionRepo.save(optionDO);
                    }
                }
            }
        }
        return save.getId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancel(List<Long> idList) {
        surveyRepoProc.updateSurveyStatus(idList, ScpUdcEnum.SURVEY_STATUS_30.getValueCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer publish(SurveyPublishParam param) {
        if(param.getId() == null || CollectionUtils.isEmpty(param.getPublishList())){
            throw new RuntimeException("参数错误");
        }
        // 可多次发布，每次发布不影响原有数据
        List<SurveyPublishDO> publishDOS = surveyPublishRepo.findBySuvId(param.getId());
        List<String> ouCodes = publishDOS.stream().map(SurveyPublishDO::getOuCode).toList();

        List<SurveyPublishDO> publishList = new ArrayList<>();
        for(SurveyPublishSaveParam publishParam : param.getPublishList()){
            if(!ouCodes.contains(publishParam.getOuCode())){
                SurveyPublishDO publishDO = SurveyPublishConvert.INSTANCE.saveParamToDo(publishParam);
                publishDO.setSuvId(param.getId());
                publishDO.setFillStatus(ScpUdcEnum.SURVEY_FILL_STATUS_00.getValueCode());
                publishList.add(publishDO);
            }
        }
        if(CollectionUtils.isNotEmpty(publishList)){
            surveyPublishRepo.saveAll(publishList);
        }
        surveyRepoProc.updateSurveyStatus(Collections.singletonList(param.getId()), ScpUdcEnum.SURVEY_STATUS_20.getValueCode());
        return publishList.size();
    }

    @Override
    @SysCodeProc
    public SurveyRespVO get(Long id) {
        Optional<SurveyDO> byId = surveyRepo.findById(id);
        if(byId.isEmpty()){
            throw new RuntimeException("问卷不存在");
        }
        SurveyRespVO surveyRespVO = SurveyConvert.INSTANCE.doToRespVO(byId.get());
        // 查询问题及选项
        List<SurveyQuestionDO> questionDOList = surveyQuestionRepo.findBySuvId(id);
        if(CollectionUtils.isNotEmpty(questionDOList)){
            List<SurveyQuestionRespVO> questionRespVOList = SurveyQuestionConvert.INSTANCE.doListToRespVOList(questionDOList);

            List<Long> qusIdList = questionRespVOList.stream().map(SurveyQuestionRespVO::getId).distinct().toList();
            Map<Long, List<SurveyOptionDO>> collect = surveyOptionRepo.findByQusIdIn(qusIdList).stream().collect(Collectors.groupingBy(SurveyOptionDO::getQusId));
            for(SurveyQuestionRespVO questionRespVO : questionRespVOList){
                if(collect.containsKey(questionRespVO.getId())){
                    List<SurveyOptionDO> surveyOptionDOList = collect.get(questionRespVO.getId());
                    List<SurveyOptionRespVO> optionRespVOList = SurveyOptionConvert.INSTANCE.doListToRespVOList(surveyOptionDOList);
                    questionRespVO.setOptionList(optionRespVOList);
                }
            }
            surveyRespVO.setQuestionList(questionRespVOList);
        }
        return surveyRespVO;
    }

    @Override
    public SurveyRespVO findSurvey() {
        // 获取当前登陆人信息，只有门店角色才能填写问卷
        GeneralUserDetails generalUserDetails = SecurityContextUtil.currentUser();
        if(generalUserDetails == null || generalUserDetails.getUser() == null){
            log.info("获取问卷：当前用户信息为空");
            return null;
        }
        SalesmanInfoQueryVO queryVO = new SalesmanInfoQueryVO();
        queryVO.setLoginAccount(generalUserDetails.getUser().getUsername());
        JPAQuery<SalesmanInfoRespVO> jpaQuery = scpsmanInfoRepoProc.findSalesmanInfo(queryVO);
        List<SalesmanInfoRespVO> salesmanInfoList = jpaQuery.fetch();
        if(CollectionUtils.isEmpty(salesmanInfoList)){
            log.info("获取问卷：订货账号为空");
            return null;
        }
        SalesmanInfoRespVO salesmanInfoRespVO = salesmanInfoList.get(0);
        if(UdcEnum.SALE_SCPSMAN_TYPE_SHOPOWNER.getValueCode().equals(salesmanInfoRespVO.getScpsmanType())){
            List<ScpsmanRegionDO> regionDO = salesmanRegionRepo.findByMasIdIn(List.of(salesmanInfoRespVO.getId()));
            if(CollectionUtils.isNotEmpty(regionDO)){
                String regionCode = regionDO.get(0).getRegionCode();
                // 查询在有效期内，并且发布至改门店的，未填写的问卷
                SurveyPagingParam param = new SurveyPagingParam();
                param.setNeedFilterValidTime("Y");
                param.setOuCode(regionCode);
                param.setSurveyStatus(ScpUdcEnum.SURVEY_STATUS_20.getValueCode());
                param.setFillStatus(ScpUdcEnum.SURVEY_FILL_STATUS_00.getValueCode());
                List<SurveyRespVO> survey = surveyRepoProc.findSurvey(param);
                if(CollectionUtils.isNotEmpty(survey)){
                    return this.get(survey.get(0).getId());
                }
            }
        }
        return null;
    }

}
