package com.elitesland.scp.infr.repo.survey;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.domain.entity.survey.QSurveyDO;
import com.elitesland.scp.domain.entity.survey.QSurveyPublishDO;
import com.elitesland.scp.domain.entity.survey.SurveyDO;
import com.elitesland.scp.domain.vo.survey.SurveyPagingParam;
import com.elitesland.scp.domain.vo.survey.SurveyRespVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.JPAExpressions;
import com.querydsl.jpa.JPQLQuery;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAUpdateClause;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

@Repository
public class SurveyRepoProc extends BaseRepoProc<SurveyDO> {

    private final static QSurveyDO surveyDO = QSurveyDO.surveyDO;
    private final static QSurveyPublishDO surveyPublishDO = QSurveyPublishDO.surveyPublishDO;

    protected SurveyRepoProc() {
        super(surveyDO);
    }


    /**
     * 分页查询
     *
     * @param queryVO 条件
     * @return 分页结果
     */
    public PagingVO<SurveyRespVO> search(SurveyPagingParam queryVO) {
        JPAQuery<SurveyRespVO> jpaQuery = select(SurveyRespVO.class)
                .from(surveyDO)
                .where(where(queryVO))
                .orderBy(surveyDO.createTime.desc());;
        queryVO.setPaging(jpaQuery);
        return PagingVO.<SurveyRespVO>builder()
                .total(jpaQuery.fetchCount())
                .records(jpaQuery.fetch())
                .build();
    }

    private Predicate where(SurveyPagingParam param) {
        Predicate predicate = Expressions.ONE.eq(Expressions.ONE);
        if (StringUtils.isNotEmpty(param.getSurveyTitle())) {
            predicate = ExpressionUtils.and(predicate, surveyDO.surveyTitle.like("%" + param.getSurveyTitle() + "%"));
        }
        if (StringUtils.isNotEmpty(param.getSurveyStatus())) {
            predicate = ExpressionUtils.and(predicate, surveyDO.surveyStatus.eq(param.getSurveyStatus()));
        }
        if (StringUtils.isNotEmpty(param.getPublisherName())) {
            predicate = ExpressionUtils.and(predicate, surveyDO.publisherName.like("%" + param.getPublisherName() + "%"));
        }
        if (param.getCreateTimeFrom() != null) {
            predicate = ExpressionUtils.and(predicate, surveyDO.createTime.goe(param.getCreateTimeFrom()).or(surveyDO.createTime.eq(param.getCreateTimeFrom())));
        }
        if (param.getCreateTimeTo() != null) {
            predicate = ExpressionUtils.and(predicate, surveyDO.createTime.loe(param.getCreateTimeTo()).or(surveyDO.createTime.eq(param.getCreateTimeTo())));
        }
        if (StringUtils.isNotEmpty(param.getNeedFilterValidTime()) && "Y".equals(param.getNeedFilterValidTime())) {
            predicate = ExpressionUtils.and(predicate, surveyDO.beginTime.loe(LocalDateTime.now()).and(surveyDO.endTime.goe(LocalDateTime.now())));
        }
        if (StringUtils.isNotEmpty(param.getOuCode())) {
            var subQuery = getPublishQuery(surveyDO.id.eq(surveyPublishDO.suvId), param);
            predicate = ExpressionUtils.and(predicate, subQuery.exists());
        }
        return predicate;
    }

    private JPQLQuery<Integer> getPublishQuery(BooleanExpression id, SurveyPagingParam param) {
        var subQuery = JPAExpressions.selectOne().from(surveyPublishDO)
                .where(id);
        if (StringUtils.isNotEmpty(param.getOuCode())) {
            subQuery.where(surveyPublishDO.ouCode.eq(param.getOuCode()));
        }
        if (StringUtils.isNotEmpty(param.getFillStatus())) {
            subQuery.where(surveyPublishDO.fillStatus.eq(param.getFillStatus()));
        }
        return subQuery;
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                surveyDO.id,
                surveyDO.surveyTitle,
                surveyDO.surveyDesc,
                surveyDO.beginTime,
                surveyDO.endTime,
                surveyDO.surveyStatus,
                surveyDO.publisherId,
                surveyDO.publisherName,
                surveyDO.fillCount,
                surveyDO.createTime,
                surveyDO.creator,
                surveyDO.modifyTime,
                surveyDO.updater
        )).from(surveyDO);
    }

    @Transactional
    public void updateSurveyStatus(List<Long> idList, String surveyStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(surveyDO)
                .set(surveyDO.surveyStatus, surveyStatus)
                .set(surveyDO.modifyTime, LocalDateTime.now())
                .where(surveyDO.id.in(idList));
        update.execute();
    }

    public List<SurveyRespVO> findSurvey(SurveyPagingParam param) {
        JPAQuery<SurveyRespVO> jpaQuery = select(SurveyRespVO.class)
                .from(surveyDO)
                .where(where(param))
                .orderBy(surveyDO.createTime.desc());;
        return jpaQuery.fetch();
    }
}
