package com.elitesland.scp.application.service.calendar;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetLineRespVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetImportDTO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetImportVO;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpStoreCalendarSetImportImpl implements DataImport<ScpStoreCalendarSetImportVO> {

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final ScpStoreCalendarSetService scpStoreCalendarSetService;
    private final RmiOrgStoreRpcService rmiOrgStoreService;
    private final TaskExecutor taskExecutor;

    @Override
    public String getTmplCode() {
        return "scp_store_calendar_set_import";
    }

    @Override
    public List<String> executeImport(List<ScpStoreCalendarSetImportVO> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        log.info("门店订货日历导入入参：{}", JSON.toJSONString(dataList));
        List<String> errors = new ArrayList<>(dataList.size());
        try {
            return checkAndSaveData(dataList, startRowIndex);
        } catch (Exception e) {
            for (ScpStoreCalendarSetImportVO entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
//        return scpStoreCalendarSetService.executeImport(dataList, startRowIndex);
    }

    private List<String> checkAndSaveData(List<ScpStoreCalendarSetImportVO> dataList, int startRowIndex) throws Exception {
        if (CollUtil.isEmpty(dataList)) {
            log.error("计划员权限导入数据为空");
            return Lists.newArrayList();
        }
        //门店信息查询
        CompletableFuture<List<OrgStoreDetailRpcDTO>> completeOrgStoreDetailRpcDTOS = CompletableFuture.supplyAsync(() -> {
            return this.storeQuery(dataList);
        }, taskExecutor);
        //重复数据校验
        Map<String, List<ScpStoreCalendarSetImportVO>> scpMap = dataList.stream()
                .collect(Collectors.groupingBy(t -> t.getType() + t.getStoreCode() + t.getDeliveryType() + t.getYear()));

        List<OrgStoreDetailRpcDTO> orgStoreInfoList = completeOrgStoreDetailRpcDTOS.get();
        Map<String, OrgStoreDetailRpcDTO> orgStoreInfoMap;
        if (CollUtil.isNotEmpty(orgStoreInfoList)) {
            orgStoreInfoMap = orgStoreInfoList.stream().collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode,
                    Function.identity(), (old, newValue) -> old));
        } else {
            orgStoreInfoMap = Collections.emptyMap();
        }

        List<String> errorList = new ArrayList<>();

        for (ScpStoreCalendarSetImportVO importVO : dataList) {
            String errorLog = "";
            if (StringUtils.isBlank(importVO.getType())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "类型", "类型不能为空");
            }
            if (StringUtils.isBlank(importVO.getStoreCode())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码", "门店编码不能为空");
            }
            if (StringUtils.isBlank(importVO.getDeliveryType())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "配送类型", "配送类型不能为空");
            }
            if (StringUtils.isBlank(importVO.getYear())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "年份", "年份不能为空");
            }
            if (StringUtils.isBlank(importVO.getMonday())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "星期一", "星期一不能为空");
            }
            if (StringUtils.isBlank(importVO.getTuesday())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "星期二", "星期二不能为空");
            }
            if (StringUtils.isBlank(importVO.getWednesday())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "星期三", "星期三不能为空");
            }
            if (StringUtils.isBlank(importVO.getThursday())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "星期四", "星期四不能为空");
            }
            if (StringUtils.isBlank(importVO.getFriday())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "星期五", "星期五不能为空");
            }
            if (StringUtils.isBlank(importVO.getSaturday())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "星期六", "星期六不能为空");
            }
            if (StringUtils.isBlank(importVO.getSunday())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "星期日", "星期日不能为空");
            }

            //门店状态校验
            if (!orgStoreInfoMap.isEmpty() && StringUtils.isNotBlank(importVO.getStoreCode())) {
                Optional<OrgStoreDetailRpcDTO> store = Optional.ofNullable(orgStoreInfoMap.get(importVO.getStoreCode()));
                if (store.isPresent()) {
                    // 更新门店名称
                    importVO.setStoreName(store.get().getStoreName());
                    // 更新门店ID
                    importVO.setStoreId(store.get().getId());
                } else {
                    if (Objects.equals(importVO.getType(), "STORE")) {
                        errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码", "门店不存在");
                    }
                }
            } else {
                if (StringUtils.isNotBlank(importVO.getType()) && (Objects.equals(importVO.getType(), "STORE"))) {
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码", "门店不存在");
                }
            }

            //重复数据校验
            String checkMessage = importVO.getType() + importVO.getStoreCode() + importVO.getDeliveryType() + importVO.getYear();
            if (MapUtils.isNotEmpty(scpMap)) {
                if (scpMap.get(checkMessage) != null && scpMap.get(checkMessage).size() > 1) {
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "数据", "存在多行重复数据");
                }
            }
            errorList.add(StringUtils.isNotBlank(errorLog) ? errorLog : null);
            startRowIndex++;
        }
        if (CollectionUtil.isNotEmpty(errorList) && !errorList.stream().allMatch(Objects::isNull)) {
            return errorList;
        } else {
            List<ScpStoreCalendarSetImportDTO> scpStoreCalendarSetDTOList = new ArrayList<>();

            Map<String, List<ScpStoreCalendarSetImportVO>> stringListMap = dataList.stream().collect(Collectors.groupingBy(t -> t.getType() + t.getDeliveryType() + t.getYear()));
            stringListMap.forEach((k, v) -> {
                ScpStoreCalendarSetImportVO scpStoreCalendarSetImportVO = v.get(0);
                ScpStoreCalendarSetImportDTO scpStoreCalendarSetDTO = new ScpStoreCalendarSetImportDTO();
                scpStoreCalendarSetDTO.setDeliveryType(scpStoreCalendarSetImportVO.getDeliveryType());
                scpStoreCalendarSetDTO.setType(scpStoreCalendarSetImportVO.getType());
                int year = Integer.parseInt(scpStoreCalendarSetImportVO.getYear());
                scpStoreCalendarSetDTO.setStartYear(year);
                scpStoreCalendarSetDTO.setEndYear(year);

                List<String> storeCodeList = v.stream().map(ScpStoreCalendarSetImportVO::getStoreCode).toList();
                scpStoreCalendarSetDTO.setStoreRegionCodeList(storeCodeList);

                List<ScpStoreCalendarSetLineRespVO> monday = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> monday2 = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> tuesday = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> tuesday2 = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> wednesday = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> wednesday2 = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> thursday = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> thursday2 = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> friday = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> friday2 = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> saturday = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> saturday2 = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> sunday = new ArrayList<>();
                List<ScpStoreCalendarSetLineRespVO> sunday2 = new ArrayList<>();
                for (ScpStoreCalendarSetImportVO storeCalendarSetImportVO : v) {
                    // 星期一处理
                    setWeekInfo(storeCalendarSetImportVO, storeCalendarSetImportVO.getMonday(), monday, monday2);

                    // 星期二处理
                    setWeekInfo(storeCalendarSetImportVO, storeCalendarSetImportVO.getTuesday(), tuesday, tuesday2);

                    // 星期三处理
                    setWeekInfo(storeCalendarSetImportVO, storeCalendarSetImportVO.getWednesday(), wednesday, wednesday2);

                    // 星期四处理
                    setWeekInfo(storeCalendarSetImportVO, storeCalendarSetImportVO.getThursday(), thursday, thursday2);

                    // 星期五处理
                    setWeekInfo(storeCalendarSetImportVO, storeCalendarSetImportVO.getFriday(), friday, friday2);

                    // 星期六处理
                    setWeekInfo(storeCalendarSetImportVO, storeCalendarSetImportVO.getSaturday(), saturday, saturday2);

                    // 星期日处理
                    setWeekInfo(storeCalendarSetImportVO, storeCalendarSetImportVO.getSunday(), sunday, sunday2);
                }

                scpStoreCalendarSetDTO.setMonday(monday);
                scpStoreCalendarSetDTO.setMonday2(monday2);
                scpStoreCalendarSetDTO.setTuesday(tuesday);
                scpStoreCalendarSetDTO.setTuesday2(tuesday2);
                scpStoreCalendarSetDTO.setWednesday(wednesday);
                scpStoreCalendarSetDTO.setWednesday2(wednesday2);
                scpStoreCalendarSetDTO.setThursday(thursday);
                scpStoreCalendarSetDTO.setThursday2(thursday2);
                scpStoreCalendarSetDTO.setFriday(friday);
                scpStoreCalendarSetDTO.setFriday2(friday2);
                scpStoreCalendarSetDTO.setSaturday(saturday);
                scpStoreCalendarSetDTO.setSaturday2(saturday2);
                scpStoreCalendarSetDTO.setSunday(sunday);
                scpStoreCalendarSetDTO.setSunday2(sunday2);
                scpStoreCalendarSetDTOList.add(scpStoreCalendarSetDTO);
            });
            scpStoreCalendarSetService.executeImport(scpStoreCalendarSetDTOList);
        }
        return null;
    }

    private void setWeekInfo(ScpStoreCalendarSetImportVO storeCalendarSetImportVO, String sunday1, List<ScpStoreCalendarSetLineRespVO> sunday, List<ScpStoreCalendarSetLineRespVO> sunday2) {
        ScpStoreCalendarSetLineRespVO scpStoreCalendarSetLineRespVO = new ScpStoreCalendarSetLineRespVO();
        scpStoreCalendarSetLineRespVO.setStoreId(storeCalendarSetImportVO.getStoreId());
        scpStoreCalendarSetLineRespVO.setStoreCode(storeCalendarSetImportVO.getStoreCode());
        scpStoreCalendarSetLineRespVO.setStoreName(storeCalendarSetImportVO.getStoreName());
        if (Objects.equals(sunday1, "W")) {
            sunday.add(scpStoreCalendarSetLineRespVO);
        } else if (Objects.equals(sunday1, "E")) {
            sunday2.add(scpStoreCalendarSetLineRespVO);
        }
    }

    /**
     * 门店查询
     *
     * @param dataList
     * @return
     */
    private List<OrgStoreDetailRpcDTO> storeQuery(List<ScpStoreCalendarSetImportVO> dataList) {
        List<String> storeCodeList = dataList.stream()
                .filter(t -> t.getType().equals("STORE"))
                .map(ScpStoreCalendarSetImportVO::getStoreCode)
                .distinct().toList();
        if (CollectionUtil.isEmpty(storeCodeList)) {
            return null;
        }
        return rmiOrgStoreService.queryByStoreCodes(storeCodeList);
    }

    @Override
    public Integer stepSize() {
        return 1000;
    }

    @Override
    public Set<Integer> sheetNoList() {
        return Set.of(1);
    }
}
