package com.elitesland.scp.application.service.calendar;

import com.el.coordinator.core.common.utils.BeanCopyUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.calendar.*;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreDemandCalendarParam;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpManAuthorityPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarExportRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreDemandCalendarRespVO;
import com.elitesland.scp.application.service.scpsman.BaseScpmanAuthorityParam;
import com.elitesland.scp.domain.convert.calendar.ScpStoreDemandCalendarConvert;
import com.elitesland.scp.domain.entity.calendar.ScpStoreDemandCalendarDO;
import com.elitesland.scp.domain.service.authority.ScpDemandAuthorityService;
import com.elitesland.scp.domain.service.calendar.ScpStoreDemandCalendarDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.calendar.ScpStoreDemandCalendarDTO;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.utils.BeanUtils;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
* @description:  
* @author: jeesie.jiang
* @create: 2024-03-28
* @Version 1.0
**/
@RequiredArgsConstructor
@Service
@Slf4j
public class ScpStoreDemandCalendarServiceImpl  implements ScpStoreDemandCalendarService {

    private final ScpStoreDemandCalendarDomainService storeDemandCalendarDomainService;

    private final ScpDemandAuthorityService scpDemandAuthorityService;
    private final UdcProvider udcProvider;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;

    @Override
    @SysCodeProc
    public PagingVO<ScpStoreDemandCalendarPageVO> searchPage(ScpStoreDemandCalendarPageParamVO param) {
        // 如果param中的scpmanAuthority为true，则获取当前用户的权限
        if(Boolean.TRUE.equals(param.getScpmanAuthority())){
            PagingVO<ScpManAuthorityPageRespVO> userAuthority = scpDemandAuthorityService.getCurrentUserAuthority();
            // 如果用户权限为空，则返回一个空的分页对象
            if(userAuthority.isEmpty()){
                return PagingVO.<ScpStoreDemandCalendarPageVO>builder()
                        .total(0L)
                        .records(new ArrayList<>())
                        .build();
            }
            // 提取权限参数
            extractedAuthorityParam(param, userAuthority);
        }
        // 根据参数查询分页对象
        PagingVO<ScpStoreDemandCalendarPageVO> page = storeDemandCalendarDomainService.searchPage(param);
        // 如果分页对象中的记录不为空，则进行处理
        if(CollectionUtils.isNotEmpty(page.getRecords())){
            // 将记录中的type、storeCode、year、month拼接成字符串
            Set<String> typeSet = new HashSet<>();
            Set<String> storeCodeSet = new HashSet<>();
            Set<String> yearSet = new HashSet<>();
            Set<String> monthSet = new HashSet<>();
            Set<String> deliverySet = new HashSet<>();
            for (ScpStoreDemandCalendarPageVO record : page.getRecords()) {
                Optional.ofNullable(record.getStoreCode()).ifPresent(storeCodeSet::add);
                Optional.ofNullable(record.getDeliveryType()).ifPresent(deliverySet::add);
                Optional.ofNullable(record.getYear()).ifPresent(yearSet::add);
                Optional.ofNullable(record.getMonth()).ifPresent(monthSet::add);
                Optional.ofNullable(record.getType()).ifPresent(typeSet::add);
            }

            List<String> strings = page.getRecords().stream().map(d ->
                    d.getType() + d.getStoreCode() + d.getYear() + d.getMonth() + d.getDeliveryType()).toList();
            // 根据拼接的字符串查询对应的日历对象
            List<ScpStoreDemandCalendarDO> calendarDos = storeDemandCalendarDomainService.findByConcatYearAndMonthAndDeliveryTypeKey(typeSet,
                    storeCodeSet,
                    yearSet,
                    monthSet,
                    deliverySet);
            // 将日历对象按type、storeCode、year、month进行分组
            Map<String, List<ScpStoreDemandCalendarDO>> listMap = calendarDos.stream()
                    .filter(v -> strings.contains(v.getType() + v.getStoreCode() + v.getYear() + v.getMonth() + v.getDeliveryType()))
                    .collect(Collectors.groupingBy(v -> v.getType() + v.getStoreCode() + v.getYear() + v.getMonth() + v.getDeliveryType()));
            // 获取配送类型对应的名称
            Map<String, String> deliveryTypeMap = udcProvider.getValueMapByUdcCode("yst-supp", "ITEM_TYPE2");

            // 获取日历配置方法
            Map<String, String> calendarConfigMethod = udcProvider.getValueMapByUdcCode("yst-supp", "CALENDAR_CONFIG_METHOD");

            // 遍历分页对象中的记录
            page.getRecords().forEach(d -> {
                // 如果记录中的配送类型不为空，则将配送类型转换为名称
                if (StringUtils.isNotBlank(d.getDeliveryType())) {
                    d.setDeliveryTypeName(Stream.of(d.getDeliveryType().split(",")).filter(deliveryTypeMap::containsKey).map(deliveryTypeMap::get).collect(Collectors.joining(",")));
                }

                if (StringUtils.isNotBlank(d.getType())){
                    d.setTypeName(calendarConfigMethod.get(d.getType()));
                }

                // 根据type、storeCode、year、month获取对应的日历对象
                List<ScpStoreDemandCalendarDO> calendarDOList = listMap.get(d.getType() + d.getStoreCode() + d.getYear() + d.getMonth() + d.getDeliveryType());
                // 如果日历对象不为空，则进行处理
                if(CollectionUtils.isNotEmpty(calendarDOList)){
                    // 设置记录的id
                    d.setId(calendarDOList.get(0).getId());
                    // 将日历对象转换为页面VO对象
                    var details = calendarDOList.stream().map(m -> {
                        ScpStoreDemandCalendarPageVO.CalendarDayDetail detail = ScpStoreDemandCalendarConvert.INSTANCE.doToPageVo(m);
                        detail.setDay(m.getDay());
                        detail.setWorkStatus(m.getWorkStatus());
                        return detail;
                    }).toList();
                    // 对页面VO对象按day进行排序
                    var sortList = details.stream()
                            .filter(v -> v.getDay() != null)
                            .sorted(getComparing())
                            .collect(Collectors.toList());
                    // 设置记录的dayDetails
                    d.setDayDetails(sortList);
                }
            });
        }
        // 返回分页对象
        return page;
    }

    private void extractedAuthorityParam(BaseScpmanAuthorityParam queryParam, PagingVO<ScpManAuthorityPageRespVO> userAuthority) {
        List<String> whCodes = userAuthority.stream()
                .filter(v -> 1 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        List<String> stores = userAuthority.stream()
                .filter(v -> 0 == v.getType())
                .map(ScpManAuthorityPageRespVO::getStWhCode)
                .filter(Objects::nonNull).collect(Collectors.toList());
        queryParam.setWhCodes(whCodes);
        queryParam.setStores(stores);
        queryParam.setScpmanAuthority(Boolean.TRUE);
    }

    private Comparator<ScpStoreDemandCalendarPageVO.CalendarDayDetail> getComparing() {
        return Comparator.comparing(v -> Integer.parseInt(v.getDay()), Comparator.nullsLast(Comparator.naturalOrder()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveStoreCalendar(ScpStoreDemandCalendarSaveVO createParam) {
        List<ScpStoreDemandCalendarDO> calendarDOList = buildDate(createParam);
        if (ScpUdcEnum.CALENDAR_TYPE_REGION.getValueCode().equals(createParam.getType())) {
            // 当类型是区域时，先删除门店数据再添加区域数据
            // 添加门店数据
            List<OrgStoreBaseRpcDTO> storeList = rmiOrgStoreRpcService.findSimpleByRegion(List.of(createParam.getStoreCode()));
            if (CollectionUtils.isNotEmpty(storeList)) {
                List<String> storeCodes = storeList.stream().map(OrgStoreBaseRpcDTO::getStoreCode).distinct().collect(Collectors.toList());
                storeDemandCalendarDomainService.deleteByStoreAndYearAndMonthAndDeliveryType(ScpUdcEnum.CALENDAR_TYPE_STORE.getValueCode(), storeCodes, createParam.getYear(), createParam.getMonth(), createParam.getDeliveryType());
            }

        }
        // 全删全插
        storeDemandCalendarDomainService.deleteByStoreAndYearAndMonthAndDeliveryType(createParam.getType(), List.of(createParam.getStoreCode()), createParam.getYear(), createParam.getMonth(), createParam.getDeliveryType());
        // 拆分类型
        List<ScpStoreDemandCalendarDO> cdo = new ArrayList<>();
        for(ScpStoreDemandCalendarDO demandCalendarDO : calendarDOList) {
            String[] typeList = demandCalendarDO.getDeliveryType().split(",");
            for(String type : typeList) {
                ScpStoreDemandCalendarDO demandCalendarDO1 = new ScpStoreDemandCalendarDO();
                BeanUtils.copyProperties(demandCalendarDO, demandCalendarDO1);
                demandCalendarDO1.setDeliveryType(type);
                cdo.add(demandCalendarDO1);
            }
        }
        storeDemandCalendarDomainService.createBatch(cdo);
    }

    private List<ScpStoreDemandCalendarDO> buildDate(ScpStoreDemandCalendarSaveVO createParam) {
        return createParam.getDayDetails().stream().map(d -> {
            ScpStoreDemandCalendarDO demandCalendarDO = ScpStoreDemandCalendarConvert.INSTANCE.saveVoToDo(d);
            demandCalendarDO.setDay(d.getDay());
            demandCalendarDO.setWorkStatus(d.getWorkStatus());
            demandCalendarDO.setStoreCode(createParam.getStoreCode());
            demandCalendarDO.setStoreId(createParam.getStoreId());
            demandCalendarDO.setStoreName(createParam.getStoreName());
            demandCalendarDO.setYear(createParam.getYear());
            demandCalendarDO.setMonth(createParam.getMonth());
            demandCalendarDO.setType(createParam.getType());
            demandCalendarDO.setDeliveryType(createParam.getDeliveryType());
            return demandCalendarDO;
        }).collect(Collectors.toList());
    }


    @Override
    public List<ScpStoreDemandCalendarRespVO> listCalendarVOs(ScpStoreDemandCalendarParam scpStoreDemandCalendarParam) {
        return ScpStoreDemandCalendarConvert.INSTANCE.dtoToRespVO(storeDemandCalendarDomainService.listCalendarDTOs(scpStoreDemandCalendarParam));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteStoreCalendar(List<ScpStoreDemandCalendarParamVO> deleteParams) {
        long nullcount = deleteParams.stream().filter(d -> d.getYear() == null).count();
        if (nullcount > 0) {
            throw new BusinessException(ApiCode.FAIL, "年不能为空");
        }
        long nullcount2 = deleteParams.stream().filter(d -> d.getMonth() == null).count();
        if (nullcount2 > 0) {
            throw new BusinessException(ApiCode.FAIL, "月不能为空");
        }
        Set<String> typeSet = new HashSet<>();
        Set<String> storeCodeSet = new HashSet<>();
        Set<String> yearSet = new HashSet<>();
        Set<String> monthSet = new HashSet<>();
        Set<String> deliverySet = new HashSet<>();
        for (ScpStoreDemandCalendarParamVO record : deleteParams) {
            Optional.ofNullable(record.getStoreCode()).ifPresent(storeCodeSet::add);
            Optional.ofNullable(record.getDeliveryType()).ifPresent(deliverySet::add);
            Optional.ofNullable(record.getYear()).ifPresent(yearSet::add);
            Optional.ofNullable(record.getMonth()).ifPresent(monthSet::add);
            Optional.ofNullable(record.getType()).ifPresent(typeSet::add);
        }
        List<String> concatKey = deleteParams.stream().map(d -> d.getType() + d.getStoreCode() + d.getYear() + d.getMonth() + d.getDeliveryType()).collect(Collectors.toList());
        List<ScpStoreDemandCalendarDO> demandCalendarDOS = storeDemandCalendarDomainService.findByConcatYearAndMonthAndDeliveryTypeKey(typeSet,
                storeCodeSet,
                yearSet,
                monthSet,
                deliverySet);
        if (CollectionUtils.isNotEmpty(demandCalendarDOS)) {
            List<Long> ids = demandCalendarDOS.stream().filter(v -> concatKey.contains(v.getType() + v.getStoreCode() + v.getYear() + v.getMonth() + v.getDeliveryType()))
                    .map(ScpStoreDemandCalendarDO::getId).distinct().toList();
            if (CollectionUtils.isNotEmpty(ids)) {
                storeDemandCalendarDomainService.deleteBatch(ids);
            }
        }
    }

    @Override
    public ScpStoreDemandCalendarRespVO findStoreCalendarDetail(ScpStoreDemandCalendarParamVO param) {
        ScpStoreDemandCalendarRespVO vo = new ScpStoreDemandCalendarRespVO();
        List<ScpStoreDemandCalendarDTO> calendarDOS = storeDemandCalendarDomainService.findStoreCalendarByParam(param);
        if(CollectionUtils.isEmpty(calendarDOS)){
            return vo;
        }
        if(param.getMonth() != null && param.getYear() != null){
            List<ScpStoreDemandCalendarRespVO> calendarRespVOS = calendarDOS.stream()
                    .map(ScpStoreDemandCalendarConvert.INSTANCE::dtoToRespVo)
                    .collect(Collectors.toList());
            ScpStoreDemandCalendarRespVO calendarRespVO = calendarRespVOS.get(0);
            BeanCopyUtil.beanCopyWithIngore(calendarRespVO,vo,BeanCopyUtil.getNullAndIgnorePropertyNames(calendarRespVO));
            var vos = calendarRespVOS.stream().filter(d -> d.getDay() != null)
                    .sorted(getComparing2()).collect(Collectors.toList());
            vo.setDayDetails(vos);
            List<String> years = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            List<String> months = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getMonth).distinct().collect(Collectors.toList());
            List<String> days = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getDay).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
            vo.setExistMonthsOfYear(months);
            vo.setExistDaysOfMonth(days);

        }else if( param.getYear() != null && param.getMonth() == null){
            List<String> years = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            List<String> months = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getMonth).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
            vo.setExistMonthsOfYear(months);
        }else if( param.getYear() == null && param.getMonth() == null){
            List<String> years = calendarDOS.stream().map(ScpStoreDemandCalendarDTO::getYear).distinct().collect(Collectors.toList());
            vo.setExistYears(years);
        }
        if (StringUtils.isNotBlank(vo.getDeliveryType())) {
            Map<String, String> deliveryTypeMap = udcProvider.getValueMapByUdcCode("yst-supp", "ITEM_TYPE2");
            vo.setDeliveryTypeName(Stream.of(vo.getDeliveryType().split(",")).filter(deliveryTypeMap::containsKey).map(deliveryTypeMap::get).collect(Collectors.joining(",")));
        }
        return vo;
    }

    @Override
    public Boolean isExistStoreByDateAndWorkStatus(ScpCalendarStoreQueryParamVO paramVO) {
        return CollectionUtils.isNotEmpty(storeDemandCalendarDomainService.listStoreByQueryParam(paramVO));
    }

    @Override
    public List<ScpCalendarStoreRespVO> listCalendarStoreByParam(ScpCalendarStoreQueryParamVO paramVO) {
        return storeDemandCalendarDomainService.listStoreByQueryParam(paramVO);
    }

    private Comparator<ScpStoreDemandCalendarRespVO> getComparing2() {
        return Comparator.comparing(v -> Integer.parseInt(v.getDay()), Comparator.nullsLast(Comparator.naturalOrder()));
    }

    @Override
    public PagingVO<ScpStoreCalendarExportRespVO> export(ScpStoreDemandCalendarPageParamVO paramVO) {
        PagingVO<ScpStoreDemandCalendarPageVO> scpStoreDemandCalendarPageVOPagingVO = this.searchPage(paramVO);
        if (scpStoreDemandCalendarPageVOPagingVO.isEmpty()) {
            return PagingVO.empty();
        }
        List<ScpStoreDemandCalendarPageVO> records = scpStoreDemandCalendarPageVOPagingVO.getRecords();
        List<ScpStoreCalendarExportRespVO> scpStoreCalendarExportRespVOS = records.stream().map(p -> {
            ScpStoreCalendarExportRespVO exportVO = ScpStoreDemandCalendarConvert.INSTANCE.pageVOtoExportVO(p);
            List<ScpStoreDemandCalendarPageVO.CalendarDayDetail> dayDetails = p.getDayDetails();

            // 按照dayDetails里的day进行排序，分别赋值给exportVO里的
            if (CollectionUtils.isNotEmpty(dayDetails)) {
                Map<String, String> dayMap = dayDetails.stream()
                        .filter(detail -> detail.getDay() != null && detail.getWorkStatus() != null)
                        .collect(Collectors.toMap(
                                ScpStoreDemandCalendarPageVO.CalendarDayDetail::getDay,
                                ScpStoreDemandCalendarPageVO.CalendarDayDetail::getWorkStatus,
                                (existing, replacement) -> existing
                        ));
                exportVO.setOne(dayMap.get("1"));
                exportVO.setTwo(dayMap.get("2"));
                exportVO.setThree(dayMap.get("3"));
                exportVO.setFour(dayMap.get("4"));
                exportVO.setFive(dayMap.get("5"));
                exportVO.setSix(dayMap.get("6"));
                exportVO.setSeven(dayMap.get("7"));
                exportVO.setEight(dayMap.get("8"));
                exportVO.setNine(dayMap.get("9"));
                exportVO.setTen(dayMap.get("10"));
                exportVO.setEleven(dayMap.get("11"));
                exportVO.setTwelve(dayMap.get("12"));
                exportVO.setThirteen(dayMap.get("13"));
                exportVO.setFourteen(dayMap.get("14"));
                exportVO.setFifteen(dayMap.get("15"));
                exportVO.setSixteen(dayMap.get("16"));
                exportVO.setSeventeen(dayMap.get("17"));
                exportVO.setEighteen(dayMap.get("18"));
                exportVO.setNineteen(dayMap.get("19"));
                exportVO.setTwenty(dayMap.get("20"));
                exportVO.setTwentyOne(dayMap.get("21"));
                exportVO.setTwentyTwo(dayMap.get("22"));
                exportVO.setTwentyThree(dayMap.get("23"));
                exportVO.setTwentyFour(dayMap.get("24"));
                exportVO.setTwentyFive(dayMap.get("25"));
                exportVO.setTwentySix(dayMap.get("26"));
                exportVO.setTwentySeven(dayMap.get("27"));
                exportVO.setTwentyEight(dayMap.get("28"));
                exportVO.setTwentyNine(dayMap.get("29"));
                exportVO.setThirty(dayMap.get("30"));
                exportVO.setThirtyOne(dayMap.get("31"));
            }
            return exportVO;
        }).toList();
        return PagingVO.<ScpStoreCalendarExportRespVO>builder()
                .total(scpStoreDemandCalendarPageVOPagingVO.getTotal())
                .records(scpStoreCalendarExportRespVOS)
                .build();
    }
}