package com.elitesland.scp.domain.service.alloc;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitesland.scp.application.facade.vo.resp.alloc.ScpAllocSettingItemRespVO;
import com.elitesland.scp.application.facade.vo.save.alloc.ScpAllocSettingItemSaveVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.alloc.ScpAllocSettingItemConvert;
import com.elitesland.scp.domain.entity.alloc.ScpAllocSettingItemDO;
import com.elitesland.scp.domain.entity.alloc.ScpAllocSettingRedisDO;
import com.elitesland.scp.infr.repo.alloc.ScpAllocSettingItemRepo;
import com.elitesland.scp.infr.repo.alloc.ScpAllocSettingItemRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepoProc;
import com.elitesland.scp.utils.SysUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;


@Slf4j
@Service
@RequiredArgsConstructor
public class ScpAllocSettingItemDomainServiceImpl implements ScpAllocSettingItemDomainService {

    private final RedisTemplate redisTemplate;
    private final ScpAllocSettingItemRepo scpAllocSettingItemRepo;
    private final ScpAllocSettingItemRepoProc scpAllocSettingItemRepoProc;
    private final ScpDemandOrderRepoProc scpDemandOrderRepoProc;

    @Override
    @SysCodeProc
    public List<ScpAllocSettingItemRespVO> findByMasId(Long masId) {
        List<ScpAllocSettingItemDO> settingItemDOS = scpAllocSettingItemRepo.findByMasId(masId);
        if (CollUtil.isNotEmpty(settingItemDOS)) {
            return ScpAllocSettingItemConvert.INSTANCE.dosToRespVOS(settingItemDOS);
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSaveAllocSettingItem(List<ScpAllocSettingItemSaveVO> saveVOS) {
        scpAllocSettingItemRepo.saveAll(ScpAllocSettingItemConvert.INSTANCE.saveVosDOS(saveVOS));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByMasId(Long masId) {
        scpAllocSettingItemRepo.deleteByMasId(masId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpAllocSettingItemRepo.deleteByIds(ids);
    }

    @Override
    public List<ScpAllocSettingItemRespVO> getEnableAllocSettingItemByStoreCode(String storeCode, String businessType, List<String> itemTypes) {
        List<ScpAllocSettingItemRespVO> respVOS = new ArrayList<>();
        // 只有门店自采需要强配
        String allocBusinessType = SysUtils.getAllocBusinessType();
        if (Objects.equals(allocBusinessType, businessType)) {
            Object obj = redisTemplate.opsForValue().get(ScpConstant.ALLOC_SETTING + storeCode);
            if (obj == null) {
                return new ArrayList<>();
            }
            String str = (String) obj;
            var setting = JSON.parseObject(str, ScpAllocSettingRedisDO.class);
            List<ScpAllocSettingRedisDO.ActivityDO> activityList = setting.getActivityList();
            for (ScpAllocSettingRedisDO.ActivityDO activityDO : activityList) {
                log.info("getEnableAllocSettingItemByStoreCode:{}", JSONUtil.toJsonStr(activityDO));
                Object key = redisTemplate.opsForValue().get(ScpConstant.ALLOC_SETTING + activityDO.getActivityId());
                if (key == null) {
                    log.info("强配活动ID:{}，缓存不存在", activityDO.getActivityId());
                    continue;
                }
                if (LocalDateTime.now().isBefore(activityDO.getValidFrom()) || LocalDateTime.now().isAfter(activityDO.getValidTo())) {
                    log.info("强配活动ID:{}，已过期", activityDO.getActivityId());
                    continue;
                }
                // 查看强配时间段内，已经有过这个强配商品，并且订单数量已经达到了最大强配次数
                Long maxNum = scpDemandOrderRepoProc.countAllocatedOrderByItemCodes(storeCode, activityDO.getActivityCode(), activityDO.getValidFrom(), activityDO.getValidTo());
                if (activityDO.getMaxNum() <= 0 || (maxNum != null && maxNum >= activityDO.getMaxNum())) {
                    log.info("强配活动ID:{},已达到最大强配次数,不参与强配", activityDO.getActivityId());
                    continue;
                }

                List<ScpAllocSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = activityDO.getInnerItemDOS();
                for (ScpAllocSettingRedisDO.ActivityDO.InnerItemDO innerItemDO : innerItemDOS) {
                    if (!itemTypes.contains(innerItemDO.getItemType2())) {
                        log.info("强配活动ID:{}，商品分类:{}不存在", activityDO.getActivityId(), innerItemDO.getItemType2());
                        continue;
                    }
                    ScpAllocSettingItemRespVO scpAllocSettingItemRespVO = new ScpAllocSettingItemRespVO();
                    scpAllocSettingItemRespVO.setItemId(innerItemDO.getItemId());
                    scpAllocSettingItemRespVO.setItemCode(innerItemDO.getItemCode());
                    scpAllocSettingItemRespVO.setItemName(innerItemDO.getItemName());
                    scpAllocSettingItemRespVO.setMasId(innerItemDO.getMasId());
                    scpAllocSettingItemRespVO.setMinNum(innerItemDO.getMinNum());
                    scpAllocSettingItemRespVO.setActivityCode(activityDO.getActivityCode());
                    scpAllocSettingItemRespVO.setImgUrl(innerItemDO.getImgUrl());
                    respVOS.add(scpAllocSettingItemRespVO);
                }
            }
        } else {
            log.info("业务类型必须是{}才能强配，当前业务类型:{}", allocBusinessType, businessType);
        }
        log.info("门店:{},获取成功的强配:{}", storeCode, JSONUtil.toJsonStr(respVOS));
        return respVOS;
    }


    public List<ScpAllocSettingItemRespVO> getEnableAllocSettingItemByStoreCode(String storeCode, String businessType) {
        List<ScpAllocSettingItemRespVO> respVOS = new ArrayList<>();
        String allocBusinessType = SysUtils.getAllocBusinessType();
        if (Objects.equals(allocBusinessType, businessType)) {
            Object obj = redisTemplate.opsForValue().get(ScpConstant.ALLOC_SETTING + storeCode);
            if (obj == null) {
                return new ArrayList<>();
            }
            String str = (String) obj;
            var setting = JSON.parseObject(str, ScpAllocSettingRedisDO.class);
            List<ScpAllocSettingRedisDO.ActivityDO> activityList = setting.getActivityList();
            for (ScpAllocSettingRedisDO.ActivityDO activityDO : activityList) {
                log.info("getEnableAllocSettingItemByStoreCode:{}", JSONUtil.toJsonStr(activityDO));
                Object key = redisTemplate.opsForValue().get(ScpConstant.ALLOC_SETTING + activityDO.getActivityId());
                if (key == null) {
                    log.info("强配活动ID:{}，缓存不存在", activityDO.getActivityId());
                    continue;
                }
                if (LocalDateTime.now().isBefore(activityDO.getValidFrom()) || LocalDateTime.now().isAfter(activityDO.getValidTo())) {
                    log.info("强配活动ID:{}，已过期", activityDO.getActivityId());
                    continue;
                }
                if (activityDO.getMaxNum() <= 0) {
                    log.info("强配活动ID:{},最大强配次数小于等于0,不参与强配", activityDO.getActivityId());
                    continue;
                }
                List<ScpAllocSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = activityDO.getInnerItemDOS();
                for (ScpAllocSettingRedisDO.ActivityDO.InnerItemDO innerItemDO : innerItemDOS) {
                    ScpAllocSettingItemRespVO scpAllocSettingItemRespVO = new ScpAllocSettingItemRespVO();
                    scpAllocSettingItemRespVO.setItemId(innerItemDO.getItemId());
                    scpAllocSettingItemRespVO.setItemCode(innerItemDO.getItemCode());
                    scpAllocSettingItemRespVO.setItemName(innerItemDO.getItemName());
                    scpAllocSettingItemRespVO.setMasId(innerItemDO.getMasId());
                    scpAllocSettingItemRespVO.setMinNum(innerItemDO.getMinNum());
                    scpAllocSettingItemRespVO.setActivityCode(activityDO.getActivityCode());
                    scpAllocSettingItemRespVO.setImgUrl(innerItemDO.getImgUrl());
                    respVOS.add(scpAllocSettingItemRespVO);
                }
            }
        } else {
            log.info("业务类型必须是{}才能强配，当前业务类型:{}", allocBusinessType, businessType);
        }
        log.info("门店:{},获取成功的强配:{}", storeCode, JSONUtil.toJsonStr(respVOS));
        return respVOS;
    }

    @Override
    public List<Long> findMasIdByItemId(Long itemId) {
        return scpAllocSettingItemRepoProc.findMasIdByItemId(itemId);
    }
}
