package com.elitesland.scp.infr.repo.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandSetPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandSetParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetPageRespVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.order.QScpDemandSetDO;
import com.elitesland.scp.infr.dto.order.ScpDemandSetDTO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.List;

@Component
@RequiredArgsConstructor
public class ScpDemandSetRepoProc {
    private final JPAQueryFactory jpaQueryFactory;
    private static final QScpDemandSetDO scpDemandSetDO = QScpDemandSetDO.scpDemandSetDO;

    public long countDemandSet(ScpDemandSetPageParamVO paramVO) {
        var jpaQuery = jpaQueryFactory.select(scpDemandSetDO.count())
                .from(scpDemandSetDO);
        jpaQuery.where(this.whereDemandSetPage(paramVO));
        return jpaQuery.fetchCount();
    }

    private final QBean<ScpDemandSetPageRespVO> pageList = Projections.bean(
            ScpDemandSetPageRespVO.class,
            scpDemandSetDO.id,
            scpDemandSetDO.type,
            scpDemandSetDO.demandCode,
            scpDemandSetDO.demandName,
            scpDemandSetDO.demandDate,
            scpDemandSetDO.status,
            scpDemandSetDO.computeMsg,
            scpDemandSetDO.isPushed,
            scpDemandSetDO.creator,
            scpDemandSetDO.createUserId,
            scpDemandSetDO.createTime,
            scpDemandSetDO.updater,
            scpDemandSetDO.modifyUserId,
            scpDemandSetDO.modifyTime,
            scpDemandSetDO.remark,
            scpDemandSetDO.expireTime,
            scpDemandSetDO.storeCode,
            scpDemandSetDO.storeName
    );

    private final QBean<ScpDemandSetDTO> demandSetList = Projections.bean(
            ScpDemandSetDTO.class,
            scpDemandSetDO.id,
            scpDemandSetDO.type,
            scpDemandSetDO.demandCode,
            scpDemandSetDO.demandName,
            scpDemandSetDO.demandDate,
            scpDemandSetDO.status,
            scpDemandSetDO.computeMsg,
            scpDemandSetDO.isPushed,
            scpDemandSetDO.creator,
            scpDemandSetDO.createUserId,
            scpDemandSetDO.createTime,
            scpDemandSetDO.updater,
            scpDemandSetDO.modifyUserId,
            scpDemandSetDO.modifyTime,
            scpDemandSetDO.remark,
            scpDemandSetDO.expireTime
    );

    public List<ScpDemandSetPageRespVO> queryDemandSet(ScpDemandSetPageParamVO paramVO) {
        var jpaQuery = jpaQueryFactory.select(pageList)
                .from(scpDemandSetDO);
        paramVO.setPaging(jpaQuery);
        paramVO.fillOrders(jpaQuery, scpDemandSetDO);
        jpaQuery.where(this.whereDemandSetPage(paramVO));
        return jpaQuery.fetch();
    }


    private Predicate whereDemandSetPage(ScpDemandSetPageParamVO paramVO) {
        List<Predicate> predicates = new ArrayList<>();
        if (StrUtil.isNotBlank(paramVO.getDemandCode())) {
            predicates.add(scpDemandSetDO.demandCode.like(paramVO.getDemandCode() + "%"));
        }
        if (StrUtil.isNotBlank(paramVO.getType())) {
            predicates.add(scpDemandSetDO.type.eq(paramVO.getType()));
        }
        if (StrUtil.isNotBlank(paramVO.getDemandName())) {
            predicates.add(scpDemandSetDO.demandName.like(paramVO.getDemandName() + "%"));
        }
        if (!ObjectUtils.isEmpty(paramVO.getDemandDateFrom()) && !ObjectUtils.isEmpty(paramVO.getDemandDateTo())) {
            predicates.add(scpDemandSetDO.demandDate.between(paramVO.getDemandDateFrom(), paramVO.getDemandDateTo()));
        }
        if (paramVO.getIsPushed() != null) {
            predicates.add(scpDemandSetDO.isPushed.eq(paramVO.getIsPushed()));
        }
        if (paramVO.getStatus() != null) {
            predicates.add(scpDemandSetDO.status.eq(paramVO.getStatus()));
        }
        return ExpressionUtils.allOf(predicates);
    }

    private Predicate whereDemandSet(ScpDemandSetParamVO paramVO) {
        List<Predicate> predicates = new ArrayList<>();
        if (StrUtil.isNotBlank(paramVO.getDemandCode())) {
            predicates.add(scpDemandSetDO.demandCode.eq(paramVO.getDemandCode()));
        }
        if (StrUtil.isNotBlank(paramVO.getType())) {
            predicates.add(scpDemandSetDO.type.eq(paramVO.getType()));
        }
        if (StrUtil.isNotBlank(paramVO.getDemandName())) {
            predicates.add(scpDemandSetDO.demandName.eq(paramVO.getDemandName()));
        }
        if (paramVO.getDemandDate() != null) {
            predicates.add(scpDemandSetDO.demandDate.eq(paramVO.getDemandDate()));
        }
        if (paramVO.getIsPushed() != null) {
            predicates.add(scpDemandSetDO.isPushed.eq(paramVO.getIsPushed()));
        }
        if (paramVO.getStatus() != null) {
            predicates.add(scpDemandSetDO.status.eq(paramVO.getStatus()));
        }
        if (paramVO.getExpireTime() != null) {
            LocalDateTime endDateTime = paramVO.getExpireTime().toLocalDate().atTime(LocalTime.MAX);
            predicates.add(scpDemandSetDO.expireTime.goe(paramVO.getExpireTime()).and(scpDemandSetDO.expireTime.loe(endDateTime)));
        }
        if (CollUtil.isNotEmpty(paramVO.getDemandNames())) {
            predicates.add(scpDemandSetDO.demandName.in(paramVO.getDemandNames()));
        }
        if (paramVO.getStoreCode() != null) {
            BooleanExpression condition = Expressions.asBoolean(false).isTrue();
            String pattern = "%," + paramVO.getStoreCode() + ",%";
            BooleanExpression match = scpDemandSetDO.storeCode.startsWith(paramVO.getStoreCode() + ",")
                    .or(scpDemandSetDO.storeCode.endsWith("," + paramVO.getStoreCode()))
                    .or(scpDemandSetDO.storeCode.eq(paramVO.getStoreCode()))
                    .or(scpDemandSetDO.storeCode.like(pattern));
            condition = condition.or(match);
            predicates.add(condition);
        }

        if (paramVO.getNotUseStoreFlag() != null && paramVO.getNotUseStoreFlag()) {
            predicates.add(scpDemandSetDO.storeCode.isNull());
        }
        return ExpressionUtils.allOf(predicates);
    }

    public List<ScpDemandSetDTO> findDemandSetByParam(ScpDemandSetParamVO paramVO) {
        var jpaQuery = jpaQueryFactory.select(demandSetList)
                .from(scpDemandSetDO);
        jpaQuery.where(this.whereDemandSet(paramVO));
        jpaQuery.orderBy(scpDemandSetDO.createTime.desc());
        return jpaQuery.fetch();
    }

    public void enableDemandSet(List<Long> ids, Boolean enable) {
        JPAUpdateClause update = jpaQueryFactory.update(scpDemandSetDO)
                .set(scpDemandSetDO.status, enable)
                .where(scpDemandSetDO.id.in(ids));
        update.execute();
    }

    public long deleteByIds(List<Long> ids) {
        List<Predicate> predicates = new ArrayList<>();
        predicates.add(scpDemandSetDO.id.in(ids));
        var delete = jpaQueryFactory.delete(scpDemandSetDO)
                .where(ExpressionUtils.allOf(predicates));
        return delete.execute();
    }

    public void updateDemandSetMsgByIds(List<Long> ids, String msg) {
        JPAUpdateClause update = jpaQueryFactory.update(scpDemandSetDO)
                .set(scpDemandSetDO.computeMsg, msg)
                .where(scpDemandSetDO.id.in(ids));
        update.execute();
    }

    public void updateDemandSetPushMsgById(List<Long> ids, String msg) {
        JPAUpdateClause update = jpaQueryFactory.update(scpDemandSetDO)
                .set(scpDemandSetDO.pushMsg, msg)
                .where(scpDemandSetDO.id.in(ids));
        update.execute();
    }

    public void completePushById(Long id) {
        JPAUpdateClause update = jpaQueryFactory.update(scpDemandSetDO)
                .set(scpDemandSetDO.isPushed, Boolean.TRUE)
                .set(scpDemandSetDO.pushMsg, ScpConstant.PUSH_DONE)
                .where(scpDemandSetDO.id.eq(id));
        update.execute();
    }

    public Boolean getDemandSetStatusById(Long demandSetId) {
        var jpaQuery = jpaQueryFactory.select(scpDemandSetDO.status)
                .from(scpDemandSetDO);
        jpaQuery.where(scpDemandSetDO.id.eq(demandSetId));
        return jpaQuery.fetchOne();
    }
}
