package com.elitesland.scp.utils;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.scp.domain.entity.item.ScpStoreItemDO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

@Slf4j
@Component
@RequiredArgsConstructor
public class ScpCacheUtils {

    private final RedisTemplate redisClient;

    /**
     * 获取商品详情
     *
     * @param spuId
     * @return
     */
    public List<ScpStoreItemDO> getItemBySpuId(Long spuId) {
        // 生成商品价格缓存key
        String redisKey = generateItemKey(spuId);
        Object obj = redisClient.opsForValue().get(redisKey);
        if (obj == null) {
            return null;
        }
        String str = (String) obj;
        return JSON.parseArray(str, ScpStoreItemDO.class);
    }

    /**
     * 生成商品价格缓存key
     *
     * @param spuId
     * @return
     */
    public String generateItemKey(Long spuId) {
        return "SCP:ITEM:" + spuId;
    }

    /**
     * 生成商品锁的key
     *
     * @param spuId
     * @return
     */
    public String generateLockKey(Long spuId) {
        return "SCP:ITEM:LOCK:" + spuId;
    }

    /**
     * 保存商品信息
     *
     * @param spuId
     * @param scpStoreItemDOS
     */
    public void saveItem(Long spuId, List<ScpStoreItemDO> scpStoreItemDOS) {
        // 生成商品价格缓存key
        String redisKey = generateItemKey(spuId);
        String jsonString = JSON.toJSONString(scpStoreItemDOS);
        //每天凌晨12点失效
        LocalDateTime now = LocalDateTime.now();
        long expire = ChronoUnit.SECONDS.between(now, now.with(LocalTime.MAX));
        redisClient.opsForValue().set(redisKey, jsonString, expire, TimeUnit.SECONDS);
    }


    /**
     * 删除指定SPU的缓存
     *
     * @param spuId 商品ID
     */
    public void deleteKeys(Long spuId) {
        if (spuId == null) {
            throw new BusinessException("spuId不能为空");
        }
        String cartKey = "*SCP:ITEM:" + spuId;
        Set scan = this.scan(redisClient, cartKey);
        if (scan != null && scan.size() > 0) {
            redisClient.delete(scan);
        }
    }

    /**
     * 删除所有商品的缓存
     */
    public void deleteAllKeys() {
        String cartKey = "*SCP:ITEM*";
        Set scan = this.scan(redisClient, cartKey);
        if (scan != null && scan.size() > 0) {
            redisClient.delete(scan);
        }
    }

    /**
     * scan 实现
     *
     * @param redisTemplate redisTemplate
     * @param pattern       表达式，如：abc*，找出所有以abc开始的键
     */
    public Set<String> scan(RedisTemplate<String, Object> redisTemplate, String pattern) {
        return redisTemplate.execute((RedisCallback<Set<String>>) connection -> {
            Set<String> keysTmp = new HashSet<>();
            try (Cursor<byte[]> cursor = connection.scan(ScanOptions.scanOptions()
                    .match(pattern)
                    .count(10000).build())) {
                while (cursor.hasNext()) {
                    keysTmp.add(new String(cursor.next(), StandardCharsets.UTF_8));
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            return keysTmp;
        });
    }

    /**
     * 清除指定key模糊缓存
     */
    public void deleteSpecKeys(String spec) {
        String cartKey = "*" + spec + "*";
        Set scan = this.scan(redisClient, cartKey);
        if (scan != null && scan.size() > 0) {
            redisClient.delete(scan);
        }
    }
}
