package com.elitesland.scp.application.service.item;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.item.*;
import com.elitesland.scp.application.facade.vo.param.item.ScpCateItemParam.CateItem;
import com.elitesland.scp.application.facade.vo.resp.app.ScpCateItemPageVO;
import com.elitesland.scp.application.facade.vo.resp.app.ScpCateItemRespVO;
import com.elitesland.scp.domain.entity.item.ScpCateItemDO;
import com.elitesland.scp.domain.entity.item.ScpCateItemSaleScopeRegionDO;
import com.elitesland.scp.domain.entity.item.ScpCateItemSaleScopeStoreDO;
import com.elitesland.scp.domain.service.item.ScpCateItemDomainService;
import com.elitesland.scp.dto.item.ScpCateItemDTO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.repo.item.ScpCateItemRepo;
import com.elitesland.scp.infr.repo.item.ScpCateItemRepoProc;
import com.elitesland.scp.infr.repo.item.ScpCateItemSaleScopeRegionRepo;
import com.elitesland.scp.infr.repo.item.ScpCateItemSaleScopeStoreRepo;
import com.elitesland.scp.param.CateItemRpcParam;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.scp.utils.BeanUtils;
import com.elitesland.scp.utils.PinyinUtil;
import com.elitesland.support.provider.item.dto.ItmItemBaseRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemSimpleRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemBaseRpcParam;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBaseRpcParam;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @description: TODO
 * @author: huruizhong
 * @modified By: huruizhong
 * @date: Created in 2024/4/3 11:42 上午
 * @version:v1.0
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ScpCateItemServiceImpl implements ScpCateItemService {

    private final ScpCateItemDomainService scpCateItemDomainService;
    private final ScpCateItemRepo scpCateItemRepo;
    private final ScpCateItemRepoProc scpCateItemRepoProc;
    private final RmiItemService rmiItemService;
    private final RmiSysUDCService rmiSysUDCService;
    private final ScpCateItemSaleScopeStoreRepo scpCateItemSaleScopeStoreRepo;
    private final ScpCateItemSaleScopeRegionRepo scpCateItemSaleScopeRegionRepo;
    private final OrgStoreRpcService orgStoreRpcService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void bindItem(ScpCateItemParam cateStoreParam) {
        Map<Long, List<String>> rootBrandMap = rmiItemService.findRootBrandByCateId(cateStoreParam.getCateId());
        List<String> brands = rootBrandMap.values().stream().findAny().get();
        if (CollUtil.isEmpty(brands)) {
            throw new BusinessException("品类下无品牌信息");
        }
        Long rootId = rootBrandMap.keySet().stream().findAny().get();
        String brand = String.join(";", brands);
        List<CateItem> items = cateStoreParam.getItems();
        List<Long> itemIds = items.stream().map(CateItem::getItemId).distinct().collect(Collectors.toList());
        ItmItemBaseRpcParam rpcParam = new ItmItemBaseRpcParam();
        rpcParam.setItemIds(itemIds);
        Map<Long, ItmItemBaseRpcDTO> itmMap = rmiItemService.findItemBaseRpcDtoByParam(rpcParam).stream().collect(Collectors.toMap(ItmItemBaseRpcDTO::getId, Function.identity()));
        scpCateItemDomainService.saveBatch(items.stream().map(item -> {
            ScpCateItemDO scpCateItemDO = BeanUtils.toBean(item, ScpCateItemDO.class);
            if (ObjectUtils.isNotEmpty(item.getItemId()) && MapUtils.isNotEmpty(itmMap)) {
                ItmItemBaseRpcDTO itmItemRpcDTO = itmMap.get(item.getItemId());
                scpCateItemDO.setSpuId(itmItemRpcDTO.getSpuId());
                scpCateItemDO.setSpuCode(itmItemRpcDTO.getSpuCode());
                scpCateItemDO.setSpuName(itmItemRpcDTO.getSpuName());
                scpCateItemDO.setItemType(itmItemRpcDTO.getItemType());
                scpCateItemDO.setAnotherName(itmItemRpcDTO.getAnotherName());
                scpCateItemDO.setItemType2(itmItemRpcDTO.getItemType2());
                scpCateItemDO.setCombineItemFlag(ScpUdcEnum.ITM_ITM_TYPE_70.getValueCode().equals(itmItemRpcDTO.getItemType()));
            }

            try {
                String firstLetter = PinyinUtil.getFirstLetter(scpCateItemDO.getSpuName());
                scpCateItemDO.setFirstLetter(firstLetter);
            } catch (Exception e) {
                log.error("获取首字母失败", e);
            }

            scpCateItemDO.setCateCode(cateStoreParam.getCateCode());
            scpCateItemDO.setCateId(cateStoreParam.getCateId());
            scpCateItemDO.setBrand(brand);
            scpCateItemDO.setRootCateId(rootId);
            return scpCateItemDO;
        }).collect(Collectors.toList()));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void remove(List<Long> ids) {
        scpCateItemDomainService.delete(ids);
    }

    @Override
    public ApiResult<List<ScpCateItemDTO>> query(CateItemRpcParam cateItemRpcParam) {
        return scpCateItemDomainService.query(cateItemRpcParam);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeHotFlag(ScpCateItemHotParamVO paramVO) {
        scpCateItemDomainService.changeHotFlag(paramVO);
    }

    @Override
    public PagingVO<ScpCateItemRespVO> page(ScpCateItemPageParamVO paramVO) {
        return scpCateItemDomainService.page(paramVO);
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpCateItemPageVO> skuPage(ScpCateItemPageParamVO paramVO) {
        if (CollUtil.isNotEmpty(paramVO.getItemStatus2List())) {
            ItmItemBaseRpcParam itemParam = new ItmItemBaseRpcParam();
            itemParam.setItemStatus2List(paramVO.getItemStatus2List());
            List<ItmItemBaseRpcDTO> itemList = rmiItemService.findItemBaseRpcDtoByParam(itemParam);
            if (CollectionUtils.isEmpty(itemList)) {
                return PagingVO.empty();
            }
            paramVO.setItemIds(itemList.stream().map(ItmItemBaseRpcDTO::getId).filter(Objects::nonNull).distinct().toList());
        }
        PagingVO<ScpCateItemPageVO> scpCateItemPage = scpCateItemDomainService.skuPage(paramVO);
        List<Long> itemIds = scpCateItemPage.getRecords().stream().map(ScpCateItemPageVO::getItemId).distinct().collect(Collectors.toList());
        Map<Long, ItmItemSimpleRpcDTO> baseItemMap = rmiItemService.findBaseItemMap(itemIds);
        for (ScpCateItemPageVO record : scpCateItemPage.getRecords()) {
            ItmItemSimpleRpcDTO itmItemSimpleRpcDTO = baseItemMap.get(record.getItemId());
            if (ObjectUtils.isEmpty(itmItemSimpleRpcDTO)) {
                continue;
            }
            record.setItemName(itmItemSimpleRpcDTO.getItemName());
            record.setSpec(itmItemSimpleRpcDTO.getSpec());
            record.setItemCateFullName(itmItemSimpleRpcDTO.getItemCateFullName());
            record.setBrandName(itmItemSimpleRpcDTO.getBrandName());
            record.setTaxRate(itmItemSimpleRpcDTO.getTaxRate());
            record.setTaxRate2(itmItemSimpleRpcDTO.getTaxRate2());
            record.setItemType(itmItemSimpleRpcDTO.getItemType());
            // 不展示商品原本的配送类型，只展示报货商城的配送类型
//            record.setItemType2(itmItemSimpleRpcDTO.getItemType2());
            record.setItemSupplySource(itmItemSimpleRpcDTO.getItemSupplySource());
            record.setLotFlag(itmItemSimpleRpcDTO.getLotFlag());
            record.setAllocType(itmItemSimpleRpcDTO.getAllocType());
            record.setGuaranteePeriod(itmItemSimpleRpcDTO.getGuaranteePeriod());
            record.setGuaranteePeriodUnit(itmItemSimpleRpcDTO.getGuaranteePeriodUnit());
            record.setSuppCode(itmItemSimpleRpcDTO.getSuppCode());
            record.setSuppName(itmItemSimpleRpcDTO.getSuppName());

            record.setItemStatus2(itmItemSimpleRpcDTO.getItemStatus2());
            record.setItemStatus2Name(itmItemSimpleRpcDTO.getItemStatus2Name());
        }
        return scpCateItemPage;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void setSaleScope(ScpCateItemOperationParam param) {
        // 保存可售范围，建新表存数据
        List<ScpCateItemDO> scpCateItemDOList = scpCateItemRepo.findAllById(param.getIdList());
        if (CollUtil.isEmpty(scpCateItemDOList)) {
            throw new BusinessException("设置商品信息不存在!");
        }

        Map<Long, List<ScpCateItemSaleScopeRegionDO>> scpCateItemSaleScopeRegionDOSMap = new HashMap<>();
        Map<Long, List<ScpCateItemSaleScopeStoreDO>> scpCateItemSaleScopeStoreDOListMap = new HashMap<>();
        for (ScpCateItemDO scpCateItemDO : scpCateItemDOList) {
            scpCateItemDO.setSaleStatus(true);
            scpCateItemDO.setSaleDateStart(param.getDateStart());
            scpCateItemDO.setSaleDateEnd(param.getDateEnd());

            // 保存区域信息
            if (CollUtil.isNotEmpty(param.getRegionCodes())) {
                List<ScpCateItemSaleScopeRegionDO> scpCateItemSaleScopeRegionDOS = param.getRegionCodes().stream().map(regionCode -> {
                    ScpCateItemSaleScopeRegionDO scpCateItemSaleScopeRegionDO = new ScpCateItemSaleScopeRegionDO();
                    scpCateItemSaleScopeRegionDO.setMasId(scpCateItemDO.getId());
                    scpCateItemSaleScopeRegionDO.setRegionCode(regionCode);
                    return scpCateItemSaleScopeRegionDO;
                }).collect(Collectors.toList());
                scpCateItemSaleScopeRegionDOSMap.put(scpCateItemDO.getId(), scpCateItemSaleScopeRegionDOS);
            }

            // 保存门店信息
            if (CollUtil.isNotEmpty(param.getStoreCodes())) {
                List<ScpCateItemSaleScopeStoreDO> scpCateItemSaleScopeStoreDOList = param.getStoreCodes().stream().map(storeCode -> {
                    ScpCateItemSaleScopeStoreDO scpCateItemSaleScopeStoreDO = new ScpCateItemSaleScopeStoreDO();
                    scpCateItemSaleScopeStoreDO.setMasId(scpCateItemDO.getId());
                    scpCateItemSaleScopeStoreDO.setStoreCode(storeCode);
                    return scpCateItemSaleScopeStoreDO;
                }).collect(Collectors.toList());
                scpCateItemSaleScopeStoreDOListMap.put(scpCateItemDO.getId(), scpCateItemSaleScopeStoreDOList);
            }
        }

        scpCateItemRepo.saveAll(scpCateItemDOList);

        if (CollUtil.isNotEmpty(scpCateItemSaleScopeRegionDOSMap)) {
            scpCateItemSaleScopeRegionRepo.deleteByMasIdIn(scpCateItemSaleScopeRegionDOSMap.keySet());
            List<ScpCateItemSaleScopeRegionDO> collect = scpCateItemSaleScopeRegionDOSMap.values().stream()
                    .filter(Objects::nonNull)
                    .flatMap(Collection::stream)
                    .peek(v -> v.setId(null)).collect(Collectors.toList());
            if (CollUtil.isNotEmpty(collect)) {
                scpCateItemSaleScopeRegionRepo.saveAll(collect);
            }
        }

        if (CollUtil.isNotEmpty(scpCateItemSaleScopeStoreDOListMap)) {
            scpCateItemSaleScopeStoreRepo.deleteByMasIdIn(scpCateItemSaleScopeStoreDOListMap.keySet());
            List<ScpCateItemSaleScopeStoreDO> collect = scpCateItemSaleScopeStoreDOListMap.values().stream()
                    .filter(Objects::nonNull)
                    .flatMap(Collection::stream)
                    .peek(v -> v.setId(null))
                    .collect(Collectors.toList());
            if (CollUtil.isNotEmpty(collect)) {
                scpCateItemSaleScopeStoreRepo.saveAll(collect);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSaleScope(ScpCateItemOperationAddParam param) {
        // 保存可售范围，建新表存数据
        List<ScpCateItemDO> scpCateItemDOList = scpCateItemRepo.findAllById(param.getIdList());
        if (CollUtil.isEmpty(scpCateItemDOList)) {
            throw new BusinessException("设置商品信息不存在!");
        }
        // 过滤空数据
        List<Long> idList = param.getIdList();
        List<String> storeCodes = param.getStoreCodes();
        for (Long masId : idList) {
            scpCateItemSaleScopeStoreRepo.deleteByMasIdAndStoreCodeList(masId, storeCodes);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addSaleScope(ScpCateItemOperationAddParam param) {
        // 保存可售范围，建新表存数据
        List<ScpCateItemDO> scpCateItemDOList = scpCateItemRepo.findAllById(param.getIdList());
        if (CollUtil.isEmpty(scpCateItemDOList)) {
            throw new BusinessException("设置商品信息不存在!");
        }

        Map<Long, List<ScpCateItemSaleScopeStoreDO>> scpCateItemSaleScopeStoreDOListMap = new HashMap<>();
        List<Long> masIds = scpCateItemDOList.stream().map(ScpCateItemDO::getId).collect(Collectors.toList());
        Map<Long, List<String>> existingStoreCodeMap = scpCateItemSaleScopeStoreRepo.findAllByMasIdInAndStoreCodeIn(masIds, param.getStoreCodes())
                .stream()
                .collect(Collectors.groupingBy(ScpCateItemSaleScopeStoreDO::getMasId,
                        Collectors.mapping(ScpCateItemSaleScopeStoreDO::getStoreCode, Collectors.toList())));
        // 保存门店信息
        scpCateItemDOList.forEach(scpCateItemDO -> {
            scpCateItemDO.setSaleStatus(true);
            if (CollUtil.isNotEmpty(param.getStoreCodes())) {
                List<String> storeCodeList = existingStoreCodeMap.get(scpCateItemDO.getId());
                List<ScpCateItemSaleScopeStoreDO> scpCateItemSaleScopeStoreDOList = param.getStoreCodes().parallelStream()
                        .filter(v -> {
                            // 筛选出storeCodeList里不存在的
                            // 如果这个门店在storeCodeList里,则 跳过
                            if (CollUtil.isNotEmpty(storeCodeList)) {
                                return !storeCodeList.contains(v);
                            }
                            return true;
                        })
                        .map(storeCode -> {
                            ScpCateItemSaleScopeStoreDO scpCateItemSaleScopeStoreDO = new ScpCateItemSaleScopeStoreDO();
                            scpCateItemSaleScopeStoreDO.setMasId(scpCateItemDO.getId());
                            scpCateItemSaleScopeStoreDO.setStoreCode(storeCode);
                            scpCateItemSaleScopeStoreDO.setId(null);
                            return scpCateItemSaleScopeStoreDO;
                        }).toList();
                scpCateItemSaleScopeStoreDOListMap.put(scpCateItemDO.getId(), scpCateItemSaleScopeStoreDOList);
            }
        });

        // 保存数据
        if (CollUtil.isNotEmpty(scpCateItemSaleScopeStoreDOListMap)) {
            // 过滤空数据
            List<ScpCateItemSaleScopeStoreDO> collect = scpCateItemSaleScopeStoreDOListMap.values().stream().flatMap(Collection::stream).filter(Objects::nonNull).toList();
            if (CollUtil.isNotEmpty(collect)) {
                scpCateItemRepo.saveAll(scpCateItemDOList);
                scpCateItemSaleScopeStoreRepo.saveAll(collect);
            }
        }
    }

    @Override
    public ScpCateItemOperationVO getSaleScope(Long id) {
        ScpCateItemDO scpCateItemDO = scpCateItemRepo.findById(id).orElseThrow(() -> new BusinessException("设置商品信息不存在!"));

        ScpCateItemOperationVO result = new ScpCateItemOperationVO();
        result.setId(scpCateItemDO.getId());
        result.setDateStart(scpCateItemDO.getSaleDateStart());
        result.setDateEnd(scpCateItemDO.getSaleDateEnd());

        List<ScpCateItemSaleScopeStoreDO> storeDOList = scpCateItemSaleScopeStoreRepo.findAllByMasId(id);

        List<ScpCateItemOperationVO.StoreInfo> storeInfos = new ArrayList<>();
        List<String> storeCodes = new ArrayList<>();
        if (CollUtil.isNotEmpty(storeDOList)){
            for (ScpCateItemSaleScopeStoreDO scpCateItemSaleScopeStoreDO : storeDOList) {
                ScpCateItemOperationVO.StoreInfo storeInfo = new ScpCateItemOperationVO.StoreInfo();
                storeCodes.add(scpCateItemSaleScopeStoreDO.getStoreCode());
                storeInfo.setStoreCode(scpCateItemSaleScopeStoreDO.getStoreCode());
                storeInfo.setVisibleStatus(scpCateItemSaleScopeStoreDO.getVisibleStatus());
                storeInfo.setVisibleDateStart(scpCateItemSaleScopeStoreDO.getVisibleDateStart());
                storeInfo.setVisibleDateEnd(scpCateItemSaleScopeStoreDO.getVisibleDateEnd());
                storeInfos.add(storeInfo);
            }
            result.setStoreCodes(storeCodes);
            result.setStoreList(storeInfos);
        }
        return result;
    }

    @Override
    public void setVisibleScope(ScpCateItemOperationVisibleParam param) {
        // 保存门店信息
        if (CollUtil.isEmpty(param.getStoreCodes())) {
            throw new BusinessException("门店不能为空!");
        }

        List<ScpCateItemSaleScopeStoreDO> scpCateItemSaleScopeStoreDOList = new ArrayList<>();
        List<ScpCateItemSaleScopeStoreDO> masIdIn = scpCateItemSaleScopeStoreRepo.findAllByMasIdIn(param.getIdList());
        Map<Long, List<ScpCateItemSaleScopeStoreDO>> masIdMap = masIdIn.stream().collect(Collectors.groupingBy(ScpCateItemSaleScopeStoreDO::getMasId));
        for (List<ScpCateItemSaleScopeStoreDO> list : masIdMap.values()) {
            Map<String, ScpCateItemSaleScopeStoreDO> storeCodeMap = list.stream().collect(Collectors.toMap(ScpCateItemSaleScopeStoreDO::getStoreCode, Function.identity()));
            for (String storeCode : param.getStoreCodes()) {
                ScpCateItemSaleScopeStoreDO scpCateItemSaleScopeStoreDO = storeCodeMap.get(storeCode);
                if (scpCateItemSaleScopeStoreDO == null) {
                    throw new BusinessException("存在商品可见范围不属于可售范围，可见状态设置失败，请重新设置！");
                }
                // 这里直接修改了 masIdMap 中的对象
                scpCateItemSaleScopeStoreDO.setVisibleStatus(param.getLimitFlag());
                if (param.getLimitFlag()) {
                    if (param.getDateStart() == null || param.getDateEnd() == null) {
                        throw new BusinessException("请选择商品可见范围生效时间！");
                    }
                    scpCateItemSaleScopeStoreDO.setVisibleDateStart(param.getDateStart());
                    scpCateItemSaleScopeStoreDO.setVisibleDateEnd(param.getDateEnd());
                } else {
                    scpCateItemSaleScopeStoreDO.setVisibleDateStart(null);
                    scpCateItemSaleScopeStoreDO.setVisibleDateEnd(null);
                }
                scpCateItemSaleScopeStoreDOList.add(scpCateItemSaleScopeStoreDO);
            }
        }

        // 设置外部可见状态标记
        List<ScpCateItemDO> allById = scpCateItemRepo.findAllById(param.getIdList());
        allById = allById.stream().peek(scpCateItemDO -> {
            List<ScpCateItemSaleScopeStoreDO> scpCateItemSaleScopeStoreDOS = masIdMap.get(scpCateItemDO.getId());
            Optional<ScpCateItemSaleScopeStoreDO> any = scpCateItemSaleScopeStoreDOS.stream().filter(ScpCateItemSaleScopeStoreDO::getVisibleStatus).findAny();
            scpCateItemDO.setVisibleStatus(any.isPresent());
        }).collect(Collectors.toList());
        scpCateItemRepo.saveAll(allById);

        if (CollUtil.isNotEmpty(scpCateItemSaleScopeStoreDOList)) {
            scpCateItemSaleScopeStoreRepo.saveAll(scpCateItemSaleScopeStoreDOList);
        }
    }

    @Override
    public ScpCateItemOperationVO getVisibleScope(Long id) {
        ScpCateItemDO scpCateItemDO = scpCateItemRepo.findById(id).orElseThrow(() -> new BusinessException("商品信息不存在!"));

        ScpCateItemOperationVO result = new ScpCateItemOperationVO();
        result.setId(scpCateItemDO.getId());
        List<ScpCateItemSaleScopeStoreDO> storeDOList = scpCateItemSaleScopeStoreRepo.findAllByMasIdAndVisibleStatus(id, true);

        List<ScpCateItemOperationVO.StoreInfo> storeInfos = new ArrayList<>();
        List<String> storeCodes = new ArrayList<>();
        if (CollUtil.isNotEmpty(storeDOList)) {
            for (ScpCateItemSaleScopeStoreDO scpCateItemSaleScopeStoreDO : storeDOList) {
                ScpCateItemOperationVO.StoreInfo storeInfo = new ScpCateItemOperationVO.StoreInfo();
                storeCodes.add(scpCateItemSaleScopeStoreDO.getStoreCode());
                storeInfo.setStoreCode(scpCateItemSaleScopeStoreDO.getStoreCode());
                storeInfo.setVisibleStatus(scpCateItemSaleScopeStoreDO.getVisibleStatus());
                storeInfo.setVisibleDateStart(scpCateItemSaleScopeStoreDO.getVisibleDateStart());
                storeInfo.setVisibleDateEnd(scpCateItemSaleScopeStoreDO.getVisibleDateEnd());
                storeInfos.add(storeInfo);
            }
            result.setStoreCodes(storeCodes);
            result.setStoreList(storeInfos);
        }
        return result;
    }

    @Override
    public ApiResult<String> setPreSaleStatus(ScpCateItemPreSaleParam param) {
        scpCateItemRepo.updatePreStatusByIdIn(param.getIdList(), param.getStatus());

        return ApiResult.ok();
    }

    @Override
    public ApiResult<String> setUpDownStatus(ScpCateItemPreSaleParam param) {
        List<ScpCateItemDO> allById = scpCateItemRepo.findAllById(param.getIdList());
        if (CollUtil.isEmpty(allById)) {
            throw new BusinessException("设置商品信息不存在!");
        }

        if (param.getStatus()) {
            List<ScpCateItemDO> collect = allById.stream().filter(v -> !Objects.equals(v.getSaleStatus(), Boolean.TRUE)).collect(Collectors.toList());
            if (CollUtil.isNotEmpty(collect)) {
                return ApiResult.ok("批量上架失败，存在未维护可售范围的商品，请维护后再进行上架！");
            }
        }

        scpCateItemRepo.updateUpDownStatusByIdIn(param.getIdList(), param.getStatus());
        return ApiResult.ok();
    }

    @Override
    public List<String> getVisibleAndSaleAndUpItemList(List<String> itemCodeList, String storeCode) {
        // 获取可售可见已上架的商品
        return scpCateItemRepoProc.getVisibleAndSaleAndUpItemList(itemCodeList, storeCode);
    }

    @Override
    public List<String> getSaleAndUpItemList(List<String> itemCodeList, String storeCode) {
        // 获取可售已上架的商品
        return scpCateItemRepoProc.getSaleAndUpItemList(itemCodeList, storeCode);
    }

    @Override
    public List<String> getPreSaleItemList(List<String> itemCodeList) {
        // 获取预售商品
        return scpCateItemRepoProc.getPreSaleItemList(itemCodeList);
    }

    @Override
    public void saveSortNo(Long id, Integer sorNo) {
        scpCateItemDomainService.saveSorNo(id, sorNo);
    }

    @Override
    @Transactional
    public ApiResult<Object> importSaleScope(List<ScpItemSaleScopeStoreImportVO> list, ScpItemSaleScopeImportParam param) {
        if (CollUtil.isEmpty(list)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入的数据不能为空");
        }
        List<String> storeCodes = list.stream().map(ScpItemSaleScopeStoreImportVO::getStoreCode).filter(StringUtils::isNotBlank).toList();
        if (CollUtil.isEmpty(storeCodes)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入的数据不能为空");
        }
        if (param.getId() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "报货商品id不能为空");
        }

        OrgStoreBaseRpcParam orgStoreBaseRpcParam = new OrgStoreBaseRpcParam();
        orgStoreBaseRpcParam.setStoreCodeList(storeCodes);
        ApiResult<List<OrgStoreBaseRpcDTO>> apiResult = orgStoreRpcService.findSimpleStoreByParam(orgStoreBaseRpcParam);
        if (apiResult.isFailed() || CollectionUtils.isEmpty(apiResult.getData())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "门店信息查询失败或门店不存在");
        }
        List<OrgStoreBaseRpcDTO> storeList = apiResult.getData();
        List<String> existStoreCodes = storeList.stream().map(OrgStoreBaseRpcDTO::getStoreCode).toList();
        List<String> errorStoreCodes = storeCodes.stream().filter(storeCode -> !existStoreCodes.contains(storeCode)).toList();
        ScpCateItemOperationParam itemOperationParam = new ScpCateItemOperationParam();
        itemOperationParam.setDateStart(param.getDateStart());
        itemOperationParam.setDateEnd(param.getDateEnd());

        if (itemOperationParam.getDateStart() == null) {
            itemOperationParam.setDateStart(LocalDateTime.of(LocalDate.now(), LocalTime.MIN));
        }
        if (itemOperationParam.getDateEnd() == null) {
            itemOperationParam.setDateEnd(LocalDateTime.of(9999, 12, 31, 23, 59, 59));
        }
        itemOperationParam.setIdList(List.of(param.getId()));
        itemOperationParam.setStoreCodes(existStoreCodes);
        this.setSaleScope(itemOperationParam);
        ApiResult<Object> result = ApiResult.ok();
        if (CollUtil.isNotEmpty(errorStoreCodes)) {
            result.setErrorMsg("门店编码不存在：" + String.join(",", errorStoreCodes));
        }
        return result;
    }

    @Override
    public ApiResult<Object> importVisibleScope(List<ScpItemSaleScopeStoreImportVO> list, ScpItemVisibleScopeImportParam param) {
        if (CollUtil.isEmpty(list)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入的数据不能为空");
        }
        List<String> storeCodes = list.stream().map(ScpItemSaleScopeStoreImportVO::getStoreCode).filter(StringUtils::isNotBlank).toList();
        if (CollUtil.isEmpty(storeCodes)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导入的数据不能为空");
        }
        if (param.getId() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "报货商品id不能为空");
        }

        OrgStoreBaseRpcParam orgStoreBaseRpcParam = new OrgStoreBaseRpcParam();
        orgStoreBaseRpcParam.setStoreCodeList(storeCodes);
        ApiResult<List<OrgStoreBaseRpcDTO>> apiResult = orgStoreRpcService.findSimpleStoreByParam(orgStoreBaseRpcParam);
        if (apiResult.isFailed() || CollectionUtils.isEmpty(apiResult.getData())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "门店信息查询失败或门店不存在");
        }
        List<OrgStoreBaseRpcDTO> storeList = apiResult.getData();
        List<String> existStoreCodes = storeList.stream().map(OrgStoreBaseRpcDTO::getStoreCode).toList();
        List<String> errorStoreCodes = storeCodes.stream().filter(storeCode -> !existStoreCodes.contains(storeCode)).toList();

        ScpCateItemOperationVisibleParam itemOperationParam = new ScpCateItemOperationVisibleParam();
        itemOperationParam.setLimitFlag(param.getLimitFlag());
        itemOperationParam.setDateStart(param.getDateStart());
        itemOperationParam.setDateEnd(param.getDateEnd());
        itemOperationParam.setIdList(List.of(param.getId()));
        itemOperationParam.setStoreCodes(existStoreCodes);
        this.setVisibleScope(itemOperationParam);
        ApiResult<Object> result = ApiResult.ok();
        if (CollUtil.isNotEmpty(errorStoreCodes)) {
            result.setErrorMsg("门店编码不存在：" + String.join(",", errorStoreCodes));
        }
        return result;
    }
}
