package com.elitesland.scp.application.service.serviceconfig;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.serviceconfig.*;
import com.elitesland.scp.domain.convert.serviceconifg.ScpServiceConfigConvert;
import com.elitesland.scp.domain.entity.serviceconfig.ScpServiceConfigDO;
import com.elitesland.scp.domain.service.serviceconfig.ScpServiceConfigDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.repo.serviceconfig.ScpServiceConfigRepo;
import com.elitesland.scp.infr.repo.serviceconfig.ScpServiceConfigRepoProc;
import com.elitesland.scp.rmi.RmiOrgOuService;
import com.elitesland.scp.rmi.RmiOrgRegionRpcService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.dto.OrgRegionDTO;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author chaofeng.xia
 * @since 2025/5/26
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ScpServiceConfigServiceImpl implements ScpServiceConfigService {

    private final ScpServiceConfigDomainService scpServiceConfigDomainService;

    private final UdcProvider udcProvider;

    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;

    private final RmiOrgOuService rmiOrgOuService;

    private final RmiOrgRegionRpcService rmiOrgRegionRpcService;

    private final ScpServiceConfigRepo scpServiceConfigRepo;

    private final ScpServiceConfigRepoProc scpServiceConfigRepoProc;

    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";


    @Override
    @SysCodeProc
    public PagingVO<ScpServiceConfigPageVO> page(ScpServiceConfigPageParamVO queryParam) {
        PagingVO<ScpServiceConfigPageVO> pagingVO = scpServiceConfigDomainService.page(queryParam);
        if (pagingVO.isNotEmpty()) {
            Map<String, String> storeLevelMap = udcProvider.getValueMapByUdcCode("yst-supp", "STORE_LEVEL");
            pagingVO.getRecords().forEach(scpServiceConfig -> {
                if(StringUtils.isNotEmpty(scpServiceConfig.getStoreLevel())){
                    String storeLevelName = Stream.of(scpServiceConfig.getStoreLevel().split(",")).map(storeLevelMap::get).collect(Collectors.joining(","));
                    scpServiceConfig.setStoreLevelName(storeLevelName);
                }
            });
        }
        return pagingVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void createServiceConfig(List<ScpServiceConfigSaveVO> createParams) {
        if (CollectionUtils.isEmpty(createParams)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "数据不能为空");
        }

        List<ScpServiceConfigDO> createList = ScpServiceConfigConvert.INSTANCE.saveVoDO(createParams);
        List<String> businessKeys = createList.stream().map(c -> {
            c.setBusinessKey(c.getFeeType() + "_" + c.getType() + "_" + c.getRegionStoreCode());
            return c.getBusinessKey();
        }).distinct().collect(Collectors.toList());
        List<ScpServiceConfigDO> existList = scpServiceConfigDomainService.findByBusinessKey(businessKeys);

        if (CollectionUtils.isNotEmpty(existList)) {
            existList.addAll(createList);
            // 校验唯一性
            checkEqual(existList);
        }

        // 保存
        scpServiceConfigRepo.saveAll(createList);
    }

    public void checkEqual(List<ScpServiceConfigDO> existList) {
        Map<String, List<ScpServiceConfigDO>> configMap = existList.stream().collect(Collectors.groupingBy(c -> c.getFeeType() + "_" + c.getType() + "_" + c.getRegionStoreId()));

        configMap.forEach((businessKey, list) -> {
            List<String> storeLevelList = new ArrayList<>();
            list.forEach(d -> storeLevelList.addAll(d.getStoreLevels()));
            List<String> repeatStoreLevels = storeLevelList.stream().collect(Collectors.groupingBy(s -> s, Collectors.counting())).entrySet().stream().filter(e -> e.getValue() > 1).map(Map.Entry::getKey).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(repeatStoreLevels)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, String.format("数据重复，费用归属类型-区域编码/门店编码【%s-%s】配置重复或已存在，请修改",
                        list.get(0).getFeeType(),
                        list.get(0).getRegionStoreCode()));
            }
        });
    }

    @Override
    @Transactional
    public void modifyServiceConfig(ScpServiceConfigSaveVO createParam) {
        if (createParam.getId() == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "项目服务配置id不能为空");
        }
        ScpServiceConfigDO scpServiceConfigDO = ScpServiceConfigConvert.INSTANCE.saveVoDO(createParam);
        scpServiceConfigDO.setBusinessKey(scpServiceConfigDO.getFeeType() + "_" + scpServiceConfigDO.getType() + "_" + scpServiceConfigDO.getRegionStoreCode());
        List<ScpServiceConfigDO> existList = scpServiceConfigRepo.findByBusinessKeyAndIdNot(createParam.getBusinessKey(), createParam.getId());
        if (CollUtil.isNotEmpty(existList)) {
            existList.add(scpServiceConfigDO);
            checkEqual(existList);
        }
        scpServiceConfigRepo.save(scpServiceConfigDO);
    }

    @Override
    @SysCodeProc
    public ScpServiceConfigRespVO findDetailById(Long id) {
        if (id == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "id不能为空");
        }
        ScpServiceConfigRespVO serviceConfigRespVO = scpServiceConfigDomainService.findById(id);
        if (serviceConfigRespVO == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "项目服务配置不存在");
        }
        Map<String, String> storeLevelMap = udcProvider.getValueMapByUdcCode("yst-supp", "STORE_LEVEL");
        String storeLevelName = Stream.of(serviceConfigRespVO.getStoreLevel().split(",")).map(storeLevelMap::get).collect(Collectors.joining(","));
        serviceConfigRespVO.setStoreLevelName(storeLevelName);
        return serviceConfigRespVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteServiceConfig(List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "ids不能为空");
        }
        scpServiceConfigDomainService.deleteBatch(ids);
    }

    @Override
    public PagingVO<ScpServiceConfigExportRespVO> exportSearch(ScpServiceConfigPageParamVO queryParam) {
        PagingVO<ScpServiceConfigExportRespVO> exportResult = scpServiceConfigDomainService.exportSearch(queryParam);

        if (exportResult != null && exportResult.isNotEmpty()) {
            Map<String, String> storeLevelMap = udcProvider.getValueMapByUdcCode("yst-supp", "STORE_LEVEL");
            Map<String, String> feeTypeMap = udcProvider.getValueMapByUdcCode("yst-suplan", "SC_FEE_TYPE");
            Map<String, String> configTypeMap = udcProvider.getValueMapByUdcCode("yst-suplan", "SERVICE_CONFIG_TYPE");
            exportResult.getRecords().forEach(sc -> {
                sc.setFeeType(feeTypeMap.get(sc.getFeeType()));
                sc.setTypeName(configTypeMap.get(sc.getType()));
                sc.setFeePercentageStr(sc.getFeePercentage().stripTrailingZeros().toPlainString() + "%");
                String storeLevelName = Stream.of(sc.getStoreLevel().split(",")).map(storeLevelMap::get).collect(Collectors.joining(","));
                sc.setStoreLevelName(storeLevelName);
            });
        }
        return exportResult;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<String> executeImport(List<ScpServiceConfigImportVO> dataList, int rowIdx) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errorResult = new ArrayList<>();
        boolean errorFlag = false;

        // 校验并补充数据
        Map<String, String> feeTypeMap = udcProvider.getValueMapByUdcCode("yst-suplan", "SC_FEE_TYPE").entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
        Map<String, String> configTypeMap = udcProvider.getValueMapByUdcCode("yst-suplan", "SERVICE_CONFIG_TYPE").entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
        Map<String, String> storeLevelMap = udcProvider.getValueMapByUdcCode("yst-supp", "STORE_LEVEL").entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));

        List<String> ouCodes = dataList.stream().map(ScpServiceConfigImportVO::getOuCode).filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        List<String> storeCodes = dataList.stream()
                .filter(sc -> ScpUdcEnum.SERVICE_CONFIG_TYPE_STORE.getValueCodeName().equals(sc.getTypeName()) && StringUtils.isNotBlank(sc.getRegionStoreCode()))
                .map(ScpServiceConfigImportVO::getRegionStoreCode).distinct().collect(Collectors.toList());
        List<String> regionCodes = dataList.stream().filter(sc -> ScpUdcEnum.SERVICE_CONFIG_TYPE_REGION.getValueCodeName().equals(sc.getTypeName()) && StringUtils.isNotBlank(sc.getRegionStoreCode()))
                .map(ScpServiceConfigImportVO::getRegionStoreCode).distinct().collect(Collectors.toList());

        Map<String, OrgStoreBaseRpcDTO> storeMap = rmiOrgStoreRpcService.getOrgStoreBaseMap(storeCodes);
        Map<String, OrgOuRpcSimpleDTO> ouMap = rmiOrgOuService.findOuDtoMapByOuCodes(ouCodes);
        Map<String, OrgRegionDTO> regionMap = rmiOrgRegionRpcService.findRegionMapByCodes(regionCodes);

        for (ScpServiceConfigImportVO sc : dataList) {
            List<String> errorList = new ArrayList<>();

            // 费用归属类型
            if (StringUtils.isBlank(sc.getFeeTypeName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "费用归属类型", "费用归属类型不能为空"));
            } else if (!feeTypeMap.containsKey(sc.getFeeTypeName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "费用归属类型", "费用归属类型不存在"));
            } else {
                sc.setFeeType(feeTypeMap.get(sc.getFeeTypeName()));
            }

            // 费用归属公司编码
            if (StringUtils.isBlank(sc.getOuCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "费用归属公司编码", "费用归属公司编码不能为空"));
            } else if (!ouMap.containsKey(sc.getOuCode())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "费用归属公司编码", "费用归属公司编码不存在"));
            } else {
                OrgOuRpcSimpleDTO ouDTO = ouMap.get(sc.getOuCode());
                sc.setOuId(ouDTO.getId());
                sc.setOuName(ouDTO.getOuName());
            }

            // 类型、门店/区域、门店等级
            if (StringUtils.isBlank(sc.getTypeName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "类型", "类型不能为空"));
            } else if (!configTypeMap.containsKey(sc.getTypeName())) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "类型", "类型不存在"));
            } else {
                sc.setType(configTypeMap.get(sc.getTypeName()));

                if (ScpUdcEnum.SERVICE_CONFIG_TYPE_STORE.getValueCode().equals(sc.getType())) {
                    if (StringUtils.isBlank(sc.getRegionStoreCode())) {
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "门店编码", "门店编码不能为空"));
                    } else if (!storeMap.containsKey(sc.getRegionStoreCode())) {
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "门店编码", "门店编码不存在"));
                    } else {
                        OrgStoreBaseRpcDTO storeDTO = storeMap.get(sc.getRegionStoreCode());
                        sc.setRegionStoreId(storeDTO.getId());
                        sc.setRegionStoreName(storeDTO.getStoreName());
                        sc.setStoreLevel(storeDTO.getStoreLevel());
//                        if (!StringUtils.isBlank(sc.getStoreLevelName())) {
//                            sc.setStoreLevel(Stream.of(sc.getStoreLevelName().split(",")).map(storeLevelMap::get).collect(Collectors.joining(",")));
//                        }
                    }
                } else {
                    if (StringUtils.isBlank(sc.getRegionStoreCode())) {
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "区域编码", "区域编码不能为空"));
                    } else if (!regionMap.containsKey(sc.getRegionStoreCode())) {
                        errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "区域编码", "区域编码不存在"));
                    } else {
                        OrgRegionDTO regionDTO = regionMap.get(sc.getRegionStoreCode());
                        sc.setRegionStoreId(regionDTO.getId());
                        sc.setRegionStoreName(regionDTO.getRegionName());
                        if (StringUtils.isBlank(sc.getStoreLevelName())) {
                            errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "适用门店等级", "类型为区域适用门店等级不能为空"));
                        }
                        sc.setStoreLevel(Stream.of(sc.getStoreLevelName().split(",")).map(storeLevelMap::get).collect(Collectors.joining(",")));
                    }
                }
            }

            // 费用比例
            if (sc.getFeePercentageStr() == null || sc.getFeePercentageStr().compareTo(BigDecimal.ZERO) <= 0) {
                errorList.add(MessageFormat.format(ERROR_TEMPLATE, rowIdx, "费用比例", "费用比例不能为空且需要大于0"));
            }

            // 如果该条数据存在错误 添加错误信息
            if (CollectionUtil.isNotEmpty(errorList)) {
                errorFlag = true;
                errorResult.add(StringUtils.join(errorList, "\n"));
            } else {
                // 不存在错误信息
                sc.setBusinessKey(sc.getFeeType() + "_" + sc.getType() + "_" + sc.getRegionStoreCode());
                errorResult.add(null);
            }
            // 行数自增
            rowIdx++;
        }

        // 如果没有错误则保存
        if (!errorFlag) {
            List<ScpServiceConfigDO> serviceConfigDOList = dataList.stream().map(entry -> {
                ScpServiceConfigDO scpServiceConfigDO = ScpServiceConfigConvert.INSTANCE.importToDO(entry);
                if (entry.getFeePercentageStr() != null) {
                    scpServiceConfigDO.setFeePercentage(entry.getFeePercentageStr().multiply(BigDecimal.valueOf(100)));
                }
                return scpServiceConfigDO;
            }).collect(Collectors.toList());
            checkEqual(serviceConfigDOList);
            scpServiceConfigRepo.saveAll(serviceConfigDOList);
        }

        return CollectionUtils.isEmpty(errorResult) ? null : errorResult;
    }

    @Override
    public Map<String,ScpServiceConfigRespVO>findServiceConfigGroupByStore(String storeCode,String regionCode,String level){
        Map<String, String> feeTypeMap = udcProvider.getValueMapByUdcCode("yst-suplan", "SC_FEE_TYPE");
        Set<String> freeTypes = feeTypeMap.keySet();
        log.info("门店编码【{}】区域编码【{}】适用门店等级【{}】费用类型 【{}】", storeCode, regionCode, level, JSON.toJSONString(freeTypes));
        List<ScpServiceConfigRespVO> byStoreCode = scpServiceConfigRepoProc.findByStoreCode(storeCode);
        if(CollectionUtils.isEmpty(byStoreCode)){
            List<ScpServiceConfigRespVO> byRegionCode = scpServiceConfigRepoProc.findByRegionCode(regionCode, level);
            if(CollectionUtils.isEmpty(byRegionCode)){
                return null;
            }else{
                return byRegionCode.stream().collect(Collectors.toMap(ScpServiceConfigRespVO::getFeeType,
                        v -> v, (t1, t2) -> t1));
            }
        }else{
            List<String> existfeeTypes = byStoreCode.stream().map(ScpServiceConfigRespVO::getFeeType).filter(Objects::nonNull).distinct()
                    .toList();
            List<String> nonFeeTyoes = freeTypes.stream().filter(d -> !existfeeTypes.contains(d)).collect(Collectors.toList());
            log.info("门店编码【{}】存在非免费类型【{}】，请检查", storeCode, nonFeeTyoes );
            if(CollectionUtils.isNotEmpty(nonFeeTyoes )){
                log.info("门店编码【{}】存在非免费类型【{}】，请检查", storeCode,nonFeeTyoes );
                List<ScpServiceConfigRespVO> byRegionCode = scpServiceConfigRepoProc.findByRegionCode2(regionCode, level, nonFeeTyoes );
                if(CollectionUtils.isNotEmpty(byRegionCode)){
                    byStoreCode.addAll(byRegionCode);
                    return byStoreCode.stream().collect(Collectors.toMap(ScpServiceConfigRespVO::getFeeType,
                        v -> v, (t1, t2) -> t1));
                }else{
                    return byStoreCode.stream().collect(Collectors.toMap(ScpServiceConfigRespVO::getFeeType,
                            t -> t, (t1, t2) -> t1));
                }
            }else{
                return byStoreCode.stream().collect(Collectors.toMap(ScpServiceConfigRespVO::getFeeType,
                        t -> t, (t1, t2) -> t1));
            }
        }
    }

    @Override
    public Map<String, ScpServiceConfigRespVO> checkServiceConfig(String storeCode, String regionCode, String level) {
        Map<String, ScpServiceConfigRespVO> configRespVOMap = this.findServiceConfigGroupByStore(storeCode, regionCode, level);
        log.info("configRespVOMap:{}", JSONUtil.toJsonStr(configRespVOMap));
        if (CollUtil.isEmpty(configRespVOMap)) {
            log.error("门店信息未完善,未找到门店对应的项目服务配置");
            throw new BusinessException(ApiCode.FAIL, "门店信息未完善");
        }
        return configRespVOMap;
    }
}
