package com.elitesland.scp.domain.service.survey;

import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.scp.application.facade.vo.scpsman.SalesmanInfoQueryVO;
import com.elitesland.scp.application.facade.vo.scpsman.SalesmanInfoRespVO;
import com.elitesland.scp.domain.convert.survey.*;
import com.elitesland.scp.domain.entity.scpsman.ScpsmanRegionDO;
import com.elitesland.scp.domain.entity.survey.*;
import com.elitesland.scp.domain.vo.survey.*;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.repo.scpsman.SalesmanRegionRepo;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepoProc;
import com.elitesland.scp.infr.repo.survey.*;
import com.querydsl.jpa.impl.JPAQuery;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@Slf4j
public class SurveyServiceImpl implements SurveyService {

    private final SurveyRepo surveyRepo;
    private final SurveyRepoProc surveyRepoProc;
    private final SurveyQuestionRepo surveyQuestionRepo;
    private final SurveyOptionRepo surveyOptionRepo;
    private final SurveyPublishRepo surveyPublishRepo;
    private final ScpsmanInfoRepoProc scpsmanInfoRepoProc;
    private final SalesmanRegionRepo salesmanRegionRepo;
    private final SurveyAnswerRepo surveyAnswerRepo;
    private final SurveyAnswerRepoProc surveyAnswerRepoProc;

    @Override
    @SysCodeProc
    public PagingVO<SurveyRespVO> search(SurveyPagingParam searchParam) {
        PagingVO<SurveyRespVO> search = surveyRepoProc.search(searchParam);
        if(search.getTotal() > 0){
            List<SurveyRespVO> records = search.getRecords();

            // 补充发布对象
            List<Long> suvIdList = records.stream().map(SurveyRespVO::getId).toList();
            Map<Long, List<SurveyPublishDO>> publishMap = surveyPublishRepo.findBySuvIdIn(suvIdList).stream().collect(Collectors.groupingBy(SurveyPublishDO::getSuvId));

            for(SurveyRespVO surveyRespVO : records){
                if(publishMap.containsKey(surveyRespVO.getId())){
                    String collect = publishMap.get(surveyRespVO.getId()).stream().map(SurveyPublishDO::getOuName).collect(Collectors.joining(","));
                    surveyRespVO.setPublishOuName(collect);
                }
            }
        }
        return search;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveOrUpdate(SurveySaveParam saveVO) {
        SurveyDO surveyDO = SurveyConvert.INSTANCE.saveVOToDo(saveVO);
        surveyDO.setSurveyStatus(ScpUdcEnum.SURVEY_STATUS_10.getValueCode());
        surveyDO.setFillCount(0); // 默认0
        SurveyDO save = surveyRepo.save(surveyDO);
        // 删除问题和选项
        List<SurveyQuestionDO> questionDOList = surveyQuestionRepo.findBySuvId(save.getId());
        if(CollectionUtils.isNotEmpty(questionDOList)){
            List<Long> qusIdList = questionDOList.stream().map(SurveyQuestionDO::getId).toList();
            surveyQuestionRepo.deleteByIdIn(qusIdList);
            surveyOptionRepo.deleteByQusIdIn(qusIdList);
        }
        // 保存问题和选项
        if(CollectionUtils.isNotEmpty(saveVO.getQuestionList())){
            for(SurveyQuestionSaveParam questionVO : saveVO.getQuestionList()){
                SurveyQuestionDO questionDO = SurveyQuestionConvert.INSTANCE.saveParamToDo(questionVO);
                questionDO.setSuvId(save.getId());
                SurveyQuestionDO question = surveyQuestionRepo.save(questionDO);
                if(CollectionUtils.isNotEmpty(questionVO.getOptionList())){
                    for(SurveyOptionSaveParam optionVO : questionVO.getOptionList()){
                        SurveyOptionDO optionDO = SurveyOptionConvert.INSTANCE.saveParamToDo(optionVO);
                        optionDO.setQusId(question.getId());
                        surveyOptionRepo.save(optionDO);
                    }
                }
            }
        }
        return save.getId();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancel(List<Long> idList) {
        surveyRepoProc.updateSurveyStatus(idList, ScpUdcEnum.SURVEY_STATUS_30.getValueCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer publish(SurveyPublishParam param) {
        if(param.getId() == null || CollectionUtils.isEmpty(param.getPublishList())){
            throw new RuntimeException("参数错误");
        }
        // 可多次发布，每次发布不影响原有数据
        List<SurveyPublishDO> publishDOS = surveyPublishRepo.findBySuvId(param.getId());
        List<String> ouCodes = publishDOS.stream().map(SurveyPublishDO::getOuCode).toList();

        List<SurveyPublishDO> publishList = new ArrayList<>();
        for(SurveyPublishSaveParam publishParam : param.getPublishList()){
            if(!ouCodes.contains(publishParam.getOuCode())){
                SurveyPublishDO publishDO = SurveyPublishConvert.INSTANCE.saveParamToDo(publishParam);
                publishDO.setSuvId(param.getId());
                publishDO.setFillStatus(ScpUdcEnum.SURVEY_FILL_STATUS_00.getValueCode());
                publishList.add(publishDO);
            }
        }
        if(CollectionUtils.isNotEmpty(publishList)){
            surveyPublishRepo.saveAll(publishList);
        }
        // 更新状态同时记录发布人
        SysUserDTO userDTO = SecurityContextUtil.currentUser().getUser();
        surveyRepoProc.publish(param.getId(), userDTO);
        // todo 如果问卷非必填，则给对应的门店发送消息
        return publishList.size();
    }

    @Override
    @SysCodeProc
    public SurveyRespVO get(Long id) {
        Optional<SurveyDO> byId = surveyRepo.findById(id);
        if(byId.isEmpty()){
            throw new RuntimeException("问卷不存在");
        }
        SurveyRespVO surveyRespVO = SurveyConvert.INSTANCE.doToRespVO(byId.get());
        // 查询问题及选项
        List<SurveyQuestionDO> questionDOList = surveyQuestionRepo.findBySuvId(id);
        if(CollectionUtils.isNotEmpty(questionDOList)){
            List<SurveyQuestionRespVO> questionRespVOList = SurveyQuestionConvert.INSTANCE.doListToRespVOList(questionDOList);

            List<Long> qusIdList = questionRespVOList.stream().map(SurveyQuestionRespVO::getId).distinct().toList();
            Map<Long, List<SurveyOptionDO>> collect = surveyOptionRepo.findByQusIdIn(qusIdList).stream().collect(Collectors.groupingBy(SurveyOptionDO::getQusId));
            for(SurveyQuestionRespVO questionRespVO : questionRespVOList){
                if(collect.containsKey(questionRespVO.getId())){
                    List<SurveyOptionDO> surveyOptionDOList = collect.get(questionRespVO.getId());
                    List<SurveyOptionRespVO> optionRespVOList = SurveyOptionConvert.INSTANCE.doListToRespVOList(surveyOptionDOList);
                    questionRespVO.setOptionList(optionRespVOList);
                }
            }
            surveyRespVO.setQuestionList(questionRespVOList);
        }
        return surveyRespVO;
    }

    @Override
    public List<SurveyCheckRespVO> findSurvey() {
        // 获取当前登陆人信息，只有门店角色才能填写问卷
        GeneralUserDetails generalUserDetails = SecurityContextUtil.currentUser();
        if(generalUserDetails == null || generalUserDetails.getUser() == null){
            log.info("获取问卷：当前用户信息为空");
            return null;
        }
        String username = generalUserDetails.getUser().getUsername();
        return findSurvey(username, null);
    }

    @Override
    public List<SurveyCheckRespVO> findSurvey(String username, List<String> storeCodeList) {
        SalesmanInfoQueryVO queryVO = new SalesmanInfoQueryVO();
        queryVO.setLoginAccount(username);
        JPAQuery<SalesmanInfoRespVO> jpaQuery = scpsmanInfoRepoProc.findSalesmanInfo(queryVO);
        List<SalesmanInfoRespVO> salesmanInfoList = jpaQuery.fetch();
        if(CollectionUtils.isEmpty(salesmanInfoList)){
            log.info("获取问卷：订货账号为空");
            return new ArrayList<>();
        }
        SalesmanInfoRespVO salesmanInfoRespVO = salesmanInfoList.get(0);
        if(UdcEnum.SALE_SCPSMAN_TYPE_SHOPOWNER.getValueCode().equals(salesmanInfoRespVO.getScpsmanType())){
            List<ScpsmanRegionDO> regionDO = salesmanRegionRepo.findByMasIdIn(List.of(salesmanInfoRespVO.getId()));
            if(CollectionUtils.isNotEmpty(regionDO)){
                List<String> finalStoreCodeList = storeCodeList;
                if(CollectionUtils.isEmpty(storeCodeList)){
                    finalStoreCodeList = regionDO.stream().map(ScpsmanRegionDO::getRegionCode).toList();
                }
                // 查询在有效期内，并且发布至改门店的，未填写的问卷
                SurveyPagingParam param = new SurveyPagingParam();
                param.setNeedFilterValidTime("Y");
                param.setOuCodeList(finalStoreCodeList);
                param.setSurveyStatus(ScpUdcEnum.SURVEY_STATUS_20.getValueCode());
                param.setRequired(true);
//                param.setFillStatus(ScpUdcEnum.SURVEY_FILL_STATUS_00.getValueCode());
                List<SurveyRespVO> survey = surveyRepoProc.findSurvey(param);
                List<SurveyCheckRespVO> resultList = new ArrayList<>();
                if(CollectionUtils.isNotEmpty(survey)){
                    List<Long> suvIdList = survey.stream().map(SurveyRespVO::getId).toList();
                    Map<String, List<SurveyAnswerDO>> collect = surveyAnswerRepo.findBySuvIdInAndOuCodeIn(suvIdList, finalStoreCodeList)
                            .stream().collect(Collectors.groupingBy(t -> t.getSuvId() + "-" + t.getOuCode()));
                    for(String storeCode : finalStoreCodeList){
                        SurveyCheckRespVO respVO = new SurveyCheckRespVO();
                        respVO.setUsername(username);
                        respVO.setStoreCode(storeCode);
                        respVO.setHaveSurvey(true);
                        respVO.setSuvId(survey.get(0).getId());
                        // 判断用户是否填写
                        respVO.setFillStatus(collect.containsKey(respVO.getSuvId() + "-" + storeCode));
                        resultList.add(respVO);
                        return resultList;
                    }
                }
            }
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void fillSurvey(String storeCode, List<SurveyAnswerSaveParam> saveParamList) {
        if(CollectionUtils.isEmpty(saveParamList)){
            throw new BusinessException("请填写问卷");
        }
        SurveyRespVO surveyRespVO = this.get((saveParamList.get(0).getSuvId()));
        if(!ScpUdcEnum.SURVEY_STATUS_20.getValueCode().equals(surveyRespVO.getSurveyStatus())){
            throw new RuntimeException("该问卷状态不是已发布，无法填写");
        }
        // 判断问卷是否在有效期内
        if(surveyRespVO.getBeginTime().isAfter(LocalDateTime.now()) || surveyRespVO.getEndTime().isBefore(LocalDateTime.now())){
            throw new RuntimeException("问卷不在答题时间段内");
        }
        // 判断该用户是否可以填写，是否已填写过
        GeneralUserDetails generalUserDetails = SecurityContextUtil.currentUser();
        if(generalUserDetails == null || generalUserDetails.getUser() == null){
            throw new RuntimeException("用户不存在");
        }
        SalesmanInfoQueryVO queryVO = new SalesmanInfoQueryVO();
        queryVO.setLoginAccount(generalUserDetails.getUser().getUsername());
        JPAQuery<SalesmanInfoRespVO> jpaQuery = scpsmanInfoRepoProc.findSalesmanInfo(queryVO);
        List<SalesmanInfoRespVO> salesmanInfoList = jpaQuery.fetch();
        if(CollectionUtils.isEmpty(salesmanInfoList)){
            throw new RuntimeException("订货账号为空");
        }
        SalesmanInfoRespVO salesmanInfoRespVO = salesmanInfoList.get(0);
        if(!UdcEnum.SALE_SCPSMAN_TYPE_SHOPOWNER.getValueCode().equals(salesmanInfoRespVO.getScpsmanType())){
            throw new RuntimeException("仅店长可以填写");
        }
        List<SurveyAnswerDO> surveyAnswerDOList = surveyAnswerRepo.findBySuvIdAndUserId(surveyRespVO.getId(), generalUserDetails.getUser().getId());
        if(CollectionUtils.isNotEmpty(surveyAnswerDOList)){
            throw new RuntimeException("该用户已填写过问卷");
        }
        List<ScpsmanRegionDO> regionDO = salesmanRegionRepo.findByMasIdIn(List.of(salesmanInfoRespVO.getId()));
        if(CollectionUtils.isEmpty(regionDO)){
            throw new RuntimeException("用户无门店信息");
        }
        Optional<ScpsmanRegionDO> first = regionDO.stream().filter(t -> t.getRegionCode().equals(storeCode)).findFirst();
        if(first.isEmpty()){
            throw new RuntimeException("用户未绑定" + storeCode + "门店信息");
        }

        List<SurveyAnswerDO> answerDOList = new ArrayList<>();
        for(SurveyAnswerSaveParam saveParam : saveParamList){
            SurveyAnswerDO answerDO = SurveyAnswerConvert.INSTANCE.saveParamToDo(saveParam);
            answerDO.setUserId(generalUserDetails.getUser().getId());
            answerDO.setUsername(generalUserDetails.getUser().getUsername());
            answerDO.setOuCode(storeCode);
            answerDOList.add(answerDO);
        }
        surveyAnswerRepo.saveAll(answerDOList);
        // 更新公司填写状态
        surveyPublishRepo.updateFillStatusByCode(surveyRespVO.getId(), storeCode);
        // 更新问卷填写份数
        surveyRepo.updateFillCountById(surveyRespVO.getId());
    }

    @Override
    public SurveyRespVO fillDetail(Long id) {
        Optional<SurveyDO> byId = surveyRepo.findById(id);
        if(byId.isEmpty()){
            throw new RuntimeException("问卷不存在");
        }
        SurveyRespVO surveyRespVO = SurveyConvert.INSTANCE.doToRespVO(byId.get());
        // 查询问卷填写明细
        SurveyPagingParam param = new SurveyPagingParam();
        param.setSuvId(id);
        param.setSize(10);
        PagingVO<SurveyAnswerRespVO> voPagingVO = fillDetailList(param);
        surveyRespVO.setAnswerList(voPagingVO.getRecords());
        return surveyRespVO;
    }

    @Override
    public PagingVO<SurveyAnswerRespVO> fillDetailList(SurveyPagingParam param) {
        PagingVO<SurveyAnswerRespVO> search = surveyAnswerRepoProc.search(param);
        if(search.getTotal() > 0){
            List<SurveyAnswerRespVO> records = search.getRecords();
            SurveyAnswerParam answerParam = new SurveyAnswerParam();
            answerParam.setSuvId(param.getSuvId());
            answerParam.setOuCodeList(records.stream().map(SurveyAnswerRespVO::getOuCode).toList());
            Map<String, List<SurveyAnswerRespVO>> collect = surveyAnswerRepoProc.findAnswer(answerParam).stream().collect(Collectors.groupingBy(SurveyAnswerRespVO::getOuCodeAndSuvId));
            records.forEach(item -> {
                if(collect.containsKey(item.getOuCodeAndSuvId())){
                    item.setAnswerList(collect.get(item.getOuCodeAndSuvId()));
                }
            });
        }
        return search;
    }

    @Override
    public List<SurveyAnalysisRespVO> surveyAnalysis(Long id) {
        List<SurveyAnalysisRespVO> results = new ArrayList<>();
        SurveyAnswerParam answerParam = new SurveyAnswerParam();
        answerParam.setSuvId(id);
        List<SurveyAnswerRespVO> answerList = surveyAnswerRepoProc.findAnswer(answerParam);
        // 根据题目分组
        if(CollectionUtils.isNotEmpty(answerList)){
            SurveyRespVO surveyRespVO = this.get(id);
            Map<Long, SurveyQuestionRespVO> questionMap = surveyRespVO.getQuestionList().stream().collect(Collectors.toMap(SurveyQuestionRespVO::getId, t -> t));

            Map<Long, List<SurveyAnswerRespVO>> collect = answerList.stream().collect(Collectors.groupingBy(SurveyAnswerRespVO::getQusId));
            // 填写总人数
            int total = answerList.stream().map(SurveyAnswerRespVO::getOuCode).distinct().toList().size();
            for(Long qusId : collect.keySet()){
                List<SurveyAnswerRespVO> surveyAnswerRespVOS = collect.get(qusId);
                SurveyAnalysisRespVO analysisRespVO = new SurveyAnalysisRespVO();
                analysisRespVO.setSuvId(id);
                analysisRespVO.setQusType(surveyAnswerRespVOS.get(0).getQusType());
                analysisRespVO.setQusTitle(surveyAnswerRespVOS.get(0).getQusTitle());

                SurveyQuestionRespVO question = questionMap.get(qusId);
                List<SurveyAnalysisDetailRespVO> details = new ArrayList<>();
                for(SurveyOptionRespVO option : question.getOptionList()){
                    SurveyAnalysisDetailRespVO detail = new SurveyAnalysisDetailRespVO();
                    detail.setOpContent(option.getOpContent());
                    detail.setCount(surveyAnswerRespVOS.stream().filter(t -> t.getOptId().equals(option.getId())).count());
                    detail.setPercentage(new BigDecimal(detail.getCount()).divide(new BigDecimal(total), 2, RoundingMode.HALF_UP));
                    details.add(detail);
                }
                analysisRespVO.setOptionList(details);
                results.add(analysisRespVO);
            }
        }
        return results;
    }

}
