package com.elitesland.scp.domain.service.calendar;

import cn.hutool.core.collection.CollectionUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetPageParamVO;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreCalendarSetParamVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.calendar.ScpStoreCalendarSetRespVO;
import com.elitesland.scp.application.facade.vo.save.calendar.ScpStoreCalendarSetSaveVO;
import com.elitesland.scp.domain.convert.calendar.ScpStoreCalendarSetConvert;
import com.elitesland.scp.domain.entity.calendar.ScpStoreCalendarSetDO;
import com.elitesland.scp.infr.dto.calendar.ScpStoreCalendarSetDTO;
import com.elitesland.scp.infr.repo.calendar.ScpStoreCalendarSetLineRepoProc;
import com.elitesland.scp.infr.repo.calendar.ScpStoreCalendarSetRepo;
import com.elitesland.scp.infr.repo.calendar.ScpStoreCalendarSetRepoProc;
import com.elitesland.workflow.enums.ProcInstStatus;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@RequiredArgsConstructor
@Service
@Slf4j
public class ScpStoreCalendarSetDomainServiceImpl implements ScpStoreCalendarSetDomainService {
    private final ScpStoreCalendarSetRepoProc scpStoreCalendarSetRepoProc;
    private final ScpStoreCalendarSetRepo scpStoreCalendarSetRepo;
    private final ScpStoreCalendarSetLineRepoProc scpStoreCalendarSetLineRepoProc;
    private final UdcProvider udcProvider;

    @Override
    public PagingVO<ScpStoreCalendarSetPageRespVO> queryCalendarSetList(ScpStoreCalendarSetPageParamVO paramVO) {
        long count = scpStoreCalendarSetRepoProc.count(paramVO);
        if (count > 0) {
            var demandSet = scpStoreCalendarSetRepoProc.queryCalendarSet(paramVO);
            if (CollectionUtil.isNotEmpty(demandSet)) {
                Map<String, String> deliveryTypeMap = udcProvider.getValueMapByUdcCode("yst-supp", "ITEM_TYPE2");
                demandSet.forEach(set -> {
                    if (StringUtils.isNotBlank(set.getDeliveryType())) {
                        set.setDeliveryTypeName(Stream.of(set.getDeliveryType().split(",")).filter(deliveryTypeMap::containsKey).map(deliveryTypeMap::get).collect(Collectors.joining(",")));
                    }
                });

            }
            return new PagingVO<>(count, demandSet);
        }
        return new PagingVO<>();
    }

    @Override
    public Optional<ScpStoreCalendarSetDTO> findCalendarSetById(Long id) {
        Optional<ScpStoreCalendarSetDTO> setOptional = scpStoreCalendarSetRepo.findById(id).map(ScpStoreCalendarSetConvert.INSTANCE::doToDTO);
        setOptional.ifPresent(set -> {
            Map<String, String> deliveryTypeMap = udcProvider.getValueMapByUdcCode("yst-supp", "ITEM_TYPE2");
            if (StringUtils.isNotBlank(set.getDeliveryType())) {
                set.setDeliveryTypeName(Stream.of(set.getDeliveryType().split(",")).filter(deliveryTypeMap::containsKey).map(deliveryTypeMap::get).collect(Collectors.joining(",")));
            }
        });
        return setOptional;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveCalendarSet(ScpStoreCalendarSetSaveVO saveVO) {
        if (saveVO.getId() == null) {
            ScpStoreCalendarSetDO intentDO = ScpStoreCalendarSetConvert.INSTANCE.saveVOToDO(saveVO);
            Long masId = scpStoreCalendarSetRepo.save(intentDO).getId();
            return masId;
        } else {
            Optional<ScpStoreCalendarSetDO> option = scpStoreCalendarSetRepo.findById(saveVO.getId());
            if (option.isEmpty()) {
                throw new BusinessException("门店订货日历配置ID：" + saveVO.getId() + "不存在");
            }
            ScpStoreCalendarSetDO scpOrderSettingDO = option.get();
            ScpStoreCalendarSetConvert.INSTANCE.copySaveParamToDo(saveVO, scpOrderSettingDO);
            Long masId = scpStoreCalendarSetRepo.save(scpOrderSettingDO).getId();
            return masId;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpStoreCalendarSetRepoProc.deleteByIds(ids);
        scpStoreCalendarSetLineRepoProc.deleteByMasIds(ids);
    }

    @Override
    public List<ScpStoreCalendarSetRespVO> findCalendarSet(ScpStoreCalendarSetParamVO paramVO) {
        return scpStoreCalendarSetRepoProc.findCalendarSet(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void approveUpdateApplyAlter(List<Long> ids, String docStatus, ProcInstStatus procInstStatus, String approveComment, LocalDateTime apprTime) {
        scpStoreCalendarSetRepoProc.approveUpdateApplyAlter(ids, docStatus, procInstStatus, approveComment, apprTime);
    }

    @Override
    public void applyChangeUpdate(Long id, String docStatus, ProcInstStatus procInstStatus) {
        scpStoreCalendarSetRepoProc.applyChangeUpdate(id, docStatus, procInstStatus);
    }
}
