package com.elitesland.scp.mq;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitesland.scp.application.facade.vo.param.order.ScpOrderSubmitMqParam;
import com.elitesland.scp.application.service.order.ScpDemandOrderService;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.alloc.ScpAllocSettingRedisDO;
import com.elitesland.scp.domain.service.alloc.ScpAllocSettingStoreDomainService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

@Component
@Slf4j
@RequiredArgsConstructor
public class ScpOrderSubmitListener implements MessageQueueListener<ScpOrderSubmitMqParam> {

    private final ScpDemandOrderService scpDemandOrderService;
    private final ScpAllocSettingStoreDomainService scpAllocSettingStoreDomainService;
    private final RedisTemplate<String,Object> redisClient;
    private final TransactionTemplate transactionTemplate;

    @Override
    public @NotEmpty String[] channels() {
        return new String[]{ScpOrderSubmitMqParam.SUBMIT_CHANNEL};
    }

    @Override
    public void onConsume(@NotBlank String s, @NotNull ScpOrderSubmitMqParam orderSubmitMqParam) {
        log.info("订单提交：{}", JSON.toJSONString(orderSubmitMqParam));
        if (CollUtil.isNotEmpty(orderSubmitMqParam.getActivityIds())) {
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            transactionTemplate.execute(trans -> {
                try {
                    //更新门店已强配次数
                    ScpAllocSettingRedisDO scpAllocSettingRedisDO = new ScpAllocSettingRedisDO();
                    String key = ScpConstant.ALLOC_SETTING + orderSubmitMqParam.getDemandWhStCode();
                    Object obj = redisClient.opsForValue().get(key);
                    if (obj != null) {
                        String str = (String) obj;
                        var setting = JSON.parseObject(str, ScpAllocSettingRedisDO.class);
                        List<ScpAllocSettingRedisDO.ActivityDO> activityList = setting.getActivityList().stream().filter(row -> {
                            if (row.getValidFrom().isAfter(LocalDateTime.now()) || row.getValidTo().isBefore(LocalDateTime.now())) {
                                return false;
                            }
                            if (!orderSubmitMqParam.getActivityIds().contains(row.getActivityId())) {
                                return true;
                            }
                            return (row.getAllocNum() + 1) < row.getMaxNum();
                        }).peek(row -> row.setAllocNum(row.getAllocNum() + 1)).collect(Collectors.toList());
                        scpAllocSettingRedisDO.setActivityList(activityList);
                        redisClient.opsForValue().set(key, JSONUtil.toJsonStr(scpAllocSettingRedisDO));
                    }
                    scpAllocSettingStoreDomainService.updateAllocNumByParam(orderSubmitMqParam.getActivityIds(), orderSubmitMqParam.getDemandWhStCode());
                    return null;
                } catch (Exception e) {
                    trans.setRollbackOnly();
                    throw new BusinessException(ApiCode.FAIL, e.getMessage());
                }
            });
        }
        // 开始推送
        scpDemandOrderService.push(orderSubmitMqParam.getDemandOrderId(), orderSubmitMqParam.getAgentEmpId(), Boolean.TRUE);
    }
}
