package com.elitesland.scp.application.service.app;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.core.common.utils.UUIDUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.util.RedLockUtils;
import com.elitescloud.cloudt.system.cacheable.SysCacheUserRpcService;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.req.SysAreaQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.SysAreaRespDTO;
import com.elitescloud.cloudt.system.provider.extend.SysAreaRpcService;
import com.elitesland.inv.dto.invTro.InvTroPayRpcDTO;
import com.elitesland.pur.dto.po.PurPoPayRpcDTO;
import com.elitesland.scp.application.facade.vo.cart.StoreCartSaveVO;
import com.elitesland.scp.application.facade.vo.param.app.AppItemActivityItemPriceParamVO;
import com.elitesland.scp.application.facade.vo.resp.app.*;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderDMgmtRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetSelectRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDSaveVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDSubmitVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderSubmitVO;
import com.elitesland.scp.application.service.UserService;
import com.elitesland.scp.application.service.cart.ScpStoreCartService;
import com.elitesland.scp.application.service.common.CommonPriceService;
import com.elitesland.scp.application.service.order.ScpDemandOrderDService;
import com.elitesland.scp.application.service.order.ScpDemandOrderService;
import com.elitesland.scp.application.service.serviceconfig.ScpServiceConfigService;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderConvert;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderDConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDO;
import com.elitesland.scp.domain.entity.scpsman.ScpsmanInfoDO;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandSetDomainService;
import com.elitesland.scp.domain.service.setting.ScpOrderSettingDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepo;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepo;
import com.elitesland.scp.rmi.*;
import com.elitesland.scp.utils.*;
import com.elitesland.support.provider.item.dto.ItmItemAttachmentRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemScpBaseRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemScpBaseRpcParam;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import com.tenpay.business.entpay.mse.sdk.api.Payment;
import com.tenpay.business.entpay.mse.sdk.exception.EntpayException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.redisson.RedissonRedLock;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
@Service
@RequiredArgsConstructor
public class AppDemandOrderServiceImpl implements AppDemandOrderService {

    private final OrgStoreRpcService orgStoreRpcService;
    private final ScpDemandOrderService scpDemandOrderService;
    private final ScpDemandOrderDService scpDemandOrderDService;
    private final ScpDemandOrderDDomainService scpDemandOrderDDomainService;
    private final RmiItemService rmiItemService;
    private final CartRedisUtil cartRedisUtil;
    private final SysAreaRpcService sysAreaRpcService;
    private final RmiSysUDCService rmiSysUDCService;
    private final ScpStoreCartService scpStoreCartService;
    private final CommonPriceService commonPriceService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final RmiPurRpcService rmiPurRpcService;
    private final MessageQueueTemplate messageQueueTemplate;
    private final ScpOrderSettingDomainService scpOrderSettingDomainService;
    private final ScpDemandOrderDomainService scpDemandOrderDomainService;
    private final RedisUtils redisUtils;
    private final MessageDelyQueueService messageDelyQueueService;
    private final TransactionCommitHandler transactionCommitHandler;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;
    private final TransactionTemplate transactionTemplate;
    private final ScpDemandOrderDRepo scpDemandOrderDRepo;
    private final ScpDemandOrderRepo scpDemandOrderRepo;
    private final RmiSysUserRpcService rmiSysUserRpcService;
    private final ScpDemandSetDomainService scpDemandSetDomainService;
    private final ScpsmanInfoRepo scpsmanInfoRepo;
    private final SysCacheUserRpcService cacheUserRpcService;
    private final RedLockUtils redLockUtils;
    private final ScpServiceConfigService scpServiceConfigService;

    @Override
//    @Transactional(rollbackFor = Exception.class)
    public long submit(ScpDemandOrderSubmitVO saveVO) {
        RedissonRedLock redLock = null;
        try {
            redLock = redLockUtils.getRedLock("app-demand-order-submit-" + saveVO.getDemandWhStCode());
            boolean lockFlag = redLock.tryLock(3, 5, TimeUnit.SECONDS);
            if (!lockFlag) {
                log.error("订货单下单已在处理中,锁失败");
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "订货单下单已在处理中,请稍后再试！");
            }

            // 获取订货单明细
            List<ScpDemandOrderDSubmitVO> itemList = saveVO.getItemList();
            // 如果订货单明细为空，则抛出异常
            if (CollUtil.isEmpty(itemList)) {
                throw new BusinessException("订货单明细不能为空");
            }
            // 根据门店编码获取门店信息
            OrgStoreDetailRpcDTO orgStore = Optional.ofNullable(orgStoreRpcService.getByCode(saveVO.getDemandWhStCode())).orElseThrow(() -> new BusinessException("门店不存在"));
            // 设置门店等级和门店类型
            saveVO.setStoreLevel(orgStore.getStoreLevel());
            saveVO.setStoreType2(orgStore.getStoreType2());

            // 门店项目服务配置校验
            scpServiceConfigService.checkServiceConfig(orgStore.getStoreCode(), orgStore.getRegion(), orgStore.getStoreLevel());

            //获取当前用户信息
            Long currentUserId = UserService.currentUserId();
            if (currentUserId == null) {
                throw new BusinessException("用户不存在");
            }
            var cartOps = cartRedisUtil.getStoreCartOps(saveVO.getDemandWhStCode(), saveVO.getBusinessType(), currentUserId);

            // 判断是否重复提交，购物车如果找不到此商品，则重复提交
            ScpDemandOrderDSubmitVO scpDemandOrderDSubmitVO = itemList.get(0);
            ScpDemandOrderDSubmitVO.ItemSpecObjectVO itemSpecObjectVO = scpDemandOrderDSubmitVO.getItemSpecObjectVOS().get(0);
            String itemKey = SysUtils.generateRedisKey(ScpConstant.SKU, scpDemandOrderDSubmitVO.getCombineItemCode(), itemSpecObjectVO.getItemCode(), itemSpecObjectVO.getActivityCode());
            Object o = cartOps.get(itemKey);
            if (o == null) {
                throw new BusinessException("请勿重复提交");
            }

            // 设计预计收货时间
            Integer expectArrivePriod = orgStore.getExpectArrivePriod();
            saveVO.setEtaDate(LocalDateTime.now().plusDays(Objects.requireNonNullElse(expectArrivePriod, 0)));

            //构建订货单明细
            List<ScpDemandOrderDSaveVO> saveItemList = buildSaveItemList(itemList, null);

            // 设置默认订货集
            ScpDemandSetSelectRespVO scpDemandSetDTO = scpDemandOrderService.getScpDemandSetSelectRespVO(saveVO.getType(), orgStore.getStoreCode());
            saveVO.setDemandId(scpDemandSetDTO.getId());
            saveVO.setDemandCode(scpDemandSetDTO.getDemandCode());
            saveVO.setDemandName(scpDemandSetDTO.getDemandName());
            saveVO.setDemandDate(scpDemandSetDTO.getDemandDate());

            AtomicReference<ScpDemandOrderDO> scpDemandOrderDO = new AtomicReference<>();


            // 设置事务传播行为为REQUIRES_NEW
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            // 执行事务
            transactionTemplate.execute(
                    transactionStatus -> {
                        try {
                            // 保存订货单
                            scpDemandOrderDO.set(scpDemandOrderService.saveAndSubmitDemandOrder(ScpDemandOrderConvert.INSTANCE.submitVoToSaveVO(saveVO)));
                            // 批量保存订货单明细
                            scpDemandOrderDService.batchSaveDemandOrderD(scpDemandOrderDO.get().getId(), "MG", saveItemList);
                        } catch (Exception e) {
                            // 记录日志
                            log.error("订货单保存报错：{}", e.getMessage(), e);
                            // 回滚数据
                            transactionStatus.setRollbackOnly();
                            throw new BusinessException(e.getMessage());
                        }

                        // 注册事务提交后的回调
                        Long id = scpDemandOrderDO.get().getId();
                        //清空采购车
                        if (saveVO.getWithoutCart() == null || !saveVO.getWithoutCart()) {
                            clearCheckedStoreCartItems(saveVO, saveItemList);
                        }
                        return null;
                    }
            );
            //工作流判断
            if (scpDemandOrderDO.get().getApprStatus() != null) {
                if (scpDemandOrderDO.get().getApprStatus()
                        .equals(ScpUdcEnum.COM_APPLY_STATUS_DRAFT.getValueCode())) {
                    // 如果是草稿，则开始工作流
                    scpDemandOrderService.startWorkFlow(scpDemandOrderDO.get());
                }
            }

            // 如果事务真正提交，则注册回调
            if (TransactionSynchronizationManager.isActualTransactionActive()) {
                TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronization() {
                    @Override
                    public void afterCommit() {
                        // 在事务真正提交后执行
                        //事物提交之后进行订单推送 拆单
                        splitDoc(scpDemandOrderDO, scpDemandOrderDO.get().getId());
                    }
                });
            } else {
                // 否则直接拆单
                splitDoc(scpDemandOrderDO, scpDemandOrderDO.get().getId());
            }
            // 返回订货单ID
            return scpDemandOrderDO.get().getId();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        } finally {
            if (redLock != null) {
                redLock.unlock();
                log.info("订货单支付，释放锁成功");
            }
        }
    }


    @Override
    public long reOrder(Long id) {
        ScpDemandOrderDO oldScpDemandOrderDO = scpDemandOrderRepo.findById(id).orElseThrow(() -> new BusinessException("原订货单数据不存在"));
        String orderNo = rmiSysUserRpcService.sysNumberRuleGenerateCode(ScpConstant.OB, new ArrayList<>());

        ScpDemandOrderDO scpDemandOrderDO = new ScpDemandOrderDO();
        BeanUtils.copyProperties(oldScpDemandOrderDO, scpDemandOrderDO);
        scpDemandOrderDO.setDocCode(orderNo);
        scpDemandOrderDO.setDocStatus(ScpUdcEnum.DEO_STATUS_DRAFT.getValueCode());
        scpDemandOrderDO.setId(null);
        scpDemandOrderDO.setCreateTime(null);
        scpDemandOrderDO.setCreator(null);
        scpDemandOrderDO.setModifyTime(null);

        // 设置默认订货集
        ScpDemandSetSelectRespVO scpDemandSetDTO = scpDemandOrderService.getScpDemandSetSelectRespVO(scpDemandOrderDO.getType(), scpDemandOrderDO.getDemandWhStCode());
        scpDemandOrderDO.setDemandId(scpDemandSetDTO.getId());
        scpDemandOrderDO.setDemandCode(scpDemandSetDTO.getDemandCode());
        scpDemandOrderDO.setDemandName(scpDemandSetDTO.getDemandName());
        scpDemandOrderDO.setDemandDate(scpDemandSetDTO.getDemandDate());

        boolean shopUser = scpDemandOrderService.isShopUser();
        // 通过订货集 管控下单时间, 只有员工和店长会被管控
        if (shopUser) {
            if (scpDemandSetDTO.getExpireTime() != null && scpDemandSetDTO.getExpireTime().isBefore(LocalDateTime.now())) {
                throw new BusinessException("不在订货时间内，无法订货，重新下单失败");
            }
            // 员工和店长建的单会支付超时
            scpDemandOrderDO.setAllowPayTimeout(true);
        } else {
            scpDemandOrderDO.setAllowPayTimeout(false);
        }

        List<ScpDemandOrderDDTO> demandOrderDByMasId = scpDemandOrderDDomainService.findDemandOrderDByMasIdAndProjectFeeFlag(id, false);
        if (CollUtil.isEmpty(demandOrderDByMasId)) {
            throw new BusinessException("原订货单明细不存在");
        }
        List<ScpDemandOrderDSaveVO> saveItemList = new ArrayList<>();
        for (ScpDemandOrderDDTO scpDemandOrderDDTO : demandOrderDByMasId) {
            ScpDemandOrderDSaveVO scpDemandOrderDSaveVO = ScpDemandOrderDConvert.INSTANCE.dtoToSaveVO(scpDemandOrderDDTO);
            scpDemandOrderDSaveVO.setId(null);
            scpDemandOrderDSaveVO.setMasId(null);
            scpDemandOrderDSaveVO.setIsPushed(false);
            scpDemandOrderDSaveVO.setIsProjFeeCharged(scpDemandOrderDDTO.getTechFee() != null
                    || scpDemandOrderDDTO.getMarketingFee() != null
                    || scpDemandOrderDDTO.getOperationFee() != null);
            saveItemList.add(scpDemandOrderDSaveVO);
        }

        var orderSetting = scpOrderSettingDomainService.findCacheByBusinessType(scpDemandOrderDO.getBusinessType(), scpDemandOrderDO.getType());
        Boolean needApproval = orderSetting.getNeedApproval();
        if (needApproval != null) {
            if (needApproval) {
                scpDemandOrderDO.setApprStatus(ScpUdcEnum.COM_APPLY_STATUS_DRAFT.getValueCode());
            } else {
                scpDemandOrderDO.setApprStatus(ScpUdcEnum.COM_APPLY_STATUS_COMPLETE.getValueCode());
            }
        }

        AtomicReference<ScpDemandOrderDO> result = new AtomicReference<>();
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionTemplate.execute(
                transactionStatus -> {
                    try {
                        ScpDemandOrderDO save = scpDemandOrderRepo.save(scpDemandOrderDO);
                        result.set(save);
                        List<ScpDemandOrderDSaveVO> collect = saveItemList.stream().peek(vo -> vo.setMasId(save.getId())).collect(Collectors.toList());
                        scpDemandOrderDService.batchSaveDemandOrderD(save.getId(), "MG", collect);

                        // 关闭原单
                        oldScpDemandOrderDO.setDocStatus(ScpUdcEnum.DEO_STATUS_CLOSE.getValueCode());
                        scpDemandOrderRepo.save(oldScpDemandOrderDO);
                    } catch (Exception e) {
                        log.error("订货单保存报错：{}", e.getMessage(), e);
                        // 回滚数据
                        transactionStatus.setRollbackOnly();
                        throw new BusinessException(e.getMessage());
                    }
                    return null;
                }
        );

        long orderId = result.get().getId();
        splitDocOrWorkflow(result, orderId);
        return orderId;
    }

    private void splitDocOrWorkflow(AtomicReference<ScpDemandOrderDO> result, long orderId) {
        //事物提交之后进行订单推送 拆单
        if (result.get().getApprStatus() != null) {
            if (result.get().getApprStatus().equals(ScpUdcEnum.COM_APPLY_STATUS_COMPLETE.getValueCode())) {
                // 如果是审批完成，则直接拆单
//                List<ScpDemandOrderDDO> byMasId = scpDemandOrderDRepo.findByMasId(orderId);
                scpDemandOrderService.splitDoc(result.get(), orderId, new ArrayList<>());
            } else if (result.get().getApprStatus().equals(ScpUdcEnum.COM_APPLY_STATUS_DRAFT.getValueCode())) {
                // 如果是草稿，则开始工作流
                scpDemandOrderService.startWorkFlow(result.get());
            }
        }
    }

    private void splitDoc(AtomicReference<ScpDemandOrderDO> result, long orderId) {
        //事物提交之后进行订单推送 拆单
        if (result.get().getApprStatus() != null) {
            if (result.get().getApprStatus().equals(ScpUdcEnum.COM_APPLY_STATUS_COMPLETE.getValueCode())) {
                // 如果是审批完成，则直接拆单
//                List<ScpDemandOrderDDO> byMasId = scpDemandOrderDRepo.findByMasId(orderId);
                scpDemandOrderService.splitDoc(result.get(), orderId, new ArrayList<>());
            }
        }
    }

    /**
     * 构建保存订货单明细
     *
     * @param itemList
     * @param itemCode
     * @return
     */
    private static List<ScpDemandOrderDSaveVO> buildSaveItemList(List<ScpDemandOrderDSubmitVO> itemList, String itemCode) {
        return itemList.stream()
                .flatMap(item -> item.getItemSpecObjectVOS().stream()
                        .flatMap(row -> {
                            String uuid = UUIDUtil.getUUID();
                            // 找到 ratio 最大的 WhAllocObject
                            Optional<ScpDemandOrderDSubmitVO.WhAllocObject> maxRatioWhAllocObject = row.getWhAllocObjects().stream()
                                    .max(Comparator.comparing(ScpDemandOrderDSubmitVO.WhAllocObject::getRatio));

                            // 构建 ScpDemandOrderDSaveVO
                            Stream<ScpDemandOrderDSaveVO> dvoStream = row.getWhAllocObjects().stream()
                                    .map(whObj -> buildScpDemandOrderDVO(item, row, whObj, uuid, itemCode));

                            Stream<ScpDemandOrderDSaveVO> giftStream = Stream.empty();
                            if (maxRatioWhAllocObject.isPresent()) {
                                ScpDemandOrderDSubmitVO.WhAllocObject wh = maxRatioWhAllocObject.get();
                                // 检查 row.getGiftItemVOS() 是否为 null 或空列表
                                List<ScpDemandOrderDSubmitVO.GiftItemVO> giftItemVOS = row.getGiftItemVOS();
                                if (giftItemVOS != null && !giftItemVOS.isEmpty()) {
                                    giftStream = giftItemVOS.stream()
                                            .map(gift -> buildScpDemandOrderGiftLine(item, row, gift, wh, uuid));
                                }
                            }
                            // 合并两个流
                            return Stream.concat(dvoStream, giftStream);
                        }))
                .collect(Collectors.toList());
    }




    /**
     * 清空采购车
     *
     * @param saveVO
     * @param saveItemList
     */
    private void clearCheckedStoreCartItems(ScpDemandOrderSubmitVO saveVO, List<ScpDemandOrderDSaveVO> saveItemList) {
        for (ScpDemandOrderDSaveVO item : saveItemList) {
            if(!Boolean.TRUE.equals(item.getGiftLineFlag())){
                cartRedisUtil.clearCheckedStoreCartItems(saveVO.getDemandWhStCode(), UserService.currentUserId(), saveVO.getBusinessType(), SysUtils.getItemKey(item.getCombineItemCode(), item.getItemCode()));
            }
        }
    }

    /**
     * 构建订货单明细
     *
     * @param item
     * @param row
     * @param whAllocObject
     * @return
     */
    private static ScpDemandOrderDSaveVO buildScpDemandOrderDVO(ScpDemandOrderDSubmitVO item,
                                                                ScpDemandOrderDSubmitVO.ItemSpecObjectVO row,
                                                                ScpDemandOrderDSubmitVO.WhAllocObject whAllocObject,
                                                                String uuid,
                                                                String freightItemCode) {

        Integer amtPlace = SysUtils.getAmtPlace();

        ScpDemandOrderDSaveVO scpDemandOrderDSaveVO = new ScpDemandOrderDSaveVO();
        scpDemandOrderDSaveVO.setAllocationDeQuantity(whAllocObject.getQty());
        scpDemandOrderDSaveVO.setRatio(whAllocObject.getRatio());
        scpDemandOrderDSaveVO.setSupplyType(whAllocObject.getSupplyType());
        scpDemandOrderDSaveVO.setCombineItemCode(item.getCombineItemCode());
        scpDemandOrderDSaveVO.setCombineItemName(item.getCombineItemName());
        scpDemandOrderDSaveVO.setCombineItemFlag(item.getCombineItemFlag());

        if (ScpConstant.WH_DELIVERY_TYPES.contains(row.getDeliveryType())) {
            scpDemandOrderDSaveVO.setSuppWhId(whAllocObject.getSupplyWhId());
            scpDemandOrderDSaveVO.setSuppWhCode(whAllocObject.getSupplyWhCode());
            scpDemandOrderDSaveVO.setSuppWhName(whAllocObject.getSupplyWhName());
        } else {
            if (whAllocObject.getSuppId() == null || whAllocObject.getSuppCode() == null) {
                throw new BusinessException("直送商品【" + row.getItemName() + "】未获取到供应商");
            }
            scpDemandOrderDSaveVO.setSuppWhId(whAllocObject.getSuppId());
            scpDemandOrderDSaveVO.setSuppWhCode(whAllocObject.getSuppCode());
            scpDemandOrderDSaveVO.setSuppWhName(whAllocObject.getSuppName());
        }

        scpDemandOrderDSaveVO.setSpuItemCode(item.getSpuCode());
        scpDemandOrderDSaveVO.setSpuItemName(item.getSpuName());
        scpDemandOrderDSaveVO.setItemId(row.getItemId());
        scpDemandOrderDSaveVO.setItemCode(row.getItemCode());
        scpDemandOrderDSaveVO.setItemName(row.getItemName());
        scpDemandOrderDSaveVO.setDemandQuantity(row.getDemandQuantity());
        scpDemandOrderDSaveVO.setUnit(row.getUom());
        scpDemandOrderDSaveVO.setUnitName(row.getUomName());
        scpDemandOrderDSaveVO.setPrice(whAllocObject.getPrice());
        scpDemandOrderDSaveVO.setSalePrice(whAllocObject.getSalePrice());
        if (whAllocObject.getSettlementPrice() == null) {
            scpDemandOrderDSaveVO.setSettlementPrice(whAllocObject.getPrice());
        } else{
            scpDemandOrderDSaveVO.setSettlementPrice(whAllocObject.getSettlementPrice());
        }
        if (whAllocObject.getSettlementSalePrice() == null) {
            scpDemandOrderDSaveVO.setSettlementSalePrice(whAllocObject.getSalePrice());
        } else {
            scpDemandOrderDSaveVO.setSettlementSalePrice(whAllocObject.getSettlementSalePrice());
        }

        scpDemandOrderDSaveVO.setDiscount(row.getDiscount());
        scpDemandOrderDSaveVO.setOuId(whAllocObject.getOuId());
        scpDemandOrderDSaveVO.setOuCode(whAllocObject.getOuCode());
        scpDemandOrderDSaveVO.setOuName(whAllocObject.getOuName());
        scpDemandOrderDSaveVO.setIsCalculated(Boolean.FALSE);
        scpDemandOrderDSaveVO.setIsPushed(Boolean.FALSE);
        scpDemandOrderDSaveVO.setUom2(row.getUom2());
        scpDemandOrderDSaveVO.setUom2Name(row.getUom2Name());
        scpDemandOrderDSaveVO.setUomRatio(row.getUomRatio());
        if (row.getUomRatio() != null) {
            scpDemandOrderDSaveVO.setQty2(whAllocObject.getQty().multiply(row.getUomRatio())
                    .setScale(row.getDecimalPlaces(), RoundingMode.HALF_UP));
        }
        if (whAllocObject.getPrice() != null) {
            scpDemandOrderDSaveVO.setDemandAmt(SysUtils.processAmtScale(row.getDemandQuantity()
                    .multiply(whAllocObject.getPrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
            scpDemandOrderDSaveVO.setSaleAmt(SysUtils.processAmtScale(row.getDemandQuantity()
                    .multiply(whAllocObject.getSalePrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
        }

        if (scpDemandOrderDSaveVO.getSettlementPrice() != null) {
            scpDemandOrderDSaveVO.setSettlementAmt(SysUtils.processAmtScale(row.getDemandQuantity()
                    .multiply(scpDemandOrderDSaveVO.getSettlementPrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
        }
        if (scpDemandOrderDSaveVO.getSettlementSalePrice() != null) {
            scpDemandOrderDSaveVO.setSettlementSaleAmt(SysUtils.processAmtScale(row.getDemandQuantity()
                    .multiply(scpDemandOrderDSaveVO.getSettlementSalePrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
        }

        // 预售状态
        scpDemandOrderDSaveVO.setPreSaleStatus(row.getPreSaleStatus());

        scpDemandOrderDSaveVO.setFreightFlag(whAllocObject.getFreightFlag());
        scpDemandOrderDSaveVO.setFreightRatio(whAllocObject.getFreightRatio());
        scpDemandOrderDSaveVO.setSaleCustCode(whAllocObject.getSaleCustCode());
        scpDemandOrderDSaveVO.setSaleOuCode(whAllocObject.getSaleOuCode());
        scpDemandOrderDSaveVO.setSaleOuName(whAllocObject.getSaleOuName());

        scpDemandOrderDSaveVO.setTefPrice(whAllocObject.getTefPrice());
        scpDemandOrderDSaveVO.setMefPrice(whAllocObject.getMefPrice());
        scpDemandOrderDSaveVO.setOefPrice(whAllocObject.getOefPrice());
        scpDemandOrderDSaveVO.setIsProjFeeCharged(whAllocObject.getIsProjFeeCharged());
        scpDemandOrderDSaveVO.setProjectFeeFlag(Boolean.FALSE);

        scpDemandOrderDSaveVO.setTefFeeOuCode(whAllocObject.getTefFeeOuCode());
        scpDemandOrderDSaveVO.setTefFeeOuName(whAllocObject.getTefFeeOuName());
        scpDemandOrderDSaveVO.setOefFeeOuCode(whAllocObject.getOefFeeOuCode());
        scpDemandOrderDSaveVO.setOefFeeOuName(whAllocObject.getOefFeeOuName());
        scpDemandOrderDSaveVO.setMefFeeOuCode(whAllocObject.getMefFeeOuCode());
        scpDemandOrderDSaveVO.setMefFeeOuName(whAllocObject.getMefFeeOuName());

        if (whAllocObject.getIsProjFeeCharged() != null && whAllocObject.getIsProjFeeCharged()) {
            Optional.ofNullable(scpDemandOrderDSaveVO.getTefPrice())
                    .ifPresent(price -> scpDemandOrderDSaveVO.setTechFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity()).setScale(amtPlace, RoundingMode.HALF_UP)));
            Optional.ofNullable(scpDemandOrderDSaveVO.getMefPrice())
                    .ifPresent(price -> scpDemandOrderDSaveVO.setMarketingFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity()).setScale(amtPlace, RoundingMode.HALF_UP)));
            Optional.ofNullable(scpDemandOrderDSaveVO.getOefPrice())
                    .ifPresent(price -> scpDemandOrderDSaveVO.setOperationFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity()).setScale(amtPlace, RoundingMode.HALF_UP)));
        }

        scpDemandOrderDSaveVO.setItemType(row.getItemType());
        scpDemandOrderDSaveVO.setDeliveryType(row.getDeliveryType());
        scpDemandOrderDSaveVO.setPreRootUuid(uuid);
        scpDemandOrderDSaveVO.setImgUrl(item.getUrl());
        // 爷爷茶没有运费，这里设置为默认false 2025-06-17
        scpDemandOrderDSaveVO.setFreightLineFlag(freightItemCode != null && freightItemCode.equals(row.getItemCode()));
        scpDemandOrderDSaveVO.setActivityId(row.getActivityId());
        scpDemandOrderDSaveVO.setActivityCode(row.getActivityCode());
        scpDemandOrderDSaveVO.setMinNum(row.getMinNum());
        scpDemandOrderDSaveVO.setCostType("standard");
        whAllocObject.setUuid(uuid);
        // 设置默认的币种为 CNY
        scpDemandOrderDSaveVO.setCurrency("CNY");
        return scpDemandOrderDSaveVO;
    }

    private static ScpDemandOrderDSaveVO buildScpDemandOrderGiftLine(ScpDemandOrderDSubmitVO item,
                                                                     ScpDemandOrderDSubmitVO.ItemSpecObjectVO row,
                                                                     ScpDemandOrderDSubmitVO.GiftItemVO gift,
                                                                     ScpDemandOrderDSubmitVO.WhAllocObject wh,
                                                                     String uuid) {
        ScpDemandOrderDSaveVO scpDemandOrderDSaveVO = new ScpDemandOrderDSaveVO();
        scpDemandOrderDSaveVO.setGiftLineFlag(Boolean.TRUE);
        scpDemandOrderDSaveVO.setAllocationDeQuantity(gift.getQty());
        scpDemandOrderDSaveVO.setRatio(wh.getRatio());
        scpDemandOrderDSaveVO.setSupplyType(wh.getSupplyType());

        if (ScpConstant.WH_DELIVERY_TYPES.contains(row.getDeliveryType())) {
            scpDemandOrderDSaveVO.setSuppWhId(wh.getSupplyWhId());
            scpDemandOrderDSaveVO.setSuppWhCode(wh.getSupplyWhCode());
            scpDemandOrderDSaveVO.setSuppWhName(wh.getSupplyWhName());
        } else {
            scpDemandOrderDSaveVO.setSuppWhId(wh.getSuppId());
            scpDemandOrderDSaveVO.setSuppWhCode(wh.getSuppCode());
            scpDemandOrderDSaveVO.setSuppWhName(wh.getSuppName());
        }

        scpDemandOrderDSaveVO.setSpuItemCode(gift.getSpuCode());
        scpDemandOrderDSaveVO.setSpuItemName(gift.getSpuName());
        scpDemandOrderDSaveVO.setItemId(gift.getDiscountGiftItemId());
        scpDemandOrderDSaveVO.setItemCode(gift.getDiscountGiftItemCode());
        scpDemandOrderDSaveVO.setItemName(gift.getDiscountGiftItemName());
        scpDemandOrderDSaveVO.setDemandQuantity(row.getDemandQuantity());
        scpDemandOrderDSaveVO.setUnit(row.getUom());
        scpDemandOrderDSaveVO.setUnitName(row.getUomName());
        scpDemandOrderDSaveVO.setPrice(BigDecimal.ZERO);
        scpDemandOrderDSaveVO.setSalePrice(BigDecimal.ZERO);
        scpDemandOrderDSaveVO.setSettlementPrice(BigDecimal.ZERO);
        scpDemandOrderDSaveVO.setSettlementSalePrice(BigDecimal.ZERO);
        scpDemandOrderDSaveVO.setOuId(wh.getOuId());
        scpDemandOrderDSaveVO.setOuCode(wh.getOuCode());
        scpDemandOrderDSaveVO.setOuName(wh.getOuName());
        scpDemandOrderDSaveVO.setIsCalculated(Boolean.FALSE);
        scpDemandOrderDSaveVO.setIsPushed(Boolean.FALSE);
        scpDemandOrderDSaveVO.setUom2(row.getUom2());
        scpDemandOrderDSaveVO.setUom2Name(row.getUom2Name());
        scpDemandOrderDSaveVO.setUomRatio(row.getUomRatio());
        if (row.getUomRatio() != null) {
            scpDemandOrderDSaveVO.setQty2(gift.getQty().multiply(row.getUomRatio())
                    .setScale(row.getDecimalPlaces(), RoundingMode.HALF_UP));
        }
        scpDemandOrderDSaveVO.setDemandAmt(BigDecimal.ZERO);
        scpDemandOrderDSaveVO.setFreightFlag(Boolean.FALSE);
        scpDemandOrderDSaveVO.setSaleCustCode(wh.getSaleCustCode());
        scpDemandOrderDSaveVO.setSaleOuCode(wh.getSaleOuCode());
        scpDemandOrderDSaveVO.setSaleOuName(wh.getSaleOuName());

        scpDemandOrderDSaveVO.setTefPrice(wh.getTefPrice());
        scpDemandOrderDSaveVO.setMefPrice(wh.getMefPrice());
        scpDemandOrderDSaveVO.setOefPrice(wh.getOefPrice());
        scpDemandOrderDSaveVO.setIsProjFeeCharged(wh.getIsProjFeeCharged());
        scpDemandOrderDSaveVO.setProjectFeeFlag(Boolean.FALSE);
        scpDemandOrderDSaveVO.setTefFeeOuCode(wh.getTefFeeOuCode());
        scpDemandOrderDSaveVO.setTefFeeOuName(wh.getTefFeeOuName());
        scpDemandOrderDSaveVO.setOefFeeOuCode(wh.getOefFeeOuCode());
        scpDemandOrderDSaveVO.setOefFeeOuName(wh.getOefFeeOuName());
        scpDemandOrderDSaveVO.setMefFeeOuCode(wh.getMefFeeOuCode());
        scpDemandOrderDSaveVO.setMefFeeOuName(wh.getMefFeeOuName());

        if (wh.getIsProjFeeCharged() != null && wh.getIsProjFeeCharged()) {
            Optional.ofNullable(scpDemandOrderDSaveVO.getTefPrice())
                    .ifPresent(price -> scpDemandOrderDSaveVO.setTechFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity())));
            Optional.ofNullable(scpDemandOrderDSaveVO.getMefPrice())
                    .ifPresent(price -> scpDemandOrderDSaveVO.setMarketingFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity())));
            Optional.ofNullable(scpDemandOrderDSaveVO.getOefPrice())
                    .ifPresent(price -> scpDemandOrderDSaveVO.setOperationFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity())));
        }

        scpDemandOrderDSaveVO.setItemType(row.getItemType());
        scpDemandOrderDSaveVO.setPreRootUuid(uuid);
        scpDemandOrderDSaveVO.setImgUrl(gift.getUrl());
        scpDemandOrderDSaveVO.setFreightLineFlag(Boolean.FALSE);
        scpDemandOrderDSaveVO.setActivityId(row.getActivityId());
        scpDemandOrderDSaveVO.setActivityCode(row.getActivityCode());
        scpDemandOrderDSaveVO.setMinNum(row.getMinNum());
        return scpDemandOrderDSaveVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSaveDemandOrderD(List<ScpDemandOrderDSaveVO> saveVOS) {
        BigDecimal maxLineNo = scpDemandOrderDDomainService.findMaxLineNoPushedByMasId(saveVOS.get(0).getMasId());
        //设置行号
        AtomicInteger lineNo = new AtomicInteger(maxLineNo.add(BigDecimal.ONE).intValue());
        for (ScpDemandOrderDSaveVO saveVO : saveVOS) {
            if (saveVO.getId() != null) {
                saveVO.setLineNo(BigDecimal.valueOf(lineNo.getAndIncrement()));
            }
        }
        scpDemandOrderDDomainService.batchSave(saveVOS, lineNo);
    }

    @Override
    public void addCart(Long id) {
        ScpDemandOrderRespVO demandOrder = scpDemandOrderService.findDemandOrderById(id).orElseThrow(new BusinessException("数据不存在"));
        List<ScpDemandOrderDMgmtRespVO> demandItemList = scpDemandOrderDService.findDemandOrderDByMasId(id);
        if (CollUtil.isEmpty(demandItemList)) {
            return;
        }
        //门店编码
        String storeCode = demandOrder.getDemandWhStCode();
        //单据类型
        String docType = demandOrder.getDocType();
        // 业务类型
        String busType = demandOrder.getBusinessType();
        List<StoreCartSaveVO> cartSaveVOS = new ArrayList<>();
        String orderItemSetting = SysUtils.getOrderItemSetting();
        //查询商品图片
        List<Long> itemIds = demandItemList.stream().map(ScpDemandOrderDMgmtRespVO::getItemId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        ItmItemScpBaseRpcParam itmItemScpBaseRpcParam = new ItmItemScpBaseRpcParam();
        itmItemScpBaseRpcParam.setItemIds(itemIds);
        List<ItmItemScpBaseRpcDTO> itemList = rmiItemService.findItemScpBaseRpcDtoByParam(itmItemScpBaseRpcParam);
        Map<Long, ItmItemScpBaseRpcDTO> itemMap = itemList.stream().collect(Collectors.toMap(ItmItemScpBaseRpcDTO::getId, Function.identity()));

        //构建购物车保存属性
        for (ScpDemandOrderDMgmtRespVO item : demandItemList) {
            //过滤掉运费商品
            if (orderItemSetting.equals(item.getItemCode())) {
                continue;
            }
            StoreCartSaveVO cartSaveVO = new StoreCartSaveVO();
            ItmItemScpBaseRpcDTO itmItemScpBaseRpcDTO = itemMap.get(item.getItemId());
            cartSaveVO.setDemandWhStCode(storeCode);
            cartSaveVO.setItemType2(itmItemScpBaseRpcDTO.getItemType2());
            cartSaveVO.setCartType(docType);
            cartSaveVO.setItemId(item.getItemId());
            cartSaveVO.setItemCode(item.getItemCode());
            cartSaveVO.setItemName(itmItemScpBaseRpcDTO.getItemName());
            cartSaveVO.setUom(item.getUnit());
            cartSaveVO.setUomName(item.getUnitName());
            cartSaveVO.setSpuId(itmItemScpBaseRpcDTO.getSpuId());
            cartSaveVO.setSpuCode(itmItemScpBaseRpcDTO.getSpuCode());
            cartSaveVO.setSpuName(itmItemScpBaseRpcDTO.getSpuName());
            cartSaveVO.setQty(item.getDemandQuantity());
            cartSaveVO.setSpec(itmItemScpBaseRpcDTO.getSpec());
            var skuAttchmentList = itmItemScpBaseRpcDTO.getSkuAttchmentList();
            var spuAttchmentList = itmItemScpBaseRpcDTO.getSpuAttchmentList();
            if (CollUtil.isNotEmpty(skuAttchmentList)) {
                Optional<ItmItemAttachmentRpcDTO> first = skuAttchmentList.stream().filter(ItmItemAttachmentRpcDTO::getMajor).findFirst();
                cartSaveVO.setUrl(first.isEmpty() ? skuAttchmentList.get(0).getUrl() : first.get().getUrl());
            } else if (CollUtil.isNotEmpty(spuAttchmentList)) {
                Optional<ItmItemAttachmentRpcDTO> first = spuAttchmentList.stream().filter(ItmItemAttachmentRpcDTO::getMajor).findFirst();
                cartSaveVO.setUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
            }
            cartSaveVO.setItemCateCode(itmItemScpBaseRpcDTO.getItemCateCode());
            cartSaveVO.setItemCateName(itmItemScpBaseRpcDTO.getItemCateName());
            cartSaveVOS.add(cartSaveVO);
        }

        //实时获取价格
        //获取单据类型配置
        ScpOrderSettingRespVO orderSetting = scpOrderSettingDomainService.findCacheByBusinessType(busType, ScpUdcEnum.DEMAND_SET_TYPE_0.getValueCode());
        var priceParamVOS = cartSaveVOS.stream().map(row -> {
            AppItemActivityItemPriceParamVO priceParamVO = new AppItemActivityItemPriceParamVO();
            priceParamVO.setItemCode(row.getItemCode());
            priceParamVO.setItemId(row.getItemId());
            priceParamVO.setItemCateCode(row.getItemCateCode());
            priceParamVO.setItemType2(row.getItemType2());
            priceParamVO.setUom(row.getUom());
            return priceParamVO;
        }).collect(Collectors.toList());
        Map<String, AppItemPriceRespVO> priceMap = commonPriceService.getNewPriceAndStockDetails(priceParamVOS, storeCode, orderSetting, demandOrder.getType());
        //获取价格
        cartSaveVOS.forEach(row -> {
            if (priceMap.containsKey(row.getItemCode())) {
                AppItemPriceRespVO priceRespVO = priceMap.get(row.getItemCode());
                row.setSuppId(priceRespVO.getSuppId());
                row.setSuppCode(priceRespVO.getSuppCode());
                row.setSuppName(priceRespVO.getSuppName());
                row.setStoreCode(priceRespVO.getStoreCode());
                row.setCostPrice(priceRespVO.getCostPrice());
                row.setPurPrice(priceRespVO.getPurPrice());
                row.setIsProjFeeCharged(priceRespVO.getIsProjFeeCharged());
                row.setTefPrice(priceRespVO.getTefPrice());
                row.setMefPrice(priceRespVO.getMefPrice());
                row.setOefPrice(priceRespVO.getOefPrice());
                row.setItemPrice(priceRespVO.getPrice());
                row.setSalePrice(priceRespVO.getSalePrice());
                row.setTefFeeOuCode(priceRespVO.getTefFeeOuCode());
                row.setTefFeeOuName(priceRespVO.getTefFeeOuName());
                row.setOefFeeOuCode(priceRespVO.getOefFeeOuCode());
                row.setOefFeeOuName(priceRespVO.getOefFeeOuName());
                row.setMefFeeOuCode(priceRespVO.getMefFeeOuCode());
                row.setMefFeeOuName(priceRespVO.getMefFeeOuName());
            }
        });
        //添加购物车
        scpStoreCartService.batchSaveCart(storeCode, docType, cartSaveVOS);
    }

    @Override
    @SysCodeProc
    public ScpDemandOrderRespVO findDemandOrderById(Long id) {
        ScpDemandOrderRespVO orderRespVO = scpDemandOrderService.findDemandOrderById(id).orElseThrow(() -> new BusinessException("订货单不存在！"));
        //转换区域名称
        Map<String, String> areaMap = buildAddr(orderRespVO.getRecvProvince(), orderRespVO.getRecvCity(), orderRespVO.getRecvCounty());
        if (CollUtil.isNotEmpty(areaMap)) {
            orderRespVO.setRecvProvinceName(areaMap.get(orderRespVO.getRecvProvince()));
            orderRespVO.setRecvCityName(areaMap.get(orderRespVO.getRecvCity()));
            orderRespVO.setRecvCountyName(areaMap.get(orderRespVO.getRecvCounty()));
        }
        //合计数据
        List<ScpDemandOrderDDTO> demandOrderD = scpDemandOrderDDomainService.findDemandOrderDByMasId(id);

        //循环订单明细
        AtomicReference<BigDecimal> itemTotalCount = new AtomicReference<>(BigDecimal.ZERO);

        // 总金额
        AtomicReference<BigDecimal> amt = new AtomicReference<>(BigDecimal.ZERO);

        // 货款总金额
        AtomicReference<BigDecimal> saleAmt = new AtomicReference<>(BigDecimal.ZERO);

        AtomicReference<BigDecimal> noPayAmt = new AtomicReference<>(BigDecimal.ZERO);
        AtomicReference<BigDecimal> goodAmt = new AtomicReference<>(BigDecimal.ZERO);
        AtomicReference<BigDecimal> freightAmt = new AtomicReference<>(BigDecimal.ZERO);

        // 优惠金额
        AtomicReference<BigDecimal> discountAmt = new AtomicReference<>(BigDecimal.ZERO);
        for (ScpDemandOrderDDTO row : demandOrderD) {
            if (row.getProjectFeeFlag() != null && row.getProjectFeeFlag()) {
                saleAmt.updateAndGet(v -> v.add(row.getSettlementSaleAmt()));
            } else {
                if (row.getPlanQuantity() != null) {
                    itemTotalCount.updateAndGet(v -> v.add(row.getPlanQuantity()));
                }
                if (row.getSettlementAmt() != null) {
                    amt.updateAndGet(v -> v.add(row.getSettlementAmt()));
                }
                if (row.getSettlementSaleAmt() != null) {
                    saleAmt.updateAndGet(v -> v.add(row.getSettlementSaleAmt()));
                }
                if (row.getPayStatus() != null && ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode().equals(row.getPayStatus())) {
                    noPayAmt.updateAndGet(v -> v.add(row.getPlanAmt()));
                }
                if (row.getFreightLineFlag()) {
                    freightAmt.updateAndGet(v -> v.add(row.getPlanAmt()));
                } else {
                    goodAmt.updateAndGet(v -> v.add(row.getPlanAmt()));
                }

                if (!Objects.equals(row.getPlanAmt(), row.getSettlementAmt())) {
                    discountAmt.updateAndGet(v -> v.add(row.getPlanAmt().subtract(row.getSettlementAmt())));
                }
            }
        }
        orderRespVO.setItemTotalCount(itemTotalCount.get());
        orderRespVO.setAmt(saleAmt.get());
        orderRespVO.setDiscountAmt(discountAmt.get());
        orderRespVO.setNoPayAmt(noPayAmt.get());
        orderRespVO.setGoodAmt(goodAmt.get());
        orderRespVO.setFreightAmt(freightAmt.get());
        return orderRespVO;
    }

    @Override
    @SysCodeProc
    public AppPayOrderRespVO findPayOrderBySrcDocId(Long srcDocId) {
        AppPayOrderRespVO appPayOrderRespVO = scpDemandOrderDomainService.payOrderBySrcDocId(srcDocId);
        return getAppPayOrderRespVO(appPayOrderRespVO);
    }

    @NotNull
    private AppPayOrderRespVO getAppPayOrderRespVO(AppPayOrderRespVO appPayOrderRespVO) {
        ScpDemandOrderDTO scpDemandOrderDTO = scpDemandOrderDomainService.findDemandOrderById(appPayOrderRespVO.getDocId()).get();
        appPayOrderRespVO.setDocCode(scpDemandOrderDTO.getDocCode());
        appPayOrderRespVO.setDocType(scpDemandOrderDTO.getDocType());
        appPayOrderRespVO.setBusinessType(scpDemandOrderDTO.getBusinessType());
        appPayOrderRespVO.setCreateTime(scpDemandOrderDTO.getCreateTime());
        appPayOrderRespVO.setRecvContactName(scpDemandOrderDTO.getRecvContactName());
        appPayOrderRespVO.setRecvContactTel(scpDemandOrderDTO.getRecvContactTel());
        //如果是在线支付查询微企付订单并且更新订单支付状态
        String payStatus = queryWqfAndUpdatePayStatus(appPayOrderRespVO.getSrcDocNo(), appPayOrderRespVO.getPayStatus(), appPayOrderRespVO.getOnlinePayFlag());
        appPayOrderRespVO.setPayStatus(payStatus);
        Map<String, String> stringStringMap = this.buildAddr(scpDemandOrderDTO.getRecvProvince(), scpDemandOrderDTO.getRecvCity(),
                scpDemandOrderDTO.getRecvCounty());
        if (CollUtil.isNotEmpty(stringStringMap)) {
            appPayOrderRespVO.setRecvProvinceName(stringStringMap.get(scpDemandOrderDTO.getRecvProvince()));
            appPayOrderRespVO.setRecvCityName(stringStringMap.get(scpDemandOrderDTO.getRecvCity()));
            appPayOrderRespVO.setRecvCountyName(stringStringMap.get(scpDemandOrderDTO.getRecvCounty()));
        }
        if (!redisUtils.hasKey(ScpConstant.SCP_ONLINE_PAY + appPayOrderRespVO.getSrcDocNo())) {
            return appPayOrderRespVO;
        }
        appPayOrderRespVO.setCountdown(messageDelyQueueService.getOrderExpireTime(ScpConstant.SCP_ONLINE_PAY + appPayOrderRespVO.getSrcDocNo()));
        return appPayOrderRespVO;
    }

    @Override
    @SysCodeProc
    public AppPayOrderRespVO findPayOrderById(Long id) {
        ScpDemandOrderDTO scpDemandOrderDTO = scpDemandOrderDomainService.findDemandOrderById(id).orElseThrow(new BusinessException("数据不存在"));
        AppPayOrderRespVO appPayOrderRespVO = new AppPayOrderRespVO();
        // 查询订货账号类型
        SysUserBasicDTO byId = cacheUserRpcService.getById(scpDemandOrderDTO.getCreateUserId());
        if (byId != null) {
            ScpsmanInfoDO byLoginAccount = scpsmanInfoRepo.findByLoginAccount(byId.getUsername());
            if (byLoginAccount != null) {
                String scpManType = byLoginAccount.getScpsmanType();
                appPayOrderRespVO.setCreatorScpManType(scpManType);
            }
        }
        appPayOrderRespVO.setDocCode(scpDemandOrderDTO.getDocCode());
        appPayOrderRespVO.setDocStatus(scpDemandOrderDTO.getDocStatus());
        appPayOrderRespVO.setDocType(scpDemandOrderDTO.getDocType());
        appPayOrderRespVO.setBusinessType(scpDemandOrderDTO.getBusinessType());
        appPayOrderRespVO.setCreator(scpDemandOrderDTO.getCreator());
        appPayOrderRespVO.setCreateUserId(scpDemandOrderDTO.getCreateUserId());
        appPayOrderRespVO.setCreateTime(scpDemandOrderDTO.getCreateTime());
        appPayOrderRespVO.setRecvContactName(scpDemandOrderDTO.getRecvContactName());
        appPayOrderRespVO.setRecvContactTel(scpDemandOrderDTO.getRecvContactTel());
        appPayOrderRespVO.setPayStatus(scpDemandOrderDTO.getPayStatus());
        appPayOrderRespVO.setIsPushed(scpDemandOrderDTO.getIsPushed());
        appPayOrderRespVO.setPushedTime(scpDemandOrderDTO.getPushedTime());
        appPayOrderRespVO.setApprStatus(scpDemandOrderDTO.getApprStatus());
        appPayOrderRespVO.setRecvDetailaddr(scpDemandOrderDTO.getRecvDetailaddr());

        Map<String, String> stringStringMap = this.buildAddr(scpDemandOrderDTO.getRecvProvince(), scpDemandOrderDTO.getRecvCity(), scpDemandOrderDTO.getRecvCounty());
        if (CollUtil.isNotEmpty(stringStringMap)) {
            appPayOrderRespVO.setRecvProvinceName(stringStringMap.get(scpDemandOrderDTO.getRecvProvince()));
            appPayOrderRespVO.setRecvCityName(stringStringMap.get(scpDemandOrderDTO.getRecvCity()));
            appPayOrderRespVO.setRecvCountyName(stringStringMap.get(scpDemandOrderDTO.getRecvCounty()));
        }
        return appPayOrderRespVO;
    }

    private String queryWqfAndUpdatePayStatus(String srcDocNo, String payStatus, Boolean onlinePayFlag) {
        // 如果在线支付订单需要查询微企付订单支付信息
        if (!onlinePayFlag || ScpUdcEnum.PAY_STATUS_PAYED.getValueCode().equals(payStatus)) {
            return payStatus;
        }
        try {
            Payment payment = Payment.retrieveByOutPaymentId(srcDocNo);
            List<String> payStatusList = List.of("CLOSED", "PROCESSING");
            if (payStatusList.contains(payment.getPayStatus())) {
                return payStatus;
            }
            String updatePayStatus = "SUCCEEDED".equals(payment.getPayStatus()) ? ScpUdcEnum.PAY_STATUS_PAYED.getValueCode() : ScpUdcEnum.PAY_STATUS_PAYING.getValueCode();
            transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            transactionTemplate.execute(transactionStatus -> {
                try {
                    scpDemandOrderDRepoProc.updatePayStatusBySrcDocId(srcDocNo, updatePayStatus);
                    return "ok";
                } catch (Exception e) {
                    log.error("更新订货单失败:{}", e.getMessage(), e);
                    // 回滚
                    transactionStatus.setRollbackOnly();
                    return "ok";
                }
            });
            return updatePayStatus;
        } catch (EntpayException e) {
            log.error("查询银行信息失败:{}", e.getMessage(), e);
        }
        return payStatus;
    }

    @Override
    public AppPayOrderItemRespVO findPayOrderItemBySrcDocId(Long srcDocId) {
        List<ScpDemandOrderDDTO> demandOrderDDTOS = scpDemandOrderDDomainService.findDemandOrderDBySrcDocIdAndProjectFeeFlag(srcDocId, false);
        return getAppPayOrderItemRespVO(demandOrderDDTOS);
    }

    @NotNull
    private AppPayOrderItemRespVO getAppPayOrderItemRespVO(List<ScpDemandOrderDDTO> demandOrderDDTOS) {
        AppPayOrderItemRespVO result = new AppPayOrderItemRespVO();
        if (CollUtil.isEmpty(demandOrderDDTOS)) {
            throw new BusinessException("订单明细不存在");
        }
        Map<String, String> uomCodeMap = rmiSysUDCService.getCodeMap("yst-supp", "UOM");
        Map<String, String> deoStatus = rmiSysUDCService.getCodeMap("yst-suplan", "DEO_STATUS");
        List<Long> itemIds = demandOrderDDTOS.stream().map(ScpDemandOrderDDTO::getItemId).distinct().collect(Collectors.toList());
        Map<Long, ItmItemScpBaseRpcDTO> scpItemMap = rmiItemService.findScpItemMap(itemIds);

        //构建商品明细信息
        List<AppPayOrderItemRespVO.ItemSpecObjectVO> itemList = demandOrderDDTOS.stream().map(row -> {
            AppPayOrderItemRespVO.ItemSpecObjectVO specObj = new AppPayOrderItemRespVO.ItemSpecObjectVO();
            specObj.setItemCode(row.getItemCode());
            specObj.setItemName(row.getItemName());
            specObj.setCombineItemCode(row.getCombineItemCode());
            specObj.setCombineItemName(row.getCombineItemName());
            specObj.setActivityId(row.getActivityId());
            specObj.setActivityCode(row.getActivityCode());
            specObj.setStatus(row.getStatus());
            if (deoStatus.containsKey(row.getStatus())){
                specObj.setStatusName(deoStatus.get(row.getStatus()));
            }
            specObj.setUom(row.getUnit());
            if (uomCodeMap.containsKey(row.getUnit())) {
                specObj.setUomName(uomCodeMap.get(row.getUnit()));
            }
            if (scpItemMap.containsKey(row.getItemId())) {
                ItmItemScpBaseRpcDTO item = scpItemMap.get(row.getItemId());
                specObj.setItemAttrName(StrUtil.isBlank(item.getItemAttrName()) ? item.getSpec() : item.getItemAttrName());
                specObj.setAnotherName(item.getAnotherName());
                var spuAttchmentList = item.getSpuAttchmentList();
                if (CollUtil.isNotEmpty(item.getSkuAttchmentList())) {
                    Optional<ItmItemAttachmentRpcDTO> first = item.getSkuAttchmentList().stream().filter(ItmItemAttachmentRpcDTO::getMajor).findFirst();
                    String url = first.isPresent() ? first.get().getUrl() : item.getSkuAttchmentList().get(0).getUrl();
                    specObj.setUrl(url);
                } else if (CollUtil.isNotEmpty(spuAttchmentList)) {
                    Optional<ItmItemAttachmentRpcDTO> first = spuAttchmentList.stream().filter(ItmItemAttachmentRpcDTO::getMajor).findFirst();
                    specObj.setUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                }
            }
            specObj.setAmt(row.getSettlementAmt());
            specObj.setOriginAmt(row.getPlanAmt());
            specObj.setQty(row.getPlanQuantity());
            return specObj;
        }).toList();

        // 合计商品数量
        BigDecimal totalNum = BigDecimal.ZERO;
        BigDecimal amt = BigDecimal.ZERO;
        BigDecimal discountAmt = BigDecimal.ZERO;
        // 优惠金额
        for (ScpDemandOrderDDTO dto : demandOrderDDTOS) {
            if (dto.getPlanQuantity() != null) {
                totalNum = totalNum.add(dto.getPlanQuantity());
            }

            if (dto.getSettlementAmt() != null) {
                amt = amt.add(dto.getSettlementAmt());
            }

            if (!Objects.equals(dto.getPlanAmt(), dto.getSettlementAmt())) {
                BigDecimal subtract = dto.getPlanAmt().subtract(dto.getSettlementAmt());
                discountAmt = discountAmt.add(subtract);
            }
        }

        // 组合商品处理
        List<AppPayOrderItemRespVO.ItemSpecObjectVO> itemResultList = itemList.stream().filter(d -> StringUtils.isBlank(d.getCombineItemCode())).collect(Collectors.toList());
        Map<String, List<AppPayOrderItemRespVO.ItemSpecObjectVO>> subItemMap = itemList.stream().filter(d -> StringUtils.isNotBlank(d.getCombineItemCode())).collect(Collectors.groupingBy(AppPayOrderItemRespVO.ItemSpecObjectVO::getCombineItemCode));
        if (!subItemMap.isEmpty()) {
            subItemMap.forEach((k, v) -> {
                AppPayOrderItemRespVO.ItemSpecObjectVO combineOrderDRespVO = new AppPayOrderItemRespVO.ItemSpecObjectVO();
                combineOrderDRespVO.setCombineItemCode(k);
                combineOrderDRespVO.setCombineItemName(v.get(0).getCombineItemName());
                combineOrderDRespVO.setSubItemList(v);
                itemResultList.add(combineOrderDRespVO);
            });
        }

        result.setItemList(itemResultList);
        result.setAmt(amt);
        result.setDiscountAmt(discountAmt);
        result.setTotalNum(totalNum);
        AtomicReference<BigDecimal> goodAmt = new AtomicReference<>(BigDecimal.ZERO);
        AtomicReference<BigDecimal> freightAmt = new AtomicReference<>(BigDecimal.ZERO);
        for (ScpDemandOrderDDTO row : demandOrderDDTOS) {
            if (row.getPlanAmt() == null) {
                continue;
            }
            if (row.getFreightLineFlag()) {
                freightAmt.updateAndGet(v -> v.add(row.getPlanAmt()));
            } else {
                goodAmt.updateAndGet(v -> v.add(row.getPlanAmt()));
            }
        }
        result.setGoodAmt(goodAmt.get());
        result.setFreightAmt(freightAmt.get());
        result.setCateNum(demandOrderDDTOS.size());
        return result;
    }

    @Override
    public AppPayOrderItemRespVO findPayOrderItemById(Long id) {
        List<ScpDemandOrderDDTO> demandOrderDDTOS = scpDemandOrderDDomainService.findDemandOrderDByMasIdAndProjectFeeFlag(id,false);
        return getAppPayOrderItemRespVO(demandOrderDDTOS);
    }

    @Override
    @SysCodeProc
    public List<AppRelateOrderRespVO> findRelateOrderById(Long id) {
        List<AppRelateOrderRespVO> result = new ArrayList<>();
        List<ScpDemandOrderDDTO> orderDDTOList = scpDemandOrderDDomainService.findDemandOrderDByMasId(id);
        if (CollUtil.isEmpty(orderDDTOList)) {
            throw new BusinessException("关联单据不存在");
        }
        Map<String, String> uomCodeMap = rmiSysUDCService.getCodeMap("yst-supp", "UOM");
        Map<String, List<ScpDemandOrderDDTO>> orderMap = orderDDTOList.stream().filter(row -> row.getSrcDocId() != null).collect(Collectors.groupingBy(row -> row.getSrcDocNo() + "@" + row.getSaleOuCode()));
        // 循环orderMap
        for (Map.Entry<String, List<ScpDemandOrderDDTO>> entry : orderMap.entrySet()) {
            String srcDocNo = entry.getKey().split("@")[0];
            AppRelateOrderRespVO appRelateOrderRespVO = new AppRelateOrderRespVO();
            // 获取关联单据明细
            List<AppRelateOrderRespVO.ItemRespVO> inItemList = entry.getValue().stream().map(row -> {
                AppRelateOrderRespVO.ItemRespVO itemRespVO = new AppRelateOrderRespVO.ItemRespVO();
                itemRespVO.setItemCode(row.getItemCode());
                itemRespVO.setItemName(row.getItemName());
                itemRespVO.setQty(row.getPlanQuantity());
                itemRespVO.setAmt(row.getPlanAmt());
                itemRespVO.setUom(row.getUnit());
                if (uomCodeMap.containsKey(row.getUnit())) {
                    itemRespVO.setUomName(uomCodeMap.get(row.getUnit()));
                }
                return itemRespVO;
            }).collect(Collectors.toList());
            appRelateOrderRespVO.setItemList(inItemList);
            // 获取关联单据
            ScpDemandOrderDDTO scpDemandOrderDDTO = entry.getValue().get(0);
            //如果是在线支付查询微企付订单并且更新订单支付状态
            String payStatus = queryWqfAndUpdatePayStatus(srcDocNo, scpDemandOrderDDTO.getPayStatus(), scpDemandOrderDDTO.getOnlinePayFlag());
            appRelateOrderRespVO.setPayStatus(payStatus);
            appRelateOrderRespVO.setSrcDocNo(srcDocNo);
            appRelateOrderRespVO.setSrcDocId(scpDemandOrderDDTO.getSrcDocId());
            appRelateOrderRespVO.setSrcDocCls(scpDemandOrderDDTO.getSrcDocCls());
            appRelateOrderRespVO.setSaleOuName(scpDemandOrderDDTO.getSaleOuName());
            appRelateOrderRespVO.setSaleOuCode(scpDemandOrderDDTO.getSaleOuCode());
            appRelateOrderRespVO.setCreateTime(scpDemandOrderDDTO.getCreateTime());
            appRelateOrderRespVO.setAmt(inItemList.stream().map(AppRelateOrderRespVO.ItemRespVO::getAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));
            appRelateOrderRespVO.setPayerName(scpDemandOrderDDTO.getPayerName());
            result.add(appRelateOrderRespVO);
        }
        return result;
    }

    @Override
    public ScpOrderPayRespVO payInfo(String docCls, Long id) {
        ScpOrderPayRespVO scpOrderPayRespVO = new ScpOrderPayRespVO();
        LocalDateTime createTime = scpDemandOrderDDomainService.findDocCreateTimeBySrcDocId(id);
        if (createTime == null) {
            throw new BusinessException("订单不存在");
        }
        scpOrderPayRespVO.setDocDateTime(createTime);
        if ("STKTRO".equals(docCls)) {
            List<InvTroPayRpcDTO> invTroPayList = rmiInvStkRpcService.payInfo(id);
            if (CollUtil.isNotEmpty(invTroPayList)) {
                BigDecimal totalAmt = invTroPayList.stream().map(InvTroPayRpcDTO::getReceiptAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
                List<ScpOrderPayRespVO.PayItem> payItems = ScpDemandOrderConvert.INSTANCE.troDtosToRespVOS(invTroPayList);
                scpOrderPayRespVO.setItemList(payItems);
                scpOrderPayRespVO.setAmt(totalAmt);
                scpOrderPayRespVO.setPayDateTime(invTroPayList.get(0).getReceiptDate());
            }
        }
        if ("PO".equals(docCls)) {
            List<PurPoPayRpcDTO> purPoPayList = rmiPurRpcService.payInfo(id);
            if (CollUtil.isNotEmpty(purPoPayList)) {
                BigDecimal totalAmt = purPoPayList.stream().map(PurPoPayRpcDTO::getReceiptAmt).reduce(BigDecimal.ZERO, BigDecimal::add);
                List<ScpOrderPayRespVO.PayItem> payItems = ScpDemandOrderConvert.INSTANCE.purDtosToRespVOS(purPoPayList);
                scpOrderPayRespVO.setItemList(payItems);
                scpOrderPayRespVO.setAmt(totalAmt);
                scpOrderPayRespVO.setPayDateTime(purPoPayList.get(0).getReceiptDate());
            }
        }
        return scpOrderPayRespVO;
    }

    private Map<String, String> buildAddr(String province, String city, String county) {
        //收货地址
        Set<String> areaCodes = new HashSet<>();
        areaCodes.add(province);
        areaCodes.add(city);
        areaCodes.add(county);
        SysAreaQueryDTO comCityCodeRpcDtoParam = new SysAreaQueryDTO();
        comCityCodeRpcDtoParam.setAreaCodes(areaCodes);
        List<SysAreaRespDTO> rpcDtos = sysAreaRpcService.queryList(comCityCodeRpcDtoParam).getData();
        if (CollUtil.isNotEmpty(rpcDtos)) {
            return rpcDtos.stream().collect(Collectors.toMap(SysAreaRespDTO::getAreaCode, SysAreaRespDTO::getAreaName, (t1, t2) -> t1));
        }
        return new HashMap<>();
    }

    @Override
    public Long findIdByCode(String docCode) {
        if (StrUtil.isBlank(docCode)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "订货单编码不能为空");
        }
        Long id = scpDemandOrderDomainService.findIdByCode(docCode);
        if (id == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "订货单不存在");
        }
        return id;
    }
}
