package com.elitesland.scp.application.service.mrp;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitesland.pur.dto.supp.PurSuppBaseRpcDTO;
import com.elitesland.pur.dto.supp.PurSuppBaseRpcParam;
import com.elitesland.pur.provider.PurSuppProvider;
import com.elitesland.scp.application.facade.vo.param.item.ScpMrpDImportSaveVO;
import com.elitesland.scp.application.facade.vo.param.mrp.ScpMrpDImportEntity;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpMrpDRespVO;
import com.elitesland.scp.application.facade.vo.save.mrp.ScpMrpDPlanSaveVO;
import com.elitesland.scp.domain.service.mrp.ScpMrpDDomainService;
import com.elitesland.scp.domain.service.mrp.ScpMrpDPlanDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author chaofeng.xia
 * @since 2025/1/17
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class ScpMrpDImportService implements DataImport<ScpMrpDImportEntity> {


    private static final String LINE = "\n";
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 校验异常: {2}; ";

    private final TransactionTemplate transactionTemplate;
    private final ScpMrpDDomainService scpMrpDDomainService;
    private final ScpMrpDPlanDomainService scpMrpDPlanDomainService;
    private final OrgOuRpcService orgOuRpcService;
    private final PurSuppProvider purSuppProvider;

    @Override
    public Set<Integer> sheetNoList() {
        return Collections.singleton(1);
    }

    @Override
    public Integer stepSize() {
        return 1000000;
    }

    @Override
    public String getTmplCode() {
        return "yst_scp_mrp_d_import";
    }

    @Override
    public List<String> executeImport(List<ScpMrpDImportEntity> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errors = new ArrayList<>();

        try {
            List<String> errorMsg = checkAndSaveData(dataList, startRowIndex);
            if (CollectionUtil.isNotEmpty(errorMsg)) {
                log.info("MRP计划明细导入完成,错误信息如下:{}", JSONUtil.toJsonStr(errorMsg));
            } else {
                log.info("MRP计划明细导入完成,未发生错误");
            }
            return errorMsg;
        } catch (Exception e) {
            for (ScpMrpDImportEntity entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
    }

    private List<String> checkAndSaveData(List<ScpMrpDImportEntity> dataList, int startRowIndex) {

        Map<Integer, String> errMsgMap = new LinkedHashMap<>();
        Integer index = 0;
        //初始化行号，和错误信息容器
        for (ScpMrpDImportEntity entity : dataList) {
            entity.setRowNo(index);
            errMsgMap.put(index, null);
            index++;
        }

        log.info("startRowIndex的值:{}", startRowIndex);
        // 记录明细有错误的明细id
        Set<Long> errorIdSet = new HashSet<>();
        List<ScpMrpDImportSaveVO> dataToSave = prepareData(dataList, errMsgMap, errorIdSet);
        log.info("MRP计划导入，准备好的数据:{},错误信息:{}", JSONUtil.toJsonStr(dataToSave), JSONUtil.toJsonStr(errMsgMap));
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        //一个一个保存
        for (ScpMrpDImportSaveVO vo : dataToSave) {
            transactionTemplate.execute(
                    transactionStatus -> {
                        try {
                            scpMrpDDomainService.updateNetDemand(vo.getId(), vo.getNetDemand());
                            scpMrpDPlanDomainService.deleteByMasIds(List.of(vo.getId()));
                            scpMrpDPlanDomainService.save(vo.getScpMrpDPlanSaveVOList());
                        } catch (Exception e) {
                            log.error("MRP计划导入报错：" + e);
                            // 回滚数据
                            transactionStatus.setRollbackOnly();
                        }
                        return null;
                    }
            );
        }
        long docNumber = dataList.stream().map(ScpMrpDImportEntity::getId).distinct().count();
        String orderMsg = String.format("文件共包含%d条数据，识别出订货单%d单，其中导入成功%d单，失败%d单", dataList.size(), docNumber,
                docNumber - errorIdSet.size(),
                errorIdSet.size());
        log.info("写入自定义提示语:{}", orderMsg);
        HttpServletUtil.currentRequest().setAttribute("orderMsg", orderMsg);
        return new ArrayList<>(errMsgMap.values());
    }

    private List<ScpMrpDImportSaveVO> prepareData(List<ScpMrpDImportEntity> dataList,
                                                  Map<Integer, String> errMsgMap, Set<Long> errorIdSet) {
        List<ScpMrpDImportSaveVO> resultList = new ArrayList<>();
        Map<Long, List<ScpMrpDImportEntity>> idMap = dataList.stream().collect(Collectors.groupingBy(ScpMrpDImportEntity::getId));
        List<Long> mrpDIds = dataList.stream().collect(Collectors.groupingBy(ScpMrpDImportEntity::getId)).keySet().stream().distinct().collect(Collectors.toList());
        List<ScpMrpDRespVO> scpMrpDRespVOList = scpMrpDDomainService.findByIds(mrpDIds);
        if (CollUtil.isEmpty(scpMrpDRespVOList)) {
            addErrMsg(errMsgMap, dataList.stream().map(ScpMrpDImportEntity::getRowNo).collect(Collectors.toList()), "MRP计划明细不存在");
            errorIdSet.addAll(dataList.stream().map(ScpMrpDImportEntity::getId).collect(Collectors.toList()));
            return null;
        }

        Map<Long, ScpMrpDRespVO> mrpDMap = scpMrpDRespVOList.stream().collect(Collectors.toMap(ScpMrpDRespVO::getId, d -> d));

        // 公司
        List<String> purOuCodes = dataList.stream().map(ScpMrpDImportEntity::getPurOuCode).filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        List<String> suppCodes = dataList.stream().map(ScpMrpDImportEntity::getSuppCode).filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        // 供应商
        Map<String, OrgOuRpcSimpleDTO> ouMap = orgOuRpcService.findSimpleByOuCodes(purOuCodes).stream().collect(Collectors.toMap(OrgOuRpcSimpleDTO::getOuCode, o -> o));
        PurSuppBaseRpcParam suppParam = new PurSuppBaseRpcParam();
        suppParam.setSuppCodes(suppCodes);
        ApiResult<List<PurSuppBaseRpcDTO>> suppResult = purSuppProvider.findSimpleRpcDtoByParam(suppParam);
        Map<String, PurSuppBaseRpcDTO> suppMap = new HashMap<>();
        if (suppResult.isSuccess() && CollectionUtil.isNotEmpty(suppResult.getData())) {
            suppMap = suppResult.getData().stream().collect(Collectors.toMap(PurSuppBaseRpcDTO::getSuppCode, s -> s));
        }
        for (Map.Entry<Long, List<ScpMrpDImportEntity>> entry : idMap.entrySet()) {
            ScpMrpDImportSaveVO scpMrpDImportSaveVO = new ScpMrpDImportSaveVO();
            List<ScpMrpDImportEntity> itemList = entry.getValue();
            ScpMrpDImportEntity header = itemList.get(0);
            List<Integer> rowNoList = itemList.stream().map(ScpMrpDImportEntity::getRowNo).collect(Collectors.toList());
            // MRP明细
            ScpMrpDRespVO scpMrpDRespVO = mrpDMap.get(header.getId());
            if (scpMrpDRespVO == null) {
                addErrMsg(errMsgMap, rowNoList, "MRP计划明细不存在");
                errorIdSet.add(entry.getKey());
                continue;
            }

            // 净需求=采购计划数量之和
            if (header.getNetDemand() == null || header.getNetDemand().compareTo(BigDecimal.ZERO) <= 0) {
                addErrMsg(errMsgMap, rowNoList, "净需求为空或小于等于0");
                errorIdSet.add(entry.getKey());
                continue;
            } else {
                BigDecimal sumPurQty = itemList.stream().map(ScpMrpDImportEntity::getPurQty).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
                if (sumPurQty.compareTo(header.getNetDemand()) != 0) {
                    addErrMsg(errMsgMap, rowNoList, "采购计划数量之和不等于净需求");
                    errorIdSet.add(entry.getKey());
                    continue;
                }
            }
            scpMrpDImportSaveVO.setId(header.getId());
            scpMrpDImportSaveVO.setNetDemand(header.getNetDemand());
            Map<String, PurSuppBaseRpcDTO> finalSuppMap = suppMap;
            List<ScpMrpDPlanSaveVO> planList = new ArrayList<>();
            itemList.forEach(row -> {
                ScpMrpDPlanSaveVO scpMrpDPlanSaveVO = new ScpMrpDPlanSaveVO();
                scpMrpDPlanSaveVO.setMasId(header.getId());

                // 公司
                if (CollectionUtil.isNotEmpty(ouMap) && StringUtils.isNotBlank(row.getPurOuCode()) && ouMap.containsKey(row.getPurOuCode())) {
                    OrgOuRpcSimpleDTO ouDTO = ouMap.get(row.getPurOuCode());
                    scpMrpDPlanSaveVO.setOuCode(ouDTO.getOuCode());
                    scpMrpDPlanSaveVO.setOuName(ouDTO.getOuName());
                } else {
                    addErrMsg(errMsgMap, row.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, row.getRowNo(), "采购公司", "采购公司编码:采购公司编码为空或不存在"));
                    errorIdSet.add(entry.getKey());
                }

                // 供应商
                if (CollectionUtil.isNotEmpty(finalSuppMap) && StringUtils.isNotBlank(row.getSuppCode()) && finalSuppMap.containsKey(row.getSuppCode())) {
                    PurSuppBaseRpcDTO suppDTO = finalSuppMap.get(row.getSuppCode());
                    scpMrpDPlanSaveVO.setSuppCode(suppDTO.getSuppCode());
                    scpMrpDPlanSaveVO.setSuppName(suppDTO.getSuppName());
                } else {
                    addErrMsg(errMsgMap, row.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, row.getRowNo(), "供应商", "供应商编码:供应商编码为空或不存在"));
                    errorIdSet.add(entry.getKey());
                }

                // 采购计划数量
                if (row.getPurQty() != null && row.getPurQty().compareTo(BigDecimal.ZERO) > 0) {
                    scpMrpDPlanSaveVO.setQty(row.getPurQty());
                } else {
                    addErrMsg(errMsgMap, row.getRowNo(), MessageFormat.format(ERROR_TEMPLATE, row.getRowNo(), "采购计划数量", "采购计划数量:采购计划数量不存在或小于等于0"));
                    errorIdSet.add(entry.getKey());
                }

                // 推送状态
                scpMrpDPlanSaveVO.setPushStatus(ScpUdcEnum.MRP_D_PUSH_STATUS_NOT.getValueCode());

                planList.add(scpMrpDPlanSaveVO);
            });
            scpMrpDImportSaveVO.setScpMrpDPlanSaveVOList(planList);
            resultList.add(scpMrpDImportSaveVO);
        }
        return resultList;
    }


    public static void addErrMsg(Map<Integer, String> errMsgMap, Integer rowNo, String msg) {
        addErrMsg(errMsgMap, Collections.singletonList(rowNo), msg);
    }

    public static void addErrMsg(Map<Integer, String> errMsgMap, List<Integer> rowNos, String msg) {
        for (Integer rowNo : rowNos) {
            if (errMsgMap.containsKey(rowNo) && errMsgMap.get(rowNo) != null) {
                String newMsg = errMsgMap.get(rowNo) + LINE + msg;
                errMsgMap.put(rowNo, newMsg);
            } else {
                errMsgMap.put(rowNo, msg);
            }
        }
    }
}
