package com.elitesland.scp.infr.repo.msg;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitesland.scp.domain.entity.msg.QScpMessageChannelDO;
import com.elitesland.scp.domain.entity.msg.ScpMessageChannelDO;
import com.querydsl.core.types.Expression;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/28
 */
@Repository
public class ScpMessageChannelRepoProc extends BaseRepoProc<ScpMessageChannelDO> {
    private static final QScpMessageChannelDO QDO = QScpMessageChannelDO.scpMessageChannelDO;

    public ScpMessageChannelRepoProc() {
        super(QDO);
    }

    /**
     * 根据消息ID删除
     *
     * @param messageId
     */
    public void deleteByMessageId(long messageId) {
        super.deleteByValue(QDO.messageId, messageId);
    }

    public void deleteByMessageId(@NotEmpty Collection<Long> messageIds) {
        super.deleteByValue(QDO.messageId, messageIds);
    }

    /**
     * 删除消息渠道
     *
     * @param messageId
     * @param channels
     */
    public void deleteChannels(long messageId, @NotEmpty Collection<String> channels) {
        super.delete(QDO.messageId.eq(messageId).and(QDO.channel.in(channels)));
    }

    /**
     * 更新发布ID
     *
     * @param messageId
     * @param channel
     * @param publishedId
     */
    public void updatePublishedId(long messageId, String channel, String publishedId) {
        jpaQueryFactory.update(QDO)
                .set(QDO.publishedId, publishedId)
                .where(QDO.messageId.eq(messageId).and(QDO.channel.eq(channel)))
                .execute();
    }

    /**
     * 删除发布ID
     *
     * @param messageIds
     * @param channel
     */
    public void deletePublishedId(Collection<Long> messageIds, String channel) {
        jpaQueryFactory.update(QDO)
                .setNull(QDO.publishedId)
                .where(QDO.messageId.in(messageIds).and(QDO.channel.eq(channel)))
                .execute();
    }

    /**
     * 获取消息渠道
     *
     * @param messageId
     * @return
     */
    public List<String> getChannels(long messageId) {
        return super.getValueListByValue(QDO.channel, QDO.messageId, messageId);
    }

    /**
     * 获取消息渠道
     *
     * @param messageIds
     * @return
     */
    public Map<Long, List<String>> getChannels(@NotEmpty Collection<Long> messageIds) {
        return super.getListForGroup(new Expression[]{QDO.messageId, QDO.channel}, QDO.messageId.in(messageIds), t -> t.get(QDO.messageId), t -> t.get(QDO.channel));
    }

    /**
     * 获取发布记录ID
     *
     * @param messageIds
     * @param channel
     * @return
     */
    public Map<Long, String> getPublishedId(@NotEmpty Collection<Long> messageIds, @NotBlank String channel) {
        return super.jpaQueryFactory.select(QDO.messageId, QDO.publishedId)
                .from(QDO)
                .where(QDO.messageId.in(messageIds).and(QDO.channel.eq(channel)))
                .fetch()
                .stream()
                .filter(t -> StringUtils.hasText(t.get(QDO.publishedId)))
                .collect(Collectors.toMap(t -> t.get(QDO.messageId), t -> t.get(QDO.publishedId)));
    }
}
