package com.elitesland.scp.application.service.authority;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.exception.BusinessException;
import com.elitesland.inv.dto.invwh.InvWhDetailRpcDTO;
import com.elitesland.inv.dto.invwh.InvWhRpcDtoParam;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpManAuthorityPageRespVO;
import com.elitesland.scp.application.facade.vo.save.authority.ScpManAuthorityDSaveVO;
import com.elitesland.scp.application.facade.vo.save.authority.ScpManAuthorityImportVO;
import com.elitesland.scp.application.facade.vo.save.authority.ScpManAuthoritySaveVO;
import com.elitesland.scp.domain.entity.scpsman.ScpsmanInfoDO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepo;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpsmanAuthprityImportImpl implements DataImport<ScpManAuthorityImportVO> {


    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 解析异常: {2}; ";

    private final ScpsmanAuthorityService scpsmanAuthorityService;
    private final ScpsmanInfoRepo scpsmanInfoRepo;
    private final RmiInvStkRpcService rmiInvStkService;
    private final RmiOrgStoreRpcService rmiOrgStoreService;
    private final TaskExecutor taskExecutor;



    @Override
    public Set<Integer> sheetNoList() {
        return Collections.singleton(1);
    }

    @Override
    public Integer stepSize() {
        return 5000;
    }

    @Override
    public String getTmplCode() {
        return "yst_scpsman_authority_import";
    }

    @Override
    public List<String> executeImport(List<ScpManAuthorityImportVO> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        log.info("计划员权限导入入参：{}", JSON.toJSONString(dataList));
        List<String> errors = new ArrayList<>(dataList.size());
        try {
            return checkAndSaveData(dataList, startRowIndex);
        } catch (Exception e) {
            log.error("计划员权限导入异常：", e);
            for (ScpManAuthorityImportVO entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
    }

    private List<String> checkAndSaveData(List<ScpManAuthorityImportVO> dataList, int startRowIndex) throws Exception {

        if (CollUtil.isEmpty(dataList)) {
            log.error("计划员权限导入数据为空");
            return Lists.newArrayList();
        }
        //仓库门店查询
        CompletableFuture<List<InvWhDetailRpcDTO>> completeInvWhDetailRpcDTOS = CompletableFuture.supplyAsync(() -> {
            return this.whQuery(dataList);
        }, taskExecutor);
        //门店信息查询
        CompletableFuture<List<OrgStoreDetailRpcDTO>> completeOrgStoreDetailRpcDTOS = CompletableFuture.supplyAsync(() -> {
            return this.storeQuery(dataList);
        }, taskExecutor);
        //查询计划员
        CompletableFuture<List<ScpsmanInfoDO>> completeScpsmanInfoList = CompletableFuture.supplyAsync(() -> {
            return this.queryScpsman(dataList);
        }, taskExecutor);
        //查询计划员权限
        CompletableFuture<List<ScpManAuthorityPageRespVO>> completableScpsmanAuthorityList = CompletableFuture.supplyAsync(() -> {
            return this.queryByScpsmanAuthority(dataList);
        }, taskExecutor);
        //重复数据校验
        Map<String, List<ScpManAuthorityImportVO>> scpMap = dataList.stream().collect(Collectors.groupingBy(data -> data.getScpsmanNo()+data.getTypeName()+data.getStWhCode()));

        List<InvWhDetailRpcDTO> invWhDetailRpcDTOS = completeInvWhDetailRpcDTOS.get();
        List<OrgStoreDetailRpcDTO> orgStoreDetailRpcDTOS = completeOrgStoreDetailRpcDTOS.get();
        List<ScpsmanInfoDO> scpsmanInfoDOS = completeScpsmanInfoList.get();
        List<ScpManAuthorityPageRespVO> scpsmanAuthorityDOS = completableScpsmanAuthorityList.get();
        Map<String, List<ScpManAuthorityPageRespVO>> scpsmanAuthorityMap = scpsmanAuthorityDOS.stream().collect(Collectors.groupingBy(data -> data.getScpsmanNo()+data.getTypeName()+data.getStWhCode()));

        List<String> errorList = new ArrayList<>();
        for (ScpManAuthorityImportVO t : dataList) {
            String errorLog = new String();
            if (StringUtils.isBlank(t.getScpsmanNo())) {
                errorLog = MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "计划员编码", "计划员编码不能为空");
            }
            if (StringUtils.isBlank(t.getTypeName())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "类型", "类型不能为空");
            }
            if(StringUtils.isNotBlank(t.getTypeName()) && (!Objects.equals(t.getTypeName(),"仓库") && !Objects.equals(t.getTypeName(),"门店"))) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "类型", "非仓库/门店");
            }
            if (StringUtils.isBlank(t.getStWhCode())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码/仓库编码", "门店编码/仓库编码不能为空");
            }
            if (StringUtils.isBlank(t.getEnableStatusName())) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "启用状态", "启用状态不能为空");
            }
            if(StringUtils.isNotBlank(t.getEnableStatusName()) && (!Objects.equals(t.getEnableStatusName(),"启用") && !Objects.equals(t.getEnableStatusName(),"禁用"))) {
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "启用状态", "非启用/禁用");
            }
            //仓库状态校验
            if(CollectionUtil.isNotEmpty(invWhDetailRpcDTOS) && StringUtils.isNotBlank(t.getStWhCode())){
                Optional<InvWhDetailRpcDTO> first = invWhDetailRpcDTOS.stream().filter(n -> n.getWhCode().equals(t.getStWhCode())).findFirst();
                if(first.isPresent() && Objects.equals(first.get().getWhStatus(), ScpUdcEnum.WH_STATUS_CLOSED.getValueCode())){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码/仓库编码", "仓库已停用");
                }
                if(!first.isPresent() && (Objects.equals(t.getTypeName(),"仓库"))){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码/仓库编码", "仓库不存在");
                }
            }else{
                if(StringUtils.isNotBlank(t.getTypeName()) && (Objects.equals(t.getTypeName(),"仓库") )){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码/仓库编码", "仓库不存在");
                }
            }
            //门店状态校验
            if(CollectionUtil.isNotEmpty(orgStoreDetailRpcDTOS) && StringUtils.isNotBlank(t.getStWhCode())){
                Optional<OrgStoreDetailRpcDTO> first = orgStoreDetailRpcDTOS.stream().filter(n -> n.getStoreCode().equals(t.getStWhCode())).findFirst();
                if(first.isPresent() && Objects.equals(first.get().getStoreStatus(), "CLOSED")){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码/仓库编码", "门店是闭店状态");
                }
                if(!first.isPresent() && (Objects.equals(t.getTypeName(),"门店"))){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码/仓库编码", "门店不存在");
                }
            }else{
                if(StringUtils.isNotBlank(t.getTypeName()) && (Objects.equals(t.getTypeName(),"门店") )) {
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "门店编码/仓库编码", "门店不存在");
                }
            }
            //计划员状态
            if(CollectionUtil.isNotEmpty(scpsmanInfoDOS)  && StringUtils.isNotBlank(t.getScpsmanNo())){
                Optional<ScpsmanInfoDO> first = scpsmanInfoDOS.stream().filter(n -> n.getScpsmanNo().equals(t.getScpsmanNo())).findFirst();
                if(first.isPresent() && first.get().getEnableStatus() !=1){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "计划员状态不是启用状态");
                }
                if(first.isPresent() && StringUtils.isNotBlank(t.getOuName()) && !Objects.equals(t.getOuName(),first.get().getOuName())){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "所属公司不是当前计划员所属公司数据");
                }
                if(!first.isPresent()){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "计划员不存在");
                }
            }else{
                errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "计划员不存在");
            }
            //重复数据校验
            String checkMessage = t.getScpsmanNo()+t.getTypeName()+t.getStWhCode();
            if(MapUtils.isNotEmpty(scpMap)){
                if(scpMap.get(checkMessage) != null && scpMap.get(checkMessage).size()>1){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "存在多行重复数据");
                }
            }
            //计划员权限明细数据校验
            if(MapUtils.isNotEmpty(scpsmanAuthorityMap)){
                if(scpsmanAuthorityMap.get(checkMessage) != null && scpsmanAuthorityMap.get(checkMessage).size()>=1){
                    errorLog += MessageFormat.format(ERROR_TEMPLATE, startRowIndex, "数据库已存在此数据");
                }
            }
            errorList.add(StringUtils.isNotBlank(errorLog) ? errorLog : null );
            startRowIndex++;
        };
        if (CollectionUtil.isNotEmpty(errorList) && !errorList.stream().allMatch(element -> element == null)) {
            return errorList;
        } else {
            //生成计划员权限数据
            this.createScpsmanAuthority(dataList,invWhDetailRpcDTOS,orgStoreDetailRpcDTOS, scpsmanInfoDOS,scpsmanAuthorityDOS);
        }
        return null;
    }

    private void createScpsmanAuthority(List<ScpManAuthorityImportVO> dataList,List<InvWhDetailRpcDTO> invWhDetailRpcDTOS,List<OrgStoreDetailRpcDTO> orgStoreDetailRpcDTOS,List<ScpsmanInfoDO> byScpsmanNoIn ,List<ScpManAuthorityPageRespVO> scpsmanAuthorityDOS) {

        Map<String, List<ScpManAuthorityPageRespVO>> scpmanAuthorityMap = CollectionUtil.isNotEmpty(scpsmanAuthorityDOS) ? scpsmanAuthorityDOS.stream().collect(Collectors.groupingBy(ScpManAuthorityPageRespVO::getScpsmanNo)) : new HashMap<>();
        //计划员分组合并
        Map<String, List<ScpManAuthorityImportVO>> scpMap = dataList.stream().collect(Collectors.groupingBy(data -> data.getScpsmanNo() + data.getScpsman() + data.getOuName()));
        if (MapUtils.isNotEmpty(scpMap)) {
            List<ScpManAuthoritySaveVO> scpManAuthoritySaveVOS = new ArrayList<>();
            scpMap.keySet().stream().forEach(k -> {
                Long id = null;
                List<ScpManAuthorityImportVO> scpManAuthorityImportVOS = scpMap.get(k);
                ScpManAuthorityImportVO scpManAuthorityImportVO = scpManAuthorityImportVOS.get(0);
                ScpManAuthoritySaveVO scpManAuthoritySaveVO = new ScpManAuthoritySaveVO();
                List<ScpManAuthorityDSaveVO> scpManAuthorityDSaveVOList = new ArrayList<>();

                //计划员编码
                scpManAuthoritySaveVO.setScpsmanNo(scpManAuthorityImportVO.getScpsmanNo());
                //启用状态
                scpManAuthoritySaveVO.setEnableStatus("启用".equals(scpManAuthorityImportVO.getEnableStatusName()) ? true : false);
                //计划员id
                if (CollectionUtil.isNotEmpty(byScpsmanNoIn)) {
                    Optional<ScpsmanInfoDO> first = byScpsmanNoIn.stream().filter(n -> n.getScpsmanNo().equals(scpManAuthorityImportVO.getScpsmanNo())).findFirst();
                    scpManAuthoritySaveVO.setScpsmanId(first.get().getId());
                    //计划员名称
                    scpManAuthoritySaveVO.setScpsmanName(first.get().getName());
                    //公司
                    scpManAuthoritySaveVO.setOuName(first.get().getOuName());
                    //公司编码
                    scpManAuthoritySaveVO.setOuCode(first.get().getOuCode());
                    //公司id
                    scpManAuthoritySaveVO.setOuId(first.get().getOuId());
                }
                //计划员唯一：计划员已经存在则放到已存在计划员下(计划员明细处理)
                if(MapUtils.isNotEmpty(scpmanAuthorityMap)){
                    List<ScpManAuthorityPageRespVO> scpManAuthorityPageRespVOS = scpmanAuthorityMap.get(scpManAuthorityImportVO.getScpsmanNo());
                    if(CollectionUtil.isNotEmpty(scpManAuthorityPageRespVOS)){
                        ScpManAuthorityPageRespVO scpManAuthorityPageRespVO = scpManAuthorityPageRespVOS.get(0);
                        id = scpManAuthorityPageRespVO.getId();
                        scpManAuthorityPageRespVOS.stream().forEach( scp -> {
                            ScpManAuthorityDSaveVO scpManAuthorityDSaveVO = new ScpManAuthorityDSaveVO();
                            scpManAuthorityDSaveVO.setType(scp.getType());
                            scpManAuthorityDSaveVO.setMasId(scpManAuthorityPageRespVO.getId());
                            scpManAuthorityDSaveVO.setStWhId(scp.getStWhId());
                            scpManAuthorityDSaveVO.setStWhName(scp.getStWhName());
                            scpManAuthorityDSaveVO.setStWhCode(scp.getStWhCode());
                            scpManAuthorityDSaveVOList.add(scpManAuthorityDSaveVO);
                        });
                    }
                }

                //计划员权限明细
                scpManAuthorityImportVOS.stream().forEach(scp -> {
                    ScpManAuthorityDSaveVO scpManAuthorityDSaveVO = new ScpManAuthorityDSaveVO();
                    //类型
                    scpManAuthorityDSaveVO.setType("门店".equals(scp.getTypeName()) ? 0 : ("仓库".equals(scp.getTypeName()) ? 1 : null));

                    //门店id/仓库id
                    if ("门店".equals(scp.getTypeName()) && CollectionUtil.isNotEmpty(orgStoreDetailRpcDTOS)) {
                        Optional<OrgStoreDetailRpcDTO> first = orgStoreDetailRpcDTOS.stream().filter(n -> n.getStoreCode().equals(scp.getStWhCode())).findFirst();
                        scpManAuthorityDSaveVO.setStWhId(first.isPresent() ? first.get().getId() : null);
                        //门店编码/仓库编码
                        scpManAuthorityDSaveVO.setStWhCode(first.isPresent() ? first.get().getStoreCode() : null);
                        //门店名称/仓库名称
                        scpManAuthorityDSaveVO.setStWhName(first.isPresent() ? first.get().getStoreName() : null);
                        scpManAuthorityDSaveVO.setRegion(first.isPresent() ? first.get().getRegion() : null);
                    }
                    if ("仓库".equals(scp.getTypeName()) && CollectionUtil.isNotEmpty(invWhDetailRpcDTOS)) {
                        Optional<InvWhDetailRpcDTO> first = invWhDetailRpcDTOS.stream().filter(n -> n.getWhCode().equals(scp.getStWhCode())).findFirst();
                        scpManAuthorityDSaveVO.setStWhId(first.isPresent() ? first.get().getId() : null);
                        //门店编码/仓库编码
                        scpManAuthorityDSaveVO.setStWhCode(first.isPresent() ? first.get().getWhCode() : null);
                        //门店名称/仓库名称
                        scpManAuthorityDSaveVO.setStWhName(first.isPresent() ? first.get().getWhName() : null);
                        scpManAuthorityDSaveVO.setRegion(first.isPresent() ? first.get().getRegion() : null);
                    }
                    scpManAuthorityDSaveVOList.add(scpManAuthorityDSaveVO);
                });
                scpManAuthoritySaveVO.setId(id == null ? null : id);
                scpManAuthoritySaveVO.setScpManAuthorityDSaveVOS(scpManAuthorityDSaveVOList);
                scpManAuthoritySaveVOS.add(scpManAuthoritySaveVO);
            });
            scpManAuthoritySaveVOS.stream().forEach(auth -> {
                scpsmanAuthorityService.saveScpAuthority(auth);
            });

        } else {
            throw new BusinessException("导入分组集合为空");
        }
    }

    /**
     * 门店查询
     * @param dataList
     * @return
     */
    private List<OrgStoreDetailRpcDTO> storeQuery(List<ScpManAuthorityImportVO> dataList) {
        List<String> scpsmanNoList = dataList.stream().filter(wh -> Objects.nonNull(wh.getTypeName())).filter(t -> t.getTypeName().equals("门店")).map(ScpManAuthorityImportVO::getStWhCode).collect(Collectors.toList());
        if(CollectionUtil.isEmpty(scpsmanNoList)){
            return null;
        }
        return rmiOrgStoreService.queryByStoreCodes(scpsmanNoList);
    }

    /**
     * 仓库查询
     * @param dataList
     * @return
     */
    private List<InvWhDetailRpcDTO> whQuery(List<ScpManAuthorityImportVO> dataList) {
        InvWhRpcDtoParam invWhRpcDtoParam = new InvWhRpcDtoParam();
        List<String> scpsmanNoList = dataList.stream().filter(wh -> Objects.nonNull(wh.getTypeName())).filter(t -> t.getTypeName().equals("仓库")).map(ScpManAuthorityImportVO::getStWhCode).collect(Collectors.toList());
        if(CollectionUtil.isEmpty(scpsmanNoList)){
            return null;
        }
        invWhRpcDtoParam.setWhCodes(scpsmanNoList);
        List<InvWhDetailRpcDTO> data = rmiInvStkService.findWhAreaDTOByParam(invWhRpcDtoParam);
        return data;
    }

    /**
     * 计划员查询
     * @param dataList
     * @return
     */
    private List<ScpsmanInfoDO> queryScpsman(List<ScpManAuthorityImportVO> dataList) {
        List<String> scpsmanNoList = dataList.stream().filter(wh -> Objects.nonNull(wh.getScpsmanNo())).filter(t -> Objects.nonNull(t.getScpsmanNo())).map(ScpManAuthorityImportVO::getScpsmanNo).collect(Collectors.toList());
        if(CollectionUtil.isEmpty(scpsmanNoList)){
            return null;
        }
        return scpsmanInfoRepo.findByScpsmanNoIn(scpsmanNoList);
    }

    /**
     * 计划员权限
     * @param dataList
     */
    private List<ScpManAuthorityPageRespVO> queryByScpsmanAuthority(List<ScpManAuthorityImportVO> dataList) {
        List<String> scpsmanNoList = dataList.stream().filter(wh -> Objects.nonNull(wh.getScpsmanNo())).filter(t -> Objects.nonNull(t.getScpsmanNo())).map(ScpManAuthorityImportVO::getScpsmanNo).collect(Collectors.toList());
        if(CollectionUtil.isEmpty(scpsmanNoList)){
            return null;
        }
        return scpsmanAuthorityService.findByScpsmanNoIn(scpsmanNoList);
    }
}
