package com.elitesland.scp.application.service.app;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.el.coordinator.core.common.utils.BeanCopyUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.app.OrgStoreListParam;
import com.elitesland.scp.application.facade.vo.param.authority.ScpManAuthorityParam;
import com.elitesland.scp.application.facade.vo.param.calendar.ScpStoreDemandCalendarParam;
import com.elitesland.scp.application.facade.vo.resp.app.AppItemActivityRespVO;
import com.elitesland.scp.application.facade.vo.resp.app.OrgStoreRespVO;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpManAuthorityPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.authority.ScpsmanAuthorityComponentVO;
import com.elitesland.scp.application.service.UserService;
import com.elitesland.scp.application.service.authority.ScpsmanAuthorityManager;
import com.elitesland.scp.common.CurrentUserDTO;
import com.elitesland.scp.domain.convert.AppStoreListConvert;
import com.elitesland.scp.domain.convert.authority.AppItmItemConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDO;
import com.elitesland.scp.domain.entity.storereceive.StoreReceiveDO;
import com.elitesland.scp.domain.service.authority.ScpDemandAuthorityService;
import com.elitesland.scp.domain.service.calendar.ScpStoreDemandCalendarDomainService;
import com.elitesland.scp.infr.dto.calendar.ScpStoreDemandCalendarDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepoProc;
import com.elitesland.scp.infr.repo.scpsman.ScpsmanInfoRepoProc;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveRepo;
import com.elitesland.scp.infr.repo.storereceive.StoreReceiveRepoProc;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.support.provider.org.dto.OrgStoreRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class AppStoreListServiceImpl implements AppStoreListService {

    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final ScpDemandAuthorityService scpDemandAuthorityService;
    private final ScpDemandOrderRepoProc scpDemandOrderRepoProc;
    private final StoreReceiveRepoProc storeReceiveRepoProc;
    private final ScpStoreDemandCalendarDomainService scpStoreDemandCalendarDomainService;
    private final ScpsmanInfoRepoProc scpsmanInfoRepoProc;
    private final ScpsmanAuthorityManager scpsmanAuthorityManager;

    @Override
    public List<OrgStoreRespVO> list(OrgStoreListParam param) {
        ScpManAuthorityParam scpManAuthorityParam = new ScpManAuthorityParam();
        //门店编码不能为空
        if (StringUtils.isEmpty(param.getStoreCode())) {
            throw new BusinessException("门店编码不能为空");
        }
        //查询当前登录用户
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (ObjectUtil.isEmpty(currentUserDTO)) {
            throw new BusinessException("当前登录人查询异常");
        }
        scpManAuthorityParam.setLoginAccount(currentUserDTO.getDetail().getUsername());
        scpManAuthorityParam.setType(0L);
        PagingVO<ScpManAuthorityPageRespVO> scpManAuthorityPageRespVOPagingVO = scpDemandAuthorityService.queryAuthorit(scpManAuthorityParam);
        if (scpManAuthorityPageRespVOPagingVO.isEmpty()) {
            return null;
        }
        List<ScpManAuthorityPageRespVO> records = scpManAuthorityPageRespVOPagingVO.getRecords();
        List<String> storeCodeList = records.stream().filter(t -> ObjectUtil.isNotNull(t.getStWhCode())).map(ScpManAuthorityPageRespVO::getStWhCode).collect(Collectors.toList());
        OrgStoreRpcParam orgStoreRpcParam = new OrgStoreRpcParam();
        orgStoreRpcParam.setStoreCodes(storeCodeList);
        List<OrgStoreRpcDTO> orgStoreRpcDTOS = rmiOrgStoreRpcService.appSearch(orgStoreRpcParam);
        List<OrgStoreRespVO> orgStoreRespVOS = AppStoreListConvert.INSTANCE.doToVO(orgStoreRpcDTOS);
        //汇总订货数，门店收获数，门店退货数
        this.sumQuantity(orgStoreRespVOS);
        //将当前门店放到第一位
        List<OrgStoreRespVO> orgStoreRespVOList = this.sortFirst(orgStoreRespVOS, param.getStoreCode());
        return orgStoreRespVOList;
    }

    @Override
    public PagingVO<OrgStoreRpcDTO> pageQuery(OrgStoreRpcParam param) {
        // 查询计划员权限
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        var scpsmanInfo = scpsmanInfoRepoProc.getByScpsmanNo(currentUser.getUsername());
        if (scpsmanInfo == null) {
            return PagingVO.empty();
        }

        if (Boolean.TRUE.equals(scpsmanInfo.getJurisdictionAll())) {
            // 所有区域，则不再过滤
        } else {
            var storeAuthed = scpsmanAuthorityManager.queryStoreList(currentUser, scpsmanInfo, -1, param.getStoreCodeOrNameLike());
            if (storeAuthed.isEmpty()) {
                // 无权限
                return PagingVO.empty();
            }

            List<String> storeCodeList = new ArrayList<>();
            for (String s : storeAuthed.keySet()) {
                if (CollUtil.isEmpty(param.getStoreCodes()) || param.getStoreCodes().contains(s)) {
                    storeCodeList.add(s);
                }
            }
            param.setStoreCodes(storeCodeList);
        }

        param.setCurrent(param.getCurrent());
        param.setSize(param.getSize());

        return rmiOrgStoreRpcService.rpcStoreQuery(param);
    }

    @Override
    public PagingVO<OrgStoreRpcDTO> pageQueryByDate(OrgStoreListParam param) {
        ScpManAuthorityParam scpManAuthorityParam = new ScpManAuthorityParam();
        //查询当前登录用户
        CurrentUserDTO currentUserDTO = UserService.currentUser();
        if (ObjectUtil.isEmpty(currentUserDTO)) {
            throw new BusinessException("当前登录人查询异常");
        }
        scpManAuthorityParam.setLoginAccount(currentUserDTO.getDetail().getUsername());
        scpManAuthorityParam.setType(0L);
        List<ScpManAuthorityPageRespVO> records = scpDemandAuthorityService.listQuery(scpManAuthorityParam);
        if (CollectionUtil.isEmpty(records)) {
            return null;
        }
        List<String> storeCodeList = records.stream().filter(t -> ObjectUtil.isNotNull(t.getStWhCode())).map(ScpManAuthorityPageRespVO::getStWhCode).collect(Collectors.toList());
        List<String> codeList = new ArrayList<>();
        codeList.addAll(storeCodeList);
        if (Objects.nonNull(param) && CollectionUtil.isNotEmpty(param.getStoreCodes())) {
            codeList.addAll(param.getStoreCodes());
        }
        param.setStoreCodes(codeList);
        //查询门店数据
        OrgStoreRpcParam orgStoreRpcParam = AppItmItemConvert.INSTANCE.paramToRpcPram(param);
        List<OrgStoreRpcDTO> orgStoreRpcDTOS = rmiOrgStoreRpcService.appSearch(orgStoreRpcParam);
        if (CollectionUtil.isEmpty(orgStoreRpcDTOS)) {
            return null;
        }
        //过滤门店日历数据
        ScpStoreDemandCalendarParam scpStoreDemandCalendarParam = new ScpStoreDemandCalendarParam();
        scpStoreDemandCalendarParam.setStoreCodeList(param.getStoreCodes());
        scpStoreDemandCalendarParam.setWorkStatus("W");
        scpStoreDemandCalendarParam.setYear(param.getYear());
        scpStoreDemandCalendarParam.setMonth(param.getMonth());
        scpStoreDemandCalendarParam.setDay(param.getDay());
        List<ScpStoreDemandCalendarDTO> scpStoreDemandCalendarDTOS = scpStoreDemandCalendarDomainService.listCalendarDTOs(scpStoreDemandCalendarParam);
        if (CollectionUtil.isEmpty(scpStoreDemandCalendarDTOS)) {
            return null;
        }
        //根据门店编码取交集
        List<OrgStoreRpcDTO> collect = orgStoreRpcDTOS.stream()
                .filter(item -> scpStoreDemandCalendarDTOS.stream().anyMatch(i -> i.getStoreCode().equals(item.getStoreCode())))
                .collect(Collectors.toList());
        //分页返回前端
        int startIndex = (param.getCurrent()) * param.getSize(); // 起始索引
        int endIndex = Math.min(startIndex + param.getSize(), collect.size()); // 结束索引，防止越界
        List<OrgStoreRpcDTO> orgStoreRpcDTOS1 = collect.subList(startIndex, endIndex);
        return PagingVO.<OrgStoreRpcDTO>builder().total(collect.size()).records(orgStoreRpcDTOS1).build();
    }

    private void sumQuantity(List<OrgStoreRespVO> orgStoreRespVOS) {
        LocalDateTime todt = LocalDateTime.now(); // 获取当前时间
        LocalDateTime fromdt = todt.minusMonths(1);
        List<String> collect = orgStoreRespVOS.stream().filter(t -> ObjectUtil.isNotNull(t.getStoreCode())).map(OrgStoreRespVO::getStoreCode).collect(Collectors.toList());
        List<ScpDemandOrderDO> scpDemandOrderDOS = new ArrayList<>();
        List<StoreReceiveDO> storeReceiveDOS = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(collect)) {
            //门店订货数，
            scpDemandOrderDOS = scpDemandOrderRepoProc.findByDemandWhStCodeInAndByCreateTime(collect, fromdt, todt);
        }

        //门店收货数，门店退货数
        List<Long> storeIds = orgStoreRespVOS.stream().filter(t -> ObjectUtil.isNotNull(t.getId())).map(OrgStoreRespVO::getId).collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(storeIds)) {
            storeReceiveDOS = storeReceiveRepoProc.findAllByStoreIdInAndByCreateTime(storeIds, fromdt, todt);
        }

        List<ScpDemandOrderDO> finalScpDemandOrderDOS = scpDemandOrderDOS;
        List<StoreReceiveDO> finalStoreReceiveDOS = storeReceiveDOS;
        orgStoreRespVOS.stream().forEach(t -> {
            if (CollectionUtil.isNotEmpty(finalScpDemandOrderDOS)) {
                List<ScpDemandOrderDO> scpDemandOrderDOS1 = finalScpDemandOrderDOS.stream().filter(od -> ObjectUtil.isNotNull(od.getDemandWhStCode())).filter(or -> or.getDemandWhStCode().equals(t.getStoreCode())).collect(Collectors.toList());
                t.setOrderQuantity(CollectionUtil.isEmpty(scpDemandOrderDOS1) ? BigDecimal.ZERO : BigDecimal.valueOf(scpDemandOrderDOS1.size()));
            }
            if (CollectionUtil.isNotEmpty(finalStoreReceiveDOS)) {
                List<StoreReceiveDO> storeReceiveDOList = finalStoreReceiveDOS.stream().filter(od -> ObjectUtil.isNotNull(od.getStoreId())).filter(or -> or.getStoreId().equals(t.getId())).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(storeReceiveDOList)) {
                    List<StoreReceiveDO> r = storeReceiveDOList.stream().filter(od -> ObjectUtil.isNotNull(od.getRtType())).filter(or -> or.getRtType().equals("R")).collect(Collectors.toList());
                    List<StoreReceiveDO> rtType = storeReceiveDOList.stream().filter(od -> ObjectUtil.isNotNull(od.getRtType())).filter(or -> or.getRtType().equals("T")).collect(Collectors.toList());
                    t.setStoreReceiptNumber(CollectionUtil.isEmpty(r) ? BigDecimal.ZERO : BigDecimal.valueOf(r.size()));
                    t.setStoreReturnsNumber(CollectionUtil.isEmpty(rtType) ? BigDecimal.ZERO : BigDecimal.valueOf(rtType.size()));
                }
            }
        });

    }


    /**
     * 将指定门店放到第一位
     *
     * @param orgStoreRespVOS
     * @param storeCode
     * @return
     */
    private List<OrgStoreRespVO> sortFirst(List<OrgStoreRespVO> orgStoreRespVOS, String storeCode) {
        Optional<OrgStoreRespVO> first = orgStoreRespVOS.stream().filter(t -> ObjectUtil.isNotNull(t.getStoreCode())).filter(org -> storeCode.equals(org.getStoreCode())).findFirst();
        if (first.isPresent()) {
            OrgStoreRespVO orgStoreRespVO = first.get();
            // 先移除该元素
            int index = orgStoreRespVOS.indexOf(orgStoreRespVO);
            if (index >= 0) {
                orgStoreRespVOS.remove(index);
            }

            // 再将其插入到集合的第一个位置
            orgStoreRespVOS.add(0, orgStoreRespVO);
        }
        return orgStoreRespVOS;
    }


}
