package com.elitesland.scp.application.service.purLimit;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.PageUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.alloc.ScpAllocSettingStoreRefreshParamVO;
import com.elitesland.scp.application.facade.vo.param.purLimit.ScpPurLimitSettingImportStoreParamVO;
import com.elitesland.scp.application.facade.vo.param.purLimit.ScpPurLimitSettingPageParamVO;
import com.elitesland.scp.application.facade.vo.param.purLimit.ScpPurLimitSettingStatusParamVO;
import com.elitesland.scp.application.facade.vo.param.purLimit.ScpPurLimitSettingStoreParamVO;
import com.elitesland.scp.application.facade.vo.resp.purLimit.ScpPurLimitSettingItemRespVO;
import com.elitesland.scp.application.facade.vo.resp.purLimit.ScpPurLimitSettingPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.purLimit.ScpPurLimitSettingRespVO;
import com.elitesland.scp.application.facade.vo.resp.purLimit.ScpPurLimitSettingStoreRespVO;
import com.elitesland.scp.application.facade.vo.save.purLimit.ScpPurLimitSettingItemSaveVO;
import com.elitesland.scp.application.facade.vo.save.purLimit.ScpPurLimitSettingRegionSaveVO;
import com.elitesland.scp.application.facade.vo.save.purLimit.ScpPurLimitSettingSaveVO;
import com.elitesland.scp.application.facade.vo.save.purLimit.ScpPurLimitSettingStoreSaveVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.purLimit.ScpPurLimitSettingConvert;
import com.elitesland.scp.domain.entity.purLimit.ScpPurLimitSettingDO;
import com.elitesland.scp.domain.entity.purLimit.ScpPurLimitSettingRedisDO;
import com.elitesland.scp.domain.entity.purLimit.ScpPurLimitSettingStoreDO;
import com.elitesland.scp.domain.service.purLimit.ScpPurLimitSettingDomainService;
import com.elitesland.scp.domain.service.purLimit.ScpPurLimitSettingItemDomainService;
import com.elitesland.scp.domain.service.purLimit.ScpPurLimitSettingRegionDomainService;
import com.elitesland.scp.domain.service.purLimit.ScpPurLimitSettingStoreDomainService;
import com.elitesland.scp.infr.repo.purLimit.ScpPurLimitSettingRepo;
import com.elitesland.scp.infr.repo.purLimit.ScpPurLimitSettingStoreRepo;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.scp.rmi.RmiSysUserRpcService;
import com.elitesland.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.support.provider.item.dto.SpuImageDTO;
import com.elitesland.support.provider.item.param.ItmItemRpcDtoParam;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBasePageRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpPurLimitSettingServiceImpl implements ScpPurLimitSettingService {
    private final RmiSysUserRpcService rmiSysUserRpcService;
    private final ScpPurLimitSettingDomainService scpPurLimitSettingDomainService;
    private final ScpPurLimitSettingItemDomainService scpPurLimitSettingItemDomainService;
    private final ScpPurLimitSettingRegionDomainService scpPurLimitSettingRegionDomainService;
    private final ScpPurLimitSettingStoreDomainService scpPurLimitSettingStoreDomainService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RedisTemplate<String, Object> redisClient;
    private final ScpPurLimitSettingRepo scpPurLimitSettingRepo;
    private final RmiItemService rmiItemService;
    private final RmiSysUDCService rmiSysUDCService;
    private final ScpPurLimitSettingStoreRepo scpPurLimitSettingStoreRepo;

    /**
     * 校验参数
     *
     * @param saveVO
     */
    private static void checkParam(ScpPurLimitSettingSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getRegionList())) {
            throw new BusinessException("区域不能为空");
        }
        if (CollUtil.isEmpty(saveVO.getItemList())) {
            throw new BusinessException("商品不能为空");
        }
    }

    @Override
    public PagingVO<ScpPurLimitSettingPageRespVO> page(ScpPurLimitSettingPageParamVO paramVO) {
        Set<Long> masIds = new HashSet<>();
        if (StrUtil.isNotBlank(paramVO.getRegion())) {
            List<Long> regionMasIds = scpPurLimitSettingRegionDomainService.findMasIdByRegionCode(paramVO.getRegion());
            if (CollUtil.isEmpty(regionMasIds)) {
                return new PagingVO<>();
            }
            masIds.addAll(regionMasIds);
        }
        if (paramVO.getStoreCode() != null) {
            List<Long> storeMasIds = scpPurLimitSettingStoreDomainService.findMasIdByStoreCode(paramVO.getStoreCode());
            if (CollUtil.isEmpty(storeMasIds)) {
                return new PagingVO<>();
            }
            intersectSets(masIds, storeMasIds);
        }
        if (paramVO.getItemId() != null) {
            List<Long> itemMasIds = scpPurLimitSettingItemDomainService.findMasIdByItemId(paramVO.getItemId());
            if (CollUtil.isEmpty(itemMasIds)) {
                return new PagingVO<>();
            }
            intersectSets(masIds, itemMasIds);
        }
        paramVO.setIds(new ArrayList<>(masIds));
        return scpPurLimitSettingDomainService.queryPurLimitSettingList(paramVO);
    }

    @Override
    @SysCodeProc
    public ScpPurLimitSettingRespVO findPurLimitSettingById(Long id) {
        Optional<ScpPurLimitSettingRespVO> scpPurLimitSettingRespVO = scpPurLimitSettingDomainService.findPurLimitSettingById(id).map(ScpPurLimitSettingConvert.INSTANCE::dtoToRespVO);
        if (scpPurLimitSettingRespVO.isPresent()) {
            ScpPurLimitSettingRespVO PurLimitSettingRespVO = scpPurLimitSettingRespVO.get();
            PurLimitSettingRespVO.setAllStoreFlag(Boolean.FALSE);
            PurLimitSettingRespVO.setItemRespVOList(scpPurLimitSettingItemDomainService.findByMasId(id));
            PurLimitSettingRespVO.setRegionRespVOList(scpPurLimitSettingRegionDomainService.findByMasId(id));
            PurLimitSettingRespVO.setStoreRespVOList(scpPurLimitSettingStoreDomainService.findByMasId(id));
            if (StrUtil.isNotBlank(PurLimitSettingRespVO.getDocType())) {
                PurLimitSettingRespVO.setDocTypes(StrUtil.split(PurLimitSettingRespVO.getDocType(), ','));
            }
            return PurLimitSettingRespVO;
        }
        return new ScpPurLimitSettingRespVO();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long savePurLimitSetting(ScpPurLimitSettingSaveVO saveVO) {
        //校验参数
        checkParam(saveVO);

        List<String> oldStoreCodes;
        if (saveVO.getId() == null) {
            oldStoreCodes = new ArrayList<>();
            if (CollUtil.isEmpty(saveVO.getStoreList()) && !saveVO.getAllStoreFlag()) {
                throw new BusinessException("门店不能为空");
            }
            String orderNo = rmiSysUserRpcService.sysNumberRuleGenerateCode(ScpConstant.FA, new ArrayList<>());
            saveVO.setActivityCode(orderNo);
        } else {
            ScpPurLimitSettingStoreParamVO scpPurLimitSettingStoreParamVO = new ScpPurLimitSettingStoreParamVO();
            scpPurLimitSettingStoreParamVO.setMasIds(List.of(saveVO.getId()));
            scpPurLimitSettingStoreParamVO.setUsed(Boolean.TRUE);
            List<ScpPurLimitSettingStoreRespVO> byParam = scpPurLimitSettingStoreDomainService.findByParam(scpPurLimitSettingStoreParamVO);
            if (CollUtil.isNotEmpty(byParam)) {
                throw new BusinessException("活动已被引用，不允许编辑");
            }
            oldStoreCodes = byParam.stream().map(ScpPurLimitSettingStoreRespVO::getStoreCode).distinct().toList();
        }
        //保存表头
        Long masId = scpPurLimitSettingDomainService.savePurLimitSetting(saveVO);
        scpPurLimitSettingItemDomainService.deleteByMasId(masId);
        scpPurLimitSettingRegionDomainService.deleteByMasId(masId);
        //保存活动区域
        saveVO.getRegionList().forEach(row -> row.setMasId(masId));
        scpPurLimitSettingRegionDomainService.batchSaveSettingRegion(saveVO.getRegionList());
        //保存商品
        //查询商品信息
        List<String> itemCodes = saveVO.getItemList().stream().map(ScpPurLimitSettingItemSaveVO::getItemCode).distinct().collect(Collectors.toList());
        ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
        itmItemRpcDtoParam.setItemCodes(itemCodes);
        List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
        Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));
        saveVO.getItemList().forEach(row -> {
            row.setMasId(masId);
            if (itemRpcDTOMap.containsKey(row.getItemCode())) {
                row.setItemId(itemRpcDTOMap.get(row.getItemCode()).getId());
            }
        });
        scpPurLimitSettingItemDomainService.batchSavePurLimitSettingItem(saveVO.getItemList());

        //保存门店
        scpPurLimitSettingStoreDomainService.deleteByMasId(masId);
        if (CollUtil.isNotEmpty(saveVO.getStoreList()) && !saveVO.getAllStoreFlag()) {
            saveVO.getStoreList().forEach(row -> {
                row.setMasId(masId);
                row.setId(null);
            });
            scpPurLimitSettingStoreDomainService.batchSavePurLimitSettingStore(saveVO.getStoreList());
//            refreshRedisValue(saveVO, masId, saveVO.getStoreList(), itemRpcDTOMap);
        }
        if (CollUtil.isEmpty(saveVO.getStoreList()) && saveVO.getAllStoreFlag()) {
            saveDefaultStore(saveVO, masId, itemRpcDTOMap);
        }
        String key = ScpConstant.PUR_LIMIT_SETTING + masId;
        if (!LocalDateTime.now().isAfter(saveVO.getValidTo())) {
            long between = ChronoUnit.MINUTES.between(LocalDateTime.now(), saveVO.getValidTo());
            LocalDateTime validFrom = saveVO.getValidFrom();
            String jsonString = JSON.toJSONString(validFrom);
            redisClient.opsForValue().set(key, jsonString, between, TimeUnit.MINUTES);
        }
        scpPurLimitSettingDomainService.enablePurLimitSetting(List.of(masId), Boolean.TRUE);

        // 事务执行完，刷新门店缓存
        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronization() {
            @Override
            public void afterCommit() {
                List<String> storeCodeList = new ArrayList<>(saveVO.getStoreList().stream().map(ScpPurLimitSettingStoreSaveVO::getStoreCode).toList());
                storeCodeList.addAll(oldStoreCodes);
                if (CollUtil.isNotEmpty(storeCodeList)) {
                    ScpAllocSettingStoreRefreshParamVO paramVO = new ScpAllocSettingStoreRefreshParamVO();
                    paramVO.setStoreCodes(storeCodeList);
                    storeRefresh(paramVO);
                }
            }
        });
        return masId;
    }

    /**
     * 刷新限购活动缓存
     *
     * @param saveVO
     * @param masId
     */
    private void refreshRedisValue(ScpPurLimitSettingSaveVO saveVO, Long masId, List<ScpPurLimitSettingStoreSaveVO> storeList, Map<String, ItmItemRpcDTO> itemRpcDTOMap) {
        Map<String, String> redisMap = new HashMap<>();
        List<ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = new ArrayList<>();
        for (ScpPurLimitSettingItemSaveVO item : saveVO.getItemList()) {
            ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO innerItemDO = new ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO();
            innerItemDO.setItemCode(item.getItemCode());
            innerItemDO.setMasId(masId);
            innerItemDO.setMinNum(item.getMinNum());
            innerItemDO.setItemType2(item.getItemType2());
            if (itemRpcDTOMap.containsKey(item.getItemCode())) {
                ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(item.getItemCode());
                List<SpuImageDTO> spuAttchmentList = itmItemRpcDTO.getSpuImageList();
                if (CollUtil.isNotEmpty(spuAttchmentList)) {
                    Optional<SpuImageDTO> first = spuAttchmentList.stream().filter(SpuImageDTO::getMajor).findFirst();
                    innerItemDO.setImgUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                }
                innerItemDO.setItemId(itmItemRpcDTO.getId());
                innerItemDO.setItemName(itmItemRpcDTO.getItemName());
            }
            innerItemDOS.add(innerItemDO);
        }
        for (ScpPurLimitSettingStoreSaveVO store : storeList) {
            ScpPurLimitSettingRedisDO scpPurLimitSettingRedisDO = new ScpPurLimitSettingRedisDO();
            Object obj = redisClient.opsForValue().get(ScpConstant.PUR_LIMIT_SETTING + store.getStoreCode());
            if (obj == null) {
                ScpPurLimitSettingRedisDO.ActivityDO activityDO1 = createActivityDO(masId, saveVO, store, innerItemDOS);
                scpPurLimitSettingRedisDO.setActivityList(List.of(activityDO1));
            } else {
                String str = (String) obj;
                var setting = JSON.parseObject(str, ScpPurLimitSettingRedisDO.class);
                List<ScpPurLimitSettingRedisDO.ActivityDO> activityList = setting.getActivityList();
                boolean flag = Boolean.FALSE;
                for (ScpPurLimitSettingRedisDO.ActivityDO inActivityDO : activityList) {
                    if (inActivityDO.getActivityId().equals(masId)) {
                        inActivityDO.setActivityId(masId);
                        inActivityDO.setActivityCode(saveVO.getActivityCode());
                        inActivityDO.setDocTypes(saveVO.getDocTypes());
                        inActivityDO.setMaxNum(store.getMaxNum());
                        inActivityDO.setActiveNum(0);
                        inActivityDO.setValidTo(saveVO.getValidTo());
                        inActivityDO.setValidFrom(saveVO.getValidFrom());
                        inActivityDO.setInnerItemDOS(innerItemDOS);
                        inActivityDO.setModifiedTime(LocalDateTime.now());
                        flag = Boolean.TRUE;
                    }
                }
                if (!flag) {
                    ScpPurLimitSettingRedisDO.ActivityDO activityDO1 = createActivityDO(masId, saveVO, store, innerItemDOS);
                    activityList.add(activityDO1);
                }
                scpPurLimitSettingRedisDO.setActivityList(activityList);
            }
            String key = ScpConstant.PUR_LIMIT_SETTING + store.getStoreCode();
            redisMap.put(key, JSONUtil.toJsonStr(scpPurLimitSettingRedisDO));
        }
        redisClient.opsForValue().multiSet(redisMap);
    }

    // 提取重复代码
    private ScpPurLimitSettingRedisDO.ActivityDO createActivityDO(Long masId, ScpPurLimitSettingSaveVO saveVO, ScpPurLimitSettingStoreSaveVO store, List<ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS) {
        ScpPurLimitSettingRedisDO.ActivityDO activityDO = new ScpPurLimitSettingRedisDO.ActivityDO();
        activityDO.setInnerItemDOS(innerItemDOS);
        activityDO.setActivityId(masId);
        activityDO.setActivityCode(saveVO.getActivityCode());
        activityDO.setDocTypes(saveVO.getDocTypes());
        activityDO.setMaxNum(store.getMaxNum());
        activityDO.setActiveNum(0);
        activityDO.setValidFrom(saveVO.getValidFrom());
        activityDO.setValidTo(saveVO.getValidTo());
        activityDO.setModifiedTime(LocalDateTime.now());
        return activityDO;
    }

    /**
     * 保存默认的门店配置
     *
     * @param saveVO 限购设置保存对象，包含区域、品牌等信息
     * @param masId  主配置ID
     *               <p>
     *               此方法主要用于保存SCP分配设置中的默认门店配置它首先校验最大限购次数是否为空，
     *               然后根据提供的区域和品牌列表查询符合条件的门店，并将这些门店信息保存到数据库中
     */
    private void saveDefaultStore(ScpPurLimitSettingSaveVO saveVO, Long masId, Map<String, ItmItemRpcDTO> itemRpcDTOMap) {
        List<String> regionList = saveVO.getRegionList().stream().map(ScpPurLimitSettingRegionSaveVO::getRegion).distinct().collect(Collectors.toList());
        OrgStoreBasePageRpcParam orgStoreBasePageRpcParam = new OrgStoreBasePageRpcParam();
        orgStoreBasePageRpcParam.setRegionList(regionList);

        Long count = rmiOrgStoreRpcService.countBaseStoreByParam(orgStoreBasePageRpcParam);
        if (count == null || count == 0) {
            log.info("saveDefaultStore查询符合条件门店数据为空");
            return;
        }
        int totalPage = PageUtil.totalPage(count, 1000);
        for (int i = 1; i <= totalPage; i++) {
            orgStoreBasePageRpcParam.setSize(1000);
            orgStoreBasePageRpcParam.setCurrent(i);
            PagingVO<OrgStoreBaseRpcDTO> orgStoreBaseRpcDTOPagingVO = rmiOrgStoreRpcService.queryBaseStore(orgStoreBasePageRpcParam);
            List<ScpPurLimitSettingStoreSaveVO> storeSaveVOS = orgStoreBaseRpcDTOPagingVO.getRecords().stream().map(row -> {
                ScpPurLimitSettingStoreSaveVO scpPurLimitSettingStoreSaveVO = new ScpPurLimitSettingStoreSaveVO();
                scpPurLimitSettingStoreSaveVO.setStoreId(row.getId());
                scpPurLimitSettingStoreSaveVO.setStoreCode(row.getStoreCode());
                scpPurLimitSettingStoreSaveVO.setStoreName(row.getStoreName());
                scpPurLimitSettingStoreSaveVO.setStoreType2(row.getStoreType2());
                scpPurLimitSettingStoreSaveVO.setMasId(masId);
                scpPurLimitSettingStoreSaveVO.setMaxNum(saveVO.getMaxNum());
                return scpPurLimitSettingStoreSaveVO;
            }).collect(Collectors.toList());
            scpPurLimitSettingStoreDomainService.batchSavePurLimitSettingStore(storeSaveVOS, 1000);
            refreshRedisValue(saveVO, masId, storeSaveVOS, itemRpcDTOMap);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpPurLimitSettingDomainService.deleteByIds(ids);
        List<String> keys = ids.stream().map(row -> ScpConstant.PUR_LIMIT_SETTING + row).collect(Collectors.toList());
        redisClient.delete(keys);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changePurLimitSettingStatus(ScpPurLimitSettingStatusParamVO paramVO) {
        if (Boolean.TRUE.equals(paramVO.getStatus())) {
            //校验门店是否存在其他活动中
            ScpPurLimitSettingStoreParamVO scpPurLimitSettingStoreParamVO = new ScpPurLimitSettingStoreParamVO();
            scpPurLimitSettingStoreParamVO.setMasIds(paramVO.getIds());
            List<ScpPurLimitSettingStoreRespVO> storeRespVOList = scpPurLimitSettingStoreDomainService.findByParam(scpPurLimitSettingStoreParamVO);
            if (CollUtil.isEmpty(storeRespVOList)) {
                throw new BusinessException("门店信息未维护");
            }
            List<ScpPurLimitSettingDO> settingDOS = scpPurLimitSettingRepo.findByIdIn(paramVO.getIds());
            for (ScpPurLimitSettingDO settingDO : settingDOS) {
                String key = ScpConstant.PUR_LIMIT_SETTING + settingDO.getId();
                if (!LocalDateTime.now().isAfter(settingDO.getValidTo())) {
                    long between = ChronoUnit.MINUTES.between(LocalDateTime.now(), settingDO.getValidTo());
                    LocalDateTime validFrom = settingDO.getValidFrom();
                    String jsonString = JSON.toJSONString(validFrom);
                    redisClient.opsForValue().set(key, jsonString, between, TimeUnit.MINUTES);
                } else {
                    continue;
                }
                List<ScpPurLimitSettingItemRespVO> itemList = scpPurLimitSettingItemDomainService.findByMasId(settingDO.getId());
                //查询商品信息
                List<String> itemCodes = itemList.stream().map(ScpPurLimitSettingItemRespVO::getItemCode).distinct().collect(Collectors.toList());
                ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
                itmItemRpcDtoParam.setItemCodes(itemCodes);
                List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
                Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));

                List<ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = new ArrayList<>();
                for (ScpPurLimitSettingItemRespVO item : itemList) {
                    ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO innerItemDO = new ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO();
                    innerItemDO.setItemCode(item.getItemCode());
                    innerItemDO.setMasId(settingDO.getId());
                    innerItemDO.setMinNum(item.getMinNum());
                    innerItemDO.setItemType2(item.getItemType2());
                    if (itemRpcDTOMap.containsKey(item.getItemCode())) {
                        ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(item.getItemCode());
                        List<SpuImageDTO> spuAttchmentList = itmItemRpcDTO.getSpuImageList();
                        if (CollUtil.isNotEmpty(spuAttchmentList)) {
                            Optional<SpuImageDTO> first = spuAttchmentList.stream().filter(SpuImageDTO::getMajor).findFirst();
                            innerItemDO.setImgUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                        }
                        innerItemDO.setItemId(itmItemRpcDTO.getId());
                        innerItemDO.setItemName(itmItemRpcDTO.getItemName());
                    }
                    innerItemDOS.add(innerItemDO);
                }
                //写入缓存
                Map<String, String> redisMap = new HashMap<>();
                for (ScpPurLimitSettingStoreRespVO store : storeRespVOList) {
                    ScpPurLimitSettingRedisDO scpPurLimitSettingRedisDO = new ScpPurLimitSettingRedisDO();
                    Object obj = redisClient.opsForValue().get(ScpConstant.PUR_LIMIT_SETTING + store.getStoreCode());
                    if (obj == null) {
                        ScpPurLimitSettingRedisDO.ActivityDO activityDO1 = new ScpPurLimitSettingRedisDO.ActivityDO();
                        activityDO1.setActivityId(settingDO.getId());
                        activityDO1.setActivityCode(settingDO.getActivityCode());
                        activityDO1.setDocTypes(StrUtil.split(settingDO.getDocType(), ","));
                        activityDO1.setMaxNum(store.getMaxNum());
                        activityDO1.setActiveNum(0);
                        activityDO1.setInnerItemDOS(innerItemDOS);
                        activityDO1.setValidFrom(settingDO.getValidFrom());
                        activityDO1.setValidTo(settingDO.getValidTo());
                        activityDO1.setModifiedTime(settingDO.getModifyTime());
                        scpPurLimitSettingRedisDO.setActivityList(List.of(activityDO1));
                        String storeKey = ScpConstant.PUR_LIMIT_SETTING + store.getStoreCode();
                        String jsonString = JSON.toJSONString(scpPurLimitSettingRedisDO);
                        log.info("写入缓存：{}", jsonString);
                        redisMap.put(storeKey, jsonString);
                    }
                }
                if (CollUtil.isNotEmpty(redisMap)) {
                    redisClient.opsForValue().multiSet(redisMap);
                }
            }
        } else {
            List<String> keys = paramVO.getIds().stream().map(row -> ScpConstant.PUR_LIMIT_SETTING + row).collect(Collectors.toList());
            redisClient.delete(keys);
        }
        scpPurLimitSettingDomainService.enablePurLimitSetting(paramVO.getIds(), paramVO.getStatus());
    }

    @Override
    public List<ScpPurLimitSettingStoreRespVO> importStoreData(List<ScpPurLimitSettingImportStoreParamVO> list) {
        if (CollectionUtil.isEmpty(list)) {
            throw new BusinessException("导入数据为空");
        }
        for (int i = 0; i < list.size(); i++) {
            int row = i + 3;
            ScpPurLimitSettingImportStoreParamVO importData = list.get(i);
            if (StrUtil.isBlank(importData.getStoreCode())) {
                throw new BusinessException("第" + row + "行门店编码为空");
            }
        }

        // 获取所有门店信息
        List<String> storeCodes = list.stream()
                .map(ScpPurLimitSettingImportStoreParamVO::getStoreCode)
                .distinct()
                .toList();
        List<OrgStoreBaseRpcDTO> orgStoreBaseByParam = rmiOrgStoreRpcService.findOrgStoreBaseByParam(storeCodes);
        Map<String, OrgStoreBaseRpcDTO> storeBaseMap;
        if (CollUtil.isNotEmpty(orgStoreBaseByParam)) {
            storeBaseMap = orgStoreBaseByParam.stream().collect(Collectors.toMap(OrgStoreBaseRpcDTO::getStoreCode, Function.identity(), (key, value) -> value));
        } else {
            storeBaseMap = new HashMap<>();
        }

        Map<String, String> storeType2 = rmiSysUDCService.getCodeMap("yst-supp", "STORE_TYPE2");
        List<ScpPurLimitSettingStoreRespVO> res = new ArrayList<>();
        for (int i = 0; i < list.size(); i++) {
            int row = i + 3;
            ScpPurLimitSettingImportStoreParamVO l = list.get(i);

            //查询门店信息
            OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = storeBaseMap.get(l.getStoreCode());
            if (orgStoreBaseRpcDTO == null) {
                throw new BusinessException("第" + row + "行门店不存在");
            }

            ScpPurLimitSettingStoreRespVO applyDResp = new ScpPurLimitSettingStoreRespVO();
            applyDResp.setStoreId(orgStoreBaseRpcDTO.getId());
            applyDResp.setStoreCode(orgStoreBaseRpcDTO.getStoreCode());
            applyDResp.setStoreName(orgStoreBaseRpcDTO.getStoreName());
            applyDResp.setRegion(orgStoreBaseRpcDTO.getRegion());
            applyDResp.setRegionName(orgStoreBaseRpcDTO.getRegionName());
            applyDResp.setStoreType2(orgStoreBaseRpcDTO.getStoreType2());
            String storeTypeName = storeType2.get(orgStoreBaseRpcDTO.getStoreType2());
            if (storeTypeName != null) {
                applyDResp.setStoreType2Name(storeTypeName);
            }
            res.add(applyDResp);
        }
        return res;
    }

    @Override
    public void storeRefresh(ScpAllocSettingStoreRefreshParamVO paramVO) {
        if (CollUtil.isEmpty(paramVO.getStoreCodes())) {
            return;
        }
        List<ScpPurLimitSettingStoreDO> allByStoreCodeIn = scpPurLimitSettingStoreRepo.findAllByStoreCodeIn(paramVO.getStoreCodes());
        List<String> storeCodes;
        List<String> storeCodesNotIn;
        if (CollUtil.isEmpty(allByStoreCodeIn)) {
            storeCodesNotIn = paramVO.getStoreCodes();
            storeCodes = new ArrayList<>();
        } else {
            // paramVO.getStoreCodes() 和 allByStoreCodeIn的storeCode比较，分为两个门店集合，一个是存在allByStoreCodeIn里的，还有一个是不存在allByStoreCodeIn里的
            storeCodes = allByStoreCodeIn.stream().map(ScpPurLimitSettingStoreDO::getStoreCode).toList();
            storeCodesNotIn = paramVO.getStoreCodes().stream().filter(v -> !storeCodes.contains(v)).toList();
        }

        if (CollUtil.isNotEmpty(storeCodes)) {
            // 获取门店编码和活动ID的分组
            Map<String, List<ScpPurLimitSettingStoreDO>> storeAndPurLimitStoreList = allByStoreCodeIn.stream().collect(Collectors.groupingBy(ScpPurLimitSettingStoreDO::getStoreCode));
            List<Long> idList = storeAndPurLimitStoreList.values().stream().flatMap(v -> v.stream().map(ScpPurLimitSettingStoreDO::getMasId)).toList();

            List<ScpPurLimitSettingDO> byIdIn = scpPurLimitSettingRepo.findByIdIn(idList);
            Map<Long, ScpPurLimitSettingDO> collect1 = byIdIn.stream().collect(Collectors.toMap(ScpPurLimitSettingDO::getId, Function.identity()));

            Map<String, String> redisMap = new HashMap<>();
            storeAndPurLimitStoreList.forEach((storeCode, ids) -> {
                ScpPurLimitSettingRedisDO ScpPurLimitSettingRedisDO = new ScpPurLimitSettingRedisDO();
                String storeKey = ScpConstant.PUR_LIMIT_SETTING + storeCode;
                for (ScpPurLimitSettingStoreDO id : ids) {
                    ScpPurLimitSettingDO settingDO = collect1.get(id.getMasId());
                    if (settingDO != null) {
                        List<ScpPurLimitSettingItemRespVO> itemList = scpPurLimitSettingItemDomainService.findByMasId(settingDO.getId());
                        //查询商品信息
                        List<String> itemCodes = itemList.stream().map(ScpPurLimitSettingItemRespVO::getItemCode).distinct().collect(Collectors.toList());
                        ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
                        itmItemRpcDtoParam.setItemCodes(itemCodes);
                        List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
                        Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));

                        List<ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = new ArrayList<>();
                        for (ScpPurLimitSettingItemRespVO item : itemList) {
                            ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO innerItemDO = new ScpPurLimitSettingRedisDO.ActivityDO.InnerItemDO();
                            innerItemDO.setItemCode(item.getItemCode());
                            innerItemDO.setMasId(settingDO.getId());
                            innerItemDO.setMinNum(item.getMinNum());
                            innerItemDO.setItemType2(item.getItemType2());
                            if (itemRpcDTOMap.containsKey(item.getItemCode())) {
                                ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(item.getItemCode());
                                List<SpuImageDTO> spuAttchmentList = itmItemRpcDTO.getSpuImageList();
                                if (CollUtil.isNotEmpty(spuAttchmentList)) {
                                    Optional<SpuImageDTO> first = spuAttchmentList.stream().filter(SpuImageDTO::getMajor).findFirst();
                                    innerItemDO.setImgUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                                }
                                innerItemDO.setItemId(itmItemRpcDTO.getId());
                                innerItemDO.setItemName(itmItemRpcDTO.getItemName());
                            }
                            innerItemDOS.add(innerItemDO);
                        }
                        ScpPurLimitSettingRedisDO.ActivityDO activityDO1 = new ScpPurLimitSettingRedisDO.ActivityDO();
                        activityDO1.setActivityId(settingDO.getId());
                        activityDO1.setActivityCode(settingDO.getActivityCode());
                        activityDO1.setDocTypes(StrUtil.split(settingDO.getDocType(), ","));
                        activityDO1.setMaxNum(id.getMaxNum());
                        activityDO1.setInnerItemDOS(innerItemDOS);
                        activityDO1.setValidFrom(settingDO.getValidFrom());
                        activityDO1.setValidTo(settingDO.getValidTo());
                        activityDO1.setModifiedTime(settingDO.getModifyTime());
                        List<ScpPurLimitSettingRedisDO.ActivityDO> activityList = ScpPurLimitSettingRedisDO.getActivityList();
                        if (activityList == null) {
                            activityList = new ArrayList<>();
                        }
                        activityList.add(activityDO1);
                        ScpPurLimitSettingRedisDO.setActivityList(activityList);
                    }
                }

                if (CollUtil.isNotEmpty(ScpPurLimitSettingRedisDO.getActivityList())) {
                    String jsonString = JSON.toJSONString(ScpPurLimitSettingRedisDO, com.alibaba.fastjson.serializer.SerializerFeature.WriteDateUseDateFormat);
                    redisMap.put(storeKey, jsonString);
                } else {
                    redisClient.delete(storeKey);
                }
            });
            if (CollUtil.isNotEmpty(redisMap)) {
                redisClient.opsForValue().multiSet(redisMap);
            }
        }

        if (CollUtil.isNotEmpty(storeCodesNotIn)) {
            for (String storeCode : storeCodesNotIn) {
                String storeKey = ScpConstant.PUR_LIMIT_SETTING + storeCode;
                redisClient.delete(storeKey);
            }
        }
    }

    private void intersectSets(Set<Long> set1, List<Long> list2) {
        if (set1.isEmpty()) {
            set1.addAll(list2);
        }
        set1.retainAll(list2);
    }

}
