package com.elitesland.scp.application.service.strategy;

import com.elitesland.scp.application.enums.AllocRuleType;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderComputeVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

@Slf4j
@Component
@RequiredArgsConstructor
public class AvgEventService implements EventStrategy {

    /**
     * 这个方法用于将仓库的商品数量分配给不同的门店。
     * 分配的方式分为两步：
     * 1. 在第一步中，计算每个门店的平均数量，并将其分配给每个门店。每个门店的分配不能超过其需求。
     * 2. 在第二步中，如果仓库中还有剩余的数量，就将其一一分配给门店。分配会继续，直到仓库为空或所有门店的需求都得到满足。
     *
     * @param stores    门店列表。每个Store对象都有一个名字，一个需求量，和一个初始为零的分配量。
     * @param warehouse 仓库中的商品总量。
     */
    @Override
    public void handleInvStk(List<ScpDemandOrderComputeVO> stores, BigDecimal warehouse) {
        // 计算每个门店的平均数量
        BigDecimal average = warehouse.divide(BigDecimal.valueOf(stores.size()),0, RoundingMode.DOWN);

        // 将平均数量分配给每个门店
        for (ScpDemandOrderComputeVO store : stores) {
            BigDecimal allocation = average.min(store.getAllocationDeQuantity());
            store.setPlanQuantity(allocation);
            warehouse = warehouse.subtract(allocation);
        }

        // 如果仓库中还有剩余的数量，就将其一一分配给门店
        for (ScpDemandOrderComputeVO store : stores) {
            while (store.getPlanQuantity().compareTo(store.getAllocationDeQuantity()) < 0 && warehouse.compareTo(BigDecimal.ZERO) > 0) {
                store.setPlanQuantity(store.getPlanQuantity().add(BigDecimal.ONE));
                warehouse = warehouse.subtract(BigDecimal.ONE);
            }
        }
    }

    @Override
    public String getAllocRuleType() {
        return AllocRuleType.AVERAGE.getCode();
    }

}
