package com.elitesland.scp.rmi;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.support.provider.org.dto.OrgOuRpcDTO;
import com.elitesland.support.provider.org.dto.OrgOuRpcSimpleDTO;
import com.elitesland.support.provider.org.param.OrgOuRpcDtoParam;
import com.elitesland.support.provider.org.service.OrgOuRpcService;
import com.elitesland.support.provider.org.service.OrgRegionRpcService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/3/18 19:14
 * @Description:
 */
@Component
@Slf4j
public class RmiOrgOuService {

    @Autowired
    private OrgOuRpcService orgOuRpcService;
    @Autowired
    private OrgRegionRpcService orgRegionRpcService;

    public List<OrgOuRpcDTO> findOuDtoListByParam(OrgOuRpcDtoParam param) {
        try {
            List<OrgOuRpcDTO> ouDtoResult = orgOuRpcService.findOuDtoByParam(param);
            return CollUtil.isNotEmpty(ouDtoResult) ? ouDtoResult : new ArrayList<>(0);
        } catch (Exception e) {
            log.error("远程调用支撑域公司相关服务异常:{}", e.getMessage());
            throw new BusinessException("远程调用支撑域公司相关服务异常" + e, e);
        }
    }

    public List<OrgOuRpcDTO> findOuDtoList(List<Long> ouIds, List<String> ouCodes) {
        var orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        orgOuRpcDtoParam.setOuIds(ouIds);
        orgOuRpcDtoParam.setOuCodes(ouCodes);
        return this.findOuDtoListByParam(orgOuRpcDtoParam);
    }

    public List<OrgOuRpcDTO> findOuDtoListByName(List<String> ouNames) {
        var orgOuRpcDtoParam = new OrgOuRpcDtoParam();
        orgOuRpcDtoParam.setOuNames(ouNames);
        return this.findOuDtoListByParam(orgOuRpcDtoParam);
    }


    /**
     * 根据公司编码批量查询公司
     *
     * @param ouCodes 公司编码
     * @return 公司信息
     */
    public List<OrgOuRpcDTO> findOuDtoListByOuCodes(List<String> ouCodes) {
        log.info("调用support域:根据编码查询公司,时间：{},入参：{}", LocalDateTime.now(), ouCodes);
        try {
            if (CollectionUtils.isEmpty(ouCodes)) {
                return Collections.emptyList();
            }
            OrgOuRpcDtoParam orgOuRpcDtoParam = new OrgOuRpcDtoParam();
            orgOuRpcDtoParam.setOuCodes(ouCodes);
            List<OrgOuRpcDTO> orgOuRpcDTOList = this.findOuDtoListByParam(orgOuRpcDtoParam);
            log.info("调用support域:根据编码查询公司,时间：{},出参：{}", LocalDateTime.now(), orgOuRpcDTOList);
            return orgOuRpcDTOList;
        } catch (Exception e) {
            log.error("调用support域:根据编码查询公司,error：{}", e.getMessage());
            throw new BusinessException("调用support域:" + e.getMessage());
        }
    }

    public Map<Long, OrgOuRpcSimpleDTO> findOuDtoMap(List<Long> ouIds) {
        if (CollUtil.isEmpty(ouIds)) {
            return new HashMap<>();
        }
        List<OrgOuRpcSimpleDTO> simpleOuDto = orgOuRpcService.findSimpleOuDto(ouIds);
        if (CollUtil.isEmpty(simpleOuDto)) {
            return new HashMap<>();
        }
        return simpleOuDto.stream().collect(Collectors.toMap(OrgOuRpcSimpleDTO::getId, Function.identity()));
    }

    public Map<String, OrgOuRpcSimpleDTO> findOuDtoMapByOuCodes(List<String> ouCodes) {
        if (CollUtil.isEmpty(ouCodes)) {
            return new HashMap<>();
        }
        List<OrgOuRpcSimpleDTO> simpleOuDto = orgOuRpcService.findSimpleByOuCodes(ouCodes);
        if (CollUtil.isEmpty(simpleOuDto)) {
            return new HashMap<>();
        }
        return simpleOuDto.stream().collect(Collectors.toMap(OrgOuRpcSimpleDTO::getOuCode, Function.identity()));
    }

    public Map<String, String> findRegionMapByCodes(Set<String> codes) {
        if (CollUtil.isEmpty(codes)) {
            return new HashMap<>();
        }
        ApiResult<Map<String, String>> nameByCode = orgRegionRpcService.getNameByCode(codes);

        if (!nameByCode.isSuccess()) {
            throw new BusinessException( "查询区域名称接口异常:" + nameByCode.getMsg());
        }
        return nameByCode.getData();
    }


}
