package com.elitesland.scp.utils;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitesland.scp.application.facade.vo.param.app.AddressParamVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.cart.ScpStoreCartDO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.*;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

import static com.elitesland.scp.common.ScpConstant.CART_PREFIX;

/**
 * 购物车Redis工具类
 *
 * @Author: ryan.xu
 * @since 2023/3/7
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class CartRedisUtil {
    private final RedisTemplate redisClient;

    /**
     * 获取购物车操作对象
     *
     * @param storeCode 门店编码
     * @param cartType  采购车类型
     * @param userId    用户ID
     * @return
     */
    public BoundHashOperations<String, Object, Object> getStoreCartOps(String storeCode, String cartType, Long userId) {
        //先得到当前用户信息
        String cartKey = CART_PREFIX + userId + ":" + storeCode + ":" + cartType;
        //绑定指定的key操作Redis
        return redisClient.boundHashOps(cartKey);
    }

    /**
     * 获取购物车商品
     *
     * @param itemKey    商品唯一标识
     * @param storeCode 门店编码
     * @param cartType  采购车类型
     * @param userId    用户ID
     * @return
     */
    public ScpStoreCartDO getStoreCartItem(String itemKey, String storeCode, String cartType, Long userId) {
        //拿到要操作的购物车信息
        var cartOps = this.getStoreCartOps(storeCode, cartType, userId);

        String redisValue = (String) cartOps.get(SysUtils.generateRedisKey(ScpConstant.SKU, itemKey));

        return JSON.parseObject(redisValue, ScpStoreCartDO.class);
    }

    /**
     * 查询购物车商品
     *
     * @param storeCode 门店编码
     * @param cartType  采购车类型
     * @param userId    用户ID
     * @return
     */
    public List<ScpStoreCartDO> getStoreCartItems(String storeCode, String cartType, Long userId) {
        //获取购物车里面的所有商品
        var operations = this.getStoreCartOps(storeCode, cartType, userId);
        return Optional.ofNullable(operations.values()).orElse(new ArrayList<>()).stream().map(obj -> {
            String str = (String) obj;
            return JSON.parseObject(str, ScpStoreCartDO.class);
        }).collect(Collectors.toList());
    }

    /**
     * 获取购物车选中的商品
     *
     * @param storeCode 门店编码
     * @param cartType  采购车类型
     * @param userId    用户ID
     * @return
     */
    public List<ScpStoreCartDO> getCheckedShoppingCartItems(String storeCode, String cartType, Long userId) {
        //获取购物车里面的选中的商品
        var operations = this.getStoreCartOps(storeCode, cartType, userId);
        return Optional.ofNullable(operations.values()).orElse(new ArrayList<>()).stream().map(obj -> {
            String str = (String) obj;
            return JSON.parseObject(str, ScpStoreCartDO.class);
        }).filter(row -> row.getSelectedStatus() == 1).collect(Collectors.toList());
    }

    /**
     * scan 实现
     *
     * @param redisTemplate redisTemplate
     * @param pattern       表达式，如：abc*，找出所有以abc开始的键
     */
    public Set<String> scan(RedisTemplate<String, Object> redisTemplate, String pattern) {
        return redisTemplate.execute((RedisCallback<Set<String>>) connection -> {
            Set<String> keysTmp = new HashSet<>();
            try (Cursor<byte[]> cursor = connection.scan(ScanOptions.scanOptions()
                    .match(pattern)
                    .count(10000).build())) {
                while (cursor.hasNext()) {
                    keysTmp.add(new String(cursor.next(), StandardCharsets.UTF_8));
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            return keysTmp;
        });
    }

    /**
     * 设置购物车未选中
     *
     * @param storeCode
     * @param userId
     * @param cartType
     */
    public void updateUnSelectedStoreCartItems(String storeCode, Long userId, String cartType) {
        //拿到要操作的购物车信息
        var cartOps = this.getStoreCartOps(storeCode, cartType, userId);
        List<ScpStoreCartDO> cartDOS = Optional.ofNullable(cartOps.values()).orElse(new ArrayList<>()).stream().map(obj -> {
            String str = (String) obj;
            return JSON.parseObject(str, ScpStoreCartDO.class);
        }).collect(Collectors.toList());
        cartDOS.forEach(row -> row.setSelectedStatus(0));
        cartOps.putAll(cartDOS.stream().collect(Collectors.toMap(row -> SysUtils.generateRedisKey(ScpConstant.SKU, row.getCombineItemCode(), row.getItemCode()), row -> JSON.toJSONString(row))));
    }

    /**
     * 删除已选中的购物车商品
     *
     * @param storeCode 门店编码
     * @param userId    用户ID
     * @param cartType  采购车类型
     * @param itemKey    商品唯一标识
     */
    public void clearCheckedStoreCartItems(String storeCode, Long userId, String cartType, String itemKey) {
        var cartOps = this.getStoreCartOps(storeCode, cartType, userId);
        cartOps.delete(SysUtils.generateRedisKey(ScpConstant.SKU, itemKey));
    }

    /**
     *  查询购物车 缓存的收货地址
     */
    public AddressParamVO getAddr() {
        AddressParamVO addressParam = null;
        // 1. 获取当前用户信息
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        assert userDetails != null : "用户信息不能为空";
        // 2. 从缓存中获取地址信息
        String cacheKey = "DBO_ADDRESS_CACHE:" + userDetails.getUser().getUsername();
        // 方式1：如果是 JSON 字符串存储
        String addressJson = (String) RedisUtils.instance().get(cacheKey);
        if (StringUtils.isNotBlank(addressJson)) {
            addressParam = JSON.parseObject(addressJson, AddressParamVO.class);
        }
        return addressParam;
    }
}
