package com.elitesland.scp.application.service.template;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.excel.common.DataImport;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.inv.dto.invwh.InvWhRpcDTO;
import com.elitesland.inv.dto.invwh.InvWhRpcDtoParam;
import com.elitesland.scp.application.facade.vo.template.DemandOrderSaveParamVO;
import com.elitesland.scp.application.facade.vo.template.ScpDemandTemplateImportEntity;
import com.elitesland.scp.enums.DayOfWeekEnumHelper;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemRpcDtoParam;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.support.provider.org.service.OrgRegionRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/5/10 17:01
 * @Description:
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class DemandOrderTemplateImportService implements DataImport<ScpDemandTemplateImportEntity> {


    private static final String LINE = "\n";
    private static final String ERROR_TEMPLATE = "第 {0} 行: {1} 校验异常: {2}; ";

    private final DemandOrderTemplateService demandOrderTemplateService;
    private final TransactionTemplate transactionTemplate;
    private final RmiSysUDCService rmiSysUDCService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RmiItemService rmiItemService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final OrgRegionRpcService orgRegionRpcService;


    @Override
    public Set<Integer> sheetNoList() {
        return Collections.singleton(1);
    }

    @Override
    public Integer stepSize() {
        return 1000000;
    }

    @Override
    public String getTmplCode() {
        return "yst_scp_demand_template_import";
    }

    @Override
    public List<String> executeImport(List<ScpDemandTemplateImportEntity> dataList, int startRowIndex) {
        if (CollUtil.isEmpty(dataList)) {
            return Collections.emptyList();
        }
        List<String> errors = new ArrayList<>();

        try {
            List<String> errorMsg = checkAndSaveData(dataList, startRowIndex);
            if (errorMsg != null) {
                log.info("订货模板导入完成,错误信息如下:{}", JSONUtil.toJsonStr(errorMsg));
            } else {
                log.info("订货模板导入完成,未发生错误");
            }
            return errorMsg;
        } catch (Exception e) {
            for (ScpDemandTemplateImportEntity entity : dataList) {
                errors.add(e.getMessage());
            }
            return errors;
        }
    }

    private List<String> checkAndSaveData(List<ScpDemandTemplateImportEntity> dataList, int startRowIndex) {

        Map<Integer, String> errMsgMap = new LinkedHashMap<>();
        Integer index = 0;
        //初始化行号，和错误信息容器
        for (ScpDemandTemplateImportEntity entity : dataList) {
            entity.setRowNo(index);
            errMsgMap.put(index, null);
            index++;
        }

        log.info("startRowIndex的值:{}", startRowIndex);
        List<DemandOrderSaveParamVO> dataToSave = prepareData(dataList, errMsgMap);
        log.info("订货模板导入，准备好的数据:{},错误信息:{}", JSONUtil.toJsonStr(dataToSave), JSONUtil.toJsonStr(errMsgMap));
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        //一个一个保存
        for (DemandOrderSaveParamVO vo : dataToSave) {
            transactionTemplate.execute(
                    transactionStatus -> {
                        try {
                            demandOrderTemplateService.save(vo);
                        } catch (Exception e) {
                            log.error("订货模板导入报错：" + e);
                            // addErrMsg(errMsgMap, rowNos, e.getMessage());
                            // 回滚数据
                            transactionStatus.setRollbackOnly();
                        }
                        return null;
                    }
            );

        }
        return new ArrayList<>(errMsgMap.values());
    }


    List<DemandOrderSaveParamVO> prepareData(List<ScpDemandTemplateImportEntity> dataList,
                                             Map<Integer, String> errMsgMap) {

        List<String> storeCodes =
                dataList.stream().filter(e -> "门店".equals(e.getTypeName())).map(ScpDemandTemplateImportEntity::getWhStZoCode).distinct().collect(Collectors.toList());
        log.info("门店编码集合:{}", JSONUtil.toJsonStr(storeCodes));
        List<OrgStoreDetailRpcDTO> queryByStoreCodes = rmiOrgStoreRpcService.queryByStoreCodes(storeCodes);
        Map<String, String> storeTypeMap =
                queryByStoreCodes.stream().collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode,
                        OrgStoreDetailRpcDTO::getStoreType));
        Map<String, OrgStoreDetailRpcDTO> storeDataMap =
                queryByStoreCodes.stream().collect(Collectors.toMap(OrgStoreDetailRpcDTO::getStoreCode,
                        Function.identity()));
        log.info("门店编码类型字典:{},门店信息字典数据:{}", JSONUtil.toJsonStr(storeTypeMap), JSONUtil.toJsonStr(storeDataMap));


        List<String> whCodes =
                dataList.stream().filter(e -> "仓库".equals(e.getTypeName())).map(ScpDemandTemplateImportEntity::getWhStZoCode).distinct().collect(Collectors.toList());
        log.info("仓库编码集合:{}", JSONUtil.toJsonStr(storeCodes));
        Map<String, InvWhRpcDTO> whDataMap = new HashMap<>();
        InvWhRpcDtoParam param = new InvWhRpcDtoParam();
        param.setWhCodes(whCodes);
        List<InvWhRpcDTO> whData = rmiInvStkRpcService.findWhDTOByParam(param).getData();
        if (CollUtil.isNotEmpty(whData)) {
            whDataMap = whData.stream().collect(Collectors.toMap(InvWhRpcDTO::getWhCode,
                    Function.identity()));
        }
        log.info("仓库信息字典数据:{}", JSONUtil.toJsonStr(whDataMap));

        List<String> itemCodeList =
                dataList.stream().filter(e -> StrUtil.isNotBlank(e.getItemCode())).map(ScpDemandTemplateImportEntity::getItemCode).distinct().collect(Collectors.toList());
        log.info("商品编码集合:{}", JSONUtil.toJsonStr(itemCodeList));
        Map<String, ItmItemRpcDTO> itemInfoMap = ProcessItemInfo(itemCodeList);
        log.info("商品信息字典:{}", JSONUtil.toJsonStr(itemInfoMap));

        List<DemandOrderSaveParamVO> resultList = new ArrayList<>();
        Map<String, List<ScpDemandTemplateImportEntity>> detailMap =
                dataList.stream().collect(Collectors.groupingBy(ScpDemandTemplateImportEntity::getSeqNo));
        transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);

        for (Map.Entry<String, List<ScpDemandTemplateImportEntity>> entry : detailMap.entrySet()) {

            //标识一个模板数据是否存在任意错误
            AtomicBoolean isValid = new AtomicBoolean(true);

            DemandOrderSaveParamVO saveParamVO = new DemandOrderSaveParamVO();
            List<DemandOrderSaveParamVO.WhStZoObject> whStZoObjects = new ArrayList<>();
            List<DemandOrderSaveParamVO.ItemObject> itemObjects = new ArrayList<>();
            //每个序号组第一条作为模板表头数据获取的依据
            ScpDemandTemplateImportEntity headerEntity = entry.getValue().get(0);
            List<Integer> rowNos =
                    entry.getValue().stream().map(ScpDemandTemplateImportEntity::getRowNo).distinct().collect(Collectors.toList());
            Map<String, InvWhRpcDTO> finalWhDataMap = whDataMap;
            transactionTemplate.execute(
                    transactionStatus -> {
                        try {
                            headerProcess(headerEntity, saveParamVO);
                            Map<String, List<ScpDemandTemplateImportEntity>> demandTypeMap = entry.getValue().stream().collect(Collectors.groupingBy(ScpDemandTemplateImportEntity::getDemandTypeName));
                            if (demandTypeMap.size() > 1) {
                                throw new BusinessException("序号" + headerEntity.getSeqNo() + ",要货类型不一致");
                            }
                            for (ScpDemandTemplateImportEntity detail : entry.getValue()) {
                                Integer rowNo = detail.getRowNo();
                                try {
                                    detailProcess(detail, storeTypeMap, storeDataMap, finalWhDataMap, itemInfoMap,
                                            whStZoObjects, itemObjects);
                                } catch (Exception e) {
                                    // 处理明细行异常
                                    isValid.set(false);
                                    log.error("行号:{}发生错误:{}", rowNo, e.getMessage(), e);
                                    addErrMsg(errMsgMap, rowNo, e.getMessage());
                                    transactionStatus.setRollbackOnly();
                                }
                            }

                        } catch (Exception e) {
                            // 处理表单异常
                            isValid.set(false);
                            log.error("订单序号:{}发生错误:{}", entry.getKey(), e.getMessage(), e);
                            addErrMsg(errMsgMap, rowNos, e.getMessage());
                            transactionStatus.setRollbackOnly();
                        }
                        return null;
                    }
            );

            if (isValid.get()) {
                saveParamVO.setWhStZoObjects(whStZoObjects);
                saveParamVO.setItemObjects(itemObjects);
                resultList.add(saveParamVO);
            }
        }
        return resultList;
    }

    //模板表头数据校验
    private void headerProcess(ScpDemandTemplateImportEntity headerEntity, DemandOrderSaveParamVO vo) {
        String detailedTime = headerEntity.getDetailedTime();
        detailedTimeCheck(detailedTime);
        if (StrUtil.isBlank(headerEntity.getDemandTemName())) {
            throw new BusinessException("模板名称不允许为空");
        }
        if (headerEntity.getStartDate() == null || headerEntity.getEndDate() == null) {
            throw new BusinessException("模板起止日期不允许为空");
        }
        if (StrUtil.isBlank(headerEntity.getDemandTemStatusName())) {
            throw new BusinessException("模板状态字段不允许为空");
        }
        vo.setDemandTemName(headerEntity.getDemandTemName());
        vo.setStartDate(headerEntity.getStartDate());
        vo.setEndDate(headerEntity.getEndDate());
        if (StrUtil.isNotBlank(headerEntity.getDetailedTime())) {
            vo.setDetailedTime(DayOfWeekEnumHelper.splitAndListCodeName(headerEntity.getDetailedTime()));
        }
        vo.setDemandTemStatus(getStatusByName(headerEntity.getDemandTemStatusName()));
        vo.setRemark(headerEntity.getRemark());
        vo.setDocType(getDocTypeName(headerEntity.getDocTypeName()));
        vo.setType(getDemandTypeName(headerEntity.getDemandTypeName()));
    }

    //明细行数据校验
    private void detailProcess(ScpDemandTemplateImportEntity headerEntity, Map<String, String> storeTypeMap,
                               Map<String, OrgStoreDetailRpcDTO> storeDataMap,
                               Map<String, InvWhRpcDTO> whDataMap,
                               Map<String, ItmItemRpcDTO> itemInfoMap,
                               List<DemandOrderSaveParamVO.WhStZoObject> whStZoObjects,
                               List<DemandOrderSaveParamVO.ItemObject> itemObjects) {
        if (StrUtil.isNotBlank(headerEntity.getWhStZoCode()) && StrUtil.isNotBlank(headerEntity.getItemCode())) {
            throw new BusinessException("仓库/门店/区域信息和商品信息不允许同时存在");
        }

        if (StrUtil.isNotBlank(headerEntity.getWhStZoCode())) { //仓库/门店/区域
            DemandOrderSaveParamVO.WhStZoObject whStZoObject = new DemandOrderSaveParamVO.WhStZoObject();
            whStZoObject.setType(getTypeByName(headerEntity.getTypeName()));
            whStZoObject.setWhStZoCode(headerEntity.getWhStZoCode());
            whStZoObject.setStoreType(storeTypeMap.get(headerEntity.getWhStZoCode()));
            if ("门店".equals(headerEntity.getTypeName())) {
                processStore(whStZoObject, storeDataMap);
            } else if ("仓库".equals(headerEntity.getTypeName())) {
                processWH(whStZoObject, whDataMap);
            } else { //区域
                whStZoObject.setWhStZoName(getRegionName(headerEntity.getWhStZoCode()));
            }
            whStZoObjects.add(whStZoObject);
        } else if (StrUtil.isNotBlank(headerEntity.getItemCode())) {  //商品信息
            if (itemInfoMap.containsKey(headerEntity.getItemCode())) {
                if (headerEntity.getDemandQuantity() == null) {
                    throw new BusinessException("订货数量不能为空");
                }
                BigDecimal qty = SysUtils.processQtyScale(headerEntity.getDemandQuantity());
                ItmItemRpcDTO dto = itemInfoMap.get(headerEntity.getItemCode());
                DemandOrderSaveParamVO.ItemObject itemObject = new DemandOrderSaveParamVO.ItemObject();
                itemObject.setItemId(dto.getId());
                itemObject.setItemCode(dto.getItemCode());
                itemObject.setItemName(dto.getItemName());
                itemObject.setItemCateCode(dto.getItemCateCode());
                itemObject.setItemCateName(dto.getItemCateFullName());
                itemObject.setItemSpec(dto.getSpec());
                itemObject.setBasicUnitMeasure(dto.getUomName());
                itemObject.setWeightUom(dto.getWeightUnit());
                itemObject.setWeightUomName(dto.getWeightUnitName());
                itemObject.setDemandQuantity(qty);
                if (dto.getGrossWeight() != null) {
                    BigDecimal weight = SysUtils.processQtyScale(dto.getGrossWeight());
                    itemObject.setWeight(weight);
                    itemObject.setTotalWeight(SysUtils.processQtyScale(qty.multiply(weight)));
                }
                itemObjects.add(itemObject);
            } else {
                throw new BusinessException("查询不到商品信息");
            }
        }
    }


    private void processStore(DemandOrderSaveParamVO.WhStZoObject whStZoObject,
                              Map<String, OrgStoreDetailRpcDTO> storeDataMap) {
        String whStZoCode = whStZoObject.getWhStZoCode();
        if (storeDataMap.containsKey(whStZoCode)) {
            OrgStoreDetailRpcDTO dto = storeDataMap.get(whStZoCode);
            whStZoObject.setWhStZoName(dto.getStoreName());
            whStZoObject.setDetailedAddress(dto.getAddressRpcDTO().getDetailAddr());
            whStZoObject.setBelongOrgCode(dto.getPcode());
            whStZoObject.setBelongOrgName(dto.getPname());
            whStZoObject.setBelongOuCode(dto.getOuCode());
            whStZoObject.setBelongOuName(dto.getOuName());
            whStZoObject.setStoreType(dto.getStoreType2());
        } else {
            throw new BusinessException("根据门店编码,找不到门店信息");
        }
    }

    private void processWH(DemandOrderSaveParamVO.WhStZoObject whStZoObject, Map<String, InvWhRpcDTO> whDataMap) {
        String whStZoCode = whStZoObject.getWhStZoCode();
        if (whDataMap.containsKey(whStZoCode)) {
            InvWhRpcDTO dto = whDataMap.get(whStZoCode);
            whStZoObject.setWhStZoName(dto.getWhName());
            whStZoObject.setDetailedAddress(appendDetailAddress(dto));
            whStZoObject.setBelongOrgCode(dto.getBelongBuCode());
            whStZoObject.setBelongOrgName(dto.getBelongBuName());
            whStZoObject.setBelongOuCode(dto.getOuCode());
            whStZoObject.setBelongOuName(dto.getOuName());
        } else {
            throw new BusinessException("根据仓库编码,找不到门仓库息");
        }
    }

    private String appendDetailAddress(InvWhRpcDTO dto) {
        return dto.getCountryName() + dto.getProvinceName() + dto.getCityName() + dto.getCountyName() + dto.getDetailaddr();
    }

    //填充商品详情数据
    private Map<String, ItmItemRpcDTO> ProcessItemInfo(List<String> itemCodeList) {
        if (CollUtil.isEmpty(itemCodeList)) {
            return new HashMap<>();
        }
        itemCodeList = itemCodeList.stream().distinct().collect(Collectors.toList());
        ItmItemRpcDtoParam rpcParam = new ItmItemRpcDtoParam();
        rpcParam.setItemCodes(itemCodeList);
        List<ItmItemRpcDTO> itemRpcDtoByParam = rmiItemService.findItemRpcDtoByParam(rpcParam);
        Map<String, ItmItemRpcDTO> itemCodeEntityMap =
                itemRpcDtoByParam.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode
                        , Function.identity()));
        log.info("商品信息字典数据:{}", JSONUtil.toJsonStr(itemCodeEntityMap));
        return itemCodeEntityMap;
    }


    private String getRegionName(String regionCode) {
        ApiResult<String> result = orgRegionRpcService.getNameByCode(regionCode);
        if (StrUtil.isBlank(result.getData())) {
            throw new BusinessException("根据区域编码,找不到区域信息");
        }
        return result.getData();
    }

    private Integer getTypeByName(String typeName) {
        Map<String, String> templateOrderType = rmiSysUDCService.getCodeMap("yst-suplan", "TEMPLATE_ORDER_TYPE");
        Map<String, String> map =
                templateOrderType.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
        if (!map.containsKey(typeName)) {
            throw new BusinessException("字段类型错误");
        }
        return Integer.valueOf(map.get(typeName));
    }

    private String getDocTypeName(String docTypeName) {
        Map<String, String> docTypeMap = rmiSysUDCService.getCodeMap("yst-suplan", "OB_DOC_TYPE");
        Map<String, String> map =
                docTypeMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
        if (!map.containsKey(docTypeName)) {
            throw new BusinessException("字段类型错误");
        }
        return map.get(docTypeName);
    }

    private String getDemandTypeName(String demandTypeName) {
        Map<String, String> docTypeMap = rmiSysUDCService.getCodeMap("yst-suplan", "DEMAND_SET_TYPE");
        Map<String, String> map =
                docTypeMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
        if (!map.containsKey(demandTypeName)) {
            throw new BusinessException("字段类型错误");
        }
        return map.get(demandTypeName);
    }


    private Integer getStatusByName(String statusName) {
        if ("启用".equals(statusName)) {
            return 1;
        } else if ("禁用".equals(statusName)) {
            return 0;
        } else {
            throw new BusinessException("状态字段维护错误");
        }

    }

    private void detailedTimeCheck(String detailedTime) {
        if (StrUtil.isBlank(detailedTime)) {
            throw new BusinessException("模板详情日期不允许为空");
        }
        if (!DayOfWeekEnumHelper.validDesc(detailedTime)) {
            throw new BusinessException("格式错误，包含非法数据(参考格式:周一，周二，周三，周四，周五，周六，周日)");
        }

    }


    public static void addErrMsg(Map<Integer, String> errMsgMap, Integer rowNo, String msg) {
        addErrMsg(errMsgMap, Collections.singletonList(rowNo), msg);
    }

    public static void addErrMsg(Map<Integer, String> errMsgMap, List<Integer> rowNos, String msg) {
        for (Integer rowNo : rowNos) {
            if (errMsgMap.containsKey(rowNo) && errMsgMap.get(rowNo) != null) {
                String newMsg = errMsgMap.get(rowNo) + LINE + msg;
                errMsgMap.put(rowNo, newMsg);
            } else {
                errMsgMap.put(rowNo, msg);
            }
        }
    }
}
