package com.elitesland.scp.application.service.mrp;

import cn.hutool.core.collection.CollectionUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.mrp.ScpMrpDPageParam;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpMrpDExportRespVO;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpMrpDPlanRespVO;
import com.elitesland.scp.application.facade.vo.resp.mrp.ScpMrpDRespVO;
import com.elitesland.scp.application.facade.vo.save.mrp.ScpMrpDPlanSaveVO;
import com.elitesland.scp.domain.service.mrp.ScpMrpDDomainService;
import com.elitesland.scp.domain.service.mrp.ScpMrpDPlanDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.repo.stock.ScpPredictStStockRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpMrpDServiceImpl implements ScpMrpDService {

    private final ScpMrpDDomainService scpMrpDDomainService;

    private final ScpMrpDPlanDomainService scpMrpDPlanDomainService;

    private final ScpPredictStStockRepoProc scpPredictStStockRepoProc;

    @Override
    public PagingVO<ScpMrpDRespVO> page(ScpMrpDPageParam param) {
        PagingVO<ScpMrpDRespVO> pagingVO = scpMrpDDomainService.searchPage(param);
        return pagingVO;
    }

    @Override
    public List<ScpMrpDRespVO> findByMasId(Long masId) {
        return scpMrpDDomainService.findByMasId(masId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void savePlan(List<ScpMrpDPlanSaveVO> scpMrpDPlanSaveVOList) {
        if (CollectionUtil.isEmpty(scpMrpDPlanSaveVOList)) {
            throw new BusinessException("采购计划不能为空");
        }
        Long masId = scpMrpDPlanSaveVOList.get(0).getMasId();
        ScpMrpDRespVO scpMrpDRespVO = scpMrpDDomainService.findById(masId);
        if (scpMrpDRespVO == null) {
            throw new BusinessException("MRP计划明细不存在");
        }
        BigDecimal sumQty = scpMrpDPlanSaveVOList.stream().map(ScpMrpDPlanSaveVO::getQty).reduce(BigDecimal.ZERO, BigDecimal::add);
        if (sumQty.compareTo(scpMrpDRespVO.getNetDemand()) != 0) {
            throw new BusinessException("采购计划数量之和需要等于净需求");
        }
        scpMrpDPlanSaveVOList.forEach(plan -> {
            plan.setMrpId(scpMrpDRespVO.getMasId());
            plan.setPushStatus(ScpUdcEnum.MRP_PUSH_STATUS_NOT.getValueCode());
        });

        List<ScpMrpDPlanRespVO> scpMrpDPlanRespVOList = scpMrpDPlanDomainService.findByMasIds(List.of(masId));
        if (CollectionUtil.isNotEmpty(scpMrpDPlanRespVOList)) {
            List<Long> deleteIds = scpMrpDPlanRespVOList.stream().filter(o -> scpMrpDPlanSaveVOList.stream().noneMatch(n -> o.getId().equals(n.getId()))).map(ScpMrpDPlanRespVO::getId).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(deleteIds)) {
                scpMrpDPlanDomainService.deleteByIds(deleteIds);
            }
        }

        scpMrpDDomainService.savePlan(scpMrpDPlanSaveVOList);
    }

    @Override
    public PagingVO<ScpMrpDExportRespVO> queryExport(ScpMrpDPageParam param) {
        return scpMrpDDomainService.queryExport(param);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        List<ScpMrpDRespVO> scpMrpDRespVOList = scpMrpDDomainService.findByIds(ids);
        if (CollectionUtil.isEmpty(scpMrpDRespVOList)) {
            throw new BusinessException("ids不能为空");
        }
        List<ScpMrpDPlanRespVO> scpMrpDPlanRespVOList = scpMrpDPlanDomainService.findByMasIds(ids);
        if (scpMrpDPlanRespVOList.stream().anyMatch(plan -> StringUtils.isNotBlank(plan.getPoNo()))) {
            throw new BusinessException("MRP计划明细已经推送采购订单，无法删除");
        }

        List<Long> predIds = scpMrpDRespVOList.stream().map(ScpMrpDRespVO::getRelateDocDid).collect(Collectors.toList());

        // 删除明细和采购计划
        scpMrpDDomainService.deleteByIds(ids);
        scpMrpDPlanDomainService.deleteByMasIds(ids);

        scpPredictStStockRepoProc.updateMrpFlag(predIds, Boolean.FALSE);
    }
}
