package com.elitesland.scp.domain.service.order;

import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandSetPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandSetParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandSetPageRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandSetSaveVO;
import com.elitesland.scp.domain.convert.order.ScpDemandSetConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandSetDO;
import com.elitesland.scp.infr.dto.order.ScpDemandSetDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandSetRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandSetRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpDemandSetDomainServiceImpl implements ScpDemandSetDomainService {

    private final ScpDemandSetRepo scpDemandSetRepo;
    private final ScpDemandSetRepoProc scpDemandSetRepoProc;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;

    @Override
    @SysCodeProc
    public PagingVO<ScpDemandSetPageRespVO> queryDemandSetList(ScpDemandSetPageParamVO paramVO) {
        long count = scpDemandSetRepoProc.countDemandSet(paramVO);
        if (count > 0) {
            var demandSet = scpDemandSetRepoProc.queryDemandSet(paramVO);
            List<Long> demandIds = demandSet.stream().map(ScpDemandSetPageRespVO::getId).collect(Collectors.toList());
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
            // 查询订货集下的订货明细单数量
            Map<Long, Long> itemQtyMap = scpDemandOrderDRepoProc.getDemandSumItem(demandIds);
            demandSet.forEach(row -> {
                int qty = itemQtyMap.get(row.getId()) == null ? 0 : itemQtyMap.get(row.getId()).intValue();
                row.setDemandItemQty(qty);
                if (row.getExpireTime() != null) {
                    row.setExpireTimeStr(row.getExpireTime().toLocalTime().format(formatter));
                }
            });
            return new PagingVO<>(count, demandSet);
        }
        return new PagingVO<>();
    }

    @Override
    @SysCodeProc
    public Optional<ScpDemandSetDTO> findDemandSetById(Long id) {
        return scpDemandSetRepo.findById(id).map(ScpDemandSetConvert.INSTANCE::doToDto);
    }

    @Override
    public Long saveDemandSet(ScpDemandSetSaveVO saveVO) {
        if (saveVO.getId() == null) {
            ScpDemandSetDO intentDO = ScpDemandSetConvert.INSTANCE.saveVoToDO(saveVO);
            return scpDemandSetRepo.save(intentDO).getId();
        } else {
            Optional<ScpDemandSetDO> option = scpDemandSetRepo.findById(saveVO.getId());
            if (option.isEmpty()) {
                throw new BusinessException("订货集ID：" + saveVO.getId() + "不存在");
            }
            ScpDemandSetConvert.INSTANCE.copySaveParamToDo(saveVO, option.get());
            Long masId = scpDemandSetRepo.save(option.get()).getId();
            return masId;
        }
    }

    @Override
    public void enableDemandSet(List<Long> ids, Boolean enable) {
        scpDemandSetRepoProc.enableDemandSet(ids, enable);
    }

    @Override
    public void deleteByIds(List<Long> ids) {
        scpDemandSetRepoProc.deleteByIds(ids);
    }

    @Override
    public List<ScpDemandSetDTO> findDemandSetByIds(List<Long> ids) {
        return scpDemandSetRepo.findByIdIn(ids).stream().map(ScpDemandSetConvert.INSTANCE::doToDto).collect(Collectors.toList());
    }

    @Override
    public List<ScpDemandSetDTO> findDemandSetByParam(ScpDemandSetParamVO paramVO) {
        return scpDemandSetRepoProc.findDemandSetByParam(paramVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDemandSetMsgByIds(List<Long> ids, String msg) {
        scpDemandSetRepoProc.updateDemandSetMsgByIds(ids, msg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateDemandSetPushMsgById(List<Long> ids, String msg) {
        scpDemandSetRepoProc.updateDemandSetPushMsgById(ids, msg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void completePushById(Long id) {
        scpDemandSetRepoProc.completePushById(id);
    }

    @Override
    public Boolean getDemandSetStatusById(Long id) {
        return scpDemandSetRepoProc.getDemandSetStatusById(id);
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpDemandSetPageRespVO> commonPage(ScpDemandSetPageParamVO paramVO) {
        long count = scpDemandSetRepoProc.countDemandSet(paramVO);
        if (count > 0) {
            var demandSet = scpDemandSetRepoProc.queryDemandSet(paramVO);
            return new PagingVO<>(count, demandSet);
        }
        return new PagingVO<>();
    }
}
