package com.elitesland.scp.pay.service;

import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitesland.fin.param.recorder.WeChatPayRecOrderRpcParam;
import com.elitesland.inv.dto.invTro.InvTroOnlinePayParamDTO;
import com.elitesland.pur.dto.po.PurPoOnlinePayParamDTO;
import com.elitesland.scp.application.facade.vo.param.order.ScpOrderPayMqParam;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.wqf.ScpWqfEntAccountApplyDO;
import com.elitesland.scp.domain.entity.wqf.ScpWqfEntAccountDO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.enums.UdcEnum;
import com.elitesland.scp.infr.dto.wqf.ScpWqfEntAccountApplyDDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepoProc;
import com.elitesland.scp.infr.repo.wqf.ScpWqfEntAccountApplyDRepoProc;
import com.elitesland.scp.infr.repo.wqf.ScpWqfEntAccountApplyRepo;
import com.elitesland.scp.infr.repo.wqf.ScpWqfEntAccountApplyRepoProc;
import com.elitesland.scp.infr.repo.wqf.ScpWqfEntAccountRepoProc;
import com.elitesland.scp.pay.config.EntpayProperties;
import com.elitesland.scp.pay.vo.AccountBindNotifyModel;
import com.elitesland.scp.pay.vo.PayNoficeParamVO;
import com.elitesland.scp.rmi.RmiFinRpcService;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiPurRpcService;
import com.elitesland.scp.utils.MessageDelyQueueService;
import com.elitesland.scp.utils.SysUtils;
import com.tenpay.business.entpay.mse.sdk.api.AccountBatchBind;
import com.tenpay.business.entpay.mse.sdk.api.Payment;
import com.tenpay.business.entpay.mse.sdk.exception.EntpayException;
import com.tenpay.business.entpay.mse.sdk.model.BindAccountDetail;
import com.tenpay.business.entpay.mse.sdk.model.PaymentNotifyModel;
import com.tenpay.business.entpay.mse.sdk.model.RefundNotifyModel;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;

import javax.persistence.EntityManager;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class NotifyVerifyServiceImpl implements NotifyVerifyService {

    private final RedisWrapper redisWrapper;
    private final RedisTemplate<Object, Object> redisTemplate;
    private final EntityManager entityManager;
    private final RmiPurRpcService rmiPurRpcService;
    private final RmiFinRpcService rmiFinRpcService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final MessageDelyQueueService messageDelyQueueService;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;
    private final MessageQueueTemplate messageQueueTemplate;

    private final ScpWqfEntAccountApplyRepo scpWqfEntAccountApplyRepo;

    private final ScpWqfEntAccountApplyRepoProc scpWqfEntAccountApplyRepoProc;

    private final ScpWqfEntAccountApplyDRepoProc scpWqfEntAccountApplyRepoDProc;
    private final TenantDataIsolateProvider tenantDataIsolateProvider;
    private final TransactionTemplate transactionTemplate;

    private final ScpWqfEntAccountRepoProc scpWqfEntAccountRepoProc;
    private final RedissonClient redissonClient;


    @Override
    public void processNotify(PaymentNotifyModel notifyData) throws EntpayException {
        Payment eventContent = notifyData.getEventContent();
        String outPaymentId = eventContent.getOutPaymentId();
        Payment payment = Payment.retrieveByOutPaymentId(outPaymentId);
        if (!"SUCCEEDED".equals(payment.getPayStatus())) {
            log.info("支付状态【{}】，不处理，outPaymentId:{}", payment.getPayStatus(), outPaymentId);
            return;
        }
        var entpayProperties = Optional.ofNullable(redisWrapper.apply(() -> redisTemplate.opsForValue().get(ScpConstant.ENT_PAY_PROPERTIES), null))
                .map(Object::toString)
                .map(o -> JSONUtil.toBean(o, EntpayProperties.class))
                .orElseThrow(() -> new BusinessException("请联系管理员配置微企付信息"));
        //设置租户信息
        tenantDataIsolateProvider.byTenantDirectly(() -> {
            TenantContextHolder.setCurrentTenant(TenantClient.getSessionTenant());
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
            if ("PO".equals(outPaymentId.substring(0, 2))) {
                PurPoOnlinePayParamDTO purPoOnlinePayParamDTO = new PurPoOnlinePayParamDTO();
                purPoOnlinePayParamDTO.setOnlinePayFlag(Boolean.TRUE);
                purPoOnlinePayParamDTO.setDocNo(outPaymentId);
                purPoOnlinePayParamDTO.setOnlinePayTime(LocalDateTime.parse(eventContent.getPayTime(), formatter));
                rmiPurRpcService.updateOnlinePay(purPoOnlinePayParamDTO);
            } else {
                InvTroOnlinePayParamDTO invTroOnlinePayParamDTO = new InvTroOnlinePayParamDTO();
                invTroOnlinePayParamDTO.setDocNo(outPaymentId);
                invTroOnlinePayParamDTO.setOnlinePayTime(LocalDateTime.parse(eventContent.getPayTime(), formatter));
                invTroOnlinePayParamDTO.setOnlinePayFlag(Boolean.TRUE);
                rmiInvStkRpcService.updateOnlinePay(invTroOnlinePayParamDTO);
            }
            String key = ScpConstant.SCP_ONLINE_PAY + outPaymentId;
            //删除缓存
            if (redisTemplate.delete(key)) {
                //删除延时队列
                messageDelyQueueService.remove(key);
                //更新订货单状态为支付中
                transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
                transactionTemplate.execute(transactionStatus -> {
                    try {
                        scpDemandOrderDRepoProc.updatePayStatusBySrcDocId(outPaymentId, ScpUdcEnum.PAY_STATUS_PAYING.getValueCode());
                        return "ok";
                    } catch (Exception e) {
                        log.error("更新要货单出错:{}", e.getMessage(), e);
                        // 回滚
                        transactionStatus.setRollbackOnly();
                        return "ok";
                    }
                });
                //写入mq队列,进行收货确认
                ScpOrderPayMqParam orderSubmitMqParam = new ScpOrderPayMqParam();
                orderSubmitMqParam.setSrcDocNo(outPaymentId);
                orderSubmitMqParam.setFlowNo(eventContent.getPaymentId());
                orderSubmitMqParam.setBusinessKey(ScpOrderPayMqParam.SCP_ORDER_PAY_CHANNEL);
                orderSubmitMqParam.setPayDateTime(LocalDateTime.parse(eventContent.getPayTime(), formatter));
                BigDecimal amt = SysUtils.processAmtScale(new BigDecimal(eventContent.getAmount()).divide(new BigDecimal(100)));
                orderSubmitMqParam.setRealRecAmt(amt);
                messageQueueTemplate.publishMessageSync("yst-suplan", ScpOrderPayMqParam.SCP_ORDER_PAY_CHANNEL, orderSubmitMqParam);
            } else {
                log.error("不能重复调用支付回调");
            }
            return "";
        }, entpayProperties.getTenantCode());

    }

    @Override
    public void processRefundNotify(RefundNotifyModel notifyData) {
        //todo 调用接口处理退款回调
    }

    @Override
    public void devopsPayNotify(PayNoficeParamVO paramVO) throws EntpayException {
        String outPaymentId = paramVO.getOutPaymentId();
        Payment payment = Payment.retrieveByOutPaymentId(outPaymentId);
        if (!"SUCCEEDED".equals(payment.getPayStatus())) {
            log.info("支付状态不是成功，不处理，outPaymentId:{}", outPaymentId);
            return;
        }
        var entpayProperties = Optional.ofNullable(redisWrapper.apply(() -> redisTemplate.opsForValue().get(ScpConstant.ENT_PAY_PROPERTIES), null))
                .map(Object::toString)
                .map(o -> JSONUtil.toBean(o, EntpayProperties.class))
                .orElseThrow(() -> new BusinessException("请联系管理员配置微企付信息"));
        //设置租户信息
        final SysTenantDTO sysTenantDTO = TenantClient.getTenant(entpayProperties.getTenantCode());
        TenantContextHolder.setCurrentTenant(sysTenantDTO);
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        if ("PO".equals(outPaymentId.substring(0, 2))) {
            PurPoOnlinePayParamDTO purPoOnlinePayParamDTO = new PurPoOnlinePayParamDTO();
            purPoOnlinePayParamDTO.setOnlinePayFlag(Boolean.TRUE);
            purPoOnlinePayParamDTO.setDocNo(outPaymentId);
            purPoOnlinePayParamDTO.setOnlinePayTime(LocalDateTime.parse(paramVO.getPayTime(), formatter));
            rmiPurRpcService.updateOnlinePay(purPoOnlinePayParamDTO);
        } else {
            InvTroOnlinePayParamDTO invTroOnlinePayParamDTO = new InvTroOnlinePayParamDTO();
            invTroOnlinePayParamDTO.setDocNo(outPaymentId);
            invTroOnlinePayParamDTO.setOnlinePayTime(LocalDateTime.parse(paramVO.getPayTime(), formatter));
            invTroOnlinePayParamDTO.setOnlinePayFlag(Boolean.TRUE);
            rmiInvStkRpcService.updateOnlinePay(invTroOnlinePayParamDTO);
        }
        redisTemplate.delete(ScpConstant.SCP_ONLINE_PAY + outPaymentId);
        //删除延时队列
        messageDelyQueueService.remove(ScpConstant.SCP_ONLINE_PAY + outPaymentId);
        //调用财务中心，扣款操作
        WeChatPayRecOrderRpcParam payParam = new WeChatPayRecOrderRpcParam();
        payParam.setSourceNo(outPaymentId);
        payParam.setFlowNo(paramVO.getPaymentId());
        payParam.setRealRecAmt(new BigDecimal(paramVO.getAmount() / 100));
        payParam.setPayDateTime(LocalDateTime.parse(paramVO.getPayTime(), formatter));
        rmiFinRpcService.wqfPay(payParam);
    }

    @Override
    public void processAccountBindNotify(AccountBindNotifyModel notifyData) throws EntpayException {
        AccountBatchBind eventContent = notifyData.getEventContent();
        String outRequestNo = eventContent.getOutRequestNo();
        // 设置租户信息
        var entpayProperties = Optional.ofNullable(redisWrapper.apply(() -> redisTemplate.opsForValue().get(ScpConstant.ENT_PAY_PROPERTIES), null))
                .map(Object::toString)
                .map(o -> JSONUtil.toBean(o, EntpayProperties.class))
                .orElseThrow(() -> new BusinessException("请联系管理员配置微企付信息"));

        //查询绑定结果
        AccountBatchBind accountBatchBind = AccountBatchBind.retrieveByOutRequestNo(outRequestNo);
        String lockKey = String.format("accountBind:lock:%s", outRequestNo);
        RLock lock = redissonClient.getLock(lockKey);
        try {
            if (lock.tryLock(300L, 180L, TimeUnit.SECONDS)) {
                //租户信息
                tenantDataIsolateProvider.byTenantDirectly(() -> {
                    TenantContextHolder.setCurrentTenant(TenantClient.getSessionTenant());
                    List<ScpWqfEntAccountApplyDO> accountApplyDOS = scpWqfEntAccountApplyRepoProc.findAllByDocNo(outRequestNo);
                    if (CollectionUtils.isEmpty(accountApplyDOS)) {
                        log.error("批量绑定账户失败，未查询到中台关联申请单号，不处理，outRequestNo:{}", outRequestNo);
                        return "";
                    }
                    List<ScpWqfEntAccountDO> wqfEntAccountDOS = scpWqfEntAccountRepoProc.findByApplyNo(outRequestNo);
                    transactionTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
                    transactionTemplate.execute(transactionStatus -> {
                        try {
                            ScpWqfEntAccountApplyDO accountApplyDO = accountApplyDOS.get(0);
                            if (!"SUCCESS".equals(accountBatchBind.getStatus())) {
                                log.info("批量绑定账户失败【{}】，不处理，outRequestNo:{}", accountBatchBind.getStatus(), outRequestNo);
                                accountApplyDO.setDocNo(accountBatchBind.getOutRequestNo());
                                accountApplyDO.setApplyStatus(UdcEnum.WQF_APPLY_STATUS_FAIL.getValueCode());
                                accountApplyDO.setErrorMsg(accountBatchBind.getFailReason());
                                scpWqfEntAccountApplyRepo.save(accountApplyDO);
                                return "";
                            }
                            List<String> accounts = accountBatchBind.getAccounts()
                                    .stream().map(BindAccountDetail::getBankAccountNumber)
                                    .filter(Objects::nonNull)
                                    .distinct()
                                    .collect(Collectors.toList());
                            if(CollectionUtils.isNotEmpty(wqfEntAccountDOS)){
                                List<String> exists = wqfEntAccountDOS.stream().map(ScpWqfEntAccountDO::getBankAccount).collect(Collectors.toList());
                                accounts = accounts.stream().filter(d -> !exists.contains(d)).collect(Collectors.toList());
                                if(CollectionUtils.isEmpty(accounts)){
                                    log.info("批量绑定账户失败【{}】，不处理，outRequestNo:{},已存在中台账户", accountBatchBind.getStatus(), outRequestNo);
                                    return "";
                                }
                            }
                            List<ScpWqfEntAccountApplyDDTO> bindNotifyDetail = scpWqfEntAccountApplyRepoDProc.findAccountBindNotifyDetail(accountBatchBind.getOutRequestNo(), accounts);
                            if (CollectionUtils.isEmpty(bindNotifyDetail)) {
                                log.info("批量绑定账户失败【{}】，不处理，outRequestNo:{},未查询到符合账户明细", accountBatchBind.getStatus(), outRequestNo);
                                accountApplyDO.setDocNo(accountBatchBind.getOutRequestNo());
                                accountApplyDO.setApplyStatus(UdcEnum.WQF_APPLY_STATUS_FAIL.getValueCode());
                                accountApplyDO.setErrorMsg(String.format("批量绑定账户失败【%s】，不处理，outRequestNo:%s,未查询到符合中台账户明细", accountBatchBind.getStatus(), outRequestNo));
                                scpWqfEntAccountApplyRepo.save(accountApplyDO);
                                return "";
                            }
                            accountApplyDO.setDocNo(accountBatchBind.getOutRequestNo());
                            accountApplyDO.setApplyStatus(UdcEnum.WQF_APPLY_STATUS_SUCESS.getValueCode());
                            Map<String, ScpWqfEntAccountApplyDDTO> accountApplyDMap = bindNotifyDetail.stream()
                                    .collect(Collectors.toMap(ScpWqfEntAccountApplyDDTO::getBankAccount, v -> v, (v1, v2) -> v1));
                            //生成门店账户列表
                            List<ScpWqfEntAccountDO> saveList = new ArrayList<>();
                            for (BindAccountDetail accountDetail : accountBatchBind.getAccounts()) {
                                ScpWqfEntAccountApplyDDTO accountApplyDDTO = accountApplyDMap.get(accountDetail.getBankAccountNumber());
                                if (accountApplyDDTO == null) {
                                    continue;
                                }
                                ScpWqfEntAccountDO scpWqfEntAccountDO = new ScpWqfEntAccountDO();
                                scpWqfEntAccountDO.setApplyNo(outRequestNo);
                                scpWqfEntAccountDO.setStoreId(accountApplyDDTO.getStoreId());
                                scpWqfEntAccountDO.setStoreCode(accountApplyDDTO.getStoreCode());
                                scpWqfEntAccountDO.setOuId(accountApplyDDTO.getOuId());
                                scpWqfEntAccountDO.setOuCode(accountApplyDDTO.getOuCode());
                                scpWqfEntAccountDO.setEntId(accountApplyDO.getEntId());
                                scpWqfEntAccountDO.setEntAcctId(accountDetail.getEntAcctId());
                                scpWqfEntAccountDO.setAccountType(accountApplyDDTO.getAccountType());
                                scpWqfEntAccountDO.setBankAccount(accountDetail.getBankAccountNumber());
                                scpWqfEntAccountDO.setOpenBank(accountApplyDDTO.getOpenBank());
                                scpWqfEntAccountDO.setBranchName(accountApplyDDTO.getBranchName());
                                scpWqfEntAccountDO.setEntAcctId(accountDetail.getEntAcctId());
                                scpWqfEntAccountDO.setBankBranchCode(accountApplyDDTO.getBankBranchCode());
                                scpWqfEntAccountDO.setBranchName(accountApplyDDTO.getBranchName());
                                scpWqfEntAccountDO.setStatus(Boolean.TRUE);
                                saveList.add(scpWqfEntAccountDO);
                            }
                            log.info("批量绑定账户成功，outRequestNo:{},saveList:{}", outRequestNo, saveList);
                            if (CollectionUtils.isNotEmpty(saveList)) {
                                batchInsert(saveList);
                            }
                            accountApplyDO.setErrorMsg(null);
                            scpWqfEntAccountApplyRepo.save(accountApplyDO);
                            return "ok";
                        } catch (Exception e) {
                            log.error("更新要货单出错:{}", e.getMessage(), e);
                            // 回滚
                            transactionStatus.setRollbackOnly();
                            return "ok";
                        }
                    });
                    return "";
                }, entpayProperties.getTenantCode());
            }
        } catch (BusinessException | InterruptedException e) {
            log.error("accountBind lock error:", e);
            throw new BusinessException(e.getMessage());
        } finally {
            if (lock.isLocked() && lock.isHeldByCurrentThread()) {
                lock.unlock();
                log.info("accountBind unlock success,lockKey:{}", lockKey);
            }
        }
    }

    /**
     * 批量保存
     *
     * @param dataList
     */
    public void batchInsert(List<ScpWqfEntAccountDO> dataList) {
        int index = 0;
        int batchSize = 100;
        for (ScpWqfEntAccountDO data : dataList) {
            entityManager.persist(data);
            if (batchSize > 1) {
                // 开启批量
                index++;
                if (index % batchSize == 0) {
                    entityManager.flush();
                    entityManager.clear();
                }
            }
        }
        if (!dataList.isEmpty()) {
            entityManager.flush();
        }
    }


}
