package com.elitesland.scp.utils;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.elitesland.scp.utils.excel.ExportColumnParam;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

/**
 * @Auther: Mark
 * @Date: 2024/3/18 17:33
 * @Description:
 */
public class ExcelExportUtil {
    private final Logger logger = LoggerFactory.getLogger(ExcelExportUtil.class);

    private final ExcelWriterBuilder builder;
    private List<String> fields;
    /**
     * 每次最多写入500条
     */
    private Integer size = 500;
    /**
     * 日期格式，默认为yyyy-MM-dd HH:mm:ss
     */
    private DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private ExcelExportUtil(OutputStream outputStream) {
        builder = EasyExcelFactory.write(outputStream)
                .autoCloseStream(true)
        ;
    }

    /**
     * 导出工具实例化
     *
     * @param outputStream 文件输出流
     * @return 工具实例
     * @throws IOException 异常
     */
    public static ExcelExportUtil instance(OutputStream outputStream) throws IOException {
        return new ExcelExportUtil(outputStream);
    }

    /**
     * 数据的标题
     *
     * @param columnParamList 导出的列
     * @return 异常
     */
    public ExcelExportUtil fields(List<ExportColumnParam> columnParamList) {
        if (columnParamList != null && !columnParamList.isEmpty()) {
            List<List<String>> titles = new ArrayList<>(64);
            List<String> includeFields = new ArrayList<>(64);
            for (ExportColumnParam p : columnParamList) {
                if (StrUtil.isBlank(p.getTitle()) || StrUtil.isBlank(p.getField())) {
                    continue;
                }
                titles.add(List.of(p.getTitle()));
                includeFields.add(p.getField());
            }
            this.builder.head(titles);
            this.fields = includeFields;
        }
        return this;
    }

    /**
     * 分页获取时每次获取的数据量
     *
     * @param size 数据量
     * @return 工具实例
     */
    public ExcelExportUtil batchSize(Integer size) {
        if (size == null || size < 1) {
            throw new IllegalArgumentException("size参数不合法");
        }

        this.size = size;
        return this;
    }

    /**
     * 日期格式
     *
     * @param dateFormat 日期格式
     * @return 工具实例
     */
    public ExcelExportUtil dateFormat(String dateFormat) {
        this.dateFormatter = DateTimeFormatter.ofPattern(dateFormat);
        return this;
    }

    /**
     * 开始写数据
     *
     * @param sheetName sheet名称
     * @param data      数据
     */
    public void write(String sheetName, List<?> data) {
        long start = System.currentTimeMillis();
        Assert.notEmpty(fields, "未知导出的数据列");

        List<List<String>> dataConvert = convertAllData(data);

        this.builder.sheet(sheetName).doWrite(dataConvert);

        logger.info("excel导出完毕，用时：{}ms", System.currentTimeMillis() - start);
    }

    /**
     * 开始写数据
     *
     * @param sheetName    sheet名称
     * @param dataProducer 数据生产者
     */
    public void write(String sheetName, BiFunction<Integer, Integer, List<?>> dataProducer) {
        long start = System.currentTimeMillis();
        Assert.notEmpty(fields, "未知导出的数据列");

        ExcelWriter excelWriter = this.builder.build();

        try {
            writeDataByPage(excelWriter, sheetName, dataProducer);
        } catch (Exception e) {
            logger.error("导出数据异常：", e);
        } finally {
            excelWriter.finish();
        }

        logger.info("excel导出完毕，用时：{}ms", System.currentTimeMillis() - start);
    }

    private void writeDataByPage(ExcelWriter excelWriter, String sheetName, BiFunction<Integer, Integer, List<?>> dataProducer) {
        // 获取导出数量限制
        Integer limit = getLimitNumOfExport();
        logger.info("导出数量限制：{}", limit);
        if (limit != null && limit < 1) {
            this.builder.sheet(sheetName).doWrite(Collections.emptyList());
            return;
        }

        ExcelWriterSheetBuilder writerSheetBuilder = new ExcelWriterSheetBuilder(excelWriter);
        writerSheetBuilder.sheetName(sheetName);
        WriteSheet writeSheet = writerSheetBuilder.build();

        int rows = 0;
        int page = 1;
        List<?> dataList = null;
        ObjectMapper objectMapper = getObjectMapper();
        // 分页导出数据
        while (true) {
            dataList = dataProducer.apply(page++, size);
            if (CollectionUtils.isEmpty(dataList)) {
                if (page == 2) {
                    excelWriter.write(Collections.emptyList(), writeSheet);
                }
                break;
            }

            rows += dataList.size();
            if (limit != null && rows > limit) {
                // 超出限制
                dataList = CollUtil.sub(dataList, 0, limit - rows + size);
            }

            // 转换数据格式后写入excel
            List<List<String>> convertData = objectMapper.convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
            }).stream().map(data ->
                    fields.stream().map(field -> obtainValue(data, field)
                    ).collect(Collectors.toList())
            ).collect(Collectors.toList());
            excelWriter.write(convertData, writeSheet);

            if (dataList.size() < size) {
                // 少于获取的数量，则说明已无数据
                break;
            }
        }
    }

    /**
     * 转换数据格式
     *
     * @param dataList 数据
     * @return 转换后的数据
     */
    private List<List<String>> convertAllData(List<?> dataList) {
        if (CollectionUtils.isEmpty(dataList)) {
            return Collections.emptyList();
        }

        // 获取导出数量限制
        Integer limit = getLimitNumOfExport();
        logger.info("导出数量限制：{}", limit);
        if (limit != null) {
            if (limit < 1) {
                return Collections.emptyList();
            }
            dataList = CollUtil.sub(dataList, 0, limit);
        }

        return convertData(dataList);
    }

    /**
     * 转换数据格式
     *
     * @param dataList 数据
     * @return 转换后的数据
     */
    private List<List<String>> convertData(List<?> dataList) {
        return getObjectMapper().convertValue(dataList, new TypeReference<List<Map<String, Object>>>() {
        }).stream().map(data ->
                fields.stream().map(field -> obtainValue(data, field)
                ).collect(Collectors.toList())
        ).collect(Collectors.toList());
    }

    /**
     * 获取导出数量的限制
     *
     * @return 导出的数量限制
     */
    private Integer getLimitNumOfExport() {
        return 10000;
    }

    private String obtainValue(Map<String, Object> data, String field) {
        Object value = data.get(field);
        if (value == null) {
            return "";
        }
        if (value instanceof LocalDateTime) {
            return dateFormatter.format((LocalDateTime) value);
        } else if (value instanceof Double) {
            return NumberUtil.decimalFormatMoney((Double) value);
        } else if (value instanceof Float) {
            return NumberUtil.decimalFormatMoney((Float) value);
        }
        return value.toString();
    }

    private ObjectMapper getObjectMapper() {
        return new Jackson2ObjectMapperBuilder()
                .serializerByType(LocalDateTime.class, new LocalDateTimeSerializer(dateFormatter))
                .deserializerByType(LocalDateTime.class, new LocalDateTimeDeserializer(dateFormatter))
                .build();
    }
}
