package com.elitesland.scp.application.web.app;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.app.AddressParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderAddItemParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderAppCountParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderDPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderPageParamVO;
import com.elitesland.scp.application.facade.vo.resp.app.*;
import com.elitesland.scp.application.facade.vo.resp.order.*;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDSaveVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderSubmitVO;
import com.elitesland.scp.application.service.app.AppDemandOrderService;
import com.elitesland.scp.application.service.order.ScpDemandOrderDService;
import com.elitesland.scp.application.service.order.ScpDemandOrderService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.utils.CartRedisUtil;
import com.elitesland.scp.utils.ValidatedList;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.stream.Collectors;


@Slf4j
@Api(tags = "移动端-订货订单接口")
@RestController
@RequestMapping(value = "/app/order", produces = MediaType.APPLICATION_JSON_VALUE)
@AllArgsConstructor
public class AppDemandOrderController {

    private final AppDemandOrderService appDemandOrderService;
    private final ScpDemandOrderDService scpDemandOrderDService;
    private final ScpDemandOrderService scpDemandOrderService;
    private final CartRedisUtil cartRedisUtil;

    @PostMapping("/submit")
    @ApiOperation("订货订单 - 提交")
//    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE,
//            operationDescription = "订货订单 - 提交", allowRepeatRequest = false,
//            repeatIntervalMillis = 5000, businessParamValueKey = "#p0.demandWhStCode")
    public ApiResult<Long> submit(@RequestBody @Validated ScpDemandOrderSubmitVO saveVO) {
        log.info("[APP-SCP-DEMAND-ORDER] submit  saveVO ={}", JSONUtil.toJsonStr(saveVO));
        // 判断是否为 门店装修订单
        if(ScpUdcEnum.BUSINESS_TYPE_50.getValueCode().equals(saveVO.getBusinessType())){
            AddressParamVO addr = cartRedisUtil.getAddr();
            saveVO.setRecvCountry(addr.getCountry());
            saveVO.setRecvProvince(addr.getProvince());
            saveVO.setRecvCity(addr.getCity());
            saveVO.setRecvCounty(addr.getCounty());
            saveVO.setRecvDetailaddr(addr.getDetailAddr());
            saveVO.setRecvContactName(addr.getContPerson());
            saveVO.setRecvContactTel(addr.getTel());
            String storeCodes = addr.getOrgStoreList().stream()
                    .map(OrgStoreDboVO::getStoreCode)
                    .collect(Collectors.joining(","));
            String storeNames = addr.getOrgStoreList().stream()
                    .map(OrgStoreDboVO::getStoreName)
                    .collect(Collectors.joining(","));

            saveVO.setRenovationStore(storeCodes);
            saveVO.setRenovationStoreName(storeNames);
        }
        return ApiResult.ok(appDemandOrderService.submit(saveVO));
    }

    @PostMapping("/reOrder/{id}")
    @ApiOperation("订货订单 - 重新下单")
    public ApiResult<Long> reOrder(@PathVariable Long id) {
        log.info("[APP-SCP-DEMAND-ORDER] reOrder  id ={}", id);
        return ApiResult.ok(appDemandOrderService.reOrder(id));
    }

    @ApiOperation(value = "订货订单 - 分页查询")
    @PostMapping(value = "/page")
    public ApiResult<PagingVO<ScpDemandOrderPageRespVO>> queryDemandOrderList(@RequestBody ScpDemandOrderPageParamVO paramVO) {
        log.info("[APP-SCP-DEMAND-ORDER] queryDemandOrderList  paramVO ={}", JSONUtil.toJsonStr(paramVO));

        if("PEEDING".equals(paramVO.getDocStatus()) || (CollectionUtil.isNotEmpty(paramVO.getDocStatusList())&&paramVO.getDocStatusList().contains("PEEDING"))){
            paramVO.setDocStatus(ScpUdcEnum.DEO_STATUS_SUBMIT.getValueCode());
        }
        return ApiResult.ok(scpDemandOrderService.queryDemandOrderHeader(paramVO));
    }

    @ApiOperation("订货订单明细 - 查询订货订单明细(小程序端)")
    @PostMapping("/detail/page")
    public ApiResult<PagingVO<ScpDemandOrderDRespVO>> page(@RequestBody ScpDemandOrderDPageParamVO paramVO) {
        log.info("[APP-SCP-DEMAND-ORDER] page  paramVO ={}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderDService.pageDemandOrderDByMasId(paramVO));
    }

    @ApiOperation("订货订单明细 - 删除订货订单明细(小程序端)")
    @PostMapping("/detail/remove/{id}")
    public ApiResult<PagingVO<ScpDemandOrderDRespVO>> page(@PathVariable Long id) {
        scpDemandOrderDService.remove(id);
        return ApiResult.ok();
    }

    @PostMapping("/detail/save")
    @ApiOperation("订货订单明细 - 修改")
    public ApiResult<Long> saveDemandOrderD(@RequestBody @Validated ValidatedList<ScpDemandOrderDSaveVO> saveVOS) {
        log.info("[APP-SCP-DEMAND-ORDER] saveDemandOrderD  saveVOS ={}", JSONUtil.toJsonStr(saveVOS));
        appDemandOrderService.batchSaveDemandOrderD(saveVOS);
        return ApiResult.ok();
    }

    @ApiOperation("订货订单 - 订货订单详情信息")
    @GetMapping("/findIdOne/{id}")
    public ApiResult<ScpDemandOrderRespVO> findIdOne(@PathVariable Long id) {
        return ApiResult.ok(appDemandOrderService.findDemandOrderById(id));
    }

    @PostMapping("/addItem")
    @ApiOperation("订货订单 - 追加商品")
    public ApiResult<Long> addItem(@RequestParam("masId") Long masId, @RequestBody @Validated ValidatedList<ScpDemandOrderAddItemParamVO> saveVOS) {
        log.info("[APP-SCP-DEMAND-ORDER] addItem  masId ={}, saveVOS ={}", masId, JSONUtil.toJsonStr(saveVOS));
        scpDemandOrderDService.addItem(masId, saveVOS);
        return ApiResult.ok();
    }

    @ApiOperation(value = "订货订单 - 统计数据")
    @PostMapping(value = "/countStatus")
    public ApiResult<List<AppDemandOrderCountRespVO>> countStatus(@RequestBody ScpDemandOrderAppCountParamVO paramVO) {
        log.info("[APP-SCP-DEMAND-ORDER] countStatus  param ={}", JSONUtil.toJsonStr(paramVO));
        return ApiResult.ok(scpDemandOrderService.countStatus(paramVO));
    }

    @PostMapping("/addCart")
    @ApiOperation("订货订单 - 再来一单")
    public ApiResult<Long> addCart(@RequestParam("id") Long id) {
        appDemandOrderService.addCart(id);
        return ApiResult.ok();
    }

    @ApiOperation("订货订单 - 通过关联id支付订单详情")
    @GetMapping("/payInfo/{srcDocId}")
    public ApiResult<AppPayOrderRespVO> findPayOrderBySrcDocId(@PathVariable Long srcDocId) {
        return ApiResult.ok(appDemandOrderService.findPayOrderBySrcDocId(srcDocId));
    }

    @ApiOperation("订货订单 - 通过id获取支付订单详情")
    @GetMapping("/payInfo/id/{id}")
    public ApiResult<AppPayOrderRespVO> findPayOrderById(@PathVariable Long id) {
        return ApiResult.ok(appDemandOrderService.findPayOrderById(id));
    }

    @ApiOperation("订货订单 - 支付订单详情明细")
    @GetMapping("/item/payInfo/{srcDocId}")
    public ApiResult<AppPayOrderItemRespVO> findPayOrderItemBySrcDocId(@PathVariable Long srcDocId) {
        return ApiResult.ok(appDemandOrderService.findPayOrderItemBySrcDocId(srcDocId));
    }

    @ApiOperation("订货订单 - 支付订单详情明细")
    @GetMapping("/item/payInfo/id/{id}")
    public ApiResult<AppPayOrderItemRespVO> findPayOrderItemById(@PathVariable Long id) {
        return ApiResult.ok(appDemandOrderService.findPayOrderItemById(id));
    }

    @ApiOperation("订货订单 - 订货单详情支付订单信息")
    @GetMapping("/relate/{id}")
    public ApiResult<List<AppRelateOrderRespVO>> findRelateOrderById(@PathVariable Long id) {
        return ApiResult.ok(appDemandOrderService.findRelateOrderById(id));
    }

    @ApiOperation("订货订单 - 支付明细信息")
    @GetMapping("/pay/item/{docCls}/{id}")
    public ApiResult<ScpOrderPayRespVO> payInfo(@PathVariable("docCls") String docCls, @PathVariable("id") Long id) {
        return ApiResult.ok(appDemandOrderService.payInfo(docCls, id));
    }

    @ApiOperation(value = "订货订单(待推送) - 关闭")
    @PostMapping(value = "/close/{id}")
    public ApiResult<List<ScpDemandOrderPayInfoRespVO>> close(@PathVariable Long id) {
        log.info("[APP-SCP-DEMAND-ORDER] close  param ={}", id);
        scpDemandOrderService.close(id, "APP");
        return ApiResult.ok();
    }

    @ApiOperation("订货订单 - 订货单根据编码查询id")
    @GetMapping("/findIdByCode/{docCode}")
    public ApiResult<Long> findIdByCode(@PathVariable("docCode") String docCode) {
        return ApiResult.ok(appDemandOrderService.findIdByCode(docCode));
    }
}
