package com.elitesland.scp.application.service.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.TimeInterval;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.el.coordinator.core.common.utils.UUIDUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.inv.dto.invstk.InvWhItemTotalStkRpcDTO;
import com.elitesland.inv.dto.invstk.InvWhItemTotalStkRpcParam;
import com.elitesland.inv.enums.InvDeter2TypeEnum;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderAddItemParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderDPageParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderDelParamVO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderItemParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderDMgmtRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderDPriceRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderDRespVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDMgmtSaveVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDPriceSaveVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDSaveVO;
import com.elitesland.scp.application.facade.vo.serviceconfig.ScpServiceConfigRespVO;
import com.elitesland.scp.application.service.serviceconfig.ScpServiceConfigService;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderDConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDDO;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDO;
import com.elitesland.scp.domain.service.alloc.ScpAllocSettingStoreDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDomainService;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepo;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.item.dto.ItmItemAttachmentRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemScpBaseRpcDTO;
import com.elitesland.support.provider.item.dto.ItmUomRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemScpBaseRpcParam;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.dto.OrgStoreDetailRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBaseRpcParam;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpDemandOrderDServiceImpl implements ScpDemandOrderDService {

    private final ScpDemandOrderService scpDemandOrderService;
    private final ScpDemandOrderDomainService scpDemandOrderDomainService;
    private final RmiSysUDCService rmiSysUDCService;
    private final ScpDemandOrderDDomainService scpDemandOrderDDomainService;
    private final RmiItemService rmiItemService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final RedisTemplate redisClient;
    private final ScpAllocSettingStoreDomainService scpAllocSettingStoreDomainService;
    private final ItmItemRpcService itmItemRpcService;
    private final ScpServiceConfigService scpServiceConfigService;
    private final OrgStoreRpcService orgStoreRpcService;
    private final ScpDemandOrderRepo scpDemandOrderRepo;
    private final ScpDemandOrderDRepo scpDemandOrderDRepo;

    @Override
    @SysCodeProc
    public List<ScpDemandOrderDMgmtRespVO> findDemandOrderDByMasId(Long masId) {
        List<ScpDemandOrderDDTO> itemList = scpDemandOrderDDomainService.findDemandOrderDByMasId(masId);
        if (CollUtil.isEmpty(itemList)) {
            return Collections.emptyList();
        }
        Map<String, String> docClsMap = rmiSysUDCService.getCodeMap("yst-supp", "DOC_CLS");
        Map<String, String> deoStatus = rmiSysUDCService.getCodeMap("yst-suplan", "DEO_STATUS");
        Map<String, String> payStatusMap = rmiSysUDCService.getCodeMap(ScpUdcEnum.PAY_STATUS_NO_PAY.getModel(), ScpUdcEnum.PAY_STATUS_WAIT_PAY.getCode());
        Map<String, String> suppTypeMap = rmiSysUDCService.getCodeMap(ScpUdcEnum.SUPPLY_DEMAND_TYPE_WH.getModel(), ScpUdcEnum.SUPPLY_DEMAND_TYPE_WH.getCode());
        List<ScpDemandOrderDMgmtRespVO> mgmtRespVOS = new ArrayList<>();
        Map<Long, List<ScpDemandOrderDDTO>> soureIdMap = itemList.stream().collect(Collectors.groupingBy(ScpDemandOrderDDTO::getSourceId));
        for (Map.Entry<Long, List<ScpDemandOrderDDTO>> entry : soureIdMap.entrySet()) {
            var supplyItems = entry.getValue().stream().map(row -> {
                ScpDemandOrderDMgmtRespVO.SupplyItem supplyItem = ScpDemandOrderDConvert.INSTANCE.dtoToSupplyItem(row);
                supplyItem.setSupplyTypeName(suppTypeMap.get(supplyItem.getSupplyType()));
                if (payStatusMap.containsKey(row.getPayStatus())) {
                    supplyItem.setPayStatusName(payStatusMap.get(row.getPayStatus()));
                }
                if (docClsMap.containsKey(row.getSrcDocCls())) {
                    supplyItem.setSrcDocClsName(docClsMap.get(row.getSrcDocCls()));
                }
                if (ScpConstant.PUR_DELIVERY_TYPES.contains(row.getDeliveryType())) {
                    supplyItem.setSuppId(row.getSuppWhId());
                    supplyItem.setSuppCode(row.getSuppWhCode());
                    supplyItem.setSuppName(row.getSuppWhName());
                }

                // 项目费用不展示在结算总金额和折前总金额中
                if (row.getProjectFeeFlag() != null && row.getProjectFeeFlag()) {
                    supplyItem.setSettlementAmt(BigDecimal.ZERO);
                    supplyItem.setPlanAmt(BigDecimal.ZERO);
                }

                // 设置状态
                Optional.ofNullable(deoStatus.get(row.getStatus())).ifPresent(supplyItem::setStatusName);
                return supplyItem;
            }).collect(Collectors.toList());
            ScpDemandOrderDMgmtRespVO mgmtRespVO = ScpDemandOrderDConvert.INSTANCE.dtoToMgmtRespVO(entry.getValue().get(0));
            mgmtRespVO.setItemList(supplyItems);
            mgmtRespVOS.add(mgmtRespVO);
        }

        return mgmtRespVOS.stream().sorted(Comparator.comparing(ScpDemandOrderDMgmtRespVO::getSrcDocNo, Comparator.nullsLast(Comparator.naturalOrder())).thenComparing(ScpDemandOrderDMgmtRespVO::getCreateTime)).collect(Collectors.toList());
    }

    @Override
    public void batchSaveDemandOrderDMgmt(Long masId, String storeCode, List<ScpDemandOrderDMgmtSaveVO> saveVOS, String source) {
        OrgStoreBaseRpcParam baseRpcParam = new OrgStoreBaseRpcParam();
        baseRpcParam.setStoreCodeList(List.of(storeCode));
        OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = orgStoreRpcService.findSimpleStoreByParam(baseRpcParam).computeData().get(0);
        // 获取门店服务配置
        Map<String, ScpServiceConfigRespVO> configRespVOMap =
                scpServiceConfigService.checkServiceConfig(orgStoreBaseRpcDTO.getStoreCode(), orgStoreBaseRpcDTO.getRegion(), orgStoreBaseRpcDTO.getStoreLevel());

        Integer amtPlace = SysUtils.getAmtPlace();
        Integer pricePlace = SysUtils.getAppPricePlace();

        List<ScpDemandOrderDSaveVO> batchSaveVOS = saveVOS.stream().filter(row -> !row.getFreightLineFlag())
                .flatMap(entity -> {
                    String uuid = UUIDUtil.getUUID();
                    return entity.getItemList().stream()
                            .map(listItem -> {
                                ScpDemandOrderDSaveVO scpDemandOrderDSaveVO = ScpDemandOrderDConvert.INSTANCE.mgmtSaveVoToSaveVo(listItem);
                                scpDemandOrderDSaveVO.setMasId(entity.getMasId());
                                scpDemandOrderDSaveVO.setSourceId(entity.getSourceId());
                                scpDemandOrderDSaveVO.setSpuItemCode(entity.getSpuItemCode());
                                scpDemandOrderDSaveVO.setSpuItemName(entity.getSpuItemName());
                                scpDemandOrderDSaveVO.setItemId(entity.getItemId());
                                scpDemandOrderDSaveVO.setItemCode(entity.getItemCode());
                                scpDemandOrderDSaveVO.setItemName(entity.getItemName());
                                scpDemandOrderDSaveVO.setCombineItemCode(entity.getCombineItemCode());
                                scpDemandOrderDSaveVO.setCombineItemName(entity.getCombineItemName());
                                scpDemandOrderDSaveVO.setDemandQuantity(entity.getDemandQuantity());
                                scpDemandOrderDSaveVO.setCurrency(entity.getCurrency());
                                scpDemandOrderDSaveVO.setRemark(entity.getRemark());
                                scpDemandOrderDSaveVO.setPreRootUuid(uuid);
                                scpDemandOrderDSaveVO.setItemType(entity.getItemType());
                                scpDemandOrderDSaveVO.setDeliveryType(entity.getDeliveryType());
                                scpDemandOrderDSaveVO.setActivityId(entity.getActivityId());
                                scpDemandOrderDSaveVO.setActivityCode(entity.getActivityCode());
                                scpDemandOrderDSaveVO.setMinNum(entity.getMinNum());

                                if (ScpConstant.WH_DELIVERY_TYPES.contains(entity.getDeliveryType())) {
                                    scpDemandOrderDSaveVO.setSuppWhId(listItem.getSuppWhId());
                                    scpDemandOrderDSaveVO.setSuppWhCode(listItem.getSuppWhCode());
                                    scpDemandOrderDSaveVO.setSuppWhName(listItem.getSuppWhName());
                                } else {
                                    if (listItem.getSuppId() == null || listItem.getSuppCode() == null) {
                                        throw new BusinessException("直送商品【" + entity.getItemName() + "】未获取到供应商");
                                    }
                                    scpDemandOrderDSaveVO.setSuppWhId(listItem.getSuppId());
                                    scpDemandOrderDSaveVO.setSuppWhCode(listItem.getSuppCode());
                                    scpDemandOrderDSaveVO.setSuppWhName(listItem.getSuppName());
                                }

                                Long id = (listItem.getId() == null || listItem.getId() < 0) ? null : listItem.getId();
                                scpDemandOrderDSaveVO.setId(id);

                                if (listItem.getSettlementPrice() != null && listItem.getSettlementPrice().compareTo(listItem.getPrice()) != 0) {
                                    if (listItem.getPrice().subtract(listItem.getSettlementPrice()).compareTo(BigDecimal.ONE) < 0) {
                                        throw new BusinessException("折后单价应低于原单价，且折扣差额需大于1元");
                                    }

                                    // 通过结算价格，计算结算货款价格（货款单价 / 1 + 三个比例）
                                    // 计算改价的商品，重新计算项目费用 settlementPrice
                                    if (configRespVOMap.get("TECH") != null) {
                                        scpDemandOrderDSaveVO.setTefFeeOuCode(configRespVOMap.get("TECH").getOuCode());
                                    }
                                    if (configRespVOMap.get("MARKET") != null) {
                                        scpDemandOrderDSaveVO.setMefFeeOuCode(configRespVOMap.get("MARKET").getOuCode());
                                    }
                                    if (configRespVOMap.get("OPERATE") != null) {
                                        scpDemandOrderDSaveVO.setOefFeeOuCode(configRespVOMap.get("OPERATE").getOuCode());
                                    }
                                    //服务费
                                    BigDecimal tefRatio = configRespVOMap.get("TECH") == null ? BigDecimal.ZERO
                                            : configRespVOMap.get("TECH").getFeePercentage().multiply(new BigDecimal("0.01"));
                                    BigDecimal mefRatio = configRespVOMap.get("MARKET") == null ? BigDecimal.ZERO :
                                            configRespVOMap.get("MARKET").getFeePercentage().multiply(new BigDecimal("0.01"));
                                    BigDecimal oefRatio = configRespVOMap.get("OPERATE") == null ? BigDecimal.ZERO :
                                            configRespVOMap.get("OPERATE").getFeePercentage().multiply(new BigDecimal("0.01"));

                                    // 结算货款单价
                                    BigDecimal settlementSalePrice = listItem.getSettlementPrice().divide(BigDecimal.ONE.add(tefRatio).add(mefRatio).add(oefRatio), pricePlace, RoundingMode.HALF_UP);
                                    listItem.setSettlementSalePrice(settlementSalePrice);

                                    // 重新计算
                                    scpDemandOrderDSaveVO.setTefPrice(settlementSalePrice.multiply(tefRatio).setScale(pricePlace, RoundingMode.HALF_UP));
                                    scpDemandOrderDSaveVO.setMefPrice(settlementSalePrice.multiply(mefRatio).setScale(pricePlace, RoundingMode.HALF_UP));
                                    // 通过倒减法，计算运营费
                                    scpDemandOrderDSaveVO.setOefPrice(listItem.getSettlementPrice().subtract(settlementSalePrice).subtract(scpDemandOrderDSaveVO.getTefPrice()).subtract(scpDemandOrderDSaveVO.getMefPrice()));
                                    scpDemandOrderDSaveVO.setSettlementPrice(listItem.getSettlementPrice());
                                } else {
                                    scpDemandOrderDSaveVO.setSettlementPrice(listItem.getPrice());
                                }
                                if (listItem.getSettlementSalePrice() == null) {
                                    scpDemandOrderDSaveVO.setSettlementSalePrice(listItem.getSalePrice());
                                } else {
                                    scpDemandOrderDSaveVO.setSettlementSalePrice(listItem.getSettlementSalePrice());
                                }
                                if (scpDemandOrderDSaveVO.getSettlementPrice() != null) {
                                    scpDemandOrderDSaveVO.setSettlementAmt(SysUtils.processAmtScale(scpDemandOrderDSaveVO.getDemandQuantity()
                                            .multiply(scpDemandOrderDSaveVO.getSettlementPrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
                                }
                                if (scpDemandOrderDSaveVO.getSettlementSalePrice() != null) {
                                    scpDemandOrderDSaveVO.setSettlementSaleAmt(SysUtils.processAmtScale(scpDemandOrderDSaveVO.getDemandQuantity()
                                            .multiply(scpDemandOrderDSaveVO.getSettlementSalePrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
                                }

                                if (listItem.getPrice() != null) {
                                    scpDemandOrderDSaveVO.setDemandAmt(SysUtils.processAmtScale(scpDemandOrderDSaveVO.getDemandQuantity()
                                            .multiply(listItem.getPrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
                                    scpDemandOrderDSaveVO.setSaleAmt(SysUtils.processAmtScale(scpDemandOrderDSaveVO.getDemandQuantity()
                                            .multiply(listItem.getSalePrice()).setScale(amtPlace, RoundingMode.HALF_UP)));
                                }

                                scpDemandOrderDSaveVO.setPlanQuantity(entity.getDemandQuantity());
                                scpDemandOrderDSaveVO.setDemandQuantity(entity.getDemandQuantity());
                                scpDemandOrderDSaveVO.setProjectFeeFlag(Boolean.FALSE);
                                Optional.ofNullable(scpDemandOrderDSaveVO.getTefPrice())
                                        .ifPresent(price -> scpDemandOrderDSaveVO.setTechFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity()).setScale(amtPlace, RoundingMode.HALF_UP)));
                                Optional.ofNullable(scpDemandOrderDSaveVO.getMefPrice())
                                        .ifPresent(price -> scpDemandOrderDSaveVO.setMarketingFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity()).setScale(amtPlace, RoundingMode.HALF_UP)));
                                Optional.ofNullable(scpDemandOrderDSaveVO.getOefPrice())
                                        .ifPresent(price -> scpDemandOrderDSaveVO.setOperationFee(price.multiply(scpDemandOrderDSaveVO.getDemandQuantity()).setScale(amtPlace, RoundingMode.HALF_UP)));
                                return scpDemandOrderDSaveVO;
                            });
                }).collect(Collectors.toList());
        log.info("保存需求订单明细开始:{}", JSONUtil.toJsonStr(batchSaveVOS));
        this.batchSaveDemandOrderD(masId, batchSaveVOS, source);
//        //更新门店已强配次数
//        List<Long> activitys = batchSaveVOS.stream().map(ScpDemandOrderDSaveVO::getActivityId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
//        if (CollUtil.isNotEmpty(activitys)) {
//            Optional<ScpDemandOrderDTO> demandOrderById = scpDemandOrderDomainService.findDemandOrderById(masId);
//            ScpDemandOrderDTO scpDemandOrderDTO = demandOrderById.get();
//
//            ScpAllocSettingStoreParamVO scpAllocSettingStoreParamVO = new ScpAllocSettingStoreParamVO();
//            scpAllocSettingStoreParamVO.setStoreCode(scpDemandOrderDTO.getDemandWhStCode());
//            scpAllocSettingStoreParamVO.setMasIds(activitys);
//            List<ScpAllocSettingStoreRespVO> settingStoreList = scpAllocSettingStoreDomainService.findByParam(scpAllocSettingStoreParamVO);
//            for (ScpAllocSettingStoreRespVO storeRespVO : settingStoreList) {
//                if ((storeRespVO.getAllocNum() + 1) >= storeRespVO.getMaxNum()) {
//                    redisClient.delete(ScpConstant.ALLOC_SETTING + scpDemandOrderDTO.getDemandWhStCode());
//                }
//            }
//            scpAllocSettingStoreDomainService.updateAllocNumByParam(activitys, scpDemandOrderDTO.getDemandWhStCode());
//        }
    }

    @Override
    public ScpDemandOrderDPriceRespVO updatePrice(ScpDemandOrderDPriceSaveVO scpDemandOrderDPriceSaveVO) {
        // 通过结算价格，计算结算货款价格（货款单价 / 1 + 三个比例）
        OrgStoreBaseRpcParam baseRpcParam = new OrgStoreBaseRpcParam();
        baseRpcParam.setStoreCodeList(List.of(scpDemandOrderDPriceSaveVO.getStoreCode()));
        OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = orgStoreRpcService.findSimpleStoreByParam(baseRpcParam).computeData().get(0);
        Map<String, ScpServiceConfigRespVO> configRespVOMap = scpServiceConfigService.findServiceConfigGroupByStore(orgStoreBaseRpcDTO.getStoreCode(),
                orgStoreBaseRpcDTO.getRegion(), orgStoreBaseRpcDTO.getStoreLevel());

        ScpDemandOrderDPriceRespVO result = new ScpDemandOrderDPriceRespVO();
        // 计算改价的商品，重新计算项目费用 settlementPrice
        if (configRespVOMap != null) {
            //服务费
            BigDecimal tefRatio = configRespVOMap.get("TECH") == null ? BigDecimal.ZERO
                    : configRespVOMap.get("TECH").getFeePercentage().multiply(new BigDecimal("0.01"));
            BigDecimal mefRatio = configRespVOMap.get("MARKET") == null ? BigDecimal.ZERO :
                    configRespVOMap.get("MARKET").getFeePercentage().multiply(new BigDecimal("0.01"));
            BigDecimal oefRatio = configRespVOMap.get("OPERATE") == null ? BigDecimal.ZERO :
                    configRespVOMap.get("OPERATE").getFeePercentage().multiply(new BigDecimal("0.01"));

            // 结算货款单价
            BigDecimal settlementSalePrice = scpDemandOrderDPriceSaveVO.getSettlementPrice()
                    .divide(BigDecimal.ONE.add(tefRatio).add(mefRatio).add(oefRatio), 2, RoundingMode.HALF_UP);
            result.setSettlementSalePrice(settlementSalePrice);
        } else {
            result.setSettlementSalePrice(scpDemandOrderDPriceSaveVO.getSettlementPrice());
        }
        return result;
    }

    @Override
    @SysCodeProc
    public PagingVO<ScpDemandOrderDRespVO> pageDemandOrderDByMasId(ScpDemandOrderDPageParamVO paramVO) {
        Page<ScpDemandOrderDDO> page = scpDemandOrderDDomainService.pageDemandOrderDByMasId(paramVO);
        List<ScpDemandOrderDDO> records = page.getContent();
        List<ScpDemandOrderDRespVO> scpDemandOrderDRespVOS = ScpDemandOrderDConvert.INSTANCE.doToRespVO(records);
        ItmItemScpBaseRpcParam itmItemRpcDtoParam = new ItmItemScpBaseRpcParam();
        itmItemRpcDtoParam.setItemIds(scpDemandOrderDRespVOS.stream().map(ScpDemandOrderDRespVO::getItemId).collect(Collectors.toList()));
        Map<Long, ItmItemScpBaseRpcDTO> itemMap = rmiItemService.findItemScpBaseRpcDtoByParam(itmItemRpcDtoParam).stream().collect(Collectors.toMap(ItmItemScpBaseRpcDTO::getId,
                Function.identity()));

        Map<String, String> deoStatus = rmiSysUDCService.getCodeMap("yst-suplan", "DEO_STATUS");

        // 总金额
        AtomicReference<BigDecimal> amt = new AtomicReference<>(BigDecimal.ZERO);

        // 货款总金额
        AtomicReference<BigDecimal> saleAmt = new AtomicReference<>(BigDecimal.ZERO);
        scpDemandOrderDRespVOS.forEach(row -> {
            if (row.getProjectFeeFlag() != null && row.getProjectFeeFlag()) {
                saleAmt.updateAndGet(v -> v.add(row.getSettlementSaleAmt()));
            } else {
                if (!itemMap.containsKey(row.getItemId())) {
                    return;
                }
                if (row.getSettlementAmt() != null) {
                    amt.updateAndGet(v -> v.add(row.getSettlementAmt()));
                }

                if (row.getSettlementSaleAmt() != null) {
                    saleAmt.updateAndGet(v -> v.add(row.getSettlementSaleAmt()));
                }

                if (deoStatus.containsKey(row.getStatus())) {
                    row.setStatusName(deoStatus.get(row.getStatus()));
                }
                // 获取商品的多单位
                try {
                    ApiResult<ItmUomRpcDTO> itmUomRpcByItemId = itmItemRpcService.findItmUomRpcByItemId(row.getItemId());
                    if (itmUomRpcByItemId != null && itmUomRpcByItemId.isSuccess()) {
                        ItmUomRpcDTO data = itmUomRpcByItemId.getData();
                        Optional.ofNullable(data).ifPresent(itmUomRpcDTO -> row.setItemUomList(itmUomRpcDTO.getItemUomList()));
                    }
                } catch (Exception e) {
                    log.error("获取商品单位数据失败", e);
                }
                // 设置待发数量
                if (row.getDemandQuantity() != null && row.getQuantity() != null) {
                    row.setToDeliveryQty(row.getDemandQuantity().subtract(row.getQuantity()));
                }

                ItmItemScpBaseRpcDTO itmItemScpBaseRpcDTO = itemMap.get(row.getItemId());
                row.setItemAttrName(StrUtil.isBlank(itmItemScpBaseRpcDTO.getItemAttrName()) ? itmItemScpBaseRpcDTO.getSpec() : itmItemScpBaseRpcDTO.getItemAttrName());
                row.setAnotherName(itmItemScpBaseRpcDTO.getAnotherName());
                row.setItemName(itmItemScpBaseRpcDTO.getItemName());
                //商品图片赋值
                var skuAttchmentList = itmItemScpBaseRpcDTO.getSkuAttchmentList();
                var spuAttchmentList = itmItemScpBaseRpcDTO.getSpuAttchmentList();
                if (CollUtil.isNotEmpty(skuAttchmentList)) {
                    Optional<ItmItemAttachmentRpcDTO> first = skuAttchmentList.stream().filter(ItmItemAttachmentRpcDTO::getMajor).findFirst();
                    row.setUrl(first.isEmpty() ? skuAttchmentList.get(0).getUrl() : first.get().getUrl());
                } else if (CollUtil.isNotEmpty(spuAttchmentList)) {
                    Optional<ItmItemAttachmentRpcDTO> first = spuAttchmentList.stream().filter(ItmItemAttachmentRpcDTO::getMajor).findFirst();
                    row.setUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                }
            }
        });

        scpDemandOrderDRespVOS = scpDemandOrderDRespVOS.stream().filter(row -> row.getProjectFeeFlag() == null || !row.getProjectFeeFlag()).toList();
        // 如果amt2比saleAmt2多，则最后一行商品减去差值，如果最后一行商品不够减，则往前递推
        if (amt.get().compareTo(saleAmt.get()) != 0) {
            BigDecimal diff = amt.get().subtract(saleAmt.get());
            // 查找settlementAmt最大的元素，避免完整排序
            scpDemandOrderDRespVOS.stream().max(Comparator.comparing(ScpDemandOrderDRespVO::getSettlementAmt))
                    .ifPresent(max -> max.setSettlementAmt(max.getSettlementAmt().subtract(diff)));
        }

        // 组合商品处理
        List<ScpDemandOrderDRespVO> result = scpDemandOrderDRespVOS.stream().filter(d -> StringUtils.isBlank(d.getCombineItemCode())).collect(Collectors.toList());
        Map<String, List<ScpDemandOrderDRespVO>> subItemMap = scpDemandOrderDRespVOS.stream().filter(d -> StringUtils.isNotBlank(d.getCombineItemCode())).collect(Collectors.groupingBy(ScpDemandOrderDRespVO::getCombineItemCode));
        if (!subItemMap.isEmpty()) {
            subItemMap.forEach((k, v) -> {
                ScpDemandOrderDRespVO combineOrderDRespVO = new ScpDemandOrderDRespVO();
                combineOrderDRespVO.setCombineItemCode(k);
                combineOrderDRespVO.setCombineItemName(v.get(0).getCombineItemName());
                combineOrderDRespVO.setSubItemList(v);
                result.add(combineOrderDRespVO);
            });
        }

        return PagingVO.<ScpDemandOrderDRespVO>builder().records(result).total(page.getTotalElements()).build();
    }

    @Override
    public void remove(Long id) {
        var demandOrderDDTOList = scpDemandOrderDDomainService.findDemandOrderDByIds(List.of(id));
        demandOrderDDTOList.forEach(row -> {
            if (row.getIsCalculated()) {
                throw new BusinessException("商品名称:" + row.getItemName() + "已计算，不能删除");
            }
            if (row.getIsPushed()) {
                throw new BusinessException("商品名称:" + row.getItemName() + "已推送，不能删除");
            }
        });

        scpDemandOrderDDomainService.deleteByIds(Arrays.asList(id));
    }


    /**
     * 批量保存订货订单明细
     *
     * 实现逻辑：
     * 1. 过滤出未推送的明细记录
     * 2. 获取当前最大行号并初始化行号计数器
     * 3. 删除该订单下所有未推送的明细行
     * 4. 按UUID分组处理明细数据，为每组分配需求数量
     * 5. 如果来源不为空，则校验库存可用性
     * 6. 调用领域服务批量保存明细数据
     * 7. 更新订单的分配状态和支付状态
     *
     * @param masId 主订单ID
     * @param source 数据来源标识
     * @param saveVOS 需要保存的订单明细数据列表
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSaveDemandOrderD(Long masId, String source, List<ScpDemandOrderDSaveVO> saveVOS) {
        batchSaveDemandOrderDInternal(masId, source, saveVOS, false);
    }

    /**
     * 批量保存甲指乙采类商品明细
     *
     * 实现逻辑：
     * 1. 过滤出未推送的明细记录
     * 2. 获取当前最大行号并初始化行号计数器
     * 3. 删除该订单下所有未推送的明细行
     * 4. 按UUID分组处理明细数据，为每组分配需求数量
     * 5. 如果来源不为空，则校验库存可用性
     * 6. 调用领域服务批量保存甲指乙采明细数据
     * 7. 更新订单的分配状态和支付状态
     *
     * @param masId 主订单ID
     * @param source 数据来源标识
     * @param saveVOS 需要保存的甲指乙采订单明细数据列表
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSaveDboDemandOrderD(Long masId, String source, List<ScpDemandOrderDSaveVO> saveVOS) {
        batchSaveDemandOrderDInternal(masId, source, saveVOS, true);
    }

    /**
     * 批量保存订货订单明细通用方法
     *
     * 实现逻辑：
     * 1. 过滤出未推送的明细记录
     * 2. 获取当前最大行号并初始化行号计数器
     * 3. 删除该订单下所有未推送的明细行
     * 4. 按UUID分组处理明细数据，为每组分配需求数量
     * 5. 如果来源不为空，则校验库存可用性
     * 6. 根据是否为甲指乙采类型调用相应的保存方法
     * 7. 更新订单的分配状态和支付状态
     *
     * @param masId 主订单ID
     * @param source 数据来源标识
     * @param saveVOS 需要保存的订单明细数据列表
     * @param isDbo 是否为甲指乙采类型
     */
    private void batchSaveDemandOrderDInternal(Long masId, String source, List<ScpDemandOrderDSaveVO> saveVOS, boolean isDbo) {
        // 明细相关金额、数量小数位,表头相关金额
        var modifyDetails = saveVOS.stream().filter(row -> !row.getIsPushed()).collect(Collectors.toList());
        BigDecimal maxLineNo = scpDemandOrderDDomainService.findMaxLineNoPushedByMasId(masId);
        // 删除未推送的明细行
        scpDemandOrderDDomainService.deleteUnPushedItem(masId);
        AtomicInteger lineNo = new AtomicInteger(maxLineNo.add(BigDecimal.ONE).intValue());
        TimeInterval timer = new TimeInterval();
        modifyDetails.stream().collect(Collectors.groupingBy(ScpDemandOrderDSaveVO::groupByUuid)).forEach((id, list) -> {
            // 按比例分配需求数量
            distributeDemandQty(list, masId, lineNo);
        });
        log.info("批量保存订货订单明细，分配耗时：{}", timer.intervalMs());
        if (source != null) {
            //校验库存可供量
            checkStock(modifyDetails);
        }

        if (isDbo) {
            scpDemandOrderDDomainService.batchSaveDboDemanOrderD(modifyDetails, lineNo, source);
        } else {
            scpDemandOrderDDomainService.batchSave(modifyDetails, lineNo, source);
        }
        scpDemandOrderDomainService.updateAllocStatusAndPayStatusById(masId);
    }

    @Override
    public void saveDemandOrderD(ScpDemandOrderDSaveVO scpDemandOrderDSaveVO) {
        if (Boolean.TRUE.equals(scpDemandOrderDSaveVO.getIsCalculated())) {
            throw new BusinessException(ApiCode.FAIL, "此商品已分配");
        }
        scpDemandOrderDDomainService.saveDemandOrderD(scpDemandOrderDSaveVO);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByParam(ScpDemandOrderDelParamVO paramVO) {
        ScpDemandOrderDTO demandOrderDTO = scpDemandOrderDomainService.findDemandOrderById(paramVO.getMasId()).orElseThrow(() -> new BusinessException("订货单不存在"));
        if (paramVO.getDId() == null) {
            List<String> docStatusList = List.of(ScpUdcEnum.DEO_STATUS_DRAFT.getValueCode(),
                    ScpUdcEnum.DEO_STATUS_WT.getValueCode(),
                    ScpUdcEnum.DEO_STATUS_DONE.getValueCode(),
                    ScpUdcEnum.DEO_STATUS_DOING.getValueCode());
            if (!docStatusList.contains(demandOrderDTO.getDocStatus())) {
                throw new BusinessException("订货单不允许删除");
            }
            scpDemandOrderService.deleteByIds(List.of(paramVO.getMasId()));
            return;
        }
        var demandOrderDDTOList = scpDemandOrderDDomainService.findDemandOrderDByIds(List.of(paramVO.getDId()));
        demandOrderDDTOList.forEach(row -> {
            if (row.getIsCalculated()) {
                throw new BusinessException("商品名称:" + row.getItemName() + "已计算，不能删除");
            }
            if (row.getIsPushed()) {
                throw new BusinessException("商品名称:" + row.getItemName() + "已推送，不能删除");
            }
        });
        List<Long> sourceIds = demandOrderDDTOList.stream().map(ScpDemandOrderDDTO::getSourceId).distinct().collect(Collectors.toList());
        scpDemandOrderDDomainService.deleteBySourceIds(sourceIds);
        // 如果删除了所有明细，表头也要删除
        List<ScpDemandOrderDDTO> demandOrderDDTOS = scpDemandOrderDDomainService.findDemandOrderDByMasId(paramVO.getMasId());
        if (CollUtil.isEmpty(demandOrderDDTOS)) {
            scpDemandOrderService.deleteByIds(List.of(paramVO.getMasId()));
        }
        //更新门店已强配次数
        List<Long> activityIds = demandOrderDDTOList.stream().map(ScpDemandOrderDDTO::getActivityId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollUtil.isNotEmpty(activityIds)) {
            scpAllocSettingStoreDomainService.updateAllocNumByParam(activityIds, demandOrderDTO.getDemandWhStCode());
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addItem(Long masId, List<ScpDemandOrderAddItemParamVO> saveVOS) {
        Optional<ScpDemandOrderRespVO> demandOrderRespVO = scpDemandOrderService.findDemandOrderById(masId);
        if (demandOrderRespVO.isEmpty()) {
            throw new BusinessException("订货订单不存在");
        }
        ScpDemandOrderRespVO scpDemandOrderRespVO = demandOrderRespVO.get();
        List<ScpDemandOrderDSaveVO> scpDemandOrderDSaveVOS = new ArrayList<>();
        BigDecimal maxLineNo = scpDemandOrderDDomainService.findMaxLineNoPushedByMasId(masId);
        AtomicInteger lineNo = new AtomicInteger(maxLineNo.add(BigDecimal.ONE).intValue());
        saveVOS.forEach(row -> {
            ScpDemandOrderItemParamVO scpDemandOrderItemParamVO = new ScpDemandOrderItemParamVO();
            scpDemandOrderItemParamVO.setItemId(row.getItemId());
            scpDemandOrderItemParamVO.setItemCode(row.getItemCode());
            scpDemandOrderItemParamVO.setItemName(row.getItemName());
            scpDemandOrderItemParamVO.setSpuItemCode(row.getSpuItemCode());
            scpDemandOrderItemParamVO.setBusinessType(scpDemandOrderRespVO.getBusinessType());
            scpDemandOrderItemParamVO.setType(scpDemandOrderRespVO.getType());
            scpDemandOrderItemParamVO.setDemandWhStCode(scpDemandOrderRespVO.getDemandWhStCode());
            scpDemandOrderItemParamVO.setDemandDate(scpDemandOrderRespVO.getDemandDate());
            scpDemandOrderItemParamVO.setDemandQuantity(row.getDemandQuantity());
            scpDemandOrderItemParamVO.setUnit(row.getUom());
            scpDemandOrderItemParamVO.setUnitName(row.getUomName());
            scpDemandOrderItemParamVO.setSpuItemName(row.getSpuItemName());
            List<ScpDemandOrderDRespVO> itemList = scpDemandOrderService.getItemList(scpDemandOrderItemParamVO);
            List<ScpDemandOrderDSaveVO> intentionList = ScpDemandOrderDConvert.INSTANCE.respsVosToSaveVos(itemList);
            distributeDemandQty(intentionList, masId, lineNo);
            scpDemandOrderDSaveVOS.addAll(intentionList);
        });
        scpDemandOrderDDomainService.batchSave(scpDemandOrderDSaveVOS, lineNo);
    }

    @Override
    @SysCodeProc
    public List<ScpDemandOrderDRespVO> findDemandOrderDBySrcDocId(Long srcDocId) {
        return scpDemandOrderDDomainService.findDemandOrderDBySrcDocId(srcDocId).stream().map(ScpDemandOrderDConvert.INSTANCE::dtoToRespVO).collect(Collectors.toList());
    }

    /**
     * 按比例分配需求数量
     *
     * @param list
     */
    private static void distributeDemandQty(List<ScpDemandOrderDSaveVO> list, Long masId, AtomicInteger index) {
        String orderSetting = SysUtils.getOrderSetting();
        for (ScpDemandOrderDSaveVO saveVO : list) {
            saveVO.setLineNo(new BigDecimal(index.getAndIncrement()));
            saveVO.setMasId(masId);
            saveVO.setCurrency(StrUtil.isNotBlank(saveVO.getCurrency()) ? saveVO.getCurrency() : ScpConstant.CNY);
            // 若系统配置=订货单分配，计算数量 = 分配需求数量
            if ("1".equals(orderSetting)) {
                saveVO.setIsCalculated(Boolean.TRUE);
                saveVO.setPlanQuantity(saveVO.getAllocationDeQuantity());
                if (saveVO.getPrice() == null) {
                    continue;
                }
                // 含税单价
                saveVO.setPrice(SysUtils.processPriceScale(saveVO.getPrice()));
                // 含税总价：含税金额 * 采购数量
                var amt = saveVO.getPrice().multiply(saveVO.getPlanQuantity());
                saveVO.setPlanAmt(SysUtils.processAmtScale(amt));
            }
        }
    }

    /**
     * 校验库存
     *
     * @param scpDemandOrderDSaveVOS
     */
    private void checkStock(List<ScpDemandOrderDSaveVO> scpDemandOrderDSaveVOS) {
        // 只有区配 统配需要校验库存
        List<ScpDemandOrderDSaveVO> saveVOList = scpDemandOrderDSaveVOS.stream()
                .filter(row -> row != null &&
                        ScpConstant.WH_DELIVERY_TYPES.contains(row.getDeliveryType()) &&
                        (row.getPreSaleStatus() == null || !row.getPreSaleStatus())
                ).collect(Collectors.toList());
        if (CollUtil.isEmpty(saveVOList)) {
            return;
        }
        List<Long> itemIds = saveVOList.stream().filter(row -> (row.getFreightLineFlag() == null || !row.getFreightLineFlag()) && ScpUdcEnum.SUPPLY_DEMAND_TYPE_WH.getValueCode().equals(row.getSupplyType()))
                .map(ScpDemandOrderDSaveVO::getItemId).distinct().collect(Collectors.toList());
        List<Long> whIds = saveVOList.stream().filter(row -> (row.getFreightLineFlag() == null || !row.getFreightLineFlag()) && ScpUdcEnum.SUPPLY_DEMAND_TYPE_WH.getValueCode().equals(row.getSupplyType()))
                .map(ScpDemandOrderDSaveVO::getSuppWhId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollUtil.isEmpty(itemIds) || CollUtil.isEmpty(whIds)) {
            log.info("校验库存商品或者仓库为空,无需查询库存");
            return;
        }
        InvWhItemTotalStkRpcParam param = new InvWhItemTotalStkRpcParam();
        param.setWhIds(whIds);
        param.setItemIds(itemIds);
        param.setExcludeDeter2Types(List.of(InvDeter2TypeEnum.TRANS.getType()));
        List<InvWhItemTotalStkRpcDTO> stkRpcDTOS = rmiInvStkRpcService.queryInvWhItemTotalStk(param);
        if (CollUtil.isEmpty(stkRpcDTOS)) {
            throw new BusinessException("商品【" + saveVOList.get(0).getItemName() + "】库存被抢购完，请重新检查购物车商品再下单");
        }
        Map<String, InvWhItemTotalStkRpcDTO> stkMap = stkRpcDTOS.stream().collect(Collectors.toMap(row -> row.getWhId() + "@" + row.getItemId(), Function.identity()));
        for (ScpDemandOrderDSaveVO saveVO : saveVOList) {
            if ((saveVO.getFreightLineFlag() == null || !saveVO.getFreightLineFlag()) && ScpUdcEnum.SUPPLY_DEMAND_TYPE_WH.getValueCode().equals(saveVO.getSupplyType())) {
                checkStockAvailability(saveVO, stkMap);
            }
        }

    }

    private void checkStockAvailability(ScpDemandOrderDSaveVO saveVO, Map<String, InvWhItemTotalStkRpcDTO> stkMap) {
        String key = saveVO.getSuppWhId() + "@" + saveVO.getItemId();
        InvWhItemTotalStkRpcDTO invWhItemTotalStkRpcDTO = stkMap.get(key);
        if (invWhItemTotalStkRpcDTO == null) {
            throw new BusinessException("商品【" + saveVO.getItemName() + "】库存不足");
        }
        BigDecimal avalQty = invWhItemTotalStkRpcDTO.getAvalQty();
        if (invWhItemTotalStkRpcDTO.getUomRatio() == null) {
            throw new BusinessException("商品【" + saveVO.getItemName() + "】辅助单位转换率不能为空");
        }
        if (saveVO.getPlanQuantity().multiply(invWhItemTotalStkRpcDTO.getUomRatio()).compareTo(avalQty) > 0) {
            throw new BusinessException("商品【" + saveVO.getItemName() + "】库存可用量为:" + invWhItemTotalStkRpcDTO.getAvalQty2() + ",库存不足");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ScpDemandOrderDO submitDemandOrder(Long masId, List<ScpDemandOrderDMgmtSaveVO> saveVOS) {
        ScpDemandOrderDO scpDemandOrderDO = scpDemandOrderRepo.findById(masId).orElseThrow(() -> new BusinessException("订货单不存在"));

        if (CollUtil.isNotEmpty(saveVOS)) {
            batchSaveDemandOrderDMgmt(masId, scpDemandOrderDO.getDemandWhStCode(), saveVOS, "PC");
        }
        String apprStatus = scpDemandOrderService.generateDemandOrderApprStatus(scpDemandOrderDO.getType(), scpDemandOrderDO.getBusinessType());
        scpDemandOrderDO.setApprStatus(apprStatus);
        scpDemandOrderRepo.updateApprStatusById(scpDemandOrderDO.getId(), apprStatus);
        if (scpDemandOrderDO.getEtaDate() == null) {
            // 根据门店编码获取门店信息
            OrgStoreDetailRpcDTO orgStore = Optional.ofNullable(orgStoreRpcService.getByCode(scpDemandOrderDO.getDemandWhStCode())).orElseThrow(() -> new BusinessException("门店不存在"));
            // 设计预计收货时间
            Integer expectArrivePriod = orgStore.getExpectArrivePriod();
            scpDemandOrderDO.setEtaDate(LocalDateTime.now().plusDays(expectArrivePriod));
            scpDemandOrderRepo.updateEtaDateById(scpDemandOrderDO.getId(), scpDemandOrderDO.getEtaDate());
        }
        return scpDemandOrderDO;
    }


}
