package com.elitesland.scp.pay.service;

import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitesland.inv.dto.invTro.InvTroOnlinePayRpcDTO;
import com.elitesland.pur.dto.po.PurPoOnlinePayRpcDTO;
import com.elitesland.scp.application.facade.vo.param.order.ScpOrderUpdatePayStatusMqParam;
import com.elitesland.scp.application.facade.vo.resp.wqf.ScpWqfEntAccountRespVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.infr.repo.wqf.ScpWqfEntAccountRepoProc;
import com.elitesland.scp.pay.config.EntpayProperties;
import com.elitesland.scp.pay.vo.OnlinePayOrderParamVO;
import com.elitesland.scp.pay.vo.PayOrderReqDTO;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiPurRpcService;
import com.tenpay.business.entpay.mse.sdk.api.Payment;
import com.tenpay.business.entpay.mse.sdk.api.Redirect;
import com.tenpay.business.entpay.mse.sdk.common.NotifyHandler;
import com.tenpay.business.entpay.mse.sdk.config.EntpayConfig;
import com.tenpay.business.entpay.mse.sdk.exception.EntpayException;
import com.tenpay.business.entpay.mse.sdk.model.*;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.configurationprocessor.json.JSONException;
import org.springframework.boot.configurationprocessor.json.JSONObject;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import static com.tenpay.business.entpay.mse.sdk.model.CurrencyConstant.CNY;

@Slf4j
@Service
@RequiredArgsConstructor
public class PaymentServiceImpl implements PaymentService {

    private final NotifyVerifyService notifyVerifyService;
    private final RedisWrapper redisWrapper;
    private final RedisUtils redisUtils;
    private final RmiPurRpcService rmiPurRpcService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final AccountLinkService accountLinkService;
    private final MessageQueueTemplate messageQueueTemplate;
    private final ScpWqfEntAccountRepoProc scpWqfEntAccountRepoProc;

    @Override
    public Payment createMpPay(PayOrderReqDTO param) throws EntpayException {
        // 封装小程序支付接口入参
        PaymentMpParam paymentMpParam = constructMPPaymentParam(param);
        // 调用小程序支付接口
        log.info("小程序支付调用微企付下单接口入参:{}", JSONUtil.toJsonStr(paymentMpParam));
        Payment mpPay = Payment.createMpPay(paymentMpParam);
        log.info("小程序支付下单成功，paymentId:{}", mpPay.getPaymentId());
        return mpPay;
    }

    @Override
    public Redirect createMpPayOnline(OnlinePayOrderParamVO param) {
        try {
            String docCls = param.getDocNo().substring(0, 2);
            List<Goods> goodsList = new ArrayList<>();
            //构建支付明细
            Redirect redirect = buildPayItems(param, "MP", docCls, goodsList);
            if (redirect != null) {
                return redirect;
            }
            PayOrderReqDTO paymentMpParam = buildPaymentMapParam(param, goodsList);
            Payment payment = this.createMpPay(paymentMpParam);

            //写入mq队列
            ScpOrderUpdatePayStatusMqParam updateParamVO = new ScpOrderUpdatePayStatusMqParam();
            updateParamVO.setBusinessKey(ScpOrderUpdatePayStatusMqParam.SCP_ORDER_UPDATE_PAY_STATUS_CHANNEL);
            updateParamVO.setDocNo(param.getDocNo());
            updateParamVO.setPaymentId(payment.getPaymentId());
            updateParamVO.setAmt(param.getAmt());
            updateParamVO.setPayerName(param.getPayerName());
            updateParamVO.setDocCls(docCls);
            messageQueueTemplate.publishMessageSync("yst-suplan", ScpOrderUpdatePayStatusMqParam.SCP_ORDER_UPDATE_PAY_STATUS_CHANNEL, updateParamVO);
            return accountLinkService.createPaymentLink(payment.getPaymentId(), param.getDocNo(), paymentMpParam.getExpireTime());
        } catch (EntpayException e) {
            throw new BusinessException("微企付支付失败" + e.getMessage());
        }
    }

    private Redirect buildPayItems(OnlinePayOrderParamVO param, String source, String docCls, List<Goods> goodsList) {
        Object obj = redisUtils.get(ScpConstant.SCP_ONLINE_PAY + param.getDocNo());
        if (obj != null) {
            Redirect redirect = JSON.parseObject((String) obj, Redirect.class);
            if ("MP".equals(source) && redirect.getMiniProgram() == null) {
                throw new BusinessException("请使用PC端扫码支付");
            }
            if ("QR".equals(source) && redirect.getStaticQrcode() == null) {
                throw new BusinessException("请使用小程序支付");
            }
        }
        if ("PO".equals(docCls)) {
            var onlinePayInfo = rmiPurRpcService.findOnlinePayInfo(param.getDocId());
            if (onlinePayInfo == null) {
                throw new BusinessException("支付订单不存在，请联系管理员");
            }
            if (onlinePayInfo.getOnlinePayFlag()) {
                throw new BusinessException("订单已支付");
            }
            if (obj != null && !onlinePayInfo.getOnlinePayFlag()) {
                return JSON.parseObject((String) obj, Redirect.class);
            }
            // 采购订单支付
            List<PurPoOnlinePayRpcDTO.PoItem> podList = onlinePayInfo.getPoItems();
            // 汇总数量
            BigDecimal totalQty = podList.stream().map(PurPoOnlinePayRpcDTO.PoItem::getQty).reduce(BigDecimal.ZERO, BigDecimal::add);
            Goods goods = Goods.builder()
                    .goodName("微企付支付商品")
                    .goodNumber(totalQty.intValue())
                    .goodAmount(param.getAmt().multiply(new BigDecimal("100")).longValue())
                    .build();
            goodsList.add(goods);
        } else {
            var onlinePayInfo = rmiInvStkRpcService.findOnlinePayInfo(param.getDocId());
            if (onlinePayInfo == null) {
                throw new BusinessException("支付订单不存在，请联系管理员");
            }
            if (onlinePayInfo.getOnlinePayFlag()) {
                throw new BusinessException("订单已支付");
            }
            if (obj != null && !onlinePayInfo.getOnlinePayFlag()) {
                return JSON.parseObject((String) obj, Redirect.class);
            }
            // 调拨订单支付
            List<InvTroOnlinePayRpcDTO.TroItem> troDList = onlinePayInfo.getItems();
            // 汇总数量
            BigDecimal totalQty = troDList.stream().map(InvTroOnlinePayRpcDTO.TroItem::getTroQty2).reduce(BigDecimal.ZERO, BigDecimal::add);
            Goods goods = Goods.builder()
                    .goodName("微企付支付商品")
                    .goodNumber(totalQty.intValue())
                    .goodAmount(param.getAmt().multiply(new BigDecimal("100")).longValue())
                    .build();
            goodsList.add(goods);
        }
        return null;
    }


    private PayOrderReqDTO buildPaymentMapParam(OnlinePayOrderParamVO param, List<Goods> goodsList) {
        PayOrderReqDTO paymentMpParam = new PayOrderReqDTO();
        paymentMpParam.setOutPaymentId(param.getDocNo());
        paymentMpParam.setAmount(param.getAmt().multiply(new BigDecimal("100")).longValue());
        paymentMpParam.setCurrency(CNY);
        Date now = new Date();
        paymentMpParam.setExpireTime(new Date(now.getTime() + 24 * 60 * 60 * 1000));
        paymentMpParam.setEntId(param.getEntId());
        paymentMpParam.setEntName(param.getEntName());
        paymentMpParam.setMemo(param.getRemark());
        paymentMpParam.setAttachment(param.getRemark());
        paymentMpParam.setDeviceId(param.getStoreCode());
        paymentMpParam.setPayerClientIp(param.getClientIp());
        paymentMpParam.setPayerUa(param.getPayerUa());
        paymentMpParam.setCreateTime(new Date());
        paymentMpParam.setStoreId(param.getStoreCode());
        paymentMpParam.setStoreName(param.getStoreName());
        paymentMpParam.setMpPath(param.getMpPath());
        paymentMpParam.setGoods(goodsList);
        paymentMpParam.setPayerType(param.getPayerType());
        paymentMpParam.setPayerName(param.getPayerName());
        //查询收款商户子账号ID
        ScpWqfEntAccountRespVO scpWqfEntAccountRespVO = scpWqfEntAccountRepoProc.getEntAcctIdByStoreIdAndEntId(param.getStoreCode(), param.getEntId());
        if (scpWqfEntAccountRespVO != null) {
            paymentMpParam.setEntAccId(scpWqfEntAccountRespVO.getEntAcctId());
            if (StrUtil.isBlank(scpWqfEntAccountRespVO.getBankAccount())) {
                throw new BusinessException("收款银行账号为空");
            }
            //获取后四位
            paymentMpParam.setBankAccountLast4(scpWqfEntAccountRespVO.getBankAccount().substring(scpWqfEntAccountRespVO.getBankAccount().length() - 4));
        }
        return paymentMpParam;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Redirect createQrCodePayOnline(OnlinePayOrderParamVO param) {
        try {
            String docCls = param.getDocNo().substring(0, 2);
            List<Goods> goodsList = new ArrayList<>();
            //构建支付明细
            Redirect redirect = buildPayItems(param, "QR", docCls, goodsList);
            if (redirect != null) {
                return redirect;
            }
            PayOrderReqDTO paymentMpParam = buildPaymentMapParam(param, goodsList);
            Payment qrCodePay = this.createQrCodePay(paymentMpParam);
            //写入mq队列
            ScpOrderUpdatePayStatusMqParam updateParamVO = new ScpOrderUpdatePayStatusMqParam();
            updateParamVO.setBusinessKey(ScpOrderUpdatePayStatusMqParam.SCP_ORDER_UPDATE_PAY_STATUS_CHANNEL);
            updateParamVO.setDocNo(param.getDocNo());
            updateParamVO.setPaymentId(qrCodePay.getPaymentId());
            updateParamVO.setAmt(param.getAmt());
            updateParamVO.setPayerName(param.getPayerName());
            updateParamVO.setDocCls(docCls);
            messageQueueTemplate.publishMessageSync("yst-suplan", ScpOrderUpdatePayStatusMqParam.SCP_ORDER_UPDATE_PAY_STATUS_CHANNEL, updateParamVO);
            return accountLinkService.createPaymentLink(qrCodePay.getPaymentId(), param.getDocNo(), paymentMpParam.getExpireTime());
        } catch (EntpayException e) {
            throw new BusinessException("微企付支付失败" + e.getMessage());
        }
    }

    @Override
    public Payment createQrCodePay(PayOrderReqDTO param) throws EntpayException {
        // 封装二维码支付接口入参
        PaymentQRParam paymentQRParam = constructQRPaymentParam(param);
        // 调用二维码支付接口
        log.info("二维码支付调用微企付下单接口入参:{}", JSONUtil.toJsonStr(paymentQRParam));
        Payment qrCodePay = Payment.createQrCodePay(paymentQRParam);
        log.info("二维码支付下单，返回结果:{}", JSONUtil.toJsonStr(qrCodePay));
        return qrCodePay;
    }

    @Override
    public String paymentNotify(String body, String authorization) throws EntpayException, JSONException {
        // 验签并获取model
        PaymentNotifyModel model = NotifyHandler.handlerWebhook(body, authorization, PaymentNotifyModel.class,
                EntpayConfig.getRealTbepPublicKey(null));
        log.info("支付回调通知数据：{}", JSONUtil.toJsonStr(model));
        notifyVerifyService.processNotify(model);
        // 封装响应参数
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("retcode", 0);
        jsonObject.put("retmsg", "SUCCESS");
        return jsonObject.toString();
    }

    private PaymentQRParam constructQRPaymentParam(PayOrderReqDTO param) {
        PaymentQRParam paymentQRParam = new PaymentQRParam();
        BeanUtils.copyProperties(constructCommonParam(param), paymentQRParam);
        Object o = redisWrapper.apply(() -> {
            return redisUtils.get(ScpConstant.ENT_PAY_PROPERTIES);
        }, null);
        if (o == null) {
            throw new BusinessException("请联系管理员配置微企付信息");
        }
        EntpayProperties entpayProperties = JSONUtil.toBean(o.toString(), EntpayProperties.class);
        // 支付回调url
        PaymentNotifyUrl paymentNotifyUrl = PaymentNotifyUrl.builder()
                .serverNotifyUrl(entpayProperties.getServerNotifyUrl())
                .build();
        paymentQRParam.setNotifyUrl(paymentNotifyUrl);
        return paymentQRParam;
    }

    private PaymentMpParam constructMPPaymentParam(PayOrderReqDTO param) {
        PaymentMpParam paymentMpParam = new PaymentMpParam();
        BeanUtils.copyProperties(constructCommonParam(param), paymentMpParam);
        Object o = redisWrapper.apply(() -> {
            return redisUtils.get(ScpConstant.ENT_PAY_PROPERTIES);
        }, null);
        if (o == null) {
            throw new BusinessException("请联系管理员配置微企付信息");
        }
        EntpayProperties entpayProperties = JSONUtil.toBean(o.toString(), EntpayProperties.class);
        FrontCallbackUrl frontCallbackUrl = FrontCallbackUrl.builder()
                .mpAppid(entpayProperties.getMpAppid())
                .mpUsername(entpayProperties.getMpUsername())
                .mpPath(param.getMpPath())
                .build();

        // 支付回调url
        PaymentNotifyUrl paymentNotifyUrl = PaymentNotifyUrl.builder()
                .serverNotifyUrl(entpayProperties.getServerNotifyUrl())
                .frontCallbackUrl(frontCallbackUrl)
                .build();
        paymentMpParam.setNotifyUrl(paymentNotifyUrl);
        return paymentMpParam;

    }

    /**
     * 构造支付公共参数
     *
     * @param param
     * @return
     */
    private PaymentParam constructCommonParam(PayOrderReqDTO param) {
        //收款方信息
        PaymentPayee payee = PaymentPayee.builder()
                .entId(param.getEntId())
                .entName(param.getEntName())
                .entAcctId(param.getEntAccId())
                .bankAccountNumberLast4(param.getBankAccountLast4())
                .build();

        // 订单行信息
        List<Goods> goodsList = param.getGoods();

        PaymentRiskControl paymentRiskControl = PaymentRiskControl.builder()
                .deviceId(param.getDeviceId())
                .payerClientIp(param.getPayerClientIp())
                .payerUa(param.getPayerUa())
                .createTime(param.getCreateTime())
                .pickType(param.getPickType())
                .pickDescription(param.getPickDescription())
                .build();

        // 支付下单
        StoreInfo storeInfo = StoreInfo.builder()
                .id(param.getStoreId())
                .name(param.getStoreName())
                .areaCode(param.getStoreAreaCode())
                .build();

        // 场景信息
        SceneInfo sceneInfo = SceneInfo.builder()
                .storeInfo(storeInfo)
                .build();
        // 付款人信息
        List<PayerInfo> payerOptions = new ArrayList<>();
        PayerInfo payerInfo = PayerInfo.builder().payerName(param.getPayerName())
                .assignPayType(param.getPayerType()).build();
        payerOptions.add(payerInfo);

        return PaymentParam.builder()
                .outPaymentId(param.getOutPaymentId())
                .amount(param.getAmount())
                .currency(param.getCurrency())
                .expireTime(param.getExpireTime())
                .payee(payee)
                .goods(goodsList)
                .memo(param.getMemo())
                .attachment(param.getAttachment())
                .riskControl(paymentRiskControl)
                .sceneInfo(sceneInfo)
                .profitAllocationFlag(param.getProfitAllocationFlag())
                .payerOptions(payerOptions)
                .build();
    }
}
