package com.elitesland.scp.application.web.cart;

import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.auth.util.SecurityUtil;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitesland.scp.application.facade.vo.cart.CartDetailParamVO;
import com.elitesland.scp.application.facade.vo.cart.CartDetailRespVO;
import com.elitesland.scp.application.facade.vo.param.app.AddressParamVO;
import com.elitesland.scp.application.facade.vo.resp.app.OrgStoreDboVO;
import com.elitesland.scp.application.service.cart.ScpStoreCartService;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBaseRpcParam;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.common.utils.SecurityUtils;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Simon.zhang
 * @version 1.0
 * @packageName com.elitesland.scp.application.web.cart
 * @className DboCartManagerController
 * @description TODO
 * @Remark 18810492456@163.com
 * @date 2025/10/15 10:48
 */
@Slf4j
@RestController
@RequestMapping(value = "/dbo/cart", produces = MediaType.APPLICATION_JSON_VALUE)
@AllArgsConstructor
@Api(value = "甲指乙采-采购车管理", tags = {"甲指乙采-采购车管理"})
public class DboCartManagerController {
    private final OrgStoreRpcService orgStoreRpcService;
    private final ScpStoreCartService scpStoreCartService;
    /**
     * 甲指乙采报货地址缓存
     * 1、小程序点击开始报货后，进入到选择装修门店、输入收货地址页面
     * 2、输入上述信息后，点击去报货
     * 3、点击报货后，将门店编码、收货地址缓存起来
     * 4、进入报货商城，添加甲指乙采商品
     * @return
     */
    @PostMapping("/saveAddr")
    @ApiOperation(value = "保存甲指乙采报货地址缓存")
    public ApiResult<Boolean> saveAddr(@RequestBody AddressParamVO addressParam) {
        try {

            // 1. 获取当前用户信息
            GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
            assert userDetails != null : "用户信息不能为空";

            // 3. 缓存地址信息到Redis或其他缓存系统
            // key格式: DB0_ADDRESS_CACHE:{username}
            String cacheKey = "DBO_ADDRESS_CACHE:" + userDetails.getUser().getUsername();
            // 删除历史缓存
            RedisUtils.instance().del(cacheKey);
            String addressJson = JSON.toJSONString(addressParam);
            RedisUtils.instance().set(cacheKey, addressJson); // 缓存24小时

            return ApiResult.ok(true);
        } catch (Exception e) {
            log.error("缓存地址信息异常", e);
            return ApiResult.fail("地址记录信息失败");
        }
    }
    /**
     * 删除缓存
     */
    @DeleteMapping("/deleteAddr")
    @ApiOperation(value = "删除甲指乙采报货地址缓存")
    public ApiResult<Boolean> deleteAddr() {
        // 1. 获取当前用户信息
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        assert userDetails != null : "用户信息不能为空";

        // 3. 缓存地址信息到Redis或其他缓存系统
        // key格式: DB0_ADDRESS_CACHE:{username}
        String cacheKey = "DBO_ADDRESS_CACHE:" + userDetails.getUser().getUsername();
        // 删除历史缓存
        RedisUtils.instance().del(cacheKey);
        return ApiResult.ok(true);
    }

    /**
     * 获取已缓存的甲指乙采报货地址信息
     * @return
     */
    @GetMapping("/getAddr")
    @ApiOperation(value = "查询甲指乙采报货地址缓存")
    public ApiResult<AddressParamVO> getAddr() {
        // 1. 获取当前用户信息
        GeneralUserDetails userDetails = SecurityContextUtil.currentUser();
        assert userDetails != null : "用户信息不能为空";

        // 2. 从缓存中获取地址信息
        String cacheKey = "DBO_ADDRESS_CACHE:" + userDetails.getUser().getUsername();

        // 方式1：如果是 JSON 字符串存储
        String addressJson = (String) RedisUtils.instance().get(cacheKey);
        if (StringUtils.isNotBlank(addressJson)) {
            AddressParamVO addressParam = JSON.parseObject(addressJson, AddressParamVO.class);
            return ApiResult.ok(addressParam);
        }

        return ApiResult.ok(new AddressParamVO());
    }

}
