package com.elitesland.scp.utils;

import com.alibaba.fastjson.JSON;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.collect.ScpUserCollectDO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.elitesland.scp.common.ScpConstant.COLLECT_PREFIX;

/**
 * 收藏Redis工具类
 *
 * @Author: ryan.xu
 * @since 2023/3/7
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class CollectRedisUtil {

    private final RedisTemplate redisClient;

    /**
     * 操作redis
     *
     * @param storeCode
     * @param userId
     * @return
     */
    public BoundHashOperations<String, Object, Object> getUserCollectOps(String storeCode, Long userId) {
        //先得到当前用户信息
        String cartKey = COLLECT_PREFIX + userId + ":" + storeCode;
        //绑定指定的key操作Redis
        return redisClient.boundHashOps(cartKey);
    }


    /**
     * 查询收藏商品
     *
     * @param userId 用户ID
     * @return
     */
    public List<ScpUserCollectDO> getUserCollectItems(String storeCode, Long userId) {
        //获取购物车里面的所有商品
        var operations = this.getUserCollectOps(storeCode, userId);
        return Optional.ofNullable(operations.values()).orElse(new ArrayList<>()).stream().map(obj -> {
            String str = (String) obj;
            return JSON.parseArray(str, ScpUserCollectDO.class);
        }).flatMap(Collection::stream).collect(Collectors.toList());
    }

    /**
     * 是否收藏商品
     * @param storeCode
     * @param userId
     * @param itemKey
     * @return
     */
    public Boolean existsSpuInUser(String storeCode, Long userId, String itemKey) {
        var operations = this.getUserCollectOps(storeCode, userId);
        Object obj = operations.get(SysUtils.generateRedisKey(ScpConstant.SPU, itemKey));
        return obj != null;
    }

    public Boolean existsSkuInUser(String storeCode, Long userId, String itemKey,Long skuId) {
        var operations = this.getUserCollectOps(storeCode, userId);
        Object obj = operations.get(SysUtils.generateRedisKey(ScpConstant.SPU, itemKey));
        if(obj != null){
            String str = (String) obj;
            List<ScpUserCollectDO> scpUserCollectDOS = JSON.parseArray(str, ScpUserCollectDO.class);
            return scpUserCollectDOS.stream().anyMatch(d -> d.getItemId().equals(skuId));
        }else{
            return false;
        }
    }

}
