package com.elitesland.scp.application.service.common;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.threadpool.ThreadPoolAutoConfiguration;
import com.elitesland.inv.dto.invstk.InvWhItemTotalStkRpcDTO;
import com.elitesland.inv.dto.invstk.InvWhItemTotalStkRpcParam;
import com.elitesland.inv.enums.InvDeter2TypeEnum;
import com.elitesland.pur.dto.PurPurcPriceRpcDTO;
import com.elitesland.pur.dto.supp.PurItemPriceParamRpcDTO;
import com.elitesland.pur.provider.PurPriceProvider;
import com.elitesland.scp.application.enums.ScpUdcEnum;
import com.elitesland.scp.application.facade.vo.param.app.AddressParamVO;
import com.elitesland.scp.application.facade.vo.param.app.AppItemActivityItemPriceParamVO;
import com.elitesland.scp.application.facade.vo.resp.app.AppInvStkRespVO;
import com.elitesland.scp.application.facade.vo.resp.app.AppItemPriceRespVO;
import com.elitesland.scp.application.facade.vo.resp.setting.ScpOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.serviceconfig.ScpServiceConfigRespVO;
import com.elitesland.scp.application.service.serviceconfig.ScpServiceConfigService;
import com.elitesland.scp.application.service.supalloc.ScpSupplyAllocationService;
import com.elitesland.scp.application.service.whnet.ScpWhNetRelationService;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.entity.cart.ScpStoreCartDO;
import com.elitesland.scp.dto.supalloc.ScpSupplyAllocationRpcDTO;
import com.elitesland.scp.dto.whnet.ScpWhNetRelationRpcDTO;
import com.elitesland.scp.rmi.RmiInvStkRpcService;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiPriceRpcService;
import com.elitesland.scp.utils.BeanUtils;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.item.ItmItemSuppSimpleRpcDTO;
import com.elitesland.support.provider.item.dto.ItmItemSimpleRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemSuppRegionRpcDtoParam;
import com.elitesland.support.provider.item.service.ItmItemRpcService;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBaseRpcParam;
import com.elitesland.support.provider.org.service.OrgStoreRpcService;
import com.elitesland.support.provider.pri.service.dto.PriPriceRpcDTO;
import com.elitesland.support.provider.pri.service.param.ItmBomPriceRpcDtoParam;
import com.elitesland.support.provider.pri.service.param.ItmPriPriceRpcDtoParam;
import jodd.util.StringUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;

import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class CommonPriceServiceImpl implements CommonPriceService {
    @Autowired
    @Qualifier(ThreadPoolAutoConfiguration.BEAN_NAME)
    private TaskExecutor taskExecutor;
    private final RmiPriceRpcService rmiPriceRpcService;
    private final RmiItemService rmiItemService;
    private final ScpSupplyAllocationService scpSupplyAllocationService;
    private final ScpWhNetRelationService scpWhNetRelationService;
    private final RmiInvStkRpcService rmiInvStkRpcService;
    private final OrgStoreRpcService orgStoreRpcService;

    private final ScpServiceConfigService serviceConfigService;

    private final ItmItemRpcService itmItemRpcService;

    private final PurPriceProvider purPriceProvider;

    @Override
    public Map<String, List<PriPriceRpcDTO>> getPriceAndStockDetails(List<AppItemActivityItemPriceParamVO> paramVOList, String storeCode, ScpOrderSettingRespVO orderSetting, String type,
                                                                     AtomicReference<Map<String, BigDecimal>> stockMap) {
        List<PriPriceRpcDTO> priceList = CollUtil.newCopyOnWriteArrayList(new ArrayList<>());
        List<ItmPriPriceRpcDtoParam> priceParamList = new ArrayList<>();
        //获取仓网供应关系
        List<ScpStoreCartDO> dataList = paramVOList.stream().map(row -> {
            ScpStoreCartDO storeCartDO = new ScpStoreCartDO();
            storeCartDO.setItemCode(row.getItemCode());
            storeCartDO.setItemCateCode(row.getItemCateCode());
            return storeCartDO;
        }).collect(Collectors.toList());
        Integer pricePlace = SysUtils.getPricePlace();
        Map<String, List<ScpWhNetRelationRpcDTO>> whNetMap = scpWhNetRelationService.findNewWhNetByParam(storeCode, type, dataList);
        //获取供应商份额
        List<String> itemCodes = paramVOList.stream().map(AppItemActivityItemPriceParamVO::getItemCode).distinct().collect(Collectors.toList());
        Map<String, List<ScpSupplyAllocationRpcDTO>> supplyAllocationMap = scpSupplyAllocationService.findSupplyAllocationByParam(storeCode, type, itemCodes);
        List<String> priorities = Arrays.asList(orderSetting.getFirstPriority(), orderSetting.getSecPriority());
        List<AppInvStkRespVO> suppPriorityList = new ArrayList<>();
        List<AppInvStkRespVO> whPriorityList = new ArrayList<>();
        // 优先级查询价格
        for (AppItemActivityItemPriceParamVO activity : paramVOList) {
            AppInvStkRespVO stkParam = new AppInvStkRespVO();
            for (String priority : priorities) {
                ItmPriPriceRpcDtoParam priceParam = null;
                if (ScpUdcEnum.ORDER_PRIORITY_SUPALLOC.getValueCode().equals(priority)) {
                    // 供应商份额配置销售价格
                    priceParam = buildSuppPriceParamList(supplyAllocationMap, activity, priceList, stkParam);
                    if (stkParam.getItemId() != null) {
                        suppPriorityList.add(stkParam);
                    }
                } else if (ScpUdcEnum.ORDER_PRIORITY_WHNET.getValueCode().equals(priority)) {
                    // 仓网关系配置销售价格
                    List<ScpWhNetRelationRpcDTO> whNetList = whNetMap.get(activity.getItemCode());
                    // 未查询到仓网关系,属于没有配置价格
                    if (CollUtil.isEmpty(whNetList)) {
                        return null;
                    }
                    ScpWhNetRelationRpcDTO supp = whNetList.get(0);
                    priceParam = buildWhPriceParamList(supp, activity, priceList, stkParam);
                    if (stkParam.getItemId() != null) {
                        whPriorityList.add(stkParam);
                    }
                }
                if (priceParam != null && priceParam.getItemCode() == null) {
                    break;
                }
                if (priceParam != null && priceParam.getItemCode() != null) {
                    priceParamList.add(priceParam);
                    break;
                }
            }
        }
        if (CollUtil.isEmpty(priceParamList) && CollUtil.isNotEmpty(priceList)) {
            // 构建库存查询参数
            queryInvStock(suppPriorityList, stockMap, whPriorityList);
            return priceList.stream().collect(Collectors.groupingBy(PriPriceRpcDTO::getItemCode));
        }
        if (CollUtil.isEmpty(priceParamList) && CollUtil.isEmpty(priceList)) {
            return new HashMap<>();
        }
        if (stockMap == null) {
            return getItemPrice(priceParamList, priceList,pricePlace);
        }

        CompletableFuture<Map<String, List<PriPriceRpcDTO>>> future1 = CompletableFuture.supplyAsync(() -> getItemPrice(priceParamList, priceList,pricePlace), taskExecutor);
        CompletableFuture<Void> future2 = CompletableFuture.runAsync(() -> queryInvStock(suppPriorityList, stockMap, whPriorityList), taskExecutor);

        CompletableFuture.allOf(future1, future2).join();
        try {
            return future1.get();
        } catch (Exception e) {
            log.info("查询价格报错：", e);
            throw new BusinessException("查询价格报错：" + e);
        }
    }
    /**
     * 新价格规则 - 获取商品价格和库存详情
     *
     * 实现逻辑：
     * 1. 调用通用方法 getPriceAndStockDetailsInternal 处理价格和库存查询
     * 2. 传入 null 地址参数，使用门店默认地址信息
     *
     * @param paramVOList 商品参数列表，包含需要查询价格和库存的商品信息
     * @param storeCode 门店编码，用于查询该门店相关的价格和库存信息
     * @param orderSetting 订单设置，包含价格查询的优先级配置
     * @param type 业务类型，用于区分不同的业务场景
     * @param stockMap 库存结果映射，用于存储查询到的库存信息
     * @return Map<String, AppItemPriceRespVO> 商品编码与价格信息的映射关系
     */
    @Override
    public Map<String,AppItemPriceRespVO> getNewPriceAndStockDetails(List<AppItemActivityItemPriceParamVO> paramVOList,
                                                                     String storeCode,
                                                                     ScpOrderSettingRespVO orderSetting,
                                                                     String type,
                                                                     AtomicReference<Map<String, BigDecimal>> stockMap) {
        return getPriceAndStockDetailsInternal(paramVOList, storeCode, orderSetting, type, stockMap, null);
    }

    /**
     * 采购价格查询 - 获取商品价格和库存详情
     *
     * 实现逻辑：
     * 1. 调用通用方法 getPriceAndStockDetailsInternal 处理价格和库存查询
     * 2. 传入地址参数，，替换门店默认收货地址
     *
     * @param paramVOList 商品参数列表，包含需要查询价格和库存的商品信息
     * @param storeCode 门店编码，用于查询该门店相关的价格和库存信息
     * @param orderSetting 订单设置，包含价格查询的优先级配置
     * @param type 业务类型，用于区分不同的业务场景
     * @param stockMap 库存结果映射，用于存储查询到的库存信息
     * @param addressParam 地址参数，包含收货地址信息，用于供应商匹配
     * @return Map<String, AppItemPriceRespVO> 商品编码与价格信息的映射关系
     */
    @Override
    public Map<String, AppItemPriceRespVO> getPurPriceAndStockDetails(List<AppItemActivityItemPriceParamVO> paramVOList,
                                                                      String storeCode,
                                                                      ScpOrderSettingRespVO orderSetting,
                                                                      String type,
                                                                      AtomicReference<Map<String, BigDecimal>> stockMap,
                                                                      AddressParamVO addressParam) {
        return getPriceAndStockDetailsInternal(paramVOList, storeCode, orderSetting, type, stockMap, addressParam);
    }


    /**
     * 通用的价格和库存详情获取方法，整合了新价格规则和采购价格查询的公共逻辑
     * 1. 获取仓网供应关系映射
     * 2. 查询门店基础信息
     * 3. 遍历商品参数列表，根据配送类型构建不同的价格查询参数：
     *    - 同OU配送：构建供应商直送价格参数
     *    - 组套商品：构建BOM子件价格参数
     *    - 仓配调拨：构建仓网结算价格参数
     *    - 跨公司直送：构建供应商价格参数
     * 4. 匹配供应商信息
     * 5. 查询采购价格或结算价格
     * 6. 异步查询库存信息
     * 7. 转换并组装最终的价格结果
     *
     * @param paramVOList 商品参数列表，包含需要查询价格和库存的商品信息
     * @param storeCode 门店编码，用于查询该门店相关的价格和库存信息
     * @param orderSetting 订单设置，包含价格查询的优先级配置
     * @param type 业务类型，用于区分不同的业务场景
     * @param stockMap 库存结果映射，用于存储查询到的库存信息
     * @param addressParam 地址参数，仅在采购价格查询时使用，包含收货地址信息
     * @return Map<String, AppItemPriceRespVO> 商品编码与价格信息的映射关系
     */
    private Map<String, AppItemPriceRespVO> getPriceAndStockDetailsInternal(List<AppItemActivityItemPriceParamVO> paramVOList,
                                                                            String storeCode,
                                                                            ScpOrderSettingRespVO orderSetting,
                                                                            String type,
                                                                            AtomicReference<Map<String, BigDecimal>> stockMap,
                                                                            AddressParamVO addressParam) {
        log.info("查询商品价格和库存参数：{}", JSONObject.toJSONString(paramVOList));
        //获取仓网供应关系
        Map<String, List<ScpWhNetRelationRpcDTO>> whNetMap = builderWhNetRelationMao(paramVOList, storeCode, type);

        log.info("查询仓网供应关系结果：{}", JSONObject.toJSONString(whNetMap));
        //查询门店
        OrgStoreBaseRpcParam baseRpcParam = new OrgStoreBaseRpcParam();
        baseRpcParam.setStoreCodeList(List.of(storeCode));
        OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = orgStoreRpcService.findSimpleStoreByParam(baseRpcParam).computeData().get(0);

        // 处理地址参数（仅在 getPurPriceAndStockDetails 中使用）
        if (addressParam != null) {
            Assert.notNull(addressParam, "地址参数不能为空");
            orgStoreBaseRpcDTO.setCountry(addressParam.getCountry());
            orgStoreBaseRpcDTO.setProvince(addressParam.getProvince());
            orgStoreBaseRpcDTO.setCity(addressParam.getCity());
            orgStoreBaseRpcDTO.setCounty(addressParam.getCounty());
        }

        List<PurItemPriceParamRpcDTO.Item> supplyItems = new ArrayList<>();
        // 查询价格
        List<AppInvStkRespVO> whPriorityList = new ArrayList<>();
        List<ItmPriPriceRpcDtoParam> priceParamList = new ArrayList<>();

        // 查询子件价格参数列表
        List<ItmBomPriceRpcDtoParam> bomPriceParams = new ArrayList<>();

        Integer pricePlace = SysUtils.getAppPricePlace();

        for (AppItemActivityItemPriceParamVO activity : paramVOList) {
            AppInvStkRespVO stkParam = new AppInvStkRespVO();
            log.info("商品参数：{}", JSONObject.toJSONString(activity));
            List<ScpWhNetRelationRpcDTO> whNetList = whNetMap.get(activity.getItemCode());
            // 未查询到仓网关系,属于没有配置价格
            log.info("价格仓网关系集合：{}，商品：{}", whNetList, activity.getItemCode());
            if (CollUtil.isEmpty(whNetList)) {
                continue;
            }
            ScpWhNetRelationRpcDTO supp = whNetList.get(0);
            log.info("价格仓网关系：{}，商品：{}", supp, activity.getItemCode());
            //库存查询参数
            stkParam.setItemId(activity.getItemId());
            stkParam.setItemCode(activity.getItemCode());
            stkParam.setWhId(supp.getSupplyWhId());
            whPriorityList.add(stkParam);
            activity.setWhNetId(supp.getId());
            if (Objects.equals(supp.getOuCode(), orgStoreBaseRpcDTO.getOuCode())){
                if(Boolean.TRUE.equals(orgStoreBaseRpcDTO.getDeliveryChargeFlag())){
                    activity.setDeliveryChargeFlag(Boolean.TRUE);
                }
                builderSuppDeliveryPriceParamWithSameOu(activity, supplyItems);
                activity.setSuppMatch(true);
            }else{
                if(Boolean.TRUE.equals(activity.getCombineItemFlag())){
                    //组套子件商品
                    if(Boolean.TRUE.equals(orgStoreBaseRpcDTO.getWhChargeFlag())){
                        activity.setWhChargeFlag(Boolean.TRUE);
                    }
                    // 价格参数组装
                    ItmBomPriceRpcDtoParam bomPriceParam = new ItmBomPriceRpcDtoParam();
                    bomPriceParam.setItemCode(activity.getItemCode());
                    bomPriceParam.setCombineItemCode(activity.getCombineItemCode());
                    bomPriceParam.setUom(activity.getUom());
                    bomPriceParams.add(bomPriceParam);

                    // 根据是否是 PUR 价格查询使用不同的配送类型常量
                    if (addressParam != null) {
                        builderSuppDeliveryPriceParamWithSameOu(activity, supplyItems);
                        if(ScpConstant.PUR_DELIVERY_TYPES.contains(supp.getDeliveryType())){
                            activity.setSuppMatch(true);
                        }
                    } else {
                        if(ScpConstant.PUR_TRN_DELIVERY_TYPES.contains(supp.getDeliveryType())){
                            activity.setSuppMatch(true);
                        }
                    }
                }else{
                    //仓配调拨
                    if (ScpConstant.WH_DELIVERY_TYPES.contains(supp.getDeliveryType())) {
                        if(Boolean.TRUE.equals(orgStoreBaseRpcDTO.getWhChargeFlag())){
                            activity.setWhChargeFlag(Boolean.TRUE);
                        }
                        builderWhDeliveryPriceParam(activity, supp, priceParamList);
                    } else if (addressParam != null) {
                        // PUR 价格查询分支
                        builderSuppDeliveryPriceParamWithSameOu(activity, supplyItems);
                        if(ScpConstant.PUR_DELIVERY_TYPES.contains(supp.getDeliveryType())){
                            if(Boolean.TRUE.equals(orgStoreBaseRpcDTO.getDeliveryChargeFlag())){
                                activity.setDeliveryChargeFlag(Boolean.TRUE);
                                activity.setWhChargeFlag(Boolean.TRUE);
                            }
                            builderTrnSuppDeliveryPriceParam(activity, supp, priceParamList);
                            activity.setSuppMatch(true);
                        }
                    } else {
                        // 新价格规则分支
                        if(ScpConstant.PUR_TRN_DELIVERY_TYPES.contains(supp.getDeliveryType())){
                            if(Boolean.TRUE.equals(orgStoreBaseRpcDTO.getDeliveryChargeFlag())){
                                activity.setDeliveryChargeFlag(Boolean.TRUE);
                                activity.setWhChargeFlag(Boolean.TRUE);
                            }
                            builderTrnSuppDeliveryPriceParam(activity, supp, priceParamList);
                            activity.setSuppMatch(true);
                        }
                    }
                }

            }
        }
        Map<String, AppItemActivityItemPriceParamVO> paramVOMap = paramVOList.stream()
                .collect(Collectors.toMap(c -> SysUtils.getItemKey(c.getCombineItemCode(), c.getItemCode()),
                        v -> v, (e1, e2) -> e1));

        //查询服务配置
        Map<String, ScpServiceConfigRespVO> configRespVOMap = getServiceConfigRespVOMap(orgStoreBaseRpcDTO);
        Map<String, AppItemPriceRespVO> priceResultMap = new HashMap<>();
        List<String> needSuppItems = paramVOList.stream()
                .filter(d -> Boolean.TRUE.equals(d.getSuppMatch()))
                .map(AppItemActivityItemPriceParamVO::getItemCode).distinct().collect(Collectors.toList());
        Map<String, ItmItemSuppSimpleRpcDTO> itemSuppSimpleMap;
        log.info("匹配供应商商品：{}", JSONObject.toJSONString(needSuppItems));
        if(CollectionUtils.isNotEmpty(needSuppItems)){
            //匹配供应商
            itemSuppSimpleMap = mathItemSuppHandler(needSuppItems, orgStoreBaseRpcDTO);
            log.info("匹配供应商结果：{}", JSONObject.toJSONString(itemSuppSimpleMap));
            //查询采购价
            if(itemSuppSimpleMap != null){
                supplyItems.forEach(d ->{
                    ItmItemSuppSimpleRpcDTO itemSuppRpcDTO =itemSuppSimpleMap.get(d.getItemCode());
                    if(itemSuppRpcDTO != null){
                        d.setSuppId(itemSuppRpcDTO.getSuppId());
                    }
                });
                findPurSupplyPrice(supplyItems,priceResultMap, orgStoreBaseRpcDTO, itemSuppSimpleMap,paramVOMap,
                        configRespVOMap,pricePlace);
            }
        } else {
            itemSuppSimpleMap = null;
        }
        //查询库存
        CompletableFuture<Void> future1 = CompletableFuture.runAsync(() -> queryNewInvStock(
                stockMap, whPriorityList), taskExecutor);
        //结算价
        List<PriPriceRpcDTO> priceList = CollUtil.newCopyOnWriteArrayList(new ArrayList<>());
        CompletableFuture<Map<String, List<PriPriceRpcDTO>>> future2 = null;
        if (CollectionUtils.isNotEmpty(priceParamList)) {
            future2 = CompletableFuture
                    .supplyAsync(() -> getItemPrice(priceParamList, priceList,pricePlace), taskExecutor);
        }

        // 查询组套商品子件价格
        if(CollectionUtils.isNotEmpty(bomPriceParams)){
            List<PriPriceRpcDTO> bomPriceList = rmiPriceRpcService.findBomPriceByParam(bomPriceParams);
            if (CollectionUtils.isNotEmpty(bomPriceList)) {
                try {
                    Map<String, List<PriPriceRpcDTO>> bomPriceMap = bomPriceList.stream().collect(Collectors.groupingBy(c -> SysUtils.getItemKey(c.getCombineItemCode(), c.getItemCode())));
                    convertPriceResultMap(bomPriceMap, priceResultMap, paramVOMap, configRespVOMap, itemSuppSimpleMap,pricePlace);
                } catch (Exception e) {
                    log.info("查询子件价格报错：", e);
                    throw new BusinessException("查询子件价格报错：" + e);
                }
            }
        }

        if(future2 != null){
            CompletableFuture.allOf(future1, future2).join();
            try {
                Map<String, List<PriPriceRpcDTO>> priceDtoMap = future2.get();
                convertPriceResultMap(priceDtoMap, priceResultMap,paramVOMap,configRespVOMap,itemSuppSimpleMap,pricePlace);
            } catch (Exception e) {
                log.info("查询价格报错：", e);
                throw new BusinessException("查询价格报错：" + e);
            }
        }else{
            CompletableFuture.allOf(future1).join();
        }
        log.info("查询商品价格和库存结果：{}", JSONObject.toJSONString(priceResultMap));
        return priceResultMap;
    }

    @NotNull
    private Map<String, ScpServiceConfigRespVO> getServiceConfigRespVOMap(OrgStoreBaseRpcDTO orgStoreBaseRpcDTO) {
        Map<String, ScpServiceConfigRespVO> configRespVOMap = null;
        if(Boolean.TRUE.equals(orgStoreBaseRpcDTO.getDeliveryChargeFlag())
                || Boolean.TRUE.equals(orgStoreBaseRpcDTO.getWhChargeFlag())){
            //查询服务费
            configRespVOMap = serviceConfigService.findServiceConfigGroupByStore(orgStoreBaseRpcDTO.getStoreCode(),
                    orgStoreBaseRpcDTO.getRegion(), orgStoreBaseRpcDTO.getStoreLevel());

        }
        return configRespVOMap;
    }

    private void builderSuppDeliveryPriceParamWithSameOu(AppItemActivityItemPriceParamVO activity,
                                                         List<PurItemPriceParamRpcDTO.Item> supplyItems){
        PurItemPriceParamRpcDTO.Item item = new PurItemPriceParamRpcDTO.Item();
        item.setItemCode(activity.getItemCode());
        item.setUom(activity.getUom());
        supplyItems.add(item);
    }

    //跨公司直送
    private void builderTrnSuppDeliveryPriceParam(AppItemActivityItemPriceParamVO activity,
                                               ScpWhNetRelationRpcDTO supp,
                                               List<ItmPriPriceRpcDtoParam> priceParamList) {
            ItmPriPriceRpcDtoParam priceParam = new ItmPriPriceRpcDtoParam();
            priceParam.setPriceCls(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
            priceParam.setPriceType(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
            priceParam.setItemId(activity.getItemId());
            priceParam.setCustCode(supp.getCustCode());
            priceParam.setItemCode(activity.getItemCode());
            priceParam.setUom(activity.getUom());
            priceParam.setCurrCode(ScpConstant.CNY);
            priceParam.setOuCode(supp.getPurCompanyCode());
            priceParam.setUuid(ScpConstant.INTERNAL_SETTLEMENT_PRICE
                    + SysUtils.getItemKey(activity.getCombineItemCode(),
                    activity.getItemCode(),
                    activity.getActivityCode()));
            priceParamList.add(priceParam);

    }

    private void builderWhDeliveryPriceParam(AppItemActivityItemPriceParamVO activity,
                                             ScpWhNetRelationRpcDTO supp,
                                             List<ItmPriPriceRpcDtoParam> priceParamList) {
        ItmPriPriceRpcDtoParam priceParam = new ItmPriPriceRpcDtoParam();
        priceParam.setPriceCls(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setPriceType(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setItemId(activity.getItemId());
        priceParam.setCustCode(supp.getCustCode());
        priceParam.setItemCode(activity.getItemCode());
        priceParam.setUom(activity.getUom());
        priceParam.setCurrCode(ScpConstant.CNY);
        priceParam.setOuCode(supp.getOuCode());
        priceParam.setUuid(ScpConstant.INTERNAL_SETTLEMENT_PRICE
                + SysUtils.getItemKey(activity.getCombineItemCode(),
                activity.getItemCode(),
                activity.getActivityCode()));
        priceParamList.add(priceParam);

    }

    private Map<String, List<ScpWhNetRelationRpcDTO>> builderWhNetRelationMao(List<AppItemActivityItemPriceParamVO> paramVOList, String storeCode, String type) {
        List<ScpStoreCartDO> dataList = paramVOList.stream().map(row -> {
            ScpStoreCartDO storeCartDO = new ScpStoreCartDO();
            storeCartDO.setItemCode(row.getItemCode());
            storeCartDO.setItemCateCode(row.getItemCateCode());
            storeCartDO.setItemType2(row.getItemType2());
            return storeCartDO;
        }).collect(Collectors.toList());
        log.info("获取仓网关系入参:{}", JSONUtil.toJsonStr(dataList));
        Map<String, List<ScpWhNetRelationRpcDTO>> whNetMap = scpWhNetRelationService.findNewWhNetByParam(storeCode, type, dataList);
        return whNetMap;
    }


    private void findPurSupplyPrice(List<PurItemPriceParamRpcDTO.Item> supplyItems,
                                    Map<String, AppItemPriceRespVO> priceResultMap,
                                    OrgStoreBaseRpcDTO orgStoreBaseRpcDTO,
                                    Map<String, ItmItemSuppSimpleRpcDTO> simpleItemSuppRpcDTOMap,
                                    Map<String, AppItemActivityItemPriceParamVO> paramVOMap,
                                    Map<String, ScpServiceConfigRespVO> configRespVOMap,Integer pricePlace) {
        List<PurItemPriceParamRpcDTO.Item> items = supplyItems.stream().filter(d -> d.getSuppId() != null)
                .collect(Collectors.toList());
        //构建参数
        PurItemPriceParamRpcDTO purItemPriceParamRpcDTO = new PurItemPriceParamRpcDTO();
        purItemPriceParamRpcDTO.setCurrCode(ScpConstant.CNY);
        purItemPriceParamRpcDTO.setDocTime(LocalDateTime.now());
        purItemPriceParamRpcDTO.setOuId(orgStoreBaseRpcDTO.getOuId());
        purItemPriceParamRpcDTO.setItems(items);
        Map<String, PurPurcPriceRpcDTO> purPurcPriceRpcMap = purPriceProvider.findItemPurPriceGroup(purItemPriceParamRpcDTO).computeData();
        log.info("查询直送价格参数：{}", purItemPriceParamRpcDTO);
        for(var d : items){
            if(paramVOMap.get(d.getItemCode()) != null){
                PurPurcPriceRpcDTO priceRpcDTO = purPurcPriceRpcMap.get(d.getItemCode() + "-" + d.getSuppId() + "-" + d.getUom());
                if(priceRpcDTO != null){
                    AppItemPriceRespVO priceRespVO = new AppItemPriceRespVO();
                    priceRespVO.setItemCode(d.getItemCode());
                    priceRespVO.setSuppId(priceRespVO.getSuppId());
                    priceRespVO.setSuppCode(priceRpcDTO.getSuppCode());
                    priceRespVO.setSuppName(priceRpcDTO.getSuppName());
                    priceRespVO.setStoreCode(orgStoreBaseRpcDTO.getStoreCode());
                    priceRespVO.setPurPrice(priceRpcDTO.getPrice());
                    priceRespVO.setSalePrice(priceRpcDTO.getPrice());
                    ItmItemSuppSimpleRpcDTO itmItemSuppSimpleRpcDTO = simpleItemSuppRpcDTOMap.get(d.getItemCode());
//                    priceRespVO.setSuppCode(itmItemSuppSimpleRpcDTO.getSuppCode());
//                    priceRespVO.setSuppName(itmItemSuppSimpleRpcDTO.getSuppName());
//                    priceRpcDTO.setSuppId(itmItemSuppSimpleRpcDTO.getSuppId());
                    if(Boolean.TRUE.equals(paramVOMap.get(d.getItemCode()).getDeliveryChargeFlag())){
                        if(configRespVOMap != null){
                            log.info("查询服务费参数：{}", configRespVOMap);
                            //服务费
                            BigDecimal techRatio = configRespVOMap.get("TECH") == null ? BigDecimal.ZERO
                                    : configRespVOMap.get("TECH").getFeePercentage().multiply(new BigDecimal("0.01"));
                            BigDecimal mefRatio = configRespVOMap.get("MARKET") == null ? BigDecimal.ZERO :
                                    configRespVOMap.get("MARKET").getFeePercentage().multiply(new BigDecimal("0.01"));
                            BigDecimal oefRatio = configRespVOMap.get("OPERATE") == null ? BigDecimal.ZERO:
                                    configRespVOMap.get("OPERATE").getFeePercentage().multiply(new BigDecimal("0.01"));
                            priceRespVO.setTefPrice(priceRpcDTO.getPrice().multiply(techRatio).setScale(pricePlace, RoundingMode.HALF_UP));
                            priceRespVO.setMefPrice(priceRpcDTO.getPrice().multiply(mefRatio).setScale(pricePlace, RoundingMode.HALF_UP));
                            priceRespVO.setOefPrice(priceRpcDTO.getPrice().multiply(oefRatio).setScale(pricePlace, RoundingMode.HALF_UP));
                            priceRespVO.setPrice((priceRpcDTO.getPrice()
                                    .add(priceRespVO.getMefPrice())
                                    .add(priceRespVO.getTefPrice())
                                    .add(priceRespVO.getOefPrice())).setScale(pricePlace, RoundingMode.HALF_UP));
                            if(configRespVOMap.get("TECH") != null){
                                priceRespVO.setTefFeeOuCode(configRespVOMap.get("TECH").getOuCode());
                                priceRespVO.setTefFeeOuName(configRespVOMap.get("TECH").getOuName());
                            }
                            if(configRespVOMap.get("MARKET") != null){
                                priceRespVO.setMefFeeOuCode(configRespVOMap.get("MARKET").getOuCode());
                                priceRespVO.setMefFeeOuName(configRespVOMap.get("MARKET").getOuName());
                            }
                            if (configRespVOMap.get("OPERATE") != null){
                                priceRespVO.setOefFeeOuCode(configRespVOMap.get("OPERATE").getOuCode());
                                priceRespVO.setOefFeeOuName(configRespVOMap.get("OPERATE").getOuName());
                            }
                        }else{
                            log.info("查询服务费参数为空");
                            priceRespVO.setPrice(priceRpcDTO.getPrice());
                            priceRespVO.setTefPrice(BigDecimal.ZERO);
                            priceRespVO.setMefPrice(BigDecimal.ZERO);
                            priceRespVO.setOefPrice(BigDecimal.ZERO);
                        }
                        priceRespVO.setIsProjFeeCharged(Boolean.TRUE);
                    }else{
                        priceRespVO.setPrice(priceRpcDTO.getPrice());
                        priceRespVO.setIsProjFeeCharged(Boolean.FALSE);
                    }
                    priceResultMap.put(d.getItemCode(), priceRespVO);
                }
            }

        }
    }

    private Map<String, ItmItemSuppSimpleRpcDTO> mathItemSuppHandler(List<String> itemCodes,
                                                                     OrgStoreBaseRpcDTO orgStoreBaseRpcDTO) {
        ItmItemSuppRegionRpcDtoParam regionRpcDtoParam = new ItmItemSuppRegionRpcDtoParam();
        regionRpcDtoParam.setItemCodes(itemCodes);
        regionRpcDtoParam.setCity(orgStoreBaseRpcDTO.getCity());
        regionRpcDtoParam.setCountry(orgStoreBaseRpcDTO.getCountry());
        regionRpcDtoParam.setProvince(orgStoreBaseRpcDTO.getProvince());
        regionRpcDtoParam.setCounty(orgStoreBaseRpcDTO.getCounty());
        regionRpcDtoParam.setDirectDelivery(Boolean.TRUE);
        return itmItemRpcService.findItemSuppByItemRegion(regionRpcDtoParam).computeData();
    }

    private void convertPriceResultMap(Map<String, List<PriPriceRpcDTO>> priceDtoMap,
                                       Map<String, AppItemPriceRespVO> priceResultMap,
                                       Map<String, AppItemActivityItemPriceParamVO> paramVOMap,
                                       Map<String, ScpServiceConfigRespVO> configRespVOMap,
                                       Map<String, ItmItemSuppSimpleRpcDTO> itemSuppSimpleMap,
                                       Integer pricePlace
                                       ) {
        for (Map.Entry<String, List<PriPriceRpcDTO>> entry : priceDtoMap.entrySet()) {
            String key = entry.getKey();
            if(CollectionUtils.isNotEmpty(entry.getValue())){
                PriPriceRpcDTO priPrice = entry.getValue().get(0);
                log.info("结算价priPrice2222:{}，itemSuppMap:{}", priPrice,JSONObject.toJSONString(itemSuppSimpleMap));
                if(paramVOMap.get(SysUtils.getItemKey(priPrice.getCombineItemCode(), priPrice.getItemCode())) != null){
                    AppItemPriceRespVO vo = new AppItemPriceRespVO();
                    log.info("结算价priPrice2222:{}",itemSuppSimpleMap);
                    if(itemSuppSimpleMap != null && itemSuppSimpleMap.get(priPrice.getItemCode()) != null){
                        vo.setSuppId(itemSuppSimpleMap.get(priPrice.getItemCode()).getSuppId());
                        vo.setSuppCode(itemSuppSimpleMap.get(priPrice.getItemCode()).getSuppCode());
                        vo.setSuppName(itemSuppSimpleMap.get(priPrice.getItemCode()).getSuppName());
                    }
                    vo.setItemId(priPrice.getItemId());
                    vo.setItemCode(priPrice.getItemCode());
                    vo.setCostPrice(priPrice.getPrice());
                    vo.setSalePrice(priPrice.getPrice());
                    vo.setCombineQty(priPrice.getCombineQty());
                    vo.setCombineItemCode(priPrice.getCombineItemCode());
                    vo.setCombineItemFlag(StringUtil.isNotEmpty(priPrice.getCombineItemCode()));
                    if(Boolean.TRUE.equals(paramVOMap.get(SysUtils.getItemKey(priPrice.getCombineItemCode(), priPrice.getItemCode())).getWhChargeFlag())){
                        vo.setIsProjFeeCharged(Boolean.TRUE);
                        if(configRespVOMap != null){
                            log.info("查询仓配服务费参数：{}", configRespVOMap);
                            //服务费
                            BigDecimal techRatio = configRespVOMap.get("TECH") == null ? BigDecimal.ZERO
                                    : configRespVOMap.get("TECH").getFeePercentage().multiply(new BigDecimal("0.01"));
                            BigDecimal mefRatio = configRespVOMap.get("MARKET") == null ? BigDecimal.ZERO :
                                    configRespVOMap.get("MARKET").getFeePercentage().multiply(new BigDecimal("0.01"));
                            BigDecimal oefRatio = configRespVOMap.get("OPERATE") == null ? BigDecimal.ZERO:
                                    configRespVOMap.get("OPERATE").getFeePercentage().multiply(new BigDecimal("0.01"));
                            vo.setTefPrice(priPrice.getPrice().multiply(techRatio).setScale(pricePlace, RoundingMode.HALF_UP));
                            vo.setMefPrice(priPrice.getPrice().multiply(mefRatio)
                                    .setScale(pricePlace, RoundingMode.HALF_UP));
                            vo.setOefPrice(priPrice.getPrice().multiply(oefRatio)
                                    .setScale(pricePlace, RoundingMode.HALF_UP));
                            vo.setPrice((priPrice.getPrice()
                                    .add(vo.getMefPrice())
                                    .add(vo.getTefPrice())
                                    .add(vo.getOefPrice())).setScale(pricePlace, RoundingMode.HALF_UP));
                            if(configRespVOMap.get("TECH") != null){
                                vo.setTefFeeOuCode(configRespVOMap.get("TECH").getOuCode());
                                vo.setTefFeeOuName(configRespVOMap.get("TECH").getOuName());
                            }
                            if(configRespVOMap.get("MARKET") != null){
                                vo.setMefFeeOuCode(configRespVOMap.get("MARKET").getOuCode());
                                vo.setMefFeeOuName(configRespVOMap.get("MARKET").getOuName());
                            }
                            if (configRespVOMap.get("OPERATE") != null){
                                vo.setOefFeeOuCode(configRespVOMap.get("OPERATE").getOuCode());
                                vo.setOefFeeOuName(configRespVOMap.get("OPERATE").getOuName());
                            }
                        }else{
                            log.info("查询仓配服务费参数为空");
                            vo.setPrice(priPrice.getPrice());
                            vo.setMefPrice(BigDecimal.ZERO);
                            vo.setTefPrice(BigDecimal.ZERO);
                            vo.setOefPrice(BigDecimal.ZERO);
                        }
                    }else{
                        vo.setPrice(priPrice.getPrice());
                        vo.setIsProjFeeCharged(Boolean.FALSE);
                    }
                    priceResultMap.put(key, vo);
                }
            }
        }
    }

    private void queryNewInvStock(AtomicReference<Map<String, BigDecimal>> stockMap,
                                  List<AppInvStkRespVO> whPriorityList) {
        if (stockMap == null) {
            return;
        }
        InvWhItemTotalStkRpcParam stockParam = buildStockParam(whPriorityList);
        getWhStockMap(stockParam, stockMap, whPriorityList);
    }




//    private void queryInvStock(AtomicReference<Map<String, BigDecimal>> stockMap) {
//        if (stockMap == null) {
//            return;
//        }
//        // 构建库存查询参数
//        for (AppInvStkRespVO stk : suppPriorityList) {
//            stockMap.updateAndGet(currentMap -> {
//                currentMap.put(stk.getItemCode(), new BigDecimal("9999"));
//                return currentMap;
//            });
//        }
//        InvWhItemTotalStkRpcParam stockParam = buildStockParam(whPriorityList);
//        getWhStockMap(stockParam, stockMap);
//    }


    private InvWhItemTotalStkRpcParam buildStockParam(List<AppInvStkRespVO> whPriorityList) {
        if (CollUtil.isNotEmpty(whPriorityList)) {
            List<Long> itemIds = whPriorityList.stream().map(AppInvStkRespVO::getItemId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            List<Long> whIds = whPriorityList.stream().map(AppInvStkRespVO::getWhId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            InvWhItemTotalStkRpcParam invWhItemTotalStkRpcParam = new InvWhItemTotalStkRpcParam();
            invWhItemTotalStkRpcParam.setItemIds(itemIds);
            invWhItemTotalStkRpcParam.setWhIds(whIds);
            invWhItemTotalStkRpcParam.setExcludeDeter2Types(List.of(InvDeter2TypeEnum.TRANS.getType()));
            return invWhItemTotalStkRpcParam;
        }
        return null;
    }

    private void getWhStockMap(InvWhItemTotalStkRpcParam stkRpcParam, AtomicReference<Map<String, BigDecimal>> stockMap, List<AppInvStkRespVO> whPriorityList) {
        if (stkRpcParam == null) {
            return;
        }

        Map<String, List<Long>> itemCodeAndWhId = whPriorityList.stream().collect(Collectors.groupingBy(AppInvStkRespVO::getItemCode, Collectors.mapping(AppInvStkRespVO::getWhId, Collectors.toList())));
        List<InvWhItemTotalStkRpcDTO> invWhItemTotalStkRpcDTOS = rmiInvStkRpcService.queryInvWhItemTotalStk(stkRpcParam);
        if (CollUtil.isNotEmpty(invWhItemTotalStkRpcDTOS)) {
            for (InvWhItemTotalStkRpcDTO dto : invWhItemTotalStkRpcDTOS) {
                List<Long> longs = itemCodeAndWhId.get(dto.getItemCode());
                if (longs.contains(dto.getWhId())) {
                    stockMap.updateAndGet(currentMap -> {
                        currentMap.put(dto.getItemCode(), dto.getAvalQty2());
                        return currentMap;
                    });
                }
            }
        }
    }

    private Map<String, List<PriPriceRpcDTO>> getItemPrice(List<ItmPriPriceRpcDtoParam> priceParamList,
                                                           List<PriPriceRpcDTO> priceList,Integer pricePlace) {
        // 根据 uuid 去重
        priceParamList = new ArrayList<>(priceParamList.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(
                        ItmPriPriceRpcDtoParam::getUuid,
                        Function.identity(),
                        (existing, replacement) -> existing
                )).values());
        List<ItmPriPriceRpcDtoParam> itmPriPriceParamList = BeanUtils.copyToList(priceParamList, ItmPriPriceRpcDtoParam.class);
        List<PriPriceRpcDTO> priceByParam = rmiPriceRpcService.findPriceByParam(itmPriPriceParamList);
        // 未查询到价格,查询基本单位配置的价格
        if (CollUtil.isNotEmpty(priceByParam)) {
            priceList.addAll(priceByParam);
            //查询基本单位价格
            Map<String, List<PriPriceRpcDTO>> inPriceMap =
                    priceList.stream().collect(Collectors.groupingBy(PriPriceRpcDTO::getItemCode));
            priceParamList.removeAll(priceParamList.stream().filter(row -> inPriceMap.containsKey(row.getItemCode()))
                    .collect(Collectors.toList()));
            getBasePrice(priceParamList, priceList,pricePlace);
        } else {
            //查询基本单位价格
            getBasePrice(priceParamList, priceList,pricePlace);
        }
        Map<String, List<PriPriceRpcDTO>> priceMap =
                priceList.stream().collect(Collectors.groupingBy(PriPriceRpcDTO::getItemCode));
        log.info("批量查询商品结算价格信息:{}", JSONUtil.toJsonStr(priceMap));
        return priceMap;
    }

    private void getBasePrice(List<ItmPriPriceRpcDtoParam> priceParamList, List<PriPriceRpcDTO> priceList,
                              Integer pricePlace) {
        if (CollUtil.isEmpty(priceParamList)) {
            return;
        }
        //查询商品
        List<Long> itemIds = priceParamList.stream().map(ItmPriPriceRpcDtoParam::getItemId).distinct().collect(Collectors.toList());
        Map<Long, ItmItemSimpleRpcDTO> simpleItemMap = rmiItemService.findBaseItemMap(itemIds);

        List<ItmPriPriceRpcDtoParam> baseUomList = priceParamList.stream().map(row -> {
            ItmPriPriceRpcDtoParam param = BeanUtils.copyProperties(row, ItmPriPriceRpcDtoParam.class);
            ItmItemSimpleRpcDTO simpleItem = simpleItemMap.get(row.getItemId());
            param.setUom(simpleItem.getUom());
            return param;
        }).collect(Collectors.toList());
        List<PriPriceRpcDTO> basePriceList = rmiPriceRpcService.findPriceByParam(baseUomList);
        log.info("查询基本单位价格:{}", JSONUtil.toJsonStr(basePriceList));
        if (CollUtil.isNotEmpty(basePriceList)) {
            List<PriPriceRpcDTO> resultPriceList = basePriceList.stream().map(row -> {
                PriPriceRpcDTO priPriceRpcDTO = BeanUtils.copyProperties(row, PriPriceRpcDTO.class);
                ItmItemSimpleRpcDTO simpleItem = simpleItemMap.get(row.getItemId());
                if (simpleItem.getUomRatio2() == null) {
                    return null;
                }
                if (simpleItem.getDecimalPlaces() != null) {
                    priPriceRpcDTO.setPrice(row.getPrice().multiply(simpleItem.getUomRatio2())
                            .setScale(pricePlace, BigDecimal.ROUND_HALF_UP));
                } else {
                    priPriceRpcDTO.setPrice(row.getPrice()
                            .multiply(simpleItem.getUomRatio2())
                            .setScale(pricePlace, BigDecimal.ROUND_HALF_UP));
                }
                return priPriceRpcDTO;
            }).filter(Objects::nonNull).collect(Collectors.toList());
            priceList.addAll(resultPriceList);

        }
    }

    private void queryInvStock(List<AppInvStkRespVO> suppPriorityList, AtomicReference<Map<String, BigDecimal>> stockMap,
                               List<AppInvStkRespVO> whPriorityList) {
        if (stockMap == null) {
            return;
        }
        // 构建库存查询参数
        for (AppInvStkRespVO stk : suppPriorityList) {
            stockMap.updateAndGet(currentMap -> {
                currentMap.put(stk.getItemCode(), new BigDecimal("9999"));
                return currentMap;
            });
        }
        InvWhItemTotalStkRpcParam stockParam = buildStockParam(whPriorityList);
        getWhStockMap(stockParam, stockMap, whPriorityList);
    }

    /**
     * 供应商份额销售价格
     *
     * @param suppAllocMap
     * @param item
     * @param priceList
     * @return
     */
    private ItmPriPriceRpcDtoParam buildSuppPriceParamList(Map<String, List<ScpSupplyAllocationRpcDTO>> suppAllocMap, AppItemActivityItemPriceParamVO item, List<PriPriceRpcDTO> priceList,
                                                           AppInvStkRespVO stkParam) {
        // 结算价格
        List<ScpSupplyAllocationRpcDTO> suppList = suppAllocMap.get(item.getItemCode());
        // 未查询到供应商配额,属于没有配置价格
        if (CollUtil.isEmpty(suppList)) {
            return null;
        }

        //库存查询参数
        stkParam.setItemId(item.getItemId());
        stkParam.setItemCode(item.getItemCode());

        ScpSupplyAllocationRpcDTO supp = suppList.get(0);
        // 供应商未配置销售OU,属于零价格
        if (StrUtil.isBlank(supp.getSaleOuCode())) {
            PriPriceRpcDTO priPriceRpcDTO = new PriPriceRpcDTO();
            priPriceRpcDTO.setItemId(item.getItemId());
            priPriceRpcDTO.setItemCode(item.getItemCode());
            priPriceRpcDTO.setPrice(BigDecimal.ZERO);
            priPriceRpcDTO.setNetPrice(BigDecimal.ZERO);
            priceList.add(priPriceRpcDTO);
            return new ItmPriPriceRpcDtoParam();
        }
        ItmPriPriceRpcDtoParam priceParam = new ItmPriPriceRpcDtoParam();
        priceParam.setPriceCls(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setPriceType(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setItemId(item.getItemId());
        priceParam.setCustCode(supp.getSaleCustCode());
        priceParam.setItemCode(item.getItemCode());
        priceParam.setUom(item.getUom());
        priceParam.setCurrCode(ScpConstant.CNY);
        priceParam.setOuCode(supp.getSaleOuCode());
        return priceParam;
    }

    /**
     * 仓网关系销售价格
     *
     * @param
     * @param item
     * @param priceList
     * @return
     */
    private ItmPriPriceRpcDtoParam buildWhPriceParamList(ScpWhNetRelationRpcDTO supp,
                                                         AppItemActivityItemPriceParamVO item,
                                                         List<PriPriceRpcDTO> priceList, AppInvStkRespVO stkParam) {

        //库存查询参数
        stkParam.setItemId(item.getItemId());
        stkParam.setItemCode(item.getItemCode());
        stkParam.setWhId(supp.getSupplyWhId());
        // 仓网关系未配置销售OU,属于零价格
        if (StrUtil.isBlank(supp.getOuCode())) {
            PriPriceRpcDTO priPriceRpcDTO = new PriPriceRpcDTO();
            priPriceRpcDTO.setItemId(item.getItemId());
            priPriceRpcDTO.setItemCode(item.getItemCode());
            priPriceRpcDTO.setPrice(BigDecimal.ZERO);
            priPriceRpcDTO.setNetPrice(BigDecimal.ZERO);
            priceList.add(priPriceRpcDTO);
            return new ItmPriPriceRpcDtoParam();
        }
        ItmPriPriceRpcDtoParam priceParam = new ItmPriPriceRpcDtoParam();
        priceParam.setPriceCls(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setPriceType(ScpConstant.INTERNAL_SETTLEMENT_PRICE);
        priceParam.setItemId(item.getItemId());
        priceParam.setCustCode(supp.getCustCode());
        priceParam.setItemCode(item.getItemCode());
        priceParam.setUom(item.getUom());
        priceParam.setCurrCode(ScpConstant.CNY);
        priceParam.setOuCode(supp.getOuCode());
        return priceParam;
    }
}
