package com.elitesland.scp.provider.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitesland.scp.ScpApplication;
import com.elitesland.scp.application.facade.vo.boh.StoreReceivePageParam;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderParamVO;
import com.elitesland.scp.application.service.boh.StoreReceiveService;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderConvert;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderDConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDDO;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDO;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDDomainService;
import com.elitesland.scp.domain.service.order.ScpDemandOrderDomainService;
import com.elitesland.scp.dto.order.*;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepoProc;
import com.elitesland.scp.param.ScpOrderUnRecvParamVO;
import com.elitesland.scp.service.order.ScpDemandOrderDRpcService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
@RestController
@RequiredArgsConstructor
@RequestMapping(ScpApplication.URI_PREFIX + ScpDemandOrderDRpcService.PATH)
public class ScpDemandOrderDRpcServiceImpl implements ScpDemandOrderDRpcService {

    private final ScpDemandOrderDomainService scpDemandOrderDomainService;
    private final ScpDemandOrderDDomainService scpDemandOrderDDomainService;
    private final StoreReceiveService storeReceiveService;
    private final ScpDemandOrderDRepo scpDemandOrderDRepo;
    private final ScpDemandOrderRepo scpDemandOrderRepo;
    private final ScpDemandOrderRepoProc scpDemandOrderRepoProc;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> updateDemandOrderDetail(List<ScpDemandOrderDRpcDTO> rpcDTOS) {
        log.info("【updateDemandOrderDetail】更新订货单明细：{}", JSONUtil.toJsonStr(rpcDTOS));
        for (ScpDemandOrderDRpcDTO rpcDTO : rpcDTOS) {
            scpDemandOrderDDomainService.saveDemandOrderD(ScpDemandOrderDConvert.INSTANCE.rpcDtoToSaveVO(rpcDTO));
        }
        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> updateAllocationQuantity(List<ScpDemandOrderDRpcAllocDTO> rpcDTOS) {
        log.info("【updateDemandOrderDetail】更新订货单配货数量：{}", JSONUtil.toJsonStr(rpcDTOS));
        for (ScpDemandOrderDRpcAllocDTO rpcDTO : rpcDTOS) {
            scpDemandOrderDDomainService.updateAllocQuantity(rpcDTO.getId(), rpcDTO.getAllocationQuantity());
        }

        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> updateQuantity(List<ScpDemandOrderDRpcQuantityDTO> rpcDTOS) {
        log.info("【updateDemandOrderDetail】更新订货单发货数量：{}", JSONUtil.toJsonStr(rpcDTOS));
        for (ScpDemandOrderDRpcQuantityDTO rpcDTO : rpcDTOS) {
            scpDemandOrderDDomainService.updateQuantity(rpcDTO.getId(), rpcDTO.getQuantity());
        }
        List<Long> ids = rpcDTOS.stream().map(ScpDemandOrderDRpcQuantityDTO::getId).collect(Collectors.toList());
        List<Long> masIds = scpDemandOrderDDomainService.getMasIdByDId(ids).stream().distinct().collect(Collectors.toList());
        for (Long masId : masIds) {
            scpDemandOrderDomainService.updateDocStatusById(masId, ScpConstant.DELIVERY);
        }
        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<String> updateSrcDoc2No(List<ScpDemandOrderDRpcSrcDoc2NoDTO> rpcDTOS) {
        log.info("更新采购单SRM单据信息：{}", JSONUtil.toJsonStr(rpcDTOS));
        for (ScpDemandOrderDRpcSrcDoc2NoDTO dto : rpcDTOS) {
            String docNo = dto.getSrcDoc2No();
            scpDemandOrderDDomainService.updateSrcDoc2No(dto.getId(), docNo);
        }
        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> updateRecvQty(List<ScpDemandOrderDRpcRecvQtyDTO> rpcDTOS) {
        log.info("【updateDemandOrderDetail】更新订货单收货数量：{}", JSONUtil.toJsonStr(rpcDTOS));
        for (ScpDemandOrderDRpcRecvQtyDTO rpcDTO : rpcDTOS) {
            scpDemandOrderDDomainService.updateRecvQty(rpcDTO.getId(), rpcDTO.getRecvQty());
        }
        List<Long> ids = rpcDTOS.stream().map(ScpDemandOrderDRpcRecvQtyDTO::getId).collect(Collectors.toList());
        List<Long> masIds = scpDemandOrderDDomainService.getMasIdByDId(ids).stream().distinct().collect(Collectors.toList());
        for (Long masId : masIds) {
            scpDemandOrderDomainService.updateDocStatusById(masId, ScpConstant.RECV);
        }
        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> cancelOrder(ScpDemandOrderCancelDTO cancelDTO) {
        log.info("【cancelOrder】订货单取消：{}", JSONUtil.toJsonStr(cancelDTO));
        if (CollUtil.isEmpty(cancelDTO.getDIds())) {
            return ApiResult.ok();
        }
        scpDemandOrderDDomainService.cancelOrderByDIds(cancelDTO.getDIds());
        // 判断是否所有明细行都被取消
        List<ScpDemandOrderDDO> byMasIdAndStatus = scpDemandOrderDRepoProc.findByMasIdAndStatusNotAndProjectFeeFlag(cancelDTO.getId(),
                ScpUdcEnum.DEO_STATUS_CLOSE.getValueCode(),
                false);
        if (CollUtil.isEmpty(byMasIdAndStatus)) {
            // 明细行已经全部取消，则更新主表状态为已取消
            scpDemandOrderRepo.updateDocStatusById(cancelDTO.getId(), ScpUdcEnum.DEO_STATUS_CLOSE.getValueCode());
        }
        return ApiResult.ok();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> payCallback(ScpDemandOrderCancelDTO cancelDTO) {
        log.info("【payCallback】订货单支付回传：{}", JSONUtil.toJsonStr(cancelDTO));
        scpDemandOrderDDomainService.updatePayStatus(cancelDTO.getDIds(), ScpUdcEnum.PAY_STATUS_PAYED.getValueCode());
        scpDemandOrderDomainService.updatePayStatusById(cancelDTO.getId());
        return ApiResult.ok();
    }

    @Override
    public ApiResult<Long> checkUnRecvOrder(ScpOrderUnRecvParamVO paramVO) {
        StoreReceivePageParam storeReceivePageParam = new StoreReceivePageParam();
        storeReceivePageParam.setStatusList(List.of("CE"));
        storeReceivePageParam.setRtType("R");
        storeReceivePageParam.setStoreIds(List.of(paramVO.getStoreId()));
        storeReceivePageParam.setCreateDateTo(paramVO.getDateTime());
        return ApiResult.ok(storeReceiveService.countStoreReceive(storeReceivePageParam));
    }

    @Override
    public ApiResult<List<ScpDemandOrderDRpcDTO>> findByMasId(Long masId) {
        if (masId == null) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "masId不能为空");
        }
        List<ScpDemandOrderDDTO> orderDList = scpDemandOrderDDomainService.findDemandOrderDByMasId(masId);
        if (CollectionUtils.isEmpty(orderDList)) {
            return ApiResult.ok(new ArrayList<>());
        }
        orderDList = orderDList.stream().filter(d -> !d.getProjectFeeFlag()).collect(Collectors.toList());
        return ApiResult.ok(ScpDemandOrderDConvert.INSTANCE.dto2RpcDTO(orderDList));
    }

    @Override
    public ApiResult<List<ScpDemandOrderRpcDTO>> findByDocNos(List<String> docNos) {
        log.info("【findByDocNos】查询门店订货单据信息：{}", JSONUtil.toJsonStr(docNos));
        List<ScpDemandOrderDO> demandOrderByDocNos = scpDemandOrderRepoProc.findDemandOrderByDocNos(docNos);
        List<ScpDemandOrderRpcDTO> demandOrderDRpcDTOS = ScpDemandOrderConvert.INSTANCE.dos2RPCDtos(demandOrderByDocNos);

        return ApiResult.ok(demandOrderDRpcDTOS);
    }
}
