package com.elitesland.scp.domain.service.order;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.el.coordinator.core.common.utils.UUIDUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitesland.sale.api.vo.resp.crm.CustBaseDTO;
import com.elitesland.scp.application.facade.vo.param.order.ScpDemandOrderDPageParamVO;
import com.elitesland.scp.application.facade.vo.resp.order.ScpDemandOrderPayInfoRespVO;
import com.elitesland.scp.application.facade.vo.save.order.ScpDemandOrderDSaveVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.order.ScpDemandOrderDConvert;
import com.elitesland.scp.domain.entity.order.ScpDemandOrderDDO;
import com.elitesland.scp.enums.ScpUdcEnum;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderDDTO;
import com.elitesland.scp.infr.dto.order.ScpDemandOrderRelateDTO;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepo;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderDRepoProc;
import com.elitesland.scp.infr.repo.order.ScpDemandOrderRepoProc;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiSalRpcService;
import com.elitesland.scp.utils.SysUtils;
import com.elitesland.support.provider.item.dto.ItmItemAttachmentProviderDTO;
import com.elitesland.support.provider.item.dto.ItmItemBaseRpcDTO;
import com.elitesland.support.provider.item.param.ItmItemBaseRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpDemandOrderDDomainServiceImpl implements ScpDemandOrderDDomainService {

    private final EntityManager entityManager;
    private final RmiItemService rmiItemService;
    private final RmiSalRpcService rmiSalRpcService;
    private final ScpDemandOrderRepoProc scpDemandOrderRepoProc;
    private final ScpDemandOrderDRepo scpDemandOrderDRepo;
    private final ScpDemandOrderDRepoProc scpDemandOrderDRepoProc;
    @Value("${ext.server-addr}")
    private String serverAddr;

    @Override
    public Long saveDemandOrderD(ScpDemandOrderDSaveVO saveVO) {
        if (saveVO.getId() == null) {
            ScpDemandOrderDDO intentDO = ScpDemandOrderDConvert.INSTANCE.saveVoToDO(saveVO);
            return scpDemandOrderDRepo.save(intentDO).getId();
        } else {
            Optional<ScpDemandOrderDDO> option = scpDemandOrderDRepo.findById(saveVO.getId());
            if (option.isEmpty()) {
                throw new BusinessException("订货订单明细ID：" + saveVO.getId() + "不存在");
            }
            ScpDemandOrderDConvert.INSTANCE.copySaveParamToDo(saveVO, option.get());
            return scpDemandOrderDRepo.save(option.get()).getId();
        }
    }

    @Override
    @SysCodeProc
    public List<ScpDemandOrderDDTO> findDemandOrderDByMasId(Long masId) {
        return scpDemandOrderDRepo.findByMasId(masId).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    @SysCodeProc
    public List<ScpDemandOrderDDTO> findDemandOrderDByMasIdAndProjectFeeFlag(Long masId, Boolean projectFeeFlag) {
        return scpDemandOrderDRepo.findByMasIdAndProjectFeeFlag(masId, projectFeeFlag).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public List<ScpDemandOrderDDO> findDemandOrderDBySpuCodes(Long masId, List<String> spuCode) {
        return scpDemandOrderDRepo.findByMasIdAndSpuItemCodeIn(masId, spuCode);
    }

    @Override
    public Page<ScpDemandOrderDDO> pageDemandOrderDByMasId(ScpDemandOrderDPageParamVO paramVO) {
        Page<ScpDemandOrderDDO> scpDemandOrderDDTOS = scpDemandOrderDRepo.findByMasId(paramVO.getMasId(),
                paramVO.getPageRequest());
        return scpDemandOrderDDTOS;
    }

    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDByIds(Collection<Long> ids) {
        return scpDemandOrderDRepo.findByIdIn(ids).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpDemandOrderDRepoProc.deleteByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updatePlanQtyAndAmtById(Long id, BigDecimal planQty) {
        scpDemandOrderDRepoProc.updatePlanQtyAndAmtById(id, planQty);
    }

    private ScpDemandOrderDSaveVO generateFeeSaveVO(AtomicInteger lineNo,
                                                    ScpDemandOrderDSaveVO oldSaveVO,
                                                    ScpDemandOrderDSaveVO scpDemandOrderDSaveVO,
                                                    Map<String, ItmItemBaseRpcDTO> itemMap,
                                                    Map<Long, List<ItmItemAttachmentProviderDTO>> skuImgMap,
                                                    String itemCode,
                                                    BigDecimal fee) {
        if (scpDemandOrderDSaveVO == null) {
            scpDemandOrderDSaveVO = new ScpDemandOrderDSaveVO();
            ItmItemBaseRpcDTO itmItemBaseRpcDTO = itemMap.get(itemCode);
            scpDemandOrderDSaveVO.setMasId(oldSaveVO.getMasId());
            scpDemandOrderDSaveVO.setDemandQuantity(BigDecimal.ONE);
            scpDemandOrderDSaveVO.setAllocationQuantity(BigDecimal.ONE);
            scpDemandOrderDSaveVO.setAllocationDeQuantity(BigDecimal.ONE);
            scpDemandOrderDSaveVO.setPlanQuantity(BigDecimal.ONE);
            scpDemandOrderDSaveVO.setPrice(fee);
            scpDemandOrderDSaveVO.setPlanAmt(fee);
            scpDemandOrderDSaveVO.setSalePrice(fee);
            scpDemandOrderDSaveVO.setSaleAmt(fee);
            scpDemandOrderDSaveVO.setSettlementPrice(fee);
            scpDemandOrderDSaveVO.setSettlementAmt(fee);
            scpDemandOrderDSaveVO.setSettlementSalePrice(fee);
            scpDemandOrderDSaveVO.setSettlementSaleAmt(fee);
            scpDemandOrderDSaveVO.setTechFee(BigDecimal.ZERO);
            scpDemandOrderDSaveVO.setMarketingFee(BigDecimal.ZERO);
            scpDemandOrderDSaveVO.setOperationFee(BigDecimal.ZERO);
            scpDemandOrderDSaveVO.setTefPrice(BigDecimal.ZERO);
            scpDemandOrderDSaveVO.setMefPrice(BigDecimal.ZERO);
            scpDemandOrderDSaveVO.setOefPrice(BigDecimal.ZERO);
            scpDemandOrderDSaveVO.setCurrency(ScpConstant.CNY);
            scpDemandOrderDSaveVO.setPreRootUuid(UUIDUtil.getUUID());
            scpDemandOrderDSaveVO.setItemId(itmItemBaseRpcDTO.getId());
            scpDemandOrderDSaveVO.setItemCode(itmItemBaseRpcDTO.getItemCode());
            scpDemandOrderDSaveVO.setItemName(itmItemBaseRpcDTO.getItemName());
            scpDemandOrderDSaveVO.setSpuItemCode(itmItemBaseRpcDTO.getSpuCode());
            scpDemandOrderDSaveVO.setSpuItemName(itmItemBaseRpcDTO.getSpuName());
            scpDemandOrderDSaveVO.setUom2(itmItemBaseRpcDTO.getUom());
            scpDemandOrderDSaveVO.setSaleCustCode(oldSaveVO.getSaleCustCode());
            scpDemandOrderDSaveVO.setUom2Name(itmItemBaseRpcDTO.getUomName());
            scpDemandOrderDSaveVO.setUnit(itmItemBaseRpcDTO.getUom2());
            scpDemandOrderDSaveVO.setUnitName(itmItemBaseRpcDTO.getUom2Name());
            scpDemandOrderDSaveVO.setPayStatus(ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode());
            scpDemandOrderDSaveVO.setItemType(itmItemBaseRpcDTO.getItemType2());
            scpDemandOrderDSaveVO.setSupplyType(oldSaveVO.getSupplyType());
            scpDemandOrderDSaveVO.setSuppWhId(oldSaveVO.getSuppWhId());
            scpDemandOrderDSaveVO.setSuppWhName(oldSaveVO.getSuppWhName());
            scpDemandOrderDSaveVO.setSuppWhCode(oldSaveVO.getSuppWhCode());
            scpDemandOrderDSaveVO.setRatio(BigDecimal.valueOf(100));
            scpDemandOrderDSaveVO.setUomRatio(itmItemBaseRpcDTO.getUomRatio2());
            scpDemandOrderDSaveVO.setQty2(itmItemBaseRpcDTO.getUomRatio2());
            scpDemandOrderDSaveVO.setLineNo(new BigDecimal(lineNo.getAndIncrement()));
            scpDemandOrderDSaveVO.setFreightLineFlag(Boolean.TRUE);
            scpDemandOrderDSaveVO.setIsCalculated(Boolean.TRUE);
            scpDemandOrderDSaveVO.setOuId(oldSaveVO.getOuId());
            scpDemandOrderDSaveVO.setOuCode(oldSaveVO.getOuCode());
            scpDemandOrderDSaveVO.setOuName(oldSaveVO.getOuName());
            scpDemandOrderDSaveVO.setSaleOuName(oldSaveVO.getSaleOuName());
            scpDemandOrderDSaveVO.setSaleOuCode(oldSaveVO.getSaleOuCode());
            scpDemandOrderDSaveVO.setSaleCustCode(oldSaveVO.getSaleCustCode());
            scpDemandOrderDSaveVO.setDeliveryType(oldSaveVO.getDeliveryType());
            scpDemandOrderDSaveVO.setProjectFeeFlag(Boolean.TRUE);
            scpDemandOrderDSaveVO.setCostType("SF");

            List<ItmItemAttachmentProviderDTO> imageInfo = skuImgMap.get(itmItemBaseRpcDTO.getId());
            if (CollUtil.isNotEmpty(imageInfo)) {
                ItmItemAttachmentProviderDTO attachmentProviderDTO = imageInfo.stream().filter(i -> i.getMajor() != null && i.getMajor()).findFirst().orElse(null);
                if (attachmentProviderDTO == null) {
                    attachmentProviderDTO = imageInfo.get(0);
                }
                String fileUrl = serverAddr + "/coordinator/el-fsm-service/api/fsm/download/" + attachmentProviderDTO.getFileCode();
                scpDemandOrderDSaveVO.setImgUrl(fileUrl);
            }
        } else {
            scpDemandOrderDSaveVO.setPrice(scpDemandOrderDSaveVO.getPrice().add(fee));
            scpDemandOrderDSaveVO.setPlanAmt(scpDemandOrderDSaveVO.getPlanAmt().add(fee));
            scpDemandOrderDSaveVO.setSettlementPrice(scpDemandOrderDSaveVO.getSettlementPrice().add(fee));
            scpDemandOrderDSaveVO.setSettlementAmt(scpDemandOrderDSaveVO.getSettlementAmt().add(fee));
            scpDemandOrderDSaveVO.setSalePrice(scpDemandOrderDSaveVO.getSalePrice().add(fee));
            scpDemandOrderDSaveVO.setSaleAmt(scpDemandOrderDSaveVO.getSaleAmt().add(fee));
            scpDemandOrderDSaveVO.setSettlementSalePrice(scpDemandOrderDSaveVO.getSettlementSalePrice().add(fee));
            scpDemandOrderDSaveVO.setSettlementSaleAmt(scpDemandOrderDSaveVO.getSettlementSaleAmt().add(fee));
        }
        return scpDemandOrderDSaveVO;
    }

    /**
     * 批量保存订货订单明细
     * <p>
     * 实现逻辑：
     * 1. 根据客户内外部信息设置支付状态
     * 2. 处理项目费用生成（科技费、营销费、运营费）
     * 3. 分别处理需要更新和新增的订单明细
     * 4. 批量保存订单明细数据
     * 5. 设置明细行的源ID关联关系
     *
     * @param saveVOS 需要保存的订单明细数据列表
     * @param lineNo  行号计数器
     * @param source  数据来源标识
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSave(List<ScpDemandOrderDSaveVO> saveVOS, AtomicInteger lineNo, String source) {
        batchSaveInternal(saveVOS, lineNo, source, false);
    }

    /**
     * 批量保存甲指乙采商品
     * <p>
     * 实现逻辑：
     * 1. 统一设置支付状态为待支付
     * 2. 不处理项目费用
     * 3. 分别处理需要更新和新增的订单明细
     * 4. 批量保存订单明细数据
     * 5. 设置明细行的源ID关联关系
     *
     * @param saveVOS 需要保存的订单明细数据列表
     * @param lineNo  行号计数器
     * @param source  数据来源标识
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchSaveDboDemanOrderD(List<ScpDemandOrderDSaveVO> saveVOS, AtomicInteger lineNo, String source) {
        batchSaveInternal(saveVOS, lineNo, source, true);
    }

    /**
     * 批量保存订货订单明细
     * <p>
     * 实现逻辑：
     * 1. 根据是否为甲指乙采类型设置不同的支付状态
     * 2. 对于普通订单，根据客户内外部信息设置支付状态并处理项目费用
     * 3. 对于甲指乙采订单，统一设置为待支付状态
     * 4. 处理项目费用生成（科技费、营销费、运营费）
     * 5. 分别处理需要更新和新增的订单明细
     * 6. 批量保存订单明细数据
     * 7. 设置明细行的源ID关联关系
     *
     * @param saveVOS 需要保存的订单明细数据列表
     * @param lineNo  行号计数器
     * @param source  数据来源标识
     * @param isDbo   是否为甲指乙采类型
     */
    private void batchSaveInternal(List<ScpDemandOrderDSaveVO> saveVOS, AtomicInteger lineNo, String source, boolean isDbo) {
        if (isDbo) {
            // 甲指乙采订单处理逻辑
            for (ScpDemandOrderDSaveVO row : saveVOS) {
                row.setPayStatus(ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode());
            }
        } else {
            // 普通订单处理逻辑
            // 获取saveVOS中所有不重复的saleCustCode
            List<String> custCodes = saveVOS.stream().map(ScpDemandOrderDSaveVO::getSaleCustCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            // 根据custCodes获取对应的客户信息
            Map<String, CustBaseDTO> custMap = rmiSalRpcService.getCustBaseMapByCode(custCodes);
            // 遍历saveVOS，设置payStatus
            for (ScpDemandOrderDSaveVO row : saveVOS) {
                if (row.getDeliveryType().equals(ScpUdcEnum.ITM_ITM_TYPE2_140.getValueCode())) {
                    row.setPayStatus(ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode());
                    continue;
                }
                log.info("保存订货单明细：销售公司编码：{},采购公司编码：{},客户编码：{}", row.getSaleOuCode(), row.getOuCode(), row.getSaleCustCode());
                if (StrUtil.isBlank(row.getSaleOuCode()) && StrUtil.isBlank(row.getOuCode())) {
                    row.setPayStatus(ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode());
                } else {
                    CustBaseDTO custBaseInfoDTO = custMap.get(row.getSaleCustCode());
                    log.info("客户编码：{}，客户内外信息：{}", custBaseInfoDTO.getCustCode(), custBaseInfoDTO.getInOutCust());
                    String payStatus = (!"OUT".equals(custBaseInfoDTO.getInOutCust())) ? ScpUdcEnum.PAY_STATUS_NO_PAY.getValueCode() : ScpUdcEnum.PAY_STATUS_WAIT_PAY.getValueCode();
                    row.setPayStatus(payStatus);
                }
            }
        }

        List<ScpDemandOrderDSaveVO> result = new ArrayList<>();

        if (isDbo) {
            // 甲指乙采不处理项目费用
            result.addAll(saveVOS);
        } else {
            // 普通订单处理项目费用
            // 将saveVOS按照 供应仓库/供应商、采购公司 进行分组
            Map<String, List<ScpDemandOrderDSaveVO>> groupMap = saveVOS.stream()
                    .filter(row -> row != null
                            && row.getIsProjFeeCharged() != null
                            && row.getIsProjFeeCharged())
                    .collect(Collectors.groupingBy(v -> v.getSuppWhCode() + "|" + v.getOuCode()));

            if (CollUtil.isNotEmpty(groupMap)) {
                String technologyCostsItemCode = SysUtils.getTechnologyCosts();
                String marketingCostsItemCode = SysUtils.getMarketingCosts();
                String operatingCostsItemCode = SysUtils.getOperatingCosts();

                ItmItemBaseRpcParam itmItemBaseRpcParam = new ItmItemBaseRpcParam();
                itmItemBaseRpcParam.setItemCodes(new ArrayList<>() {{
                    add(technologyCostsItemCode);
                    add(marketingCostsItemCode);
                    add(operatingCostsItemCode);
                }});
                // 根据itemCodes获取对应的商品信息
                List<ItmItemBaseRpcDTO> itemBaseRpcDTOList = rmiItemService.findItemBaseRpcDtoByParam(itmItemBaseRpcParam);
                if (CollUtil.isEmpty(itemBaseRpcDTOList)) {
                    throw new BusinessException("项目费用商品编码不存在");
                }

                List<Long> fetchImageIds = itemBaseRpcDTOList.stream().map(ItmItemBaseRpcDTO::getId).toList();
                //获取商品图片信息
                List<ItmItemAttachmentProviderDTO> skuImgList = rmiItemService.findSkuImgByItemIds(fetchImageIds);
                Map<Long, List<ItmItemAttachmentProviderDTO>> skuImgMap = new HashMap<>();
                if (CollUtil.isNotEmpty(skuImgList)) {
                    skuImgMap = skuImgList.stream().collect(Collectors.groupingBy(ItmItemAttachmentProviderDTO::getItemId));
                }

                // 将itemBaseRpcDTOList转换为Map，方便后续使用
                Map<String, ItmItemBaseRpcDTO> itemMap = itemBaseRpcDTOList.stream().collect(Collectors.toMap(ItmItemBaseRpcDTO::getItemCode, itmItemBaseRpcDTO -> itmItemBaseRpcDTO));
                for (Map.Entry<String, List<ScpDemandOrderDSaveVO>> entry : groupMap.entrySet()) {
                    List<ScpDemandOrderDSaveVO> value = entry.getValue();
                    Map<String, ScpDemandOrderDSaveVO> map = new HashMap<>();
                    // 总金额
                    AtomicReference<BigDecimal> amt = new AtomicReference<>(BigDecimal.ZERO);
                    // 货款总金额
                    AtomicReference<BigDecimal> saleAmt = new AtomicReference<>(BigDecimal.ZERO);
                    for (ScpDemandOrderDSaveVO oldSaveVO : value) {
                        saleAmt.updateAndGet(v -> v.add(oldSaveVO.getSettlementSaleAmt()));
                        amt.updateAndGet(v -> v.add(oldSaveVO.getSettlementAmt()));

                        // 生成科技费明细行
                        ScpDemandOrderDSaveVO scpDemandOrderDSaveVO = generateFeeSaveVO(lineNo, oldSaveVO, map.get(technologyCostsItemCode), itemMap, skuImgMap, technologyCostsItemCode, oldSaveVO.getTechFee());
                        scpDemandOrderDSaveVO.setTefPrice(scpDemandOrderDSaveVO.getTefPrice().add(oldSaveVO.getTefPrice()));
                        scpDemandOrderDSaveVO.setTechFee(scpDemandOrderDSaveVO.getTechFee().add(oldSaveVO.getTechFee()));
                        scpDemandOrderDSaveVO.setSaleOuCode(oldSaveVO.getTefFeeOuCode());
                        scpDemandOrderDSaveVO.setSaleOuName(oldSaveVO.getTefFeeOuName());
                        scpDemandOrderDSaveVO.setTefFeeOuCode(oldSaveVO.getTefFeeOuCode());
                        map.put(technologyCostsItemCode, scpDemandOrderDSaveVO);

                        // 生成营销费明细行
                        ScpDemandOrderDSaveVO scpDemandOrderDSaveVO1 = generateFeeSaveVO(lineNo, oldSaveVO, map.get(marketingCostsItemCode), itemMap, skuImgMap, marketingCostsItemCode, oldSaveVO.getMarketingFee());
                        scpDemandOrderDSaveVO1.setMefPrice(scpDemandOrderDSaveVO1.getMefPrice().add(oldSaveVO.getMefPrice()));
                        scpDemandOrderDSaveVO1.setMarketingFee(scpDemandOrderDSaveVO1.getMarketingFee().add(oldSaveVO.getMarketingFee()));
                        scpDemandOrderDSaveVO1.setSaleOuCode(oldSaveVO.getMefFeeOuCode());
                        scpDemandOrderDSaveVO1.setSaleOuName(oldSaveVO.getMefFeeOuName());
                        scpDemandOrderDSaveVO1.setMefFeeOuCode(oldSaveVO.getMefFeeOuCode());
                        map.put(marketingCostsItemCode, scpDemandOrderDSaveVO1);

                        // 生成运营费明细行
                        ScpDemandOrderDSaveVO scpDemandOrderDSaveVO2 = generateFeeSaveVO(lineNo, oldSaveVO, map.get(operatingCostsItemCode), itemMap, skuImgMap, operatingCostsItemCode, oldSaveVO.getOperationFee());
                        scpDemandOrderDSaveVO2.setOefPrice(scpDemandOrderDSaveVO2.getOefPrice().add(oldSaveVO.getOefPrice()));
                        scpDemandOrderDSaveVO2.setOperationFee(scpDemandOrderDSaveVO2.getOperationFee().add(oldSaveVO.getOperationFee()));
                        scpDemandOrderDSaveVO2.setSaleOuCode(oldSaveVO.getOefFeeOuCode());
                        scpDemandOrderDSaveVO2.setSaleOuName(oldSaveVO.getOefFeeOuName());
                        scpDemandOrderDSaveVO2.setOefFeeOuCode(oldSaveVO.getOefFeeOuCode());
                        map.put(operatingCostsItemCode, scpDemandOrderDSaveVO2);
                    }

                    List<ScpDemandOrderDSaveVO> projectFeeList = new ArrayList<>();
                    if (!map.isEmpty()) {
                        // 将生成的明细行添加到saveVOS中
                        projectFeeList = map.values().stream().filter(Objects::nonNull).toList();
                        for (ScpDemandOrderDSaveVO scpDemandOrderDSaveVO : projectFeeList) {
                            saleAmt.updateAndGet(v -> v.add(scpDemandOrderDSaveVO.getSettlementSaleAmt()));
                        }
                    }
                    // 如果amt2比saleAmt2多，则取value里settlementAmt最多的一行去做减去差值,然后再
                    if (amt.get().compareTo(saleAmt.get()) != 0) {
                        BigDecimal diff = amt.get().subtract(saleAmt.get());
                        // 查找settlementAmt最大的元素，避免完整排序
                        value.stream().max(Comparator.comparing(ScpDemandOrderDSaveVO::getSettlementAmt))
                                .ifPresent(max -> max.setSettlementAmt(max.getSettlementAmt().subtract(diff)));
                    }
                    result.addAll(value);

                    if (CollUtil.isNotEmpty(projectFeeList)) {
                        result.addAll(projectFeeList);
                    }
                }
            } else {
                result.addAll(saveVOS);
            }
        }

        // 将saveVOS中id不为空的元素提取出来，保存到数据库
        List<ScpDemandOrderDSaveVO> updateList = result.stream().filter(saveVO -> ObjectUtil.isNotEmpty(saveVO.getId())).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(updateList)) {
            scpDemandOrderDRepo.saveAll(ScpDemandOrderDConvert.INSTANCE.saveVoListToDOList(updateList));
        }
        // 将saveVOS中id不为空的元素提取出来，保存到数据库
        List<ScpDemandOrderDSaveVO> saveList = result.stream().filter(saveVO -> ObjectUtil.isEmpty(saveVO.getId())).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(saveList)) {
            this.batchInsert(ScpDemandOrderDConvert.INSTANCE.saveVoListToDOList(saveList));
        }
        // 查询saveVOS中第一个元素的masId对应的未推送的明细行
        List<ScpDemandOrderDDO> saveRootList = scpDemandOrderDRepoProc.findByMasIdAndIsPushed(saveVOS.get(0).getMasId(), Boolean.FALSE);
        // 将saveRootList按照groupByUuid进行分组，并取第一个元素，转换为Map
        Map<String, ScpDemandOrderDDO> rootItemMap = saveRootList.stream()
                .collect(Collectors.groupingBy(ScpDemandOrderDDO::groupByUuid))
                .values().stream()
                .map(list -> list.get(0))
                .collect(Collectors.toMap(ScpDemandOrderDDO::groupByUuid, Function.identity(),
                        (t1, t2) -> t1));
        // 遍历saveRootList，设置sourceId
        saveRootList.forEach(row -> {
            ScpDemandOrderDDO item = rootItemMap.get(row.groupByUuid());
            row.setSourceId(item != null ? item.getId() : row.getId());
        });
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateSuppAllocPlanQty(Long demandId) {
        scpDemandOrderDRepo.updateSuppAllocPlanQty(demandId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateIsPushed(List<Long> dIds, String msg) {
        String syncMsg = StrUtil.isBlank(msg) ? "成功" : msg;
        scpDemandOrderDRepo.updateIsPushed(dIds, syncMsg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateSyncMsg(List<Long> dIds, String msg) {
        scpDemandOrderDRepo.updateSyncMsg(dIds, msg);
    }

    @Override
    public void updateSrcInfo(Long srcDocId, String srcDocNo, Long dId, Integer srcDocLineNo) {
        scpDemandOrderDRepo.updateSrcDocInfoByIds(srcDocId, srcDocNo, dId, srcDocLineNo);
    }

    @Override
    public void updatePOSrcInfo(Long id, Long srcDocId, String srcDocNo, String srcLineNO) {
        scpDemandOrderDRepoProc.updateSrcDocInfoById(id, srcDocId, srcDocNo, srcLineNO, "PO");
    }

    @Override
    public void updateSrcDoc2No(Long id, String srcDocNo) {
        scpDemandOrderDRepoProc.updateSrcDoc2No(id, srcDocNo);
    }

    @Override
    public void clearSrcDocInfo(Long docId) {
        scpDemandOrderDRepoProc.clearSrcDocInfo(docId);
    }

    @Override
    public void resetSrcDocQuantity(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.resetSrcDocQuantity(dId, qty);
    }

    @Override
    public void updateAllocQuantity(Long dId, BigDecimal qty) {
        scpDemandOrderDRepoProc.updateAllocQtyById(dId, qty);
    }

    @Override
    public void updateQuantity(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.updateQtyById(dId, qty);
    }

    @Override
    public void updateErrorMsg(Long dId, String errorMsg) {
        scpDemandOrderDRepoProc.updateErrorMsgById(dId, errorMsg);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteUnPushedItem(Long masId) {
        scpDemandOrderDRepo.deleteByMasIdAndIsPushed(masId, Boolean.FALSE);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBySourceIds(List<Long> sourceIds) {
        scpDemandOrderDRepoProc.deleteBySourceIds(sourceIds);
    }

    @Override
    public BigDecimal findMaxLineNoPushedByMasId(Long masId) {
        return scpDemandOrderDRepoProc.findPushedItemMasId(masId);
    }

    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDByMasIds(List<Long> masIds) {
        return scpDemandOrderDRepo.findByMasIdIn(masIds).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public List<Long> getMasIdByDId(List<Long> dIds) {
        return scpDemandOrderDRepoProc.getMasIdByDId(dIds);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancelOrderByDIds(List<Long> dIds) {
        Optional<ScpDemandOrderDDO> byId = scpDemandOrderDRepo.findById(dIds.get(0));
        if (byId.isPresent()) {
            ScpDemandOrderDDO scpDemandOrderDDO = byId.get();
            String srcDocNo = scpDemandOrderDDO.getSrcDocNo();
            if (srcDocNo != null) {
                scpDemandOrderDRepoProc.updateStatusBySrcDocNo(srcDocNo, ScpUdcEnum.DEO_STATUS_CLOSE.getValueCode());
            } else {
                // 标记明细行取消
                scpDemandOrderDRepo.cancelOrderByDIds(dIds, ScpUdcEnum.DEO_STATUS_CLOSE.getValueCode());
            }
        }
    }

    @Override
    public List<ScpDemandOrderPayInfoRespVO> payInfo(Long masId) {
        return scpDemandOrderDRepoProc.findPayInfo(masId);
    }

    @Override
    public void updateRecvQty(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.updateRecvQtyById(dId, qty);
    }

    @Override
    public void updateReturnQty(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.updateReturnQtyById(dId, qty);
    }

    @Override
    public void updateCompensationQty(Long dId, BigDecimal qty) {
        scpDemandOrderDRepo.updateCompensateQtyById(dId, qty);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void initAllocQuantity(Long masId) {
        scpDemandOrderDRepoProc.initAllocQuantity(masId);
    }

    @Override
    public void updatePayStatus(List<Long> ids, String payStatus) {
        scpDemandOrderDRepo.updatePayStatus(ids, payStatus);
    }

    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDBySrcDocId(Long srcDocId) {
        return scpDemandOrderDRepo.findBySrcDocId(srcDocId).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public List<ScpDemandOrderDDTO> findDemandOrderDBySrcDocIdAndProjectFeeFlag(Long srcDocId, Boolean projectFeeFlag) {
        return scpDemandOrderDRepo.findBySrcDocIdAndProjectFeeFlag(srcDocId, false).stream().map(ScpDemandOrderDConvert.INSTANCE::doToDto)
                .collect(Collectors.toList());
    }

    @Override
    public LocalDateTime findDocCreateTimeBySrcDocId(Long srcDocId) {
        return scpDemandOrderDRepoProc.findDocCreateTimeBySrcDocId(srcDocId);
    }

    @Override
    public List<ScpDemandOrderRelateDTO> findRelateOrderByMasId(Long masId) {
        return scpDemandOrderDRepoProc.findRelateOrderByMasId(masId);
    }


    public void batchInsert(List<ScpDemandOrderDDO> dataList) {
        int index = 0;
        int batchSize = 500;
        for (ScpDemandOrderDDO data : dataList) {
            entityManager.persist(data);
            if (batchSize > 1) {
                // 开启批量
                index++;
                if (index % batchSize == 0) {
                    entityManager.flush();
                    entityManager.clear();
                }
            }
        }
        if (!dataList.isEmpty()) {
            entityManager.flush();
        }
    }
}
