package com.elitesland.scp.application.service.minOrder;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.PageUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.scp.application.facade.vo.param.minOrder.ScpMinOrderSettingImportStoreParamVO;
import com.elitesland.scp.application.facade.vo.param.minOrder.ScpMinOrderSettingPageParamVO;
import com.elitesland.scp.application.facade.vo.param.minOrder.ScpMinOrderSettingStatusParamVO;
import com.elitesland.scp.application.facade.vo.param.minOrder.ScpMinOrderSettingStoreParamVO;
import com.elitesland.scp.application.facade.vo.resp.minOrder.ScpMinOrderSettingItemRespVO;
import com.elitesland.scp.application.facade.vo.resp.minOrder.ScpMinOrderSettingPageRespVO;
import com.elitesland.scp.application.facade.vo.resp.minOrder.ScpMinOrderSettingRespVO;
import com.elitesland.scp.application.facade.vo.resp.minOrder.ScpMinOrderSettingStoreRespVO;
import com.elitesland.scp.application.facade.vo.save.minOrder.ScpMinOrderSettingItemSaveVO;
import com.elitesland.scp.application.facade.vo.save.minOrder.ScpMinOrderSettingRegionSaveVO;
import com.elitesland.scp.application.facade.vo.save.minOrder.ScpMinOrderSettingSaveVO;
import com.elitesland.scp.application.facade.vo.save.minOrder.ScpMinOrderSettingStoreSaveVO;
import com.elitesland.scp.common.ScpConstant;
import com.elitesland.scp.domain.convert.minOrder.ScpMinOrderSettingConvert;
import com.elitesland.scp.domain.entity.minOrder.ScpMinOrderSettingDO;
import com.elitesland.scp.domain.entity.minOrder.ScpMinOrderSettingRedisDO;
import com.elitesland.scp.domain.service.minOrder.ScpMinOrderSettingDomainService;
import com.elitesland.scp.domain.service.minOrder.ScpMinOrderSettingItemDomainService;
import com.elitesland.scp.domain.service.minOrder.ScpMinOrderSettingRegionDomainService;
import com.elitesland.scp.domain.service.minOrder.ScpMinOrderSettingStoreDomainService;
import com.elitesland.scp.infr.repo.minOrder.ScpMinOrderSettingRepo;
import com.elitesland.scp.rmi.RmiItemService;
import com.elitesland.scp.rmi.RmiOrgStoreRpcService;
import com.elitesland.scp.rmi.RmiSysUDCService;
import com.elitesland.scp.rmi.RmiSysUserRpcService;
import com.elitesland.support.provider.item.dto.ItmItemRpcDTO;
import com.elitesland.support.provider.item.dto.SpuImageDTO;
import com.elitesland.support.provider.item.param.ItmItemRpcDtoParam;
import com.elitesland.support.provider.org.dto.OrgStoreBaseRpcDTO;
import com.elitesland.support.provider.org.param.OrgStoreBasePageRpcParam;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ScpMinOrderSettingServiceImpl implements ScpMinOrderSettingService {
    private final RmiSysUserRpcService rmiSysUserRpcService;
    private final ScpMinOrderSettingDomainService scpMinOrderSettingDomainService;
    private final ScpMinOrderSettingItemDomainService scpMinOrderSettingItemDomainService;
    private final ScpMinOrderSettingRegionDomainService scpMinOrderSettingRegionDomainService;
    private final ScpMinOrderSettingStoreDomainService scpMinOrderSettingStoreDomainService;
    private final RmiOrgStoreRpcService rmiOrgStoreRpcService;
    private final RedisTemplate<String, Object> redisClient;
    private final ScpMinOrderSettingRepo scpMinOrderSettingRepo;
    private final RmiItemService rmiItemService;
    private final RmiSysUDCService rmiSysUDCService;

    /**
     * 校验参数
     *
     * @param saveVO
     */
    private static void checkParam(ScpMinOrderSettingSaveVO saveVO) {
        if (CollUtil.isEmpty(saveVO.getRegionList())) {
            throw new BusinessException("区域不能为空");
        }
        if (CollUtil.isEmpty(saveVO.getItemList())) {
            throw new BusinessException("商品不能为空");
        }
    }

    @Override
    public PagingVO<ScpMinOrderSettingPageRespVO> page(ScpMinOrderSettingPageParamVO paramVO) {
        Set<Long> masIds = new HashSet<>();
        if (StrUtil.isNotBlank(paramVO.getRegion())) {
            List<Long> regionMasIds = scpMinOrderSettingRegionDomainService.findMasIdByRegionCode(paramVO.getRegion());
            if (CollUtil.isEmpty(regionMasIds)) {
                return new PagingVO<>();
            }
            masIds.addAll(regionMasIds);
        }
        if (paramVO.getStoreCode() != null) {
            List<Long> storeMasIds = scpMinOrderSettingStoreDomainService.findMasIdByStoreCode(paramVO.getStoreCode());
            if (CollUtil.isEmpty(storeMasIds)) {
                return new PagingVO<>();
            }
            intersectSets(masIds, storeMasIds);
        }
        if (paramVO.getItemId() != null) {
            List<Long> itemMasIds = scpMinOrderSettingItemDomainService.findMasIdByItemId(paramVO.getItemId());
            if (CollUtil.isEmpty(itemMasIds)) {
                return new PagingVO<>();
            }
            intersectSets(masIds, itemMasIds);
        }
        paramVO.setIds(new ArrayList<>(masIds));
        return scpMinOrderSettingDomainService.queryMinOrderSettingList(paramVO);
    }

    @Override
    @SysCodeProc
    public ScpMinOrderSettingRespVO findMinOrderSettingById(Long id) {
        Optional<ScpMinOrderSettingRespVO> scpMinOrderSettingRespVO = scpMinOrderSettingDomainService.findMinOrderSettingById(id).map(ScpMinOrderSettingConvert.INSTANCE::dtoToRespVO);
        if (scpMinOrderSettingRespVO.isPresent()) {
            ScpMinOrderSettingRespVO MinOrderSettingRespVO = scpMinOrderSettingRespVO.get();
            MinOrderSettingRespVO.setAllStoreFlag(Boolean.FALSE);
            MinOrderSettingRespVO.setItemRespVOList(scpMinOrderSettingItemDomainService.findByMasId(id));
            MinOrderSettingRespVO.setRegionRespVOList(scpMinOrderSettingRegionDomainService.findByMasId(id));
            MinOrderSettingRespVO.setStoreRespVOList(scpMinOrderSettingStoreDomainService.findByMasId(id));
            if (StrUtil.isNotBlank(MinOrderSettingRespVO.getDocType())) {
                MinOrderSettingRespVO.setDocTypes(StrUtil.split(MinOrderSettingRespVO.getDocType(), ','));
            }
            return MinOrderSettingRespVO;
        }
        return new ScpMinOrderSettingRespVO();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long saveMinOrderSetting(ScpMinOrderSettingSaveVO saveVO) {
        //校验参数
        checkParam(saveVO);
        if (saveVO.getId() == null) {
            if (CollUtil.isEmpty(saveVO.getStoreList()) && !saveVO.getAllStoreFlag()) {
                throw new BusinessException("门店不能为空");
            }
            String orderNo = rmiSysUserRpcService.sysNumberRuleGenerateCode(ScpConstant.FA, new ArrayList<>());
            saveVO.setActivityCode(orderNo);
        } else {
            ScpMinOrderSettingStoreParamVO scpMinOrderSettingStoreParamVO = new ScpMinOrderSettingStoreParamVO();
            scpMinOrderSettingStoreParamVO.setMasIds(List.of(saveVO.getId()));
            scpMinOrderSettingStoreParamVO.setUsed(Boolean.TRUE);
            List<ScpMinOrderSettingStoreRespVO> byParam = scpMinOrderSettingStoreDomainService.findByParam(scpMinOrderSettingStoreParamVO);
            if (CollUtil.isNotEmpty(byParam)) {
                throw new BusinessException("活动已被引用，不允许编辑");
            }
        }
        //保存表头
        Long masId = scpMinOrderSettingDomainService.saveMinOrderSetting(saveVO);
        scpMinOrderSettingItemDomainService.deleteByMasId(masId);
        scpMinOrderSettingRegionDomainService.deleteByMasId(masId);
        //保存活动区域
        saveVO.getRegionList().forEach(row -> row.setMasId(masId));
        scpMinOrderSettingRegionDomainService.batchSaveSettingRegion(saveVO.getRegionList());
        //保存商品
        //查询商品信息
        List<String> itemCodes = saveVO.getItemList().stream().map(ScpMinOrderSettingItemSaveVO::getItemCode).distinct().collect(Collectors.toList());
        ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
        itmItemRpcDtoParam.setItemCodes(itemCodes);
        List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
        Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));
        saveVO.getItemList().forEach(row -> {
            row.setMasId(masId);
            if (itemRpcDTOMap.containsKey(row.getItemCode())) {
                row.setItemId(itemRpcDTOMap.get(row.getItemCode()).getId());
            }
        });
        scpMinOrderSettingItemDomainService.batchSaveMinOrderSettingItem(saveVO.getItemList());

        //保存门店
        scpMinOrderSettingStoreDomainService.deleteByMasId(masId);
        if (CollUtil.isNotEmpty(saveVO.getStoreList()) && !saveVO.getAllStoreFlag()) {
            saveVO.getStoreList().forEach(row -> {
                row.setMasId(masId);
                row.setId(null);
            });
            scpMinOrderSettingStoreDomainService.batchSaveMinOrderSettingStore(saveVO.getStoreList());
            refreshRedisValue(saveVO, masId, saveVO.getStoreList(), itemRpcDTOMap);
        }
        if (CollUtil.isEmpty(saveVO.getStoreList()) && saveVO.getAllStoreFlag()) {
            saveDefaultStore(saveVO, masId, itemRpcDTOMap);
        }
        String key = ScpConstant.MIN_ORDER_SETTING + masId;
        if (!LocalDateTime.now().isAfter(saveVO.getValidTo())) {
            long between = ChronoUnit.MINUTES.between(LocalDateTime.now(), saveVO.getValidTo());
            LocalDateTime validFrom = saveVO.getValidFrom();
            String jsonString = JSON.toJSONString(validFrom);
            redisClient.opsForValue().set(key, jsonString, between, TimeUnit.MINUTES);
        }
        scpMinOrderSettingDomainService.enableMinOrderSetting(List.of(masId), Boolean.TRUE);
        return masId;
    }

    /**
     * 刷新起订活动缓存
     *
     * @param saveVO
     * @param masId
     */
    private void refreshRedisValue(ScpMinOrderSettingSaveVO saveVO, Long masId, List<ScpMinOrderSettingStoreSaveVO> storeList, Map<String, ItmItemRpcDTO> itemRpcDTOMap) {
        Map<String, String> redisMap = new HashMap<>();
        List<ScpMinOrderSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = new ArrayList<>();
        for (ScpMinOrderSettingItemSaveVO item : saveVO.getItemList()) {
            ScpMinOrderSettingRedisDO.ActivityDO.InnerItemDO innerItemDO = new ScpMinOrderSettingRedisDO.ActivityDO.InnerItemDO();
            innerItemDO.setItemCode(item.getItemCode());
            innerItemDO.setMasId(masId);
            innerItemDO.setMinNum(item.getMinNum());
            innerItemDO.setItemType2(item.getItemType2());
            if (itemRpcDTOMap.containsKey(item.getItemCode())) {
                ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(item.getItemCode());
                List<SpuImageDTO> spuAttchmentList = itmItemRpcDTO.getSpuImageList();
                if (CollUtil.isNotEmpty(spuAttchmentList)) {
                    Optional<SpuImageDTO> first = spuAttchmentList.stream().filter(SpuImageDTO::getMajor).findFirst();
                    innerItemDO.setImgUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                }
                innerItemDO.setItemId(itmItemRpcDTO.getId());
                innerItemDO.setItemName(itmItemRpcDTO.getItemName());
            }
            innerItemDOS.add(innerItemDO);
        }
        for (ScpMinOrderSettingStoreSaveVO store : storeList) {
            ScpMinOrderSettingRedisDO scpMinOrderSettingRedisDO = new ScpMinOrderSettingRedisDO();
            Object obj = redisClient.opsForValue().get(ScpConstant.MIN_ORDER_SETTING + store.getStoreCode());
            if (obj == null) {
                ScpMinOrderSettingRedisDO.ActivityDO activityDO1 = createActivityDO(masId, saveVO, store, innerItemDOS);
                scpMinOrderSettingRedisDO.setActivityList(List.of(activityDO1));
            } else {
                String str = (String) obj;
                var setting = JSON.parseObject(str, ScpMinOrderSettingRedisDO.class);
                List<ScpMinOrderSettingRedisDO.ActivityDO> activityList = setting.getActivityList();
                Boolean flag = Boolean.FALSE;
                for (ScpMinOrderSettingRedisDO.ActivityDO inActivityDO : activityList) {
                    if (inActivityDO.getActivityId().equals(masId)) {
                        inActivityDO.setActivityId(masId);
                        inActivityDO.setActivityCode(saveVO.getActivityCode());
                        inActivityDO.setDocTypes(saveVO.getDocTypes());
                        inActivityDO.setMaxNum(store.getMaxNum());
                        inActivityDO.setActiveNum(0);
                        inActivityDO.setValidTo(saveVO.getValidTo());
                        inActivityDO.setValidFrom(saveVO.getValidFrom());
                        inActivityDO.setInnerItemDOS(innerItemDOS);
                        flag = Boolean.TRUE;
                    }
                }
                if (!flag) {
                    ScpMinOrderSettingRedisDO.ActivityDO activityDO1 = createActivityDO(masId, saveVO, store, innerItemDOS);
                    activityList.add(activityDO1);
                }
                scpMinOrderSettingRedisDO.setActivityList(activityList);
            }
            String key = ScpConstant.MIN_ORDER_SETTING + store.getStoreCode();
            redisMap.put(key, JSONUtil.toJsonStr(scpMinOrderSettingRedisDO));
        }
        redisClient.opsForValue().multiSet(redisMap);
    }

    // 提取重复代码
    private ScpMinOrderSettingRedisDO.ActivityDO createActivityDO(Long masId, ScpMinOrderSettingSaveVO saveVO, ScpMinOrderSettingStoreSaveVO store, List<ScpMinOrderSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS) {
        ScpMinOrderSettingRedisDO.ActivityDO activityDO = new ScpMinOrderSettingRedisDO.ActivityDO();
        activityDO.setInnerItemDOS(innerItemDOS);
        activityDO.setActivityId(masId);
        activityDO.setActivityCode(saveVO.getActivityCode());
        activityDO.setDocTypes(saveVO.getDocTypes());
        activityDO.setMaxNum(store.getMaxNum());
        activityDO.setActiveNum(0);
        activityDO.setValidFrom(saveVO.getValidFrom());
        activityDO.setValidTo(saveVO.getValidTo());
        return activityDO;
    }

    /**
     * 保存默认的门店配置
     *
     * @param saveVO 起订设置保存对象，包含区域、品牌等信息
     * @param masId  主配置ID
     *               <p>
     *               此方法主要用于保存SCP分配设置中的默认门店配置它首先校验最大起订次数是否为空，
     *               然后根据提供的区域和品牌列表查询符合条件的门店，并将这些门店信息保存到数据库中
     */
    private void saveDefaultStore(ScpMinOrderSettingSaveVO saveVO, Long masId, Map<String, ItmItemRpcDTO> itemRpcDTOMap) {
        if (saveVO.getMaxNum() == null) {
            throw new BusinessException("最大起订次数不能为空");
        }
        List<String> regionList = saveVO.getRegionList().stream().map(ScpMinOrderSettingRegionSaveVO::getRegion).distinct().collect(Collectors.toList());
        OrgStoreBasePageRpcParam orgStoreBasePageRpcParam = new OrgStoreBasePageRpcParam();
        orgStoreBasePageRpcParam.setRegionList(regionList);

        Long count = rmiOrgStoreRpcService.countBaseStoreByParam(orgStoreBasePageRpcParam);
        if (count == null || count == 0) {
            log.info("saveDefaultStore查询符合条件门店数据为空");
            return;
        }
        int totalPage = PageUtil.totalPage(count, 1000);
        for (int i = 1; i <= totalPage; i++) {
            orgStoreBasePageRpcParam.setSize(1000);
            orgStoreBasePageRpcParam.setCurrent(i);
            PagingVO<OrgStoreBaseRpcDTO> orgStoreBaseRpcDTOPagingVO = rmiOrgStoreRpcService.queryBaseStore(orgStoreBasePageRpcParam);
            List<ScpMinOrderSettingStoreSaveVO> storeSaveVOS = orgStoreBaseRpcDTOPagingVO.getRecords().stream().map(row -> {
                ScpMinOrderSettingStoreSaveVO scpMinOrderSettingStoreSaveVO = new ScpMinOrderSettingStoreSaveVO();
                scpMinOrderSettingStoreSaveVO.setStoreId(row.getId());
                scpMinOrderSettingStoreSaveVO.setStoreCode(row.getStoreCode());
                scpMinOrderSettingStoreSaveVO.setStoreName(row.getStoreName());
                scpMinOrderSettingStoreSaveVO.setStoreType2(row.getStoreType2());
                scpMinOrderSettingStoreSaveVO.setMasId(masId);
                scpMinOrderSettingStoreSaveVO.setMaxNum(saveVO.getMaxNum());
                return scpMinOrderSettingStoreSaveVO;
            }).collect(Collectors.toList());
            scpMinOrderSettingStoreDomainService.batchSaveMinOrderSettingStore(storeSaveVOS, 1000);
            refreshRedisValue(saveVO, masId, storeSaveVOS, itemRpcDTOMap);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<Long> ids) {
        scpMinOrderSettingDomainService.deleteByIds(ids);
        List<String> keys = ids.stream().map(row -> ScpConstant.MIN_ORDER_SETTING + row).collect(Collectors.toList());
        redisClient.delete(keys);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeMinOrderSettingStatus(ScpMinOrderSettingStatusParamVO paramVO) {
        if (Boolean.TRUE.equals(paramVO.getStatus())) {
            //校验门店是否存在其他活动中
            ScpMinOrderSettingStoreParamVO scpMinOrderSettingStoreParamVO = new ScpMinOrderSettingStoreParamVO();
            scpMinOrderSettingStoreParamVO.setMasIds(paramVO.getIds());
            List<ScpMinOrderSettingStoreRespVO> storeRespVOList = scpMinOrderSettingStoreDomainService.findByParam(scpMinOrderSettingStoreParamVO);
            if (CollUtil.isEmpty(storeRespVOList)) {
                throw new BusinessException("门店信息未维护");
            }
            List<ScpMinOrderSettingDO> settingDOS = scpMinOrderSettingRepo.findByIdIn(paramVO.getIds());
            for (ScpMinOrderSettingDO settingDO : settingDOS) {
                String key = ScpConstant.MIN_ORDER_SETTING + settingDO.getId();
                if (!LocalDateTime.now().isAfter(settingDO.getValidTo())) {
                    long between = ChronoUnit.MINUTES.between(LocalDateTime.now(), settingDO.getValidTo());
                    LocalDateTime validFrom = settingDO.getValidFrom();
                    String jsonString = JSON.toJSONString(validFrom);
                    redisClient.opsForValue().set(key, jsonString, between, TimeUnit.MINUTES);
                } else {
                    continue;
                }
                List<ScpMinOrderSettingItemRespVO> itemList = scpMinOrderSettingItemDomainService.findByMasId(settingDO.getId());
                //查询商品信息
                List<String> itemCodes = itemList.stream().map(ScpMinOrderSettingItemRespVO::getItemCode).distinct().collect(Collectors.toList());
                ItmItemRpcDtoParam itmItemRpcDtoParam = new ItmItemRpcDtoParam();
                itmItemRpcDtoParam.setItemCodes(itemCodes);
                List<ItmItemRpcDTO> itemRpcDtoList = rmiItemService.findItemRpcDtoByParam(itmItemRpcDtoParam);
                Map<String, ItmItemRpcDTO> itemRpcDTOMap = itemRpcDtoList.stream().collect(Collectors.toMap(ItmItemRpcDTO::getItemCode, Function.identity()));

                List<ScpMinOrderSettingRedisDO.ActivityDO.InnerItemDO> innerItemDOS = new ArrayList<>();
                for (ScpMinOrderSettingItemRespVO item : itemList) {
                    ScpMinOrderSettingRedisDO.ActivityDO.InnerItemDO innerItemDO = new ScpMinOrderSettingRedisDO.ActivityDO.InnerItemDO();
                    innerItemDO.setItemCode(item.getItemCode());
                    innerItemDO.setMasId(settingDO.getId());
                    innerItemDO.setMinNum(item.getMinNum());
                    innerItemDO.setItemType2(item.getItemType2());
                    if (itemRpcDTOMap.containsKey(item.getItemCode())) {
                        ItmItemRpcDTO itmItemRpcDTO = itemRpcDTOMap.get(item.getItemCode());
                        List<SpuImageDTO> spuAttchmentList = itmItemRpcDTO.getSpuImageList();
                        if (CollUtil.isNotEmpty(spuAttchmentList)) {
                            Optional<SpuImageDTO> first = spuAttchmentList.stream().filter(SpuImageDTO::getMajor).findFirst();
                            innerItemDO.setImgUrl(first.isEmpty() ? spuAttchmentList.get(0).getUrl() : first.get().getUrl());
                        }
                        innerItemDO.setItemId(itmItemRpcDTO.getId());
                        innerItemDO.setItemName(itmItemRpcDTO.getItemName());
                    }
                    innerItemDOS.add(innerItemDO);
                }
                //写入缓存
                Map<String, String> redisMap = new HashMap<>();
                for (ScpMinOrderSettingStoreRespVO store : storeRespVOList) {
                    ScpMinOrderSettingRedisDO scpMinOrderSettingRedisDO = new ScpMinOrderSettingRedisDO();
                    Object obj = redisClient.opsForValue().get(ScpConstant.MIN_ORDER_SETTING + store.getStoreCode());
                    if (obj == null) {
                        ScpMinOrderSettingRedisDO.ActivityDO activityDO1 = new ScpMinOrderSettingRedisDO.ActivityDO();
                        activityDO1.setActivityId(settingDO.getId());
                        activityDO1.setActivityCode(settingDO.getActivityCode());
                        activityDO1.setDocTypes(StrUtil.split(settingDO.getDocType(), ","));
                        activityDO1.setMaxNum(store.getMaxNum());
                        activityDO1.setActiveNum(0);
                        activityDO1.setInnerItemDOS(innerItemDOS);
                        activityDO1.setValidFrom(settingDO.getValidFrom());
                        activityDO1.setValidTo(settingDO.getValidTo());
                        scpMinOrderSettingRedisDO.setActivityList(List.of(activityDO1));
                        String storeKey = ScpConstant.MIN_ORDER_SETTING + store.getStoreCode();
                        String jsonString = JSON.toJSONString(scpMinOrderSettingRedisDO,
                                com.alibaba.fastjson.serializer.SerializerFeature.WriteDateUseDateFormat);
                        redisMap.put(storeKey, jsonString);
                    }
                }
                if (CollUtil.isNotEmpty(redisMap)) {
                    redisClient.opsForValue().multiSet(redisMap);
                }
            }
        } else {
            List<String> keys = paramVO.getIds().stream().map(row -> ScpConstant.MIN_ORDER_SETTING + row).collect(Collectors.toList());
            redisClient.delete(keys);
        }
        scpMinOrderSettingDomainService.enableMinOrderSetting(paramVO.getIds(), paramVO.getStatus());
    }

    @Override
    public List<ScpMinOrderSettingStoreRespVO> importStoreData(List<ScpMinOrderSettingImportStoreParamVO> list) {
        if (CollectionUtil.isEmpty(list)) {
            throw new BusinessException("导入数据为空");
        }
        for (int i = 0; i < list.size(); i++) {
            int row = i + 3;
            ScpMinOrderSettingImportStoreParamVO importData = list.get(i);
            if (StrUtil.isBlank(importData.getStoreCode())) {
                throw new BusinessException("第" + row + "行门店编码为空");
            }
        }

        // 获取所有门店信息
        List<String> storeCodes = list.stream()
                .map(ScpMinOrderSettingImportStoreParamVO::getStoreCode)
                .distinct()
                .toList();
        List<OrgStoreBaseRpcDTO> orgStoreBaseByParam = rmiOrgStoreRpcService.findOrgStoreBaseByParam(storeCodes);
        Map<String, OrgStoreBaseRpcDTO> storeBaseMap;
        if (CollUtil.isNotEmpty(orgStoreBaseByParam)) {
            storeBaseMap = orgStoreBaseByParam.stream().collect(Collectors.toMap(OrgStoreBaseRpcDTO::getStoreCode, Function.identity(), (key, value) -> value));
        } else {
            storeBaseMap = new HashMap<>();
        }

        Map<String, String> storeType2 = rmiSysUDCService.getCodeMap("yst-supp", "STORE_TYPE2");
        List<ScpMinOrderSettingStoreRespVO> res = new ArrayList<>();
        for (int i = 0; i < list.size(); i++) {
            int row = i + 3;
            ScpMinOrderSettingImportStoreParamVO l = list.get(i);

            //查询门店信息
            OrgStoreBaseRpcDTO orgStoreBaseRpcDTO = storeBaseMap.get(l.getStoreCode());
            if (orgStoreBaseRpcDTO == null) {
                throw new BusinessException("第" + row + "行门店不存在");
            }

            ScpMinOrderSettingStoreRespVO applyDResp = new ScpMinOrderSettingStoreRespVO();
            applyDResp.setStoreId(orgStoreBaseRpcDTO.getId());
            applyDResp.setStoreCode(orgStoreBaseRpcDTO.getStoreCode());
            applyDResp.setStoreName(orgStoreBaseRpcDTO.getStoreName());
            applyDResp.setRegion(orgStoreBaseRpcDTO.getRegion());
            applyDResp.setRegionName(orgStoreBaseRpcDTO.getRegionName());
            applyDResp.setStoreType2(orgStoreBaseRpcDTO.getStoreType2());
            String storeTypeName = storeType2.get(orgStoreBaseRpcDTO.getStoreType2());
            if (storeTypeName != null) {
                applyDResp.setStoreType2Name(storeTypeName);
            }
            res.add(applyDResp);
        }
        return res;
    }

    private void intersectSets(Set<Long> set1, List<Long> list2) {
        if (set1.isEmpty()) {
            set1.addAll(list2);
        }
        set1.retainAll(list2);
    }

}
